/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Copyright (C) 2002,2003  Thiemo Seufer <seufer@csv.ica.uni-stuttgart.de>
 * 
 * Much code is from isoinfo, original copyright info:
 *
 * Written by Eric Youngdale (1993).
 *
 *  Copyright 1993 Yggdrasil Computing, Incorporated
 */

#include <sys/types.h>
#include <sys/stat.h>

/* Hack! */
#define SIZE_T_DEFINED

#include "delo.h"
#include "stringops.h"

#define PATH_MAX 255

/*
 * The isofs filesystem constants/structures
 */

#define PAGE_BITS	11
#define PAGE		(1 << PAGE_BITS)
#define	OFFSET(TYPE, MEMBER)	((size_t) &((TYPE *)0)->MEMBER)

/* This part borrowed from the bsd386 isofs */
#define ISODCL(from, to) (to - from + 1)

struct iso_volume_descriptor {
	char type			[ISODCL(1,1)]; /* 711 */
	char id				[ISODCL(2,6)];
	char version			[ISODCL(7,7)];
	char data			[ISODCL(8,PAGE)];
};

/* volume descriptor types */
#define ISO_VD_PRIMARY		1
#define ISO_STANDARD_ID		"CD001"

struct iso_primary_descriptor {
	char type			[ISODCL (  1,   1)]; /* 711 */
	char id				[ISODCL (  2,   6)];
	char version			[ISODCL (  7,   7)]; /* 711 */
	char unused1			[ISODCL (  8,   8)];
	char system_id			[ISODCL (  9,  40)]; /* achars */
	char volume_id			[ISODCL ( 41,  72)]; /* dchars */
	char unused2			[ISODCL ( 73,  80)];
	char volume_space_size		[ISODCL ( 81,  88)]; /* 733 */
	char escape_sequences		[ISODCL ( 89, 120)];
	char volume_set_size		[ISODCL (121, 124)]; /* 723 */
	char volume_sequence_number	[ISODCL (125, 128)]; /* 723 */
	char logical_block_size		[ISODCL (129, 132)]; /* 723 */
	char path_table_size		[ISODCL (133, 140)]; /* 733 */
	char type_l_path_table		[ISODCL (141, 144)]; /* 731 */
	char opt_type_l_path_table	[ISODCL (145, 148)]; /* 731 */
	char type_m_path_table		[ISODCL (149, 152)]; /* 732 */
	char opt_type_m_path_table	[ISODCL (153, 156)]; /* 732 */
	char root_directory_record	[ISODCL (157, 190)]; /* 9.1 */
	char volume_set_id		[ISODCL (191, 318)]; /* dchars */
	char publisher_id		[ISODCL (319, 446)]; /* achars */
	char preparer_id		[ISODCL (447, 574)]; /* achars */
	char application_id		[ISODCL (575, 702)]; /* achars */
	char copyright_file_id		[ISODCL (703, 739)]; /* 7.5 dchars */
	char abstract_file_id		[ISODCL (740, 776)]; /* 7.5 dchars */
	char bibliographic_file_id	[ISODCL (777, 813)]; /* 7.5 dchars */
	char creation_date		[ISODCL (814, 830)]; /* 8.4.26.1 */
	char modification_date		[ISODCL (831, 847)]; /* 8.4.26.1 */
	char expiration_date		[ISODCL (848, 864)]; /* 8.4.26.1 */
	char effective_date		[ISODCL (865, 881)]; /* 8.4.26.1 */
	char file_structure_version	[ISODCL (882, 882)]; /* 711 */
	char unused4			[ISODCL (883, 883)];
	char application_data		[ISODCL (884, 1395)];
	char unused5			[ISODCL (1396, PAGE)];
} __attribute((packed));

/* We use this to help us look up the parent inode numbers. */

struct iso_path_table {
	unsigned char  name_len[2];	/* 721 */
	char extent[4];			/* 731 */
	char  parent[2];		/* 721 */
	char name[1];
} __attribute((packed));

/*
 * A ISO filename is: "abcde.eee;1" -> <filename> '.' <ext> ';' <version #>
 *
 * The maximum needed string length is:
 *	30 chars (filename + ext)
 * +	 2 chars ('.' + ';')
 * +	   strlen("32767")
 * +	   null byte
 * ================================
 * =	38 chars
 */
#define	LEN_ISONAME	31
#define	MAX_ISONAME	37

struct iso_directory_record {
	unsigned char length		[ISODCL (1, 1)];   /* 711 */
	char ext_attr_length		[ISODCL (2, 2)];   /* 711 */
	char extent			[ISODCL (3, 10)];  /* 733 */
	char size			[ISODCL (11, 18)]; /* 733 */
	char date			[ISODCL (19, 25)]; /* 7 by 711 */
	char flags			[ISODCL (26, 26)];
	char file_unit_size		[ISODCL (27, 27)]; /* 711 */
	char interleave			[ISODCL (28, 28)]; /* 711 */
	char volume_sequence_number	[ISODCL (29, 32)]; /* 723 */
	unsigned char name_len		[ISODCL (33, 33)]; /* 711 */
	char name			[MAX_ISONAME+1]; /* Not really, but we need something here */
} __attribute((packed));

/*
 * Iso directory flags.
 */
#define	ISO_FILE	0	/* Not really a flag...			*/
#define	ISO_EXISTENCE	1	/* Do not make existence known (hidden)	*/
#define	ISO_DIRECTORY	2	/* This file is a directory		*/
#define	ISO_ASSOCIATED	4	/* This file is an assiciated file	*/
#define	ISO_RECORD	8	/* Record format in extended attr. != 0	*/
#define	ISO_PROTECTION	16	/* No read/execute perm. in ext. attr.	*/
#define	ISO_DRESERVED1	32	/* Reserved bit 5			*/
#define	ISO_DRESERVED2	64	/* Reserved bit 6			*/
#define	ISO_MULTIEXTENT	128	/* Not final entry of a mult. ext. file	*/

struct iso_ext_attr_record {
	char owner			[ISODCL (1, 4)];     /* 723 */
	char group			[ISODCL (5, 8)];     /* 723 */
	char permissions		[ISODCL (9, 10)];    /* 16 bits */
	char creation_date		[ISODCL (11, 27)];   /* 8.4.26.1 */
	char modification_date		[ISODCL (28, 44)];   /* 8.4.26.1 */
	char expiration_date		[ISODCL (45, 61)];   /* 8.4.26.1 */
	char effective_date		[ISODCL (62, 78)];   /* 8.4.26.1 */
	char record_format		[ISODCL (79, 79)];   /* 711 */
	char record_attributes		[ISODCL (80, 80)];   /* 711 */
	char record_length		[ISODCL (81, 84)];   /* 723 */
	char system_id			[ISODCL (85, 116)];  /* achars */
	char system_use			[ISODCL (117, 180)];
	char ext_attr_version		[ISODCL (181, 181)]; /* 711 */
	char esc_seq_len		[ISODCL (182, 182)]; /* 711 */
	char reserved			[ISODCL (183, 246)]; /* for future use */
	char appl_use_len		[ISODCL (247, 250)]; /* 723 */
	char appl_use[1];		/* really more */
/*	char esc_seq[];			escape sequences recorded after appl_use */
} __attribute((packed));

/*
 * Iso extended attribute permissions.
 */
#define	ISO_GS_READ		0x0001	/* System Group Read */
#define	ISO_BIT_1		0x0002
#define	ISO_GS_EXEC		0x0004	/* System Group Execute */
#define	ISO_BIT_3		0x0008

#define	ISO_O_READ		0x0010	/* Owner Read */
#define	ISO_BIT_5		0x0020
#define	ISO_O_EXEC		0x0040	/* Owner Exexute */
#define	ISO_BIT_7		0x0080

#define	ISO_G_READ		0x0100	/* Group Read */
#define	ISO_BIT_9		0x0200
#define	ISO_G_EXEC		0x0400	/* Group Execute */
#define	ISO_BIT_11		0x0800

#define	ISO_W_READ		0x1000	/* World (other) Read */
#define	ISO_BIT_13		0x2000
#define	ISO_W_EXEC		0x4000	/* World (other) Execute */
#define	ISO_BIT_15		0x8000

#define	ISO_MB_ONE		(ISO_BIT_1|ISO_BIT_3|ISO_BIT_5|ISO_BIT_7| \
				 ISO_BIT_9|ISO_BIT_11|ISO_BIT_13|ISO_BIT_15)

/*
 * Extended Attributes record according to Yellow Book.
 */
struct iso_xa_dir_record {
	char group_id			[ISODCL (1, 2)];
	char user_id			[ISODCL (3, 4)];
	char attributes			[ISODCL (5, 6)];
	char signature			[ISODCL (7, 8)];
	char file_number		[ISODCL (9, 9)];
	char reserved			[ISODCL (10, 14)];
} __attribute((packed));

static char xtract[PAGE];
static struct stat fstat_buf;
static char name_buf[256];
static char xname[PAGE];
static unsigned char date_buf[9];
static unsigned char buffer[PAGE];

static struct todo {
	struct todo *next;
	char *name;
	unsigned int extent;
	unsigned int length;
} todo_root;
static struct todo *todo_last;

static int isonum_733(void *val);
static int parse_rr(unsigned char *pnt, int len);
static void find_rr(struct iso_directory_record *idr, unsigned char **pntp, int *lenp);
static int dump_rr(struct iso_directory_record *idr);
static void parse_dir(struct todo *td);
int readisofile(char *file, unsigned int *extent, unsigned int *len);

static inline int isonum_733(void *val)
{
	unsigned char *p = (unsigned char *) val;

	return ((p[0] & 0xff)
		| ((p[1] & 0xff) << 8)
		| ((p[2] & 0xff) << 16)
		| ((p[3] & 0xff) << 24));
}

static int parse_rr(unsigned char *pnt, int len)
{
	int slen;
	int xlen;
	int extent;
	int cont_extent, cont_offset, cont_size;
	int flag1, flag2;
	unsigned char *pnts;
	char symlinkname[PAGE];

	symlinkname[0] = 0;
	cont_extent = cont_offset = cont_size = 0;

	flag1 = flag2 = 0;
	while(len >= 4) {
		if(pnt[3] != 1 && pnt[3] != 2) {
			printf("**BAD RRVERSION (%d)\n", pnt[3]);
			return 0;	/* JS ??? Is this right ??? */
		}
		if(strncmp((char *)pnt, "RR", 2) == 0)
			flag1 = pnt[4] & 0xff;
		else if(strncmp((char *)pnt, "PX", 2) == 0) {	/* POSIX attributes */
			flag2 |= 1;
			fstat_buf.st_mode = isonum_733(pnt+4);
			fstat_buf.st_nlink = isonum_733(pnt+12);
			fstat_buf.st_uid = isonum_733(pnt+20);
			fstat_buf.st_gid = isonum_733(pnt+28);
		} else if(strncmp((char *)pnt, "PN", 2) == 0) {	/* POSIX device number */
			flag2 |= 2;
		} else if(strncmp((char *)pnt, "SL", 2) == 0) {	/* Symlink */
		        int cflag;

			flag2 |= 4;
			cflag = pnt[4];
			pnts = pnt+5;
			slen = pnt[2] - 5;
			while (slen >= 1) {
				switch (pnts[0] & 0xfe) {
				case 0:
					strncat(symlinkname, (char *)(pnts+2), pnts[1]);
					symlinkname[pnts[1]] = 0;
					break;
				case 2:
					strcat (symlinkname, ".");
					break;
				case 4:
					strcat (symlinkname, "..");
					break;
				case 8:
					strcat (symlinkname, "/");
					break;
				case 16:
					strcat(symlinkname,"/mnt");
					printf("Warning - mount point requested");
					break;
				case 32:
					strcat(symlinkname,"kafka");
					printf("Warning - host_name requested");
					break;
				default:
					printf("Reserved bit setting in symlink");
					break;
				}
				if((pnts[0] & 0xfe) && pnts[1] != 0) {
					printf("Incorrect length in symlink component");
				}
				if(xname[0] == 0)
					strcpy(xname, "-> ");
				strcat(xname, symlinkname);
				symlinkname[0] = 0;
				xlen = strlen(xname);
				if((pnts[0] & 1) == 0 && xname[xlen-1] != '/')
					strcat(xname, "/");

				slen -= (pnts[1] + 2);
				pnts += (pnts[1] + 2);
			}
		} else if(strncmp((char *)pnt, "NM", 2) == 0) {	/* Alternate Name */
			flag2 |= 8;
			strncpy(name_buf, (char *)(pnt+5), pnt[2] - 5);
			name_buf[pnt[2] - 5] = 0;
		} else if(strncmp((char *)pnt, "CL", 2) == 0) {	/* Child link */
			flag2 |= 16;
			extent = isonum_733(pnt+4);
		} else if(strncmp((char *)pnt, "PL", 2) == 0) {	/* Parent link */
			flag2 |= 32;
			extent = isonum_733(pnt+4);
		} else if(strncmp((char *)pnt, "RE", 2) == 0) {	/* Relocated Directory */
			flag2 |= 64;
		} else if(strncmp((char *)pnt, "TF", 2) == 0) {	/* Time stamp */
			flag2 |= 128;
		} else if(strncmp((char *)pnt, "CE", 2) == 0) {	/* Continuation Area */
			cont_extent = isonum_733(pnt+4);
			cont_offset = isonum_733(pnt+12);
			cont_size = isonum_733(pnt+20);
		}

		len -= pnt[2];
		pnt += pnt[2];
		if(len <= 3 && cont_extent) {
			unsigned char sector[PAGE];
			bootread(cont_extent * 4, sector, sizeof(sector));
			flag2 |= parse_rr(&sector[cont_offset], cont_size);
		}
	}
	/*
	 * for symbolic links, strip out the last '/'
	 */
	if (xname[0] != 0 && xname[strlen(xname)-1] == '/')
		xname[strlen(xname)-1] = '\0';

	return flag2;
}

static void find_rr(struct iso_directory_record *idr, unsigned char **pntp, int *lenp)
{
	struct iso_xa_dir_record *xadp;
	int len;
	unsigned char * pnt;

	len = idr->length[0] & 0xff;
	len -= OFFSET(struct iso_directory_record, name[0]);
	len -= idr->name_len[0];

	pnt = (unsigned char *) idr;
	pnt += OFFSET(struct iso_directory_record, name[0]);
	pnt += idr->name_len[0];
	if((idr->name_len[0] & 1) == 0){
		pnt++;
		len--;
	}
	if (len >= 14) {
		xadp = (struct iso_xa_dir_record *)pnt;

		if (xadp->signature[0] == 'X' && xadp->signature[1] == 'A'
				&& xadp->reserved[0] == '\0') {
			len -= 14;
			pnt += 14;
		}
	}
	*pntp = pnt;
	*lenp = len;
}

static int dump_rr(struct iso_directory_record *idr)
{
	int len;
	unsigned char * pnt;

	find_rr(idr, &pnt, &len);
	return parse_rr(pnt, len);
}

static void parse_dir(struct todo *td)
{
	unsigned int i;
	struct iso_directory_record *idr;
	int extent = td->extent;
	int len = td->length;

	while(len > 0) {
		bootread(extent * 4, buffer, sizeof(buffer));
		len -= sizeof(buffer);
		extent++;
		i = 0;
		while(1) {
			idr = (struct iso_directory_record *) &buffer[i];
			if(idr->length[0] == 0) break;
			memset(&fstat_buf, 0, sizeof(fstat_buf));
			name_buf[0] = xname[0] = 0;
			fstat_buf.st_size = (off_t)isonum_733(idr->size);
			if( idr->flags[0] & 2)
				fstat_buf.st_mode |= S_IFDIR;
			else
				fstat_buf.st_mode |= S_IFREG;	
			if(idr->name_len[0] == 1 && idr->name[0] == 0)
				strcpy(name_buf, ".");
			else if(idr->name_len[0] == 1 && idr->name[0] == 1)
				strcpy(name_buf, "..");
			else {
				strncpy(name_buf, idr->name, idr->name_len[0]);
				name_buf[idr->name_len[0]] = 0;
			}
			memcpy(date_buf, idr->date, 9);
			dump_rr(idr);
			if((idr->flags[0] & 2) != 0
			   && (idr->name_len[0] != 1
			       || (idr->name[0] != 0 && idr->name[0] != 1)))
			{
				/*
				 * Add this directory to the todo list.
				 */
				todo_last->next = (struct todo *)malloc(sizeof(struct todo));
				todo_last = todo_last->next;

				todo_last->next = NULL;
				todo_last->extent = isonum_733(idr->extent);
				todo_last->length = fstat_buf.st_size;
				todo_last->name = (char *) malloc(strlen(td->name)
								  + strlen(name_buf) + 2);
				strcpy(todo_last->name, td->name);
				strcat(todo_last->name, name_buf);
				strcat(todo_last->name, "/");
			} else if (xtract[0]) {
				char testname[PATH_MAX + 1];

				strcpy(testname, td->name);
				strcat(testname, name_buf);
				dprintf("Dir entry %s\n", testname);
				if(strcmp(xtract, testname) == 0) {
					if (xname[0]) {
						strcpy(xtract, "/");
						strcat(xtract, xname + 3);
						td = &todo_root;
dprintf("Symlink, new name %s\n", xtract);
						return;
					}
					if (extentlist_add_tail(isonum_733(idr->extent) * 4,
								(fstat_buf.st_size + SECTOR_SIZE - 1) / SECTOR_SIZE)) {
						extentlist_clear();
						return;
					}

					break;
				}
			}

			i += buffer[i];
			if (i > PAGE - OFFSET(struct iso_directory_record, name[0]))
				break;
		}
	}
}

int readisoblocks(char *file)
{
	struct todo *td;
	struct iso_primary_descriptor ipd;
	struct iso_directory_record *idr;

	strcpy(xtract, file);
	extentlist_clear();
	
	if (bootinit(NULL))
		return 1;

	if (bootread(16 * 4, &ipd, sizeof(ipd)) != sizeof(ipd))
		return 1;

	/*
	 * ISO 9660:
	 *
	 *	DESC TYPE	== 1 (VD_PVD)	offset 0	len 1
	 *	STR ID		== "CD001"	offset 1	len 5
	 *	STD_VER		== 1		offset 6	len 1
	 */
	if ((ipd.type[0] != ISO_VD_PRIMARY)
	    || (strncmp(ipd.id, ISO_STANDARD_ID, sizeof(ipd.id)) != 0)
	    || (ipd.version[0] != 1)) {
		/* CD-ROM is not in ISO 9660 format */
		return 1;
	}

	idr = (struct iso_directory_record *)ipd.root_directory_record;
	todo_root.next = NULL;
	todo_root.name = "/";
	todo_root.extent = isonum_733(idr->extent);
	todo_root.length = isonum_733(idr->size);
	todo_last = &todo_root;
	td = &todo_root;
	while(td) {
		parse_dir(td);
		td = td->next;
	}

#ifdef DEBUG
	extentlist_dump();
#endif
	if (!extentlist_size())
		return 1;

	return 0;
}
