# Written by John Hoffman
# Modified by Cameron Dale
# see LICENSE.txt for license information
#
# $Id: piecebuffer.py 266 2007-08-18 02:06:35Z camrdale-guest $

"""Store buffers of piece data.

@type logger: C{logging.Logger}
@var logger: the logger to send all log messages to for this module
@type _pool: L{BufferPool}
@var _pool: example buffer pool
@type PieceBuffer: L{SingleBuffer}
@var PieceBuffer: example buffer

"""

from array import array
from threading import Lock
import logging
# import inspect
    
logger = logging.getLogger('DebTorrent.piecebuffer')

class SingleBuffer:
    """A single piece buffer.
    
    @type pool: L{BufferPool}
    @ivar pool: the pool of all piece buffers
    @type buf: C{array.array}
    @ivar buf: the character buffer used to store the piece
    @type length: C{int}
    @ivar length: the amount of data in the piece buffer
    
    """
    
    def __init__(self, pool, count):
        """Initialize the instance.
        
        @type pool: L{BufferPool}
        @param pool: the pool of all piece buffers
        @type count: L{BufferPool}
        @param count: the pool of all piece buffers
        
        """
        
        self.pool = pool
        self.buf = array('c')
        self.count = count
        logger.debug('new/pooled buffer index: '+str(count))

    def init(self):
        """Initialize the new piece buffer."""
        self.length = 0

    def append(self, s):
        """Add new data to the end of the buffer.
        
        @type s: C{string}
        @param s: the new data to add
        
        """
        
        l = self.length+len(s)
        self.buf[self.length:l] = array('c',s)
        self.length = l

    def __len__(self):
        """Get the length of the buffer.
        
        @rtype: C{int}
        @return: the length of the data in the buffer
        
        """
        
        return self.length

    def __getslice__(self, a, b):
        """Get a slice of data from the buffer.
        
        @type a: C{int}
        @param a: the starting offset to get data from
        @type b: C{int}
        @param b: the ending offset to stop getting data at, can be negative 
            to count from the end of the buffer
        @rtype: C{array.array}
        @return: the requested data from the buffer
        
        """
        
        if b > self.length:
            b = self.length
        if b < 0:
            b += self.length
        if a == 0 and b == self.length and len(self.buf) == b:
            return self.buf  # optimization
        return self.buf[a:b]

    def getarray(self):
        """Get al the data from the buffer.
        
        @rtype: C{array.array}
        @return: the requested data from the buffer
        
        """
        
        return self.buf[:self.length]

    def release(self):
        """Release this buffer to the pool for reuse."""
        self.pool.release(self)


class BufferPool:
    """The pool of all created piece buffers.
    
    @type pool: C{list} of L{SingleBuffer}
    @ivar pool: the pooled buffers available for use
    @type lock: C{threading.Lock}
    @ivar lock: serializes access to the pool across different threads
    @type count: C{int}
    @ivar count: the number of created buffers (for debugging)
    
    """
    
    def __init__(self):
        """Initialize the instance."""
        self.pool = []
        self.lock = Lock()
        self.count = 0

    def new(self):
        """Get a SingleBuffer from the pool, or create a new one.
        
        @rtype: L{SingleBuffer}
        @return: a usable piece buffer
        
        """
        
        self.lock.acquire()
        if self.pool:
            x = self.pool.pop()
        else:
            self.count += 1
            x = SingleBuffer(self, self.count)
        x.init()
        self.lock.release()
        return x

    def release(self, x):
        """Add an existing buffer to the pool of available ones.
        
        @type x: L{SingleBuffer}
        @param x: the buffer to add to the pool
        
        """
        
        self.pool.append(x)


_pool = BufferPool()
PieceBuffer = _pool.new
