#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <db.h>
#include <glib.h>

#include "user_manage.h"
#include "status.h"
#include "bdb.h"

G_LOCK_DEFINE(hub_user_list);		/* the lock is required if hub_user_list or hub_user_xtra_info is modified */
											/* (not the content of data pointed but the array) */
											/* the main reason of this lock is due to the fact set_xtra_information_* can */
											/* be called from threads */

GPtrArray *hub_user_list=NULL;		/* list of all users connected to the hub */
GPtrArray *hub_op_list=NULL;		/* list of all op connected to the hub (op are included in user list) */

GPtrArray *hub_user_xtra_info=NULL;	/* array of extra data stored for each hub user */
												/* the nth pointer of this array is for the nth user of the hub_user_list */

GPtrArray *hub_user_uinfo=NULL;		/* it is the result of /UINFO command for each user */

/********************************/
/* free 1 user xtra information */
/********************************/
static void free_xtra_info(XTRA_INFO *xi)
{
	if(xi==NULL)
		return;

	if(xi->cnx_addr!=NULL)
		free(xi->cnx_addr);

	free(xi);
}

static void dump_users_info(char *nick)
{
	int i;

	printf("-----------------------------------------------dump\n");
	G_LOCK(hub_user_list);
	for(i=0;i<hub_user_list->len;i++)
	{
		XTRA_INFO *xi;
		char *ppp;

		ppp=g_ptr_array_index(hub_user_list,i);
		if(strcmp(ppp,nick))
			continue;

		printf("%.40s -> ",ppp);

		xi=g_ptr_array_index(hub_user_xtra_info,i);
		if(xi==NULL)
			printf("\n");
		else
		{
			switch(xi->cnx_type)
			{
				case UNDEFINED:	printf("UNDEFINED\n");
											break;
				case PASSIVE:		printf("PASSIVE\n");
											break;
				case ACTIVE:		printf("ACTIVE: %s\n",xi->cnx_addr);
											break;
			}
		}
	}
	G_UNLOCK(hub_user_list);
}

/*******************************************************/
/* check if the given username is inside the user_list */
/*******************************************************/
/* output: 0=no, 1=yes */
/***********************/
int user_in_list(GPtrArray *user_list, char *username)
{
	int i;

	G_LOCK(hub_user_list);
	if((user_list==NULL)||(user_list->len==0)||(username==NULL))
	{
		G_UNLOCK(hub_user_list);
		return 0;
	}

	for(i=0;i<user_list->len;i++)
	{
		if(!strcmp(username,g_ptr_array_index(user_list,i)))
		{
			G_UNLOCK(hub_user_list);
			return 1;
		}
	}

	G_UNLOCK(hub_user_list);
	return 0;
}

/*********************************/
/* empty the hub_user_list array */
/*********************************/
void reset_hub_user_list(void)
{
	int i;
	char *p;

	G_LOCK(hub_user_list);
	/* reset hub user list */
	if(hub_user_list!=NULL)
	{
		for(i=0;i<hub_user_list->len;i++)
		{
			p=g_ptr_array_index(hub_user_list,i);
			if(p)
				free(p);
		}
		g_ptr_array_free(hub_user_list,TRUE);
	}
	hub_user_list=g_ptr_array_new();

	/* reset hub user uinfo */
	if(hub_user_uinfo!=NULL)
	{
		for(i=0;i<hub_user_uinfo->len;i++)
		{
			p=g_ptr_array_index(hub_user_uinfo,i);
			if(p)
				free(p);
		}
		g_ptr_array_free(hub_user_uinfo,TRUE);
	}
	hub_user_uinfo=g_ptr_array_new();

	/* reset hub op list */
	if(hub_op_list!=NULL)
	{
		for(i=0;i<hub_op_list->len;i++)
		{
			p=g_ptr_array_index(hub_op_list,i);
			if(p)
				free(p);
		}
		g_ptr_array_free(hub_op_list,TRUE);
	}
	hub_op_list=g_ptr_array_new();

	/* reset hub user xtra info */
	if(hub_user_xtra_info!=NULL)
	{
		for(i=0;i<hub_user_xtra_info->len;i++)
		{
			free_xtra_info(g_ptr_array_index(hub_user_xtra_info,i));
		}
		g_ptr_array_free(hub_user_xtra_info,TRUE);
	}
	hub_user_xtra_info=g_ptr_array_new();

	G_UNLOCK(hub_user_list);
}

/*******************************************/
/* add the given nick to the hub user list */
/*******************************************/
void add_user_to_user_list(char *nick)
{
	int i;
	char *p;

	if(hub_user_list==NULL)
		reset_hub_user_list();		/* if the list is not yet available, reset it */

	/* don't lock before because reset_* also locks */
	G_LOCK(hub_user_list);

	/* search the given nick in the user list */
	for(i=0;i<hub_user_list->len;i++)	
	{
		if(!strcmp(nick,g_ptr_array_index(hub_user_list,i)))
		{
			G_UNLOCK(hub_user_list);
			return;			/* we have found it */
		}
	}

	/* not found, add it */
	p=strdup(nick);
	if(p!=NULL)
	{
		g_ptr_array_add(hub_user_list,p);
		g_ptr_array_add(hub_user_uinfo,NULL);			/* add the UINFO pointer at the same time */
		g_ptr_array_add(hub_user_xtra_info,NULL);		/* add the xtra_info pointer at the same time */
	}
	G_UNLOCK(hub_user_list);
	SET_GSTATUS_USERS(hub_user_list->len);
}

/*****************************************/
/* add the given nick to the hub op list */
/*****************************************/
void add_user_to_op_list(char *nick)
{
	int i;
	char *p;

	if(hub_op_list==NULL)
		reset_hub_user_list();		/* if the list is not yet available, reset it */

	/* search the given nick in the user list */
	for(i=0;i<hub_op_list->len;i++)	
	{
		if(!strcmp(nick,g_ptr_array_index(hub_op_list,i)))
			return;			/* we have found it */
	}

	/* not found, add it */
	p=strdup(nick);
	if(p!=NULL)
		g_ptr_array_add(hub_op_list,p);
}

/**********************************************/
/* remove the given nick to the hub user list */
/**********************************************/
void remove_user_to_user_list(char *nick)
{
	int i;

	G_LOCK(hub_user_list);

	if(hub_user_list==NULL)
	{
		G_UNLOCK(hub_user_list);
		return;
	}

	/* search the given nick in the user list */
	for(i=0;i<hub_user_list->len;i++)	
	{
		char *p;

		p=g_ptr_array_index(hub_user_list,i);
		if(!strcmp(nick,p))
		{
			free(p);
			g_ptr_array_remove_index_fast(hub_user_list,i);
			p=g_ptr_array_remove_index_fast(hub_user_uinfo,i);
			if(p!=NULL)
				free(p);

			free_xtra_info(g_ptr_array_remove_index_fast(hub_user_xtra_info,i));		/* free xtra information at the same time */
			break;
		}
	}

	/* the use is perhaps also an op */
	for(i=0;i<hub_op_list->len;i++)	
	{
		char *p;

		p=g_ptr_array_index(hub_op_list,i);
		if(!strcmp(nick,p))
		{
			free(p);
			g_ptr_array_remove_index_fast(hub_op_list,i);
			break;
		}
	}
	G_UNLOCK(hub_user_list);
	SET_GSTATUS_USERS(hub_user_list->len);
}

/******************************/
/* set user extra information */
/******************************/
void set_xtra_information_cnx(char *nick, UCNX type, char *addr)
{
	int i;

	G_LOCK(hub_user_list);
	/* first we must find the index of the given user */
	for(i=0;i<hub_user_list->len;i++)	
	{
		char *p;

		p=g_ptr_array_index(hub_user_list,i);
		if(!strcmp(nick,p))
		{
			/* gotcha */
			XTRA_INFO *xi;

			xi=g_ptr_array_index(hub_user_xtra_info,i);
			if(xi==NULL)
			{
				/* not yet set */
				xi=malloc(sizeof(XTRA_INFO));
				if(xi!=NULL)
				{
					xi->cnx_type=type;
					if(type!=ACTIVE)
						xi->cnx_addr=NULL;
					else
						xi->cnx_addr=strdup(addr);
	
					g_ptr_array_index(hub_user_xtra_info,i)=xi;
				}
			}
			else
			{
				/* still set, do update */
				xi->cnx_type=type;
				if(addr!=NULL)
				{
					if(xi->cnx_addr!=NULL)
					{
						if(strcmp(addr,xi->cnx_addr))
						{
							free(xi->cnx_addr);
							xi->cnx_addr=strdup(addr);
						}
					}
					else
						xi->cnx_addr=strdup(addr);
				}
				else
				{
					if(xi->cnx_addr!=NULL)
					{
						free(xi->cnx_addr);
						xi->cnx_addr=NULL;
					}
				}
			}
			break;
		}
	}
	G_UNLOCK(hub_user_list);
#if 0
	dump_users_info();
#endif
}

/******************************/
/* get user extra information */
/*************************************************************/
/* output: *type is set to UNDEFINED, PASSIVE or ACTIVE      */
/*         the returned g_string (if not NULL) must be freed */
/*         when useless.                                     */
/*************************************************************/
GString *get_xtra_information_cnx(char *nick, UCNX *type)
{
	int i;
	GString *result=NULL;
	*type=UNDEFINED;

	G_LOCK(hub_user_list);
	/* first we must find the index of the given user */
	for(i=0;i<hub_user_list->len;i++)	
	{
		char *p;

		p=g_ptr_array_index(hub_user_list,i);
		if(!strcmp(nick,p))
		{
			/* gotcha */
			XTRA_INFO *xi;

			xi=g_ptr_array_index(hub_user_xtra_info,i);
			if(xi!=NULL)
			{
				*type=xi->cnx_type;
				if(xi->cnx_addr!=NULL)
				{
					result=g_string_new(xi->cnx_addr);
				}
			}
			break;
		}
	}
	G_UNLOCK(hub_user_list);
	return result;
}

/******************/
/* set user UINFO */
/******************/
void set_cached_user_uinfo(char *nick, char *uinfo)
{
	int i;

	G_LOCK(hub_user_list);
	/* first we must find the index of the given user */
	for(i=0;i<hub_user_list->len;i++)	
	{
		char *p;

		p=g_ptr_array_index(hub_user_list,i);
		if(!strcmp(nick,p))
		{
			/* gotcha */
			char *ui;

			ui=g_ptr_array_index(hub_user_uinfo,i);
			if(ui!=NULL)
			{
				free(ui);
			}

			g_ptr_array_index(hub_user_uinfo,i)=strdup(uinfo);
			break;
		}
	}
	G_UNLOCK(hub_user_list);
}

/******************/
/* get user UINFO */
/*************************************************************/
/* output: the returned g_string (if not NULL) must be freed */
/*         when useless.                                     */
/*************************************************************/
GString *get_cached_user_uinfo(char *nick)
{
	int i;
	GString *result=NULL;

	G_LOCK(hub_user_list);
	/* first we must find the index of the given user */
	for(i=0;i<hub_user_list->len;i++)	
	{
		char *p;

		p=g_ptr_array_index(hub_user_list,i);
		if(!strcmp(nick,p))
		{
			/* gotcha */
			char *ui;

			ui=g_ptr_array_index(hub_user_uinfo,i);
			if(ui!=NULL)
			{
				result=g_string_new(ui);
			}
			break;
		}
	}
	G_UNLOCK(hub_user_list);
	return result;
}

/* ------------------------------------------------------------------------------- */
/* ------------------------------------------------------------------------------- */
/* ------------------------------------------------------------------------------- */
/* the following calls handle queries about user in the berkeleyDB 'unwanted_user' */
/* ------------------------------------------------------------------------------- */
/* ------------------------------------------------------------------------------- */
/* ------------------------------------------------------------------------------- */


/**************************************************/
/* check if the given user has the following flag */
/**************************************************/
/* output: 1= yes, the user has this flag */
/*         0= no                          */
/******************************************/
int user_has_flag(char *nickname,const char *flag)
{
	char *data_val=NULL;
	int data_len;
	int ret;
	char *pos;
	int flag_len;

	if(get_key_data(unwanted_user,nickname,strlen(nickname),(void*)&data_val,&data_len)!=0)
	{
		/* user not found -> end */
		return 0;
	}

	if(data_val==NULL)
	{
		/* user has no flag -> end */
		return 0;
	}

	ret=0;
	flag_len=strlen(flag);
	pos=data_val;

	while(flag_len<data_len)
	{
		if(!strncmp(flag,pos,flag_len))
		{	/* the beginning of flag match */
			if(pos[flag_len]=='|')		/* and the current flag immediatly ends with a pipe */
			{
				ret=1;
				break;
			}
		}
		pos++;
		data_len--;
	}

	/* check the last flag if not yet found, last chance */
	if((ret==0)&&(flag_len==data_len))
	{
		if(!strncmp(flag,pos,flag_len))
			ret=1;
	}

	if(data_val!=NULL)
		free(data_val);
	return ret;
}

