
/*
 * Copyright (c) 2000 David Stes.
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Library General Public License as published 
 * by the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: frame.m,v 1.12 2001/03/14 19:17:56 stes Exp $
 */

#include <stdio.h>
#include <stdlib.h>
#include <limits.h>
#include <assert.h>
#include <sys/types.h>
#include <signal.h>
#include <curses.h>
#include <menu.h>
#include <form.h>

#include <Object.h>
#include <ordcltn.h>
#include <ocstring.h>
#include <point.h>
#include <rectangl.h>

#include "slk.h"
#include "frame.h"
#include "cursel.h"
#include "menufram.h"
#include "cmdmenu.h"
#include "process.h"
#include "coroute.h"
#include "var.h"

id framelist;
id curframe;
id activeframe;

id openframe(char *name,int what,id args)
{
  id w;
  int x;
  char *q;

  [Var undefargs];
  [Var defargs:args];

  w = nil;
  x = strtol(name,&q,10);
  if (q != name && 0 < x && x <= [framelist size]) {
    w = [framelist at:x-1]; 
  } else {
    if (name != NULL) w=findframe(name);
  }

  if (w) {
    curframe=[w renumber];
    [w activate];
    [w display];
    [w home];
    return w; 
  }

  switch (what) {
    case C_NONE : w = openfile(name);break;
    case C_MENU : w = openmenu(name);break;
    case C_FORM : w = openform(name);break;
    case C_TEXT : w = opentext(name);break;
    default : assert(0);break;
  }

  if (w) {
    [w setargs:args];
    if (![w mustopenframe]) return nil;
    if ([w create]) {
      [framelist addIfAbsent:w];
      curframe=[w renumber];
      [w activate];
      [w display];
      [w home];
      return w; 
    }
  } else {
    id s = [String sprintf:"Could not access %s",name];
    showtransientmessage([s str]);
    return nil;
  }
}

@implementation Frame 

sigset_t alrmblocked,alrmunblocked;

+ initialize
{
  static int beenhere = 0;
  if (!beenhere) {
   beenhere++;
   /* ALARM/USR2 is blocked unless waiting for key input */
   /* when longjmp out of alarmhandler, we have to unblock those signals */
   /* we may 'miss' some signal by blocking (this is a minor bug) */
   sigemptyset(&alrmblocked);
   sigaddset(&alrmblocked,SIGALRM);
   sigaddset(&alrmblocked,SIGUSR2);
   [self blockalarm];
  }
  return self;
}

+ (int)blockalarm
{
   return sigprocmask(SIG_BLOCK,&alrmblocked,&alrmunblocked);
}

+ (int)unblockalarm
{
   return sigprocmask(SIG_SETMASK,&alrmunblocked,NULL);
}
   
- setfilename:v
{
  filename = v;
  return self;
}

- filename
{
  return filename;
}

- setargs:v
{
  args = v;return self;
}

- (STR)name
{
  return "Frame";
}

- (int)num
{
   return num;
}

- setnum:(int)i
{
  num=i;
  return self;
}

- renumber
{
  int i,n;
  for(i=0,n=[framelist size];i<n;i++) {
    [[framelist at:i] setnum:i+1];
  }
  return self;
}

- setform:f
{
  formtitle = expandstr(f);
  return self;
}

- setmenu:m
{
  if (m == nil) return self;
  menu = expandstr(m);
  return self;
}

- setrows:r
{
  if (r == nil) return self;
  rows = expandstr(r);
  return self;
}

- setcolumns:r
{
  if (r == nil) return self;
  columns = expandstr(r);
  return self;
}

- setscrlabkeys:x
{
  scrlabkeys = x;
  return self;
}

- setaltslks:x
{
   altslks = x;alternate = expandbool(x,NO); return self;
}

- altslkstoggle
{
  alternate = !alternate; return self;
}

- setmultiselect:m
{
  multiselect = m;
  return self;
}

- setlifetime:m
{
  lifetime = m;
  return self;
}

- framemsg
{
  return (framemsg) ? expandstr(framemsg) : nil;
}

- setframemsg:m
{
  framemsg = m;
  return self;
}

- setcyclic:m
{
  cyclic = expandbool(m,YES);return self;
}

- setinterrupt:x
{
  interrupt = x;return self;
}

- setoninterrupt:x
{
  oninterrupt = x;return self;
}

- setsiginthandler
{
  if (interrupt) {
    u_interrupt=expandbool(interrupt,NO);
  } else {
    u_interrupt=expandbool(g_interrupt,NO);
  }
  if (oninterrupt) {
    u_oninterrupt=oninterrupt;
  } else {
    u_oninterrupt=g_oninterrupt;
  }
  return self;
}

- setreread:m
{
  reread = m;return self;
}

- (BOOL)needsreread
{
  return expandbool(reread,NO);
}

- (BOOL)mustopenframe
{
  return expandbool(init,YES);
}

- setinit:m
{
  init = m;return self;
}

- setdone:m
{
  done = m;return self;
}

- setclose:m
{
  close = m;return self;
}

- setbegrow:m
{
  if (m == nil) return self;
  begrow = expandstr(m);
  return self;
}

- setbegcol:m
{
  if (m == nil) return self;
  begcol = expandstr(m);return self;
}

static int calcoverlap(int y1,int h1,int y2,int h2)
{
  int t;
  if (y1 < y2) {
    if (y1 + h1 < y2) return 0;
    if (y1 + h1 > y2 + h2) return h2;
    t = y1+h1-y2;
    assert(t>=0 && t<=h2);
    return t; 
  } else {
    if (y2 + h2 < y1) return 0;
    if (y2 + h2 > y1 + h1) return h1;
    t = y2+h2-y1;
    assert(t>=0 && t<=h1);
    return t; 
  }
}

static unsigned long calcweight(int y1,int x1,int h1,int w1)
{
  int d;
  int i,n;
  int v,h;
  unsigned long weight = 0;
  for(i=0,n=[framelist size];i<n;i++) {
    id f = [framelist at:i]; 
    v = calcoverlap([f top],[f height],y1,h1); 
    assert(v == calcoverlap(y1,h1,[f top],[f height])); 
    h = calcoverlap([f left],[f width],x1,w1); 
    assert(h == calcoverlap(x1,w1,[f left],[f width])); 
    weight += v*h;
#ifdef NOT_STRICT_FMLI
    /* avoid overlapping title */
    d = y1 - [f top];if (d==0||d==1) weight++;
    d = x1 - [f left];if (d==0||d==1) weight++;
#endif
  }
  return weight;
}

static void findminweight(int *yref,int *xref,int h,int w)
{
  int x,y;
  unsigned long min = ULONG_MAX;

  for(y=1;y<=DISPLAYH-h;y++) {
    for(x=0;x<=DISPLAYW-w;x++) {
      unsigned long weight = calcweight(y,x,h,w);
      if (weight < min) { min=weight;*yref=y;*xref=x; }
    }
  }
}

- (int)fixedbegrow
{
  int y;
  char *q;
  if (begrow) {
    char *s = [begrow str]; 
    y = strtol(s,&q,10);
    if (y > 0 && y < DISPLAYH) return y;
  }
  return 0;
}

- (int)fixedbegcol
{
  int y;
  char *q;
  if (begcol) {
    char *s = [begcol str]; 
    y = strtol(s,&q,10);
    if (y > 0 && y < DISPLAYW) return y;
  }
  return 0;
}

- begframerect:(int)width:(int)height
{
  id f;
  char *q;
  int w,h,x,y;

  x=0;
  y=1;

  if (begrow==nil && begcol == nil) {
     findminweight(&y,&x,height,width);
  }

  if (begrow) {
    char *s = [begrow str]; 
    if (!strcasecmp(s,"center")) {
      y = (DISPLAYH-height) / 2;
    } else {
      y = strtol(s,&q,10);
      if (y <= 0 || q == s || y >= DISPLAYH) {
         findminweight(&y,&x,height,width);
      }
    }
  }

  if (begcol) {
    char *s = [begcol str]; 
    if (!strcasecmp(s,"center")) {
      x = (DISPLAYW-width) / 2;
    } else {
      x = strtol(s,&q,10);
      if (x < 0 || q == s || x >= DISPLAYW ) {
        findminweight(&y,&x,height,width);
      }
    }
  }

  dbg("begframerect y: %i x: %i h: %i w: %i\n",y,x,height,width);
  dbg("DISPLAYW: %i DISPLAYH: %i\n",DISPLAYW,DISPLAYH);

  if (x < 0 || y < 0 || x+width > DISPLAYW || y+height > DISPLAYH) {
    return nil;
  } else {
    return [Rectangle origin:x:y corner:x+width:y+height];
  }
}

- frametoolarge
{
  char *m = "Frame can not be displayed, it may be too large for the screen";
  showtransientmessage(m);
  return nil;
}

- setframerect:aRect
{
  framerect = aRect;
  boundrect = [Rectangle origin:0:0 corner:[aRect width]:[aRect height]];
  return self;
} 

- (int)top
{
  assert(framerect); return [framerect top];
}

- (int)left
{
  assert(framerect); return [framerect left];
}

- (int)height
{
  assert(framerect); return [framerect height];
}

- (int)width
{
  assert(framerect); return [framerect width];
}

- sethelp:m
{
  help = m;return self;
}

- activate
{
  [activeframe deactivate];
  [Var undefargs];
  [Var defargs:args];
  [self setslk];
  activeframe=self;isactive=YES;
  [self needsdisplay];
  return self;
}

- closeshortterm
{
  id lt = expandstr(lifetime);
  if (ischoicemenu) [self setchoice];
  if (lt && !strcmp([lt str],"shortterm")) [self close];
  return self;
}

- deactivate
{
  isactive=NO;
  [self needsdisplay];
  return self;
}

- nextframe
{
  id f,g = curframe;
  f = [framelist after:g];
  if (!f) f = [framelist firstElement];
  curframe = [f activate];
  refresh();
  return self;
}

- prevframe
{
  id f,g = curframe;
  f = [framelist before:curframe];
  if (!f) f = [framelist lastElement];
  curframe = [f activate];
  refresh();
  return self;
}

- destroy
{
  framerect = nil;
  if (eti_win) {
    werase(eti_win);
    wrefresh(eti_win);
    delwin(eti_win);
  }
  eti_win = NULL;
  return self;
}

- create
{
  return [self subclassResponsibility:_cmd];
}

- displayifneeded
{
  if (needdisplay) { [self draw]; wnoutrefresh(eti_win); }
  assert(needdisplay==NO);
  return self;
}

- draw
{
  return [self subclassResponsibility:_cmd];
}

- refresh
{
  /* in response to refresh cmd of cmdmenu */
  return [self needsdisplay];
}

- update
{
  id w;
  char *s;
  /* in response to update cmd of cmdmenu */
  s = [filename str];
  w = openfile(s); 
  if (!w) {
    showtransientmessage("Frame not updated: Frame definition file missing");
    return self;
  }
  if ([w mustopenframe]) {
    [self unpost];
    [framelist remove:self];
    [self destroy];
    [framelist add:curframe=[w create]];
    [w renumber];
    [w needsdisplay];
  }
  [curframe home];
  [curframe activate];
  return self;
}

- display
{
  [self needsdisplay];
  displayall();
  return self;
}

- drawbox:(char*)s
{
  int dots = 0;

  if (dontdrawbox) {
    dontdrawbox = 0;
  } else {
    int i,n,x,w,h;

    w = [self width];
    h = [self height];

#if HAVE_COLOR
    if (has_colors()) {
      wcolor_set(eti_win,colorpairs[(isactive)?CP_CURFRAME:CP_FRAME],NULL);
    }
#endif

    box(eti_win,0,0);
    wstandout(eti_win);
    wattron(eti_win,A_BOLD);

#if HAVE_COLOR
    if (has_colors()) {
      wcolor_set(eti_win,colorpairs[(isactive)?CP_CURTITLE:CP_TITLE],NULL);
    }
#endif

    for(x=1;x<w-1;x++) mvwaddch(eti_win,0,x,' ');
    if (num>0) mvwprintw(eti_win,0,2,"%i",num);
    n = strlen(s);
    if (n <= w-5) {
      x = (w - n - 5)/2;
      assert(0 <= x);
      assert(x+n+4 <= w-1);
    } else {
      x = 0;n = w-9;dots++;
    }
    for(i=0;i<n;i++) mvwaddch(eti_win,0,x+4+i,s[i]); 
    if (dots) for(i=0;i<3;i++) mvwaddch(eti_win,0,x+4+n+i,'.');
    wattroff(eti_win,A_BOLD);
    wstandend(eti_win);
 }

 return self;
}

- drawscrollbar:(int)y1:(int)y2:(int)h
{
  int x,y;

  assert(0<=y1 && y1<=y2 && y2<=h);

  if (h<=3) return self;

  x=[self width]-1;
  y=[self height]/2-1;

#if HAVE_COLOR
  if (has_colors()) {
    wcolor_set(eti_win,colorpairs[(isactive)?CP_CURFRAME:CP_FRAME],NULL);
  }
#endif

  wattron(eti_win,A_REVERSE);
  wattron(eti_win,A_BOLD);
  mvwaddch(eti_win,y,x,(y1==0)?' ':'^');
  mvwaddch(eti_win,y+1,x,' ');
  mvwaddch(eti_win,y+2,x,(y2<h)?'v':' ');
  wattroff(eti_win,A_REVERSE);
  wattroff(eti_win,A_BOLD);

  return self;
}

- setslk
{
  int i,n;
  if (maxbutton(g_slk) > maxfunkey || maxbutton(scrlabkeys) > maxfunkey) {
    slktab[maxfunkey] = k_chg_keys;
    slktab[2*maxfunkey] = k_chg_keys;
  }
  for(i=0,n=[g_slk size];i<n;i++) {
    id slk = [g_slk at:i];
    slktab[[slk button]] = slk;
  } 
  for(i=0,n=[scrlabkeys size];i<n;i++) {
    id slk = [scrlabkeys at:i];
    slktab[[slk button]] = slk;
  } 
  drawslk(alternate);
  return self;
}

- enter
{
  return self;
}

- spacebar
{
  return self;
}

static void clearcommandline(void)
{
  int i;
  for(i=0;i<DISPLAYW;i++) mvwaddch(stdscr,DISPLAYH+1,i,' ');
  refresh();
}

#define CMDBUFSIZE 512

- controlj
{
  int len;
  char buf[CMDBUFSIZE+1];
  if (isnobang()) { beep(); return self; } 

  mvwaddstr(stdscr,DISPLAYH+1,0,"-->");
  
  echo();
  len = (DISPLAYW < CMDBUFSIZE)?DISPLAYW:CMDBUFSIZE; 
 
#ifdef NCURSES_VERSION
  mvwgetnstr(stdscr,DISPLAYH+1,4,buf,len);
#else
  mvwgetstr(stdscr,DISPLAYH+1,4,buf);
#endif

  noecho();
  clearcommandline();

  evalcmd([String str:buf]);
  return self;
}

- controlt
{
  beep();
  return self;
}

- controlb
{
  beep();
  return self;
}

- controly
{
  beep();
  return self;
}

- controlh
{
  beep();
  return self;
}

- end
{
  beep();
  return self;
}

- home
{
  beep();
  return self;
}

- clearEol
{
  beep();
  return self;
}

- homedown
{
  beep();
  return self;
}

- mark
{
  beep();
  return self;
}

- scrolldown
{
  beep();
  return self;
}

- scrollup
{
  beep();
  return self;
}

- reset
{
  beep();
  return self;
}

- controlf
{
  int c = wgetch(eti_win);

  if ('1' <= c && c <= '9') {
    dobutton(alternate,c - '0');return self;
  }

  switch (c) {
    case 'b' : [self home];return self;
    case 'c' : [self controlj];return self;
    case 'd' : [self scrolldown];return self;
    case 'e' : [self homedown];return self;
    case 'm' : [self mark];return self;
    case 'y' : [self clearEol];return self;
    case 'u' : [self scrollup];return self;
    case 'r' : [self reset];return self;
    default : beep();
  }

  return self;
}

- controlx
{
  beep();
  return self;
}

- controlk
{
  beep();
  return self;
}

- controld
{
  beep();
  return self;
}

- controle
{
  beep();
  return self;
}

- controla
{
  beep();
  return self;
}

- controlo
{
  beep();
  return self;
}

- controll
{
  beep();
  return self;
}

- controln
{
  beep();
  return self;
}

- controlw
{
  beep();
  return self;
}

- controlv
{
  beep();
  return self;
}

- controlp
{
  beep();
  return self;
}

- controlr
{
  beep();
  return self;
}

- controli
{
  beep();
  return self;
}

- controlu
{
  beep();
  return self;
}

- arrowup
{
  beep();
  return self;
}

- arrowdown
{
  beep();
  return self;
}

- pageup
{
  beep();
  return self;
}

- pagedown
{
  beep();
  return self;
}

- arrowleft
{
  beep();return self;
}

- arrowright
{
  beep();return self;
}

- keydown:(int)c
{
  /* clear message line if necessary */
  transientmsg=NULL;

  if (KEY_F(1) <= c && c <= KEY_F(MAXFUNKEYS)) {
    dobutton(alternate,c - KEY_F(0));return self;
  }
 
  switch(c) {
   case '\r' : [self enter];break;
   case '\n' : [self controlj];break;
   case ' ' : [self spacebar];break;
   case 127 :
   case KEY_BACKSPACE : [self controlh];break; 
   case KEY_HOME : [self home];break;
   case KEY_LL : [self end];break;
   case KEY_LEFT : [self arrowleft];break;
   case KEY_RIGHT : [self arrowright];break;
   case KEY_UP : [self arrowup];break;
   case KEY_DOWN : [self arrowdown];break;
   case KEY_NPAGE : [self pageup];break;
   case KEY_PPAGE : [self pagedown];break;
   case 20 : [self controlt];break;
   case 2  : [self controlb];break;
   case 25 : [self controly];break;
   case 6  : [self controlf];break;
   case 24 : [self controlx];break;
   case 11 : [self controlk];break;
   case 4  : [self controld];break;
   case 5  : [self controle];break;
   case 1  : [self controla];break;
   case 15 : [self controlo];break;
   case 12 : [self controll];break;
   case 14 : [self controln];break;
   case 23 : [self controlw];break;
   case 22 : [self controlv];break;
   case 16 : [self controlp];break;
   case 18 : [self controlr];break;
   case '\t' : [self controli];break;
   case 21 : [self controlu];break;

   default : beep();
  }

  return self;
}

- (int)getkey
{
   int c;
   int ok;

   signal(SIGALRM,alarmhandler);
   signal(SIGUSR2,alarmhandler);

   ok = [isa unblockalarm];
   /* tell any vsig children this is a good time to signal us */ 
   startvsig();
   c = wgetch(eti_subwin); 
   stopvsig();
   ok = [isa blockalarm];
   /* clear message line if necessary */
   transientmsg=NULL;
   return c;
}

- getevent
{
  return [self subclassResponsibility:_cmd];
}

- chgkeys
{
  [self altslkstoggle];[self setslk];refresh();return self;
}

- unpost
{
  return self;
}

- close
{
  if (self == curframe) {
    id f = [framelist before:self];
    if (!f) { showtransientmessage("Can't close this frame");return self; }
    curframe = [f activate];
  } else {
    [curframe activate];
  }
  if (close) {
    id c;
    [self setsiginthandler];
    c = expandstr(close);
    if (c) evalcmd(c);
  }
  [self unpost];
  [[framelist remove:self] renumber];
  [self destroy];
  redrawall();
  return self;
}

- cancel
{
  return [self close];
}

- help
{
  id s = expandstr(help);
  if (s && [s size]) evalcmd(s);
  return self;
}

- save
{
  beep();
  return self;
}

- choices
{
  beep();
  return self;
}

- setchoice
{
  beep();
  return self;
}

- needsdisplay
{
  needdisplay++;
  return self;
}

- showcmdmenu
{
  runframe(cmdmenu);return self;
}
  
- intermove
{
  int c;
  chtype corn[4];
  int y = [self top];
  int x = [self left];
  int w = [self width] - 1;
  int h = [self height] - 1;

  showtransientmessage("Position top-left corner and press ENTER");

  do {
    corn[0] = mvinch(y,x);
    corn[1] = mvinch(y,x+w);
    corn[2] = mvinch(y+h,x);
    corn[3] = mvinch(y+h,x+w);

    attron(A_REVERSE);
    mvaddch(y,x,ACS_ULCORNER);
    mvaddch(y,x+w,ACS_URCORNER);
    mvaddch(y+h,x,ACS_LLCORNER);
    attroff(A_REVERSE);

    mvaddch(y+h,x+w,ACS_LRCORNER);
    move(y+h,x+w);

    refresh();
    c = wgetch(stdscr);

    mvaddch(y,x,corn[0]);
    mvaddch(y,x+w,corn[1]);
    mvaddch(y+h,x,corn[2]);
    mvaddch(y+h,x+w,corn[3]);

    switch(c) {
      case '\n' :
      case '\r' :
      case KEY_ENTER : break;
      case KEY_UP : y--;break;
      case KEY_DOWN : y++;break;
      case KEY_LEFT : x--;break;
      case KEY_RIGHT : x++;break;
      default : beep();break;
    }

    if (y<1) {y=1;beep();}
    if (x<0) {x=0;beep();}
    if (y+h>DISPLAYH-1) {y=DISPLAYH-1-h;beep();}
    if (x+w>DISPLAYW-1) {x=DISPLAYW-1-w;beep();}
  } while (c != KEY_ENTER && c != '\n' && c != '\r');

  return [Rectangle origin:x:y extent:w+1:h+1];
}

- move
{
  id r = [self intermove];
  [self destroy];
  [self setframerect:r];
  [self activate];
  transientmsg=NULL;
  redrawall();
  return self;
}

- interreshape
{
  int c;
  chtype corn[4];
  int y = [self top];
  int x = [self left];
  int w = [self width] - 1;
  int h = [self height] - 1;

  showtransientmessage("Position bottom-right corner and press ENTER");

  do {
    corn[0] = mvinch(y,x);
    corn[1] = mvinch(y,x+w);
    corn[2] = mvinch(y+h,x);
    corn[3] = mvinch(y+h,x+w);

    attron(A_REVERSE);
    mvaddch(y,x,ACS_ULCORNER);
    mvaddch(y,x+w,ACS_URCORNER);
    mvaddch(y+h,x,ACS_LLCORNER);
    attroff(A_REVERSE);

    mvaddch(y+h,x+w,ACS_LRCORNER);
    move(y+h,x+w);

    refresh();
    c = wgetch(stdscr);

    mvaddch(y,x,corn[0]);
    mvaddch(y,x+w,corn[1]);
    mvaddch(y+h,x,corn[2]);
    mvaddch(y+h,x+w,corn[3]);

    switch(c) {
      case '\n' :
      case '\r' :
      case KEY_ENTER : break;
      case KEY_UP : h--;break;
      case KEY_DOWN : h++;break;
      case KEY_LEFT : w--;break;
      case KEY_RIGHT : w++;break;
      default : beep();break;
    }

    if (h<0) {h=0;beep();}
    if (w<0) {w=0;beep();}
    if (y+h>DISPLAYH-1) {h=DISPLAYH-1-y;beep();}
    if (x+w>DISPLAYW-1) {w=DISPLAYW-1-x;beep();}
  } while (c != KEY_ENTER && c != '\n' && c != '\r');

  return [Rectangle origin:x:y extent:w+1:h+1];
}

- reshape
{
  id r;
  framerect = [self intermove];
  r = [self interreshape];
  if ([r height] <= 2 || [r width] <= 3) {
    showtransientmessage("Too small, try again");
  } else {
    [self destroy];
    [self setframerect:r];
    [self activate];
    transientmsg=NULL;
    redrawall();
  }
  return self;
}

- writeOn:process delim:(char*)d
{
  return self;
}

@end
 
