# curves.rb, copyright (c) 2006 by Vincent Fourmond: 
# The class describing a curve to be plotted.
  
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
  
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details (in the COPYING file).

require 'CTioga/elements'
require 'CTioga/utils'
require 'CTioga/boundaries'
require 'CTioga/curve_style'

module CTioga

  Version::register_svn_info('$Revision: 879 $', '$Date: 2009-02-23 23:58:20 +0100 (Mon, 23 Feb 2009) $')

  # The class Curve2D stores both the data and the way it
  # should be plotted, such as it's legend, it's color, it's line
  # style and so on...
  class Curve2D  < TiogaElement

    # for Dvectors:
    include Dobjects

    # The CurveStyle object representing the curve's style.
    attr_reader :style

    # The underlying Function object:
    attr_reader :function

    def need_style?
      return true
    end

    def set_style(style)
      @style = style.dup
    end

    def initialize(style = nil, data = nil)
      # style is a CurveStyle object
      set_style(style) if style
      set_data(data) if data
      @line_cap = nil
    end

    def has_legend?
      if @style.legend
        return true
      else
        return false
      end
    end

    # Sets the data for the curve
    def set_data(ar)
      @function = ar
    end

    # This function returns the index of a point
    # from the curve according to the given _spec_, a string.
    # * if _spec_ is a number >= 1, it represents the index of the point
    #   in the Function object.
    # * if _spec_ is a number <1, it represents the relative position of the
    #   point in the object (it is then multiplied by the size to get
    #   the actual index).
    # * if _spec_ is of the form 'x,y', the closest point belonging to the
    #   function is taken. Not implemented yet.
    def parse_position(spec)
      if spec =~ /(.+),(.+)/
        raise "The (x,y) point position is not implemented yet"
      else
        val = Float(spec)
        if val < 1 
          index = (@function.size * val).round
        else
          index = val.round
        end
        index
      end
    end

    # Returns the tangent of the curve at the given point, that is a vector
    # parallel to it. _dir_ specifies if it is a left tangent or a right
    # tangent or an average of both. Nothing else than the latter is
    # implemented for now.
    def tangent(index, dir = :both)
      before = @function.point(index - 1)
      point = @function.point(index)
      after = @function.point(index + 1)
      raise "Point invalid" unless point
      tangent = Dvector[0,0]
      if index > 0
        tangent += (point - before)
      end
      if index < (@function.size - 1)
        tangent += (after - point)
      end
      return tangent
    end

    # This function returns the bouding box of the specified graphes
    # No margin adjustment is done here, as it can lead to very uneven
    # graphs
    def get_boundaries
      top = @function.y.max
      bottom = @function.y.min
      left = @function.x.min
      right = @function.x.max

      width = (left == right) ? 1 : right - left
      height = (top == bottom) ? 1 : top - bottom

      return [left,right,top,bottom]
    end

    # Computes the outmost boundaries of the given boundaries.
    # Any NaN in here will happily get ignored.
    def Curve2D.compute_boundaries(bounds)
      left = Dvector.new
      right = Dvector.new
      top = Dvector.new
      bottom = Dvector.new
      bounds.each do |a|
        left.push(a[0])
        right.push(a[1])
        top.push(a[2])
        bottom.push(a[3])
      end
      return [left.min, right.max, top.max, bottom.min]
    end

    # Creates a path for the given curve. This should be defined
    # with care, as it will be used for instance for region coloring
    # and stroking. The function should only append to the current
    # path, not attempt to create a new path or empty what was done
    # before.
    def make_path(t)
      bnds = Utils::Boundaries.new(parent.effective_bounds)
      if @style.interpolate
        for f in @function.split_monotonic
          new_f = f.bound_values(*bnds.real_bounds)
          t.append_interpolant_to_path(f.make_interpolant)
        end
      else
        f = @function.bound_values(*bnds.real_bounds)
        t.append_points_to_path(f.x, f.y)
      end
    end

    # Draw the path
    def draw_path(t)
      t.line_width = @style.linewidth if @style.linewidth
      if @style.color && @style.line_style
        t.line_type = @style.line_style
        t.stroke_transparency = @style.transparency || 0
        t.stroke_color = @style.color
        if @line_cap
          t.line_cap = @line_cap
        end
        make_path(t)
        t.stroke
      end
    end

    def draw_markers(t)
      xs = @function.x
      ys = @function.y

      if @style.marker
        t.line_type = [[], 0]   # Always solid for striking markers
        t.stroke_transparency = @style.marker_transparency || 0
        t.fill_transparency = @style.marker_transparency || 0
        t.show_marker('Xs' => xs, 'Ys' => ys,
                      'marker' => @style.marker, 
                      'scale' => @style.marker_scale, 
                      'color' => @style.marker_color)
      end
    end

    # Returns a y value suitable for fills/histograms or other kinds of
    # stuff based on a specification:
    # * false/nil: returns nil
    # * to_y_axis: y = 0
    # * to_bottom:  the bottom of the plot
    # * to_top: the top of the plot
    # * "y = ....":  the given value.
    def y_value(spec)
      case spec
      when false, nil, :old_style
        return false
      when Float                # If that is already a Float, fine !
        return spec
      when :to_y_axis
        return 0.0
      when :to_bottom
        return parent.effective_bounds[3] # bottom
      when :to_top
        return parent.effective_bounds[2] # top
      when /y\s*=\s*(.*)/
        return Float($1)
      else
        warn "Y value #{spec} not understood"
        return false                  # We don't have anything to do, then.
      end
    end

    # A function to close the path created by make_path.
    # Overridden in the histogram code.
    def close_path(t, y)
      t.append_point_to_path(@function.x.last, y)
      t.append_point_to_path(@function.x.first, y)
      t.close_path
    end

    # Draws the filled region according to the :fill_type element
    # of the style pseudo-hash. It can be:
    def draw_fill(t)
      y = y_value(@style.fill_type)
      return unless y

      t.fill_transparency = @style.fill_transparency || 0
      # Now is the tricky part. To do the actual fill, we first make a
      # path according to the make_path function.
      make_path(t)

      # Then we add two line segments that go from the end to the
      # beginning.
      close_path(t, y)

      # Now the path is ready. Just strike -- or, rather, fill !
      t.fill_color = @style.fill_color
      t.fill
    end

    def plot(t = nil)
      debug "Plotting curve #{inspect}"
      t.context do

        # The fill is always first
        draw_fill(t)

        for op in CurveStyle::DrawingOrder[@style[:drawing_order]]
          self.send("draw_#{op}".to_sym, t)
        end
      end
    end

    # The function that plots error bars.
    def draw_error_bars(t = nil)
      # We first check that we actually need to do anything 
      t.context do 
        t.stroke_transparency = @style.error_bars_transparency || 0
        if @function.errors.key?(:xmin) or @function.errors.key?(:ymin)
          error_bar = {}                 # Just create it once, anyway
          error_bar['color'] = @style.error_bar_color
          errors = @function.errors # So we won't have to worry
          # some data should be shared
          @function.errors[:x].each_index do |i|
            error_bar['x'] = errors[:x][i]
            if errors.key?(:xmin) && 
                ((errors[:xmax][i] - errors[:xmin][i]) != 0)
              error_bar['dx_plus'] = errors[:xmax][i] - errors[:x][i]
              error_bar['dx_minus'] = errors[:x][i] - errors[:xmin][i]
              error_bar.delete('dx')
            else
              %w(dx_plus dx_minus).each do |el|
                error_bar.delete(el)
              end
              error_bar['dx'] = 0
            end
            error_bar['y'] = errors[:y][i]
            if errors.key?(:ymin) && 
                ((errors[:ymax][i] - errors[:ymin][i]) != 0)
              error_bar['dy_plus'] = errors[:ymax][i] - errors[:y][i]
              error_bar['dy_minus'] = errors[:y][i] - errors[:ymin][i]
              error_bar.delete('dy')
            else
              %w(dy_plus dy_minus).each do |el|
                error_bar.delete(el)
              end
              error_bar['dy'] = 0
            end
            if (error_bar['dx'] != 0 || error_bar['dy'] != 0) 
              t.show_error_bars(error_bar)
            end
          end
        end
      end
    end

    # plot is the 'real_do' method.
    alias :do :plot    
  end

  # The basic class to create histograms.
  class Histogram2D < Curve2D

    def initialize(*args)
      super
      @line_cap = LINE_CAP_BUTT
    end

    # Creates a path for the given curve. This should be defined
    # with care, as it will be used for instance for region coloring
    # and stroking. The function should only append to the current
    # path, not attempt to create a new path or empty what was done
    # before.
    def make_path(t)
      # vectors to store the resulting path
      x_res = Dvector.new
      y_res = Dvector.new
      x_first = 2 * @function.x[0] - @function.x[1]
      y_first = 2 * @function.y[0] - @function.y[1]
      n = @function.size - 1
      x_last = 2 * @function.x[n] - @function.x[n-1]
      y_last = 2 * @function.y[n] - @function.y[n-1]
      
      t.make_steps('xs' => @function.x, 'ys' => @function.y,
                   'dest_xs' => x_res, 'dest_ys' => y_res,
                   'x_first' => x_first, 'y_first' => y_first,
                   'x_last' =>  x_last,  'y_last' => y_last)
      y_base = y_value(@style.hist_type)
      if y_base
        x_prev = y_prev = nil
        # We remove outer elements, not needed.
        x_res.shift
        y_res.shift
        x_res.pop
        y_res.pop
        for x,y in Function.new(x_res, y_res)
          if x_prev
            # We create a path according to the specs
            x_left = (x_prev + (x - x_prev) * @style.hist_left)
            x_right = (x_prev + (x - x_prev) * @style.hist_right)
            x_p = Dvector[x_left, x_right, x_right]
            y_p = Dvector[y,y,y_base]
            t.move_to_point(x_left,y_base)
            t.append_points_to_path(x_p,y_p)
            y_prev = x_prev = nil
          else
            # Accumulate to get a full step. 
            x_prev = x
            y_prev = y
          end
        end
      else
        t.append_points_to_path(x_res, y_res)
      end
    end

    # In the case of histograms with a defined level, we ignore the
    # fill_type setting for the y value, we use the same level.
    def close_path(t,y)
      if y_value(@style.hist_type)
        t.close_path
      else
        super
      end
    end

  end
end
