# boundaries.rb: Small class for manipulating boundaries 
# Copyright (c) 2008 by Vincent Fourmond: 
  
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
  
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details (in the COPYING file).

require 'CTioga/utils'

# A small add-on to the Array class, to make it easy to convert to
# frame specifications.
class Array

  def to_frame(spec = "%s")
    i = 0
    h = {}
    for side in %w(left right top bottom)
      h[ spec % side] = self[i]
      i += 1
    end
    return h
  end

end


module CTioga
  
  Version::register_svn_info('$Revision: 825 $', '$Date: 2008-07-23 14:36:18 +0200 (Wed, 23 Jul 2008) $')
  
  module Utils


    # Converting a boundary hash to an array
    def self.frame_to_array(hash, format = '%s')
      return %w(left right top bottom).map {|x| sprintf(format,x) }.map do |f|
        hash[f]
      end
    end

    # Compose two margins (in the form of arrays): you get the m2 expressed
    # in the same frame as m1, but taken relative to m1.
    def self.compose_margins(m1, m2)
      width = 1 - m1[0] - m1[1]
      height = 1 - m1[2] - m1[3]
      
      return [ m1[0] + m2[0] * width,
               m1[1] + m2[1] * width,
               m1[2] + m2[2] * height,
               m1[3] + m2[3] * height
             ]
    end


    # A small class to handle boundaries
    class Boundaries
      
      attr_accessor :left, :right, :top, :bottom
      
      def initialize(*args)
        args.flatten!

        @left = args[0]
        @right = args[1]
        @top = args[2]
        @bottom = args[3]
      end

      def xmin
        if @left <= @right
          return @left
        else
          return @right
        end
      end

      def xmax
        if @left <= @right
          return @right
        else
          return @left
        end
      end

      def ymin
        if @top <= @bottom
          return @top
        else
          return @bottom
        end
      end

      def ymax
        if @top <= @bottom
          return @bottom
        else
          return @top
        end
      end

      # Returns [xmin, xmax, ymin, ymax]. Useful, for instance,
      # for Function#bound_values
      def real_bounds
        return [xmin, xmax, ymin, ymax]
      end

      def to_a
        return [@left, @right, @top, @bottom]
      end

      # Returns a 'left' => ..., 'right' => .... hash containg the
      # boundaries.
      def to_hash
        return {
          'left' => @left,
          'right' => @right, 
          'top' => @top,
          'bottom' => @bottom
        }
      end

      # Returns true if the given X coordinate is within the bounds
      def x_inside?(x)
        return (x >= xmin && x <= xmax)
      end

      # Returns true if the given Y coordinate is within the bounds
      def y_inside?(y)
        return (y >= ymin && y <= ymax)
      end

      # The position of the given X coordinate with respect to the
      # boundaries. Returns either :inside, :left or :right
      def where_x?(x)
        if x_inside?(x)
          return :inside
        elsif @left <= @right
          if x < @left
            return :left
          else
            return :right
          end
        else                    # Right-to-left order
          if x < @right
            return :right
          else
            return :left
          end
        end
      end

      # The position of the given Y coordinate with respect to the
      # boundaries. Returns either :inside, :top or :bottom
      def where_y?(y)
        if y_inside?(y)
          return :inside
        elsif @top <= @bottom
          if y < @top
            return :top
          else
            return :bottom
          end
        else
          if y < @top
            return :bottom
          else
            return :top
          end
        end
      end
      
    end

    # A function that transforms an inset/legend specification into
    # margins. Specifications understood are:
    # * x,y:w(xh) : box centered on x,y of size w x h (or w x w if
    #   h is omitted)
    # * x1,y1;x2,y2 : the exact box 
    def self.inset_margins(spec)
      case spec
      when /(.*),(.*):([^x]*)(?:x(.*))?/
        x = $1.to_f; y = $2.to_f; w = $3.to_f
        h = ($4 || $3).to_f
        margins = [x - w/2, 1 - (x + w/2), 1 - (y+h/2), y - h/2]
      when /(.*),(.*);(.*),(.*)/
        x1 = $1.to_f; y1 = $2.to_f; 
        x2 = $3.to_f; y2 = $4.to_f; 
        left = [x1, x2].min
        right = [x1, x2].max
        top = [y1, y2].max
        bottom = [y1, y2].min
        margins = [left, 1 - right, 1 - top, bottom]
      when /(.*)x(.*)([+-])(.*)([+-])(.*)/ # X geometry-like specification
        w = $1.to_f; h = $2.to_f
        if $3 == '+'            # Left
          left = $4.to_f
          right = left + w
        else                    # Right
          right = $4.to_f
          left = right - w
        end
        if $5 == '+'            # Top
          top = $6.to_f
          bottom = top - h
        else                    # Bottom
          bottom = $6.to_f
          top = bottom + h
        end
        margins = [left, 1 - right, 1 - top, bottom]
      else
        raise "Incorrect inset specification #{spec}"
      end
      return margins
    end

  end
  
end
