/* mkdatafile -- create a Crimson Fields data file
   Copyright (C) 2000, 2001 Jens Granseuer

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* takes five file names: icons BMP, small font BMP, large font BMP, and output filename

   The data file is written in little-endian format.

   Creates a data file for Crimson Fields containing fonts and icons image data
*/

#include <stdio.h>
#include <stdlib.h>
#include "SDL.h"
#include "SDL_endian.h"

static char *PutDump( char *dest, const char *from, short nn );
static char *PutRun( char *dest, short nn, char cc );
static long ByteRun1_pack( const char *src, char *destbuf, unsigned long srcsize );

#define FONT_CAPS_ONLY	0x01

struct FontInfo {
  unsigned char w;
  unsigned char h;
  unsigned char chars;
  unsigned char flags;
};

struct PicHeader {
  unsigned short colors;
  unsigned short width;
  unsigned short height;
  unsigned long packed_size;
};


int main( int argc, char *argv[] ) {
  SDL_Surface *bmp;
  int status, i, j;
  char *packbuf, *upackbuf;
  SDL_RWops *out;

  if ( argc != 5 ) {
    fprintf(stderr, "Invalid number of arguments\n"
                    "Usage: %s <icons.bmp> <smallfont.bmp> <largefont.bmp> <outfile>\n",
                    argv[0] );
    exit(-1);
  }


  if ( SDL_Init(0) < 0 ) {
    fprintf(stderr, "Couldn't init SDL: %s\n", SDL_GetError());
    exit(-1);
  }
  atexit(SDL_Quit);

  out = SDL_RWFromFile( argv[4], "wb" );
  if ( !out ) {
    fprintf(stderr, "Couldn't open output file %s\n", argv[5] );
    exit(-1);
  }

  status = 0;

  /* icons */
  bmp = SDL_LoadBMP(argv[1]);
  if ( bmp ) {
    if ( bmp->format->BitsPerPixel == 8 ) {
      SDL_Palette *palette = bmp->format->palette;
      struct PicHeader pict = { palette->ncolors, bmp->w, bmp->h, 0 };

      packbuf = (char *)malloc( 2 * bmp->w * bmp->h );
      upackbuf = &packbuf[ bmp->w * bmp->h ];

      /* copy surface lines to buffer */
      for ( i = 0; i < bmp->h; i++ ) {
        char *pix = (Uint8 *)bmp->pixels + i * bmp->pitch;
        for ( j = 0; j < bmp->w; j++ ) {
          upackbuf[j + i * bmp->w] = *pix++;
        }
      }

      pict.packed_size = ByteRun1_pack( upackbuf, packbuf, bmp->w * bmp->h );

      SDL_WriteLE16( out, pict.colors );
      SDL_WriteLE16( out, pict.width );
      SDL_WriteLE16( out, pict.height );
      SDL_WriteLE32( out, pict.packed_size );

      for ( i=0; i<palette->ncolors; ++i ) {
        SDL_RWwrite( out, &palette->colors[i].r, 1, 1 );
        SDL_RWwrite( out, &palette->colors[i].g, 1, 1 );
        SDL_RWwrite( out, &palette->colors[i].b, 1, 1 );
      }

      SDL_RWwrite( out, packbuf, 1, pict.packed_size );
      free( packbuf );
    } else fprintf(stderr, "Couldn't find palette\n" );
    SDL_FreeSurface( bmp );
  } else {
    fprintf(stderr, "Couldn't open BMP %s\n", argv[2] );
    exit(-1);
  }

  /* small font */
  bmp = SDL_LoadBMP(argv[2]);
  if ( bmp ) {
    if ( bmp->format->BitsPerPixel == 8 ) {
      SDL_Palette *palette = bmp->format->palette;
      struct PicHeader pict = { palette->ncolors, bmp->w, bmp->h, 0 };
      struct FontInfo font = { 7, 12, 90, 0 };

      packbuf = (char *)malloc( 2 * bmp->w * bmp->h );
      upackbuf = &packbuf[ bmp->w * bmp->h ];

      /* copy surface lines to buffer */
      for ( i = 0; i < bmp->h; i++ ) {
        char *pix = (Uint8 *)bmp->pixels + i * bmp->pitch;
        for ( j = 0; j < bmp->w; j++ ) {
          upackbuf[j + i * bmp->w] = *pix++;
        }
      }

      pict.packed_size = ByteRun1_pack( upackbuf, packbuf, bmp->w * bmp->h );

      SDL_RWwrite( out, (char *)&font, sizeof(struct FontInfo), 1 );

      SDL_WriteLE16( out, pict.colors );
      SDL_WriteLE16( out, pict.width );
      SDL_WriteLE16( out, pict.height );
      SDL_WriteLE32( out, pict.packed_size );

      for ( i=0; i<palette->ncolors; ++i ) {
        SDL_RWwrite( out, &palette->colors[i].r, 1, 1 );
        SDL_RWwrite( out, &palette->colors[i].g, 1, 1 );
        SDL_RWwrite( out, &palette->colors[i].b, 1, 1 );
      }

      SDL_RWwrite( out, packbuf, 1, pict.packed_size );
      free( packbuf );
    } else fprintf(stderr, "Couldn't find palette\n" );
    SDL_FreeSurface( bmp );
  } else {
    fprintf(stderr, "Couldn't open BMP %s\n", argv[3] );
    exit(-1);
  }

  /* large font */
  bmp = SDL_LoadBMP(argv[3]);
  if ( bmp ) {
    if ( bmp->format->BitsPerPixel == 8 ) {
      SDL_Palette *palette = bmp->format->palette;
      struct PicHeader pict = { palette->ncolors, bmp->w, bmp->h, 0 };
      struct FontInfo font = { 12, 14, 64, FONT_CAPS_ONLY };

      packbuf = (char *)malloc( 2 * bmp->w * bmp->h );
      upackbuf = &packbuf[ bmp->w * bmp->h ];

      /* copy surface lines to buffer */
      for ( i = 0; i < bmp->h; i++ ) {
        char *pix = (Uint8 *)bmp->pixels + i * bmp->pitch;
        for ( j = 0; j < bmp->w; j++ ) {
          upackbuf[j + i * bmp->w] = *pix++;
        }
      }

      pict.packed_size = ByteRun1_pack( upackbuf, packbuf, bmp->w * bmp->h );

      SDL_RWwrite( out, (char *)&font, sizeof(struct FontInfo), 1 );

      SDL_WriteLE16( out, pict.colors );
      SDL_WriteLE16( out, pict.width );
      SDL_WriteLE16( out, pict.height );
      SDL_WriteLE32( out, pict.packed_size );

      for ( i=0; i<palette->ncolors; ++i ) {
        SDL_RWwrite( out, &palette->colors[i].r, 1, 1 );
        SDL_RWwrite( out, &palette->colors[i].g, 1, 1 );
        SDL_RWwrite( out, &palette->colors[i].b, 1, 1 );
      }

      SDL_RWwrite( out, packbuf, 1, pict.packed_size );
      free( packbuf );
    } else fprintf(stderr, "Couldn't find palette\n" );
    SDL_FreeSurface( bmp );
  } else {
    fprintf(stderr, "Couldn't open BMP %s\n", argv[4] );
    exit(-1);
  }

  SDL_RWclose( out );
  return 0;
}

/**********************************************************************
* cmpByteRun1 compression algorithm
*
* control bytes:
*  [0..127]   : followed by n+1 bytes of data
*  [-1..-127] : followed by byte to be repeated (-n)+1 times
*  -128       : NOOP
**********************************************************************/

#define DUMP	0
#define RUN	1

#define MINRUN 3
#define MAXRUN 128
#define MAXLEN 128	/* otherwise exceeds char range */

/* buffer size must be at least 2 * MAXLEN */
static char brbuf[2*MAXLEN];

static char *PutDump( char *dest, const char *from, short nn ) {
  short i;
  *dest++ = nn-1;
  for ( i = 0; i < nn; i++ ) *dest++ = *from++;
  return dest;
}

static char *PutRun( char *dest, short nn, char cc ) {
  *dest++ = -(nn-1);
  *dest++ = cc;
  return dest;
}

/* Compresses given data using cmpByteRun1 compression. returns size of packed data. */
static long ByteRun1_pack( const char *src, char *destbuf, unsigned long srcsize ) {
  char *dest = destbuf, c, lastc;
  short mode = DUMP;
  short nbuf = 1;			/* number of chars in buffer */
  short rstart = 0;			/* buffer index current run starts */

  brbuf[0] = lastc = *src++;

  for ( srcsize--; srcsize; srcsize-- ) {
    brbuf[nbuf++] = c = *src++;
    if ( mode == DUMP ) {
      /* If the buffer is full, write the length byte, then the data */
      if ( nbuf > MAXLEN ) {
        dest = PutDump( dest, brbuf, MAXLEN );
        brbuf[0] = c;
        nbuf = 1; rstart = 0;
      } else if ( c == lastc ) {
        if ( nbuf-rstart >= MINRUN ) {
          if ( rstart > 0 ) dest = PutDump( dest, brbuf, rstart );
          mode = RUN;
        } else if ( rstart == 0 ) mode = RUN;
      } else rstart = nbuf-1;		/* first of run */ 

    /* mode == RUN */
    } else if ( (c != lastc) || (nbuf-rstart > MAXRUN) ) {
      /* output run */
      dest = PutRun( dest, nbuf-rstart-1, lastc );
      brbuf[0] = c;
      nbuf = 1; rstart = 0;
      mode = DUMP;
    }

    lastc = c;
  }

  if ( mode == DUMP ) dest = PutDump( dest, brbuf, nbuf );
  else dest = PutRun( dest, nbuf-rstart, lastc );

  return dest-destbuf;
}

