// Crimson Fields -- a game of tactical warfare
// Copyright (C) 2000, 2001 Jens Granseuer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

////////////////////////////////////////////////////////////////////////
// misc.cpp - miscellaneous classless functions
//
// History:
//  02-06-2000 - created
//  23-06-2000 - added crypt()
//  11-05-2001 - added queue_quit_event()
//  24-06-2001 - removed queue_quit_event()
////////////////////////////////////////////////////////////////////////

#include <stdlib.h>
#include <string.h>

#include "misc.h"

////////////////////////////////////////////////////////////////////////
// NAME       : random
// DESCRIPTION: Create a pseudo-random number between min and max.
// PARAMETERS : min - lower end of random numbers
//              max - high end of random numbers
// RETURNS    : min <= r <= max
//
// HISTORY
////////////////////////////////////////////////////////////////////////

int random( int min, int max ) {
  return( rand_range( max - min + 1 ) + min );
}

////////////////////////////////////////////////////////////////////////
// NAME       : rand_range
// DESCRIPTION: Create a pseudo-random number between 0 and range.
// PARAMETERS : range - range of pseudo-random numbers;
//                      1 < range <= RAND_MAX+1
// RETURNS    : 0 <= r < range
//
// HISTORY
////////////////////////////////////////////////////////////////////////

unsigned int rand_range( unsigned int range ) {
  unsigned int rmax, r, d;
  d = (RAND_MAX+1U-range) / range + 1; 
  rmax = d * range - 1;                // -1 to avoid "overflow to zero"
  do
    r = rand();
  while (r > rmax);
  return r/d;
}

////////////////////////////////////////////////////////////////////////
// NAME       : itoa
// DESCRIPTION: Convert a decimal number to an ASCII string.
// PARAMETERS : n   - number to convert
//              buf - buffer to hold the resulting string; make sure it
//                    is large enough for the string before calling!
// RETURNS    : pointer to buffer
//
// HISTORY
////////////////////////////////////////////////////////////////////////

char *itoa( int n, char *buf ) {
  bool neg = false;
  short i = 0, j;
  char c;

  if (n < 0) {
    neg = true;
    n = -n;
  }

  // output number to buffer in reverse order
  do {
    buf[i++] = n % 10 + '0';
  } while ( (n /= 10) > 0 );

  if (neg) buf[i++] = '-';
  buf[i--] = '\0';

  // turn the string around
  for (j = 0; j < i; i--, j++) {
    c = buf[i];
    buf[i] = buf[j];
    buf[j] = c;
  }
  return buf;
}

////////////////////////////////////////////////////////////////////////
// NAME       : tolower
// DESCRIPTION: Convert a character to lower-case.
// PARAMETERS : c - character to convert
// RETURNS    : lower version of c or c if it's not an upper-case letter
//
// BUGS       : Does not handle umlauts or other fancy characters.
//
// HISTORY
////////////////////////////////////////////////////////////////////////

char tolower( char c ) {
  if ( (c >= 'A') && (c <= 'Z') ) return c - 'A' + 'a';
  return c;
}

////////////////////////////////////////////////////////////////////////
// NAME       : byterun_unpack
// DESCRIPTION: Decode data compressed using the simple cmpByteRun1
//              compression algorithm.
// PARAMETERS : src       - buffer containing compressed data
//              dest      - destination buffer; make sure it is large
//                          enough to hold the uncompressed data
//              srcbytes0 - size of source buffer in bytes
//              dstbytes0 - size of destination buffer bytes
// RETURNS    : -1 on error, 0 otherwise
//
// NOTES
//   control bytes:
//     [0..127]   : followed by n+1 bytes of data
//     [-1..-127] : followed by byte to be repeated (-n)+1 times
//     -128       : NOOP
//
// HISTORY
//   31-05-2001 - loop until srcbytes is zero, not dstbytes
////////////////////////////////////////////////////////////////////////

int byterun_unpack( char *src, char *dest, long srcbytes0, long dstbytes0 ) {
  long srcbytes = srcbytes0, dstbytes = dstbytes0;
  short n;
  char c;

  while ( srcbytes > 0 ) {
    --srcbytes;
    n = *src++;

    if ( n >= 0 ) {
      srcbytes -= ++n;
      dstbytes -= n;
      if ( (srcbytes < 0) || (dstbytes < 0) ) return -1;
      do { *dest++ = *src++; } while ( --n );
    } else if ( n != -128 ) {
      n = -n + 1;
      dstbytes -= n;
      if ( (--srcbytes < 0) || (dstbytes < 0) ) return -1;
      c = *src++;
      do { *dest++ = c; } while ( --n );
    }
  }
  return 0;
}

////////////////////////////////////////////////////////////////////////
// NAME       : crypt
// DESCRIPTION: Encrypt/decrypt ASCII strings. The "algorithm" is really
//              simple and designed to prevent users from "accidentally"
//              browsing through data files with a hex editor.
// PARAMETERS : str - string to encrypt or decrypt; the encrypted or
//                    decrypted version will be printed to that buffer
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void crypt( char *str ) {
  while ( *str ) *str++ = ~(*str);
}

