// Crimson Fields -- a hex-based game of tactical warfare
// Copyright (C) 2000, 2001 Jens Granseuer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

/////////////////////////////////////////////////////////////////////
// main.cpp -- Crimson Fields
//
// History:
//  23-03-2000 - created
//  27-04-2000 - support level files
//  01-03-2001 - initial public release
//  07-03-2001 - include string.h to fix compilation errors
//  25-03-2001 - added main menu to select level
//  02-04-2001 - added glob var AllowUpd to disallow display surface
//               updates if necessary
//  22-04-2001 - display depth now 16 bpp to use true "alpha fog"
//             - added sound support
//  04-05-2001 - reintroduced --level command line parameter
//  16-05-2001 - added --version and --help command line parameters
//  26-05-2001 - removed AllowUpd, moved into View class
//  19-06-2001 - added event filter callback function
/////////////////////////////////////////////////////////////////////

#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <fstream.h>
#include "SDL.h"
#include "SDL_endian.h"

#include "view.h"
#include "game.h"
#include "misc.h"
#include "fileio.h"
#include "initwindow.h"
#include "globals.h"
#include "sound.h"

// global vars
Game *Gam;
Color ColLight, ColDark;
Surface *Icons;
Image *Images[NUM_IMAGES] = { NULL };

struct GameOptions {
  short px_width;
  short px_height;
  short bpp;
  bool sound;
  unsigned long sdl_flags;
  const char *level;
};

// local vars
static Font *f1 = NULL, *f2 = NULL;
static View *display = NULL;

// local function prototypes
static void parse_options( int argc, char **argv, GameOptions &opts );
static void print_usage( char *prog );
static int init_data( Font **f1, Font **f2, bool sound );
static void load_settings( GameOptions &opts );
static void save_settings( View *display );
static GUI_Status event_filter( SDL_Event &event, Window *window );
static void do_exit( void );

// Start of program functions //////////////////////////////////////////

int main( int argc, char **argv ) {
  struct GameOptions options;

  parse_options( argc, argv, options );

  if ( SDL_Init( SDL_INIT_VIDEO
#ifndef DISABLE_SOUND
                |SDL_INIT_AUDIO
#endif
  ) < 0 ) {
    fprintf( stderr, "Couldn't initialize: %s\n", SDL_GetError() );
    return 1;
  }

  atexit( SDL_Quit );

  // set main window title
  SDL_WM_SetCaption( PROGRAMNAME, PACKAGE );
  SDL_EnableKeyRepeat( SDL_DEFAULT_REPEAT_DELAY, SDL_DEFAULT_REPEAT_INTERVAL );

  display = new View( options.px_width, options.px_height, options.bpp,
                      options.sdl_flags );
  if ( !display->s_surface ) fprintf( stderr, "Couldn't set video mode: %s\n", SDL_GetError() );
  else {
    srand( time(0) );			// initialize random number generator

    if ( !init_data( &f1, &f2, options.sound ) ) {

      display->SetSmallFont( f1 );
      display->SetLargeFont( f2 );
      display->SetEventFilter( event_filter );

      // create options and save games directories if possible
      create_config_dir();

      // only open intro screen if the user didn't supply a level on the command line
      int intro = 1;
      if ( options.level ) {
        Gam = new Game( 0, display );
        intro = Gam->Load( options.level );
        if ( !intro ) Gam->NextTurn();
      }

      GUI_Status status;
      do {
        if ( intro ) new InitWindow( display );
        intro = 1;

        do {
          status = display->HandleEvents();
        } while ( (status != GUI_QUIT) && (status != GUI_RESTART) );

        display->CloseAllWindows();
      } while ( status != GUI_QUIT );
    }
  }

  do_exit();
  return 0;
}

////////////////////////////////////////////////////////////////////////
// NAME       : parse_options
// DESCRIPTION: Process any options given to the program on the command
//              line.
// PARAMETERS : argc - argument count
//              argv - pointer to array of arguments
//              opts - buffer to store options in
// RETURNS    : -
//
// HISTORY
//   18-06-2001 - fullscreen option now takes an argument
//   23-09-2001 - added --sound command line parameter
////////////////////////////////////////////////////////////////////////

// parse command line arguments
void parse_options( int argc, char **argv, GameOptions &opts ) {
  // initialize with some default values
  opts.px_width = 800;
  opts.px_height = 600;
  opts.bpp = DISPLAY_BPP;
  opts.sound = true;
  opts.sdl_flags = SDL_HWSURFACE;
  opts.level = NULL;

  load_settings( opts );

  while ( argc > 1 ) {
    argc--;

    if (strcmp(argv[argc-1], "--width") == 0) {
      opts.px_width = atoi(argv[argc]);
    } else if (strcmp(argv[argc-1], "--height") == 0) {
      opts.px_height = atoi(argv[argc]);
    } else if (strcmp(argv[argc-1], "--level") == 0) {
      opts.level = argv[argc];
    } else if (strcmp(argv[argc-1], "--fullscreen") == 0) {
      if ( atoi( argv[argc] ) ) opts.sdl_flags |= SDL_FULLSCREEN;
      else opts.sdl_flags &= ~SDL_FULLSCREEN;
    } else if (strcmp(argv[argc-1], "--sound") == 0) {
      if ( atoi( argv[argc] ) ) opts.sound = true;
      else opts.sound = false;
    } else {
      if (strcmp(argv[argc], "--version") == 0)
        fprintf( stdout, PROGRAMNAME" "VERSION"\n" );
      else print_usage( argv[0] );
      exit ( 0 );
    }
    argc--;
  }
  if ( opts.px_width < 640 ) opts.px_width = 640;
  if ( opts.px_height < 480 ) opts.px_height = 480;
}

////////////////////////////////////////////////////////////////////////
// NAME       : print_usage
// DESCRIPTION: Print a usage message to stdout.
// PARAMETERS : prog - program name as given on the command line
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void print_usage( char *prog ) {
  fprintf( stdout, "Usage: %s [options]\n\n"
                   "Available options:\n"
                   "  --level <level>      load level or save file\n"
                   "  --width <width>      set screen width\n"
                   "  --height <height>    set screen height\n"
                   "  --fullscreen <1|0>   enable/disable fullscreen mode\n"
#ifndef DISABLE_SOUND
                   "  --sound <1|0>        enable/disable sound\n"
#endif
                   "  --help               display this help and exit\n"
                   "  --version            output version information and exit\n",
        prog );
}

////////////////////////////////////////////////////////////////////////
// NAME       : init_data
// DESCRIPTION: Initialize the global Icons surface and the game fonts.
// PARAMETERS : f1    - reference to small font to be initialized
//              f2    - reference to large font to be initialized
//              sound - flags indicating whether sound is enabled or
//                      muted
// RETURNS    : 0 on success, -1 on error
//
// HISTORY
////////////////////////////////////////////////////////////////////////

int init_data( Font **f1, Font **f2, bool sound ) {
  string datpath( get_data_dir() );
  datpath.append( CF_DATFILE );
  SDL_RWops *datfile = SDL_RWFromFile( datpath.c_str(), "rb" );
  if ( !datfile ) {
    fprintf( stderr, "Error: Couldn't open '%s'\n", datpath.c_str() );
    return -1;
  }

  Icons = new Surface;		// load icons surface
  if ( Icons->LoadImageData( datfile ) ) {
    fprintf( stderr, "Error reading data file\n" );
    return -1;
  }
  Icons->DisplayFormat();

  *f1 = new Font( datfile );	// load fonts
  *f2 = new Font( datfile );

  SDL_RWclose( datfile );

  init_audio( sound );		// load sound effects

  Images[ICON_SPEED] = new Image( Icons, 0 * ICON_WIDTH, 0 * ICON_HEIGHT, ICON_WIDTH, ICON_HEIGHT );
  Images[ICON_DEFENCE] = new Image( Icons, 1 * ICON_WIDTH, 0 * ICON_HEIGHT, ICON_WIDTH, ICON_HEIGHT );
  Images[ICON_GROUND] = new Image( Icons, 2 * ICON_WIDTH, 0 * ICON_HEIGHT, ICON_WIDTH, ICON_HEIGHT );
  Images[ICON_AIRCRAFT] = new Image( Icons, 3 * ICON_WIDTH, 0 * ICON_HEIGHT, ICON_WIDTH, ICON_HEIGHT );
  Images[ICON_SHIP] = new Image( Icons, 4 * ICON_WIDTH, 0 * ICON_HEIGHT, ICON_WIDTH, ICON_HEIGHT );
  Images[ICON_CRYSTALS] = new Image( Icons, 6 * ICON_WIDTH, 0 * ICON_HEIGHT, ICON_WIDTH, ICON_HEIGHT );
  Images[ICON_HEX_BG] = new Image( Icons, 9 * ICON_WIDTH, 1 * ICON_HEIGHT, ICON_WIDTH, ICON_HEIGHT );
  for ( int i = 0; i <= XP_MAX_LEVEL; i++ )
    Images[ICON_XP_BASE+i] = new Image( Icons, 256 + i * XP_ICON_WIDTH, 0, XP_ICON_WIDTH, XP_ICON_HEIGHT );

  return 0;
}

////////////////////////////////////////////////////////////////////////
// NAME       : load_settings
// DESCRIPTION: Read default display settings from the crimsonrc file.
// PARAMETERS : opts - buffer to store the settings. These should
//                     already be initialized with some defaults in
//                     case the rc file doesn't exist or this function
//                     fails.
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void load_settings( GameOptions &opts ) {
  string crimsonrc( get_config_dir() );
  crimsonrc.append( CRIMSONRC );

  FILE *fp = fopen( crimsonrc.c_str(), "r" );
  if ( fp ) {
    char linebuf[256], *val;
    unsigned short linecnt = 0;

    while ( fgets( linebuf, 255, fp ) ) {
      ++linecnt;
      if ( (linebuf[0] != '#') && (linebuf[0] != '\n') ) {
        val = strchr( linebuf, ' ' );
        if ( val ) {
          while ( *val == ' ' ) ++val;

          if ( !strncmp( linebuf, "width", 5 ) ) opts.px_width = atoi( val );
          else if ( !strncmp( linebuf, "height", 6 ) ) opts.px_height = atoi( val );
          else if ( !strncmp( linebuf, "fullscreen", 10 ) ) {
            if ( atoi( val ) != 0 ) opts.sdl_flags |= SDL_FULLSCREEN;
          } else if ( !strncmp( linebuf, "sound", 5 ) ) {
            if ( atoi( val ) == 0 ) opts.sound = 0;
            else opts.sound = 1;
          } else fprintf( stderr, "warning: unrecognized config option in line %d\n", linecnt );
        }
      }
    }
    fclose( fp );
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : save_settings
// DESCRIPTION: Save current display settings to the crimsonrc file.
// PARAMETERS : display - pointer to display
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void save_settings( View *display ) {
  string crimsonrc( get_config_dir() );
  crimsonrc.append( CRIMSONRC );

  FILE *fp = fopen( crimsonrc.c_str(), "w" );
  if ( fp ) {
    char linebuf[20];

    sprintf( linebuf, "width %d\n", display->Width() );
    fputs( linebuf, fp );
    sprintf( linebuf, "height %d\n", display->Height() );
    fputs( linebuf, fp );
    sprintf( linebuf, "fullscreen %d\n", display->IsFullScreen() ? 1 : 0 );
    fputs( linebuf, fp );
    sprintf( linebuf, "sound %d\n", get_audio() );
    fputs( linebuf, fp );

    fclose( fp );
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : event_filter
// DESCRIPTION: This is the global event filter function. It is hooked
//              to the display and called everytime the event handler
//              receives an event.
// PARAMETERS : event  - event received by the event handler
//              window - pointer to the currently active window
// RETURNS    : GUI_Status; if the filter returns GUI_SKIP the event
//              handler will not pass the event to its windows, but
//              silently drop it.
//
// HISTORY
//   13-08-2001 - F11 toggles sound on/off
////////////////////////////////////////////////////////////////////////

GUI_Status event_filter( SDL_Event &event, Window *window ) {
  GUI_Status rc = GUI_OK;

  if ( event.type == SDL_KEYDOWN ) {
    rc = GUI_SKIP;

    switch ( event.key.keysym.sym ) {
    case SDLK_F11:            // toggle sound
      toggle_audio();
      break;
    case SDLK_F12:            // toggle fullscreen mode
      window->GetView()->ToggleFullScreen();
      break;
    default:
      rc = GUI_OK;            // send to windows
    }
  } else if ( event.type == SDL_QUIT ) do_exit();

  return rc;
}

////////////////////////////////////////////////////////////////////////
// NAME       : do_exit
// DESCRIPTION: Free all resources and exit the program.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void do_exit( void ) {
  shutdown_audio();

  if ( display ) {
    save_settings( display );
    delete display;
  }

  delete f1;
  delete f2;

  for ( int i = 0; i < NUM_IMAGES; i++ ) delete Images[i];

  exit( 0 );
}

