// Crimson Fields -- a game of tactical warfare
// Copyright (C) 2000, 2001 Jens Granseuer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

////////////////////////////////////////////////////////////////////////
// game.cpp
//
// History:
//  08-12-2000 - created
//  02-03-2001 - added smooth unit and cursor movement
//  07-03-2001 - set cursor to unit at start of turn
//  09-03-2001 - modified mission file format to include player info
//  11-03-2001 - added ShowBriefing()
//  29-03-2001 - added ShowLevelInfo()
//  30-04-2001 - added PBeM support
//  10-06-2001 - added support for turn replays
//  12-08-2001 - added support for mission set sound effects
////////////////////////////////////////////////////////////////////////

#include <string.h>
#include "SDL_endian.h"

#include "game.h"
#include "combat.h"
#include "event.h"
#include "history.h"
#include "unitwindow.h"
#include "filewindow.h"
#include "initwindow.h"
#include "fileio.h"
#include "sound.h"
#include "ai.h"
#include "globals.h"


// button identifiers used by the game hook function dispatcher
#define G_BUTTON_START_TURN	1

#define G_BUTTON_END_TURN	10
#define G_BUTTON_MAP		11
#define G_BUTTON_BRIEFING	12
#define G_BUTTON_SAVE		13
#define G_BUTTON_LEV_INFO	14
#define G_BUTTON_VIDEO_OPTIONS	15
#define G_BUTTON_ABORT		16
#define G_BUTTON_QUIT		17

#define G_BUTTON_UNIT_INFO	20
#define G_BUTTON_UNIT_CONTENT	21

#define G_BUTTON_NEW_PW         30
#define G_BUTTON_CONFIRM_PW     31
#define G_BUTTON_REQUEST_PW     32



////////////////////////////////////////////////////////////////////////
// NAME       : Game::Game
// DESCRIPTION: Create a new game.
// PARAMETERS : flags - game flags (see game.h for details)
//              view  - the display surface
// RETURNS    : -
//
// HISTORY
//   30-04-2001 - added flags parameter
////////////////////////////////////////////////////////////////////////

Game::Game( unsigned short flags, View *view ) :
      MapWindow( 0, 0, view->Width(), view->Height(), 0, view ) {
  g_turn = 1;
  g_limit = 0;

  g_flags = flags;
  g_players[PLAYER_ONE] = g_players[PLAYER_TWO] = NULL;
  g_utypes = NULL;
  g_mission = NULL;
  g_set = NULL;
  g_messages = NULL;
  g_level_info = -1;
  g_password[0] = '\0';

  g_history = NULL;
  g_set_sounds = NULL;

  EnableMapDisplay();
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::~Game
// DESCRIPTION: Destroy the game.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

Game::~Game( void ) {
  delete g_players[PLAYER_ONE];
  delete g_players[PLAYER_TWO];
  delete [] g_utypes;
  delete [] g_mission;
  delete [] g_set;
  delete [] g_messages;
  delete g_history;

  if ( g_set_sounds ) {
    for ( int i = 0; g_set_sounds[i] != NULL; i++ ) delete g_set_sounds[i];
    delete g_set_sounds;
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::Quit
// DESCRIPTION: Ask the user for confirmation and exit.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
//   24-10-2001 - play sound when opening window
////////////////////////////////////////////////////////////////////////

void Game::Quit( void ) const {
  RequestWindow *req = new RequestWindow( "Do you really want to quit?",
                                          "yYes", "nNo", 0, view );
  req->button1->SetID( GUI_QUIT );
  play_audio( SND_GUI_ASK, 0 );
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::CreateSaveFileName
// DESCRIPTION: Get the name of the file to save to. Pop up a file
//              requester if necessary.
// PARAMETERS : filename - default filename
// RETURNS    : pointer to file name or NULL on error; if the function
//              returns a valid pointer the area it points to must be
//              deleted after use.
//
// HISTORY
//   12-08-2001 - save dialog couldn't be aborted
////////////////////////////////////////////////////////////////////////

string Game::CreateSaveFileName( const char *filename ) const {
  string file;
  if ( filename ) file.append( filename );

  bool filesel;
  do {
    GUI_Status rc;
    RequestWindow *rw;
    filesel = false;

    if ( file.length() == 0 ) filesel = true;
    else if ( exists( file.c_str() ) ) {
      // if file exists let user confirm the write
      char *conmsg = new char[ file.length() + 20 ];
      strcpy( conmsg, file.c_str() );
      strcat( conmsg, "\nexists. Overwrite?" );
      rw = new RequestWindow( conmsg, "yYes", "nNo", 0, view );
      rw->button0->SetID( 0 );
      rw->button1->SetID( 1 );
      rc = rw->EventLoop();
      if ( rc == 0 ) filesel = true;
      view->CloseWindow( rw );
      delete conmsg;
    }

    if ( filesel ) {
      string save_dir( get_save_dir() );
      bool done = false;
      FileWindow *fw = new FileWindow( save_dir.c_str(), NULL, ".sav", WIN_FILE_SAVE, view );
      fw->ok->SetID( 1 );
      fw->cancel->SetID( 0 );

      do {
        rc = fw->EventLoop();

        if ( rc == 1 ) {
          file = fw->GetFile();
          if ( file.length() != 0 ) {
            view->CloseWindow( fw );
            done = true;
          }
        } else if ( rc == 0 ) {
          if ( g_flags & GI_PBEM ) {
            // if saving a PBeM game is aborted the game data is lost...
            rw = new RequestWindow( "Really abort? Current game will be lost!", "wSo what?", "oOops!", 0, view );
            play_audio( SND_GUI_ASK, 0 );
            rw->button0->SetID( 0 );
            rw->button1->SetID( 1 );
            rc = rw->EventLoop();
            view->CloseWindow( rw );

            if ( rc != 0 ) {
              view->CloseWindow( fw );
              file = "";
              done = true;
              filesel = false;
            }
          } else {
            view->CloseWindow( fw );
            file.assign( "" );
            done = true;
            filesel = false;
          }
        }

      } while ( !done );
    }
  } while ( filesel );

  return file;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::Save
// DESCRIPTION: Save the current game to a file.
// PARAMETERS : filename - data file name; if NULL pops up a file
//                         selection window
// RETURNS    : 0 on successful write, non-zero otherwise
//
// HISTORY
//   11-05-2001 - show file selection dialog if filename is NULL
//              - ask before overwriting existing files
//   27-09-2001 - if pbem confirmation message after saving was closed
//                with 'ESC' the user was left with a black screen
////////////////////////////////////////////////////////////////////////

int Game::Save( const char *filename ) const {
  string fname( CreateSaveFileName( filename ) );
  if ( fname.length() == 0 ) return -1;
 
  SDL_RWops *file = SDL_RWFromFile( fname.c_str(), "wb" );
  if ( !file ) {
    new NoteWindow( "Error!", "Couldn't open file for writing", 0, view );
    return -1;
  }

  // save game info
  unsigned char version = FILE_VERSION, current_player = player->ID();
  unsigned short len, num;
  SDL_WriteLE32( file, FID_MISSION );
  SDL_RWwrite( file, &version, sizeof(unsigned char), 1 );
  SDL_RWwrite( file, &g_level_info, sizeof(char), 1 );

  SDL_WriteLE16( file, g_flags );
  SDL_WriteLE16( file, g_turn );
  if ( g_flags & GI_PASSWORD ) SDL_RWwrite( file, g_password, sizeof(char), 8 );
  SDL_RWwrite( file, &current_player, sizeof(unsigned char), 1 );
  SDL_RWwrite( file, &g_turn_phase, sizeof(unsigned char), 1 );

  len = strlen( g_mission );                      // save mission title
  SDL_WriteLE16( file, len );
  SDL_RWwrite( file, g_mission, sizeof(char), len );

  len = strlen( g_set );                          // save mission set info
  SDL_WriteLE16( file, len );
  SDL_RWwrite( file, g_set, sizeof(char), len );

  Map::Save( file );                              // save map

  g_players[PLAYER_ONE]->Save( file );            // save player data
  g_players[PLAYER_TWO]->Save( file );

  num = g_buildings.CountNodes();                 // save buildings
  SDL_WriteLE16( file, num );
  for ( Building *b = static_cast<Building *>( g_buildings.Head() );
        b; b = static_cast<Building *>( b->Next() ) ) b->Save( file );

  // save transports; basically, transports are not much different
  // from other units but we MUST make sure that transports having
  // other units on board are loaded before those units; we need to
  // make two passes through the list, first counting the actual number
  // of transports. Then we save all transports with loading and those
  // not inside other buildings or transports in the second, and all
  // the rest in the last run
  len = num = 0;
  Unit *u;
  for ( u = static_cast<Unit *>( g_units.Head() );
        u; u = static_cast<Unit *>( u->Next() ) ) {       // count transports
    if ( u->IsTransport() ) len++;
    else num++;                                           // already count normal
  }                                                       // units as well
  SDL_WriteLE16( file, len ); 

  for ( u = static_cast<Unit *>( g_units.Head() );
        u; u = static_cast<Unit *>( u->Next() ) ) {       // save "free" transports
    if ( u->IsTransport() && !u->IsSheltered() )
      static_cast<Transport *>(u)->Save( file );
  }

  for ( u = static_cast<Unit *>( g_units.Head() );
        u; u = static_cast<Unit *>( u->Next() ) ) {       // save "unfree" transports
    if ( u->IsTransport() && u->IsSheltered() )
      static_cast<Transport *>(u)->Save( file );
  }

  SDL_WriteLE16( file, num );                             // save units
  for ( u = static_cast<Unit *>( g_units.Head() );
        u; u = static_cast<Unit *>( u->Next() ) )
    if ( !u->IsTransport() ) u->Save( file );

  num = g_combat.CountNodes();                           // save combat data
  SDL_WriteLE16( file, num );
  for ( Combat *c = static_cast<Combat *>( g_combat.Head() );
        c; c = static_cast<Combat *>( c->Next() ) )
    c->Save( file );

  num = g_events.CountNodes();                           // save events
  SDL_WriteLE16( file, num );
  for ( Event *e = static_cast<Event *>( g_events.Head() );
        e; e = static_cast<Event *>( e->Next() ) )
    e->Save( file );

  SDL_WriteLE16( file, g_msg_num );                     // save text messages
  SDL_WriteLE32( file, g_msg_len );
  SDL_RWwrite( file, g_messages[0], sizeof(char), g_msg_len );

  if ( g_history ) g_history->Save( file );		// save turn history
  else SDL_WriteLE16( file, 0 );

  SDL_RWclose( file );

  if ( g_flags & GI_PBEM ) {
    fname.insert( 0, "Current PBeM game was saved to\n" );
    NoteWindow *nw = new NoteWindow( "Game saved", fname.c_str(), 0, view );
    nw->button->SetID( GUI_RESTART );
  }
  return 0;
}


////////////////////////////////////////////////////////////////////////
// NAME       : Game::Load
// DESCRIPTION: Load a game from a mission file (start a new game) or
//              a save file (resume game).
// PARAMETERS : filename - data file name
// RETURNS    : 0 on success, non-zero otherwise
//
// HISTORY
//   19-08-2001 - if no history found in file but we need one, create
//                an empty history instead of setting it to NULL
////////////////////////////////////////////////////////////////////////

int Game::Load( const char *filename ) {
  int rc = -1;
  SDL_RWops *file = SDL_RWFromFile( filename, "rb" );
  if ( !file ) return -1;

  // read game info
  unsigned char version;
  unsigned long id = SDL_ReadLE32( file );
  SDL_RWread( file, &version, 1, 1 );
  if ( (version == FILE_VERSION) && (id == FID_MISSION) ) {
    unsigned short len, i;
    unsigned char current_player;

    SDL_RWread( file, &g_level_info, 1, 1 );
    g_flags |= SDL_ReadLE16( file ) | GI_SAVEFILE;
    g_turn = SDL_ReadLE16( file );
    if ( g_flags & GI_PASSWORD ) SDL_RWread( file, g_password, 1, 8 );
    SDL_RWread( file, &current_player, 1, 1 );
    SDL_RWread( file, &g_turn_phase, 1, 1 );

    delete [] g_mission;		// load mission name
    len = SDL_ReadLE16( file );
    g_mission = new char [len + 1];
    SDL_RWread( file, g_mission, sizeof(char), len );
    g_mission[len] = '\0';

    delete [] g_set;			// load name of mission set
    len = SDL_ReadLE16( file );
    g_set = new char [len + 1];
    SDL_RWread( file, g_set, sizeof(char), len );
    g_set[len] = '\0';

    Map::Load( file );		// load map
    Init();			// adjust map window variables to map

    if ( LoadSet( g_set ) ) {
      fprintf( stderr, "Error: Level set file '%s' not available.\n", g_set );
      return -1;
    }

    g_players[PLAYER_ONE] = new Player( file );
    g_players[PLAYER_TWO] = new Player( file );
    if ( g_flags & GI_AI ) g_players[PLAYER_TWO]->SetType( COMPUTER );
    player = g_players[current_player];

    len = SDL_ReadLE16( file );		// load buildings
    for ( i = 0; i < len; i++ ) {
      Building *b = new Building( file, g_players );
      g_buildings.AddTail( b );
      SetBuilding( b, b->Position().x, b->Position().y );
    }

    len = SDL_ReadLE16( file );		// load transports
    for ( i = 0; i < len; i++ ) {
      Transport *t = new Transport( file, g_utypes, g_players );
      g_units.AddTail( t );
      SetUnit( t, t->Position().x, t->Position().y );
    }

    len = SDL_ReadLE16( file );		// load units
    for ( i = 0; i < len; i++ ) {
      Unit *u = LoadUnit( file );
      if ( u ) {
        g_units.AddTail( u );
        SetUnit( u, u->Position().x, u->Position().y );
      }
    }

    len = SDL_ReadLE16( file );		// load combats
    for ( i = 0; i < len; i++ ) {
      g_combat.AddTail( new Combat( file, this ) );
    }

    len = SDL_ReadLE16( file );		// load events
    for ( i = 0; i < len; i++ ) {
      g_events.AddTail( new Event( file, g_players ) );
    }

    // load text messages
    g_msg_num = SDL_ReadLE16( file );		// number of messages
    g_msg_len = SDL_ReadLE32( file );
    g_messages = (char **)new char[g_msg_len + (g_msg_num + 1) * sizeof(char *)];
    g_messages[g_msg_num] = 0;
    if ( g_msg_num ) {                         // load messages
      char *ptr;
      int j;

      ptr = (char *)&g_messages[g_msg_num + 1];
      SDL_RWread( file, ptr, sizeof(char), g_msg_len );
      g_messages[0] = ptr;

      i = 1; j = 0;
      while ( i < g_msg_num ) {
        if ( ptr[j++] == '\0' ) g_messages[i++] = &ptr[j];
      }
    }

    len = SDL_ReadLE16( file );
    if ( len ) g_history = new History( file, len );
    else if ( NextPlayer()->Type() == COMPUTER ) g_history = NULL;
    else g_history = new History();

    rc = 0;
  } else
    fprintf( stderr, "Error: File '%s' is not of the required type\n", filename );
 
  SDL_RWclose( file );
  return rc;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::LoadUnit
// DESCRIPTION: Load a unit from a file. This is a separate function
//              because some parameters must be suppplied which are only
//              accessible to the Game class itself.
// PARAMETERS : file - file dscriptor
// RETURNS    : pointer to loaded unit or NULL on error
//
// HISTORY
////////////////////////////////////////////////////////////////////////

Unit *Game::LoadUnit( SDL_RWops *file ) {
  return new Unit( file, g_utypes, g_players );
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::LoadSet
// DESCRIPTION: Load a mission set. A mission sets contains definitions
//              of unit and terrain types as well as the map graphics.
// PARAMETERS : set - name of set mission set; ".set" will be appended
//                    to the name to define the file to look for.
// RETURNS    : 0 on success, non-zero otherwise
//
// HISTORY
//   12-08-2001 - added sounds to mission sets
////////////////////////////////////////////////////////////////////////

int Game::LoadSet( const char *name ) {
  int rc = -1;
  string buf( get_data_dir() );
  buf.append( name );
  buf.append( ".set" );

  SDL_RWops *file = SDL_RWFromFile( buf.c_str(), "rb" );
  if ( !file ) return -1;

  unsigned long file_id;
  SDL_RWread( file, (char *)&file_id, sizeof(unsigned long), 1 );

  if ( file_id == FID_SET ) {
    int i;
    unsigned char sounds;
    unsigned short units = SDL_ReadLE16( file );
    unsigned short terrains = SDL_ReadLE16( file );
    SDL_RWread( file, &sounds, sizeof(unsigned char), 1 );

    if ( sounds != 0 ) {             // load sound effects
      string sndd( get_sounds_dir() );
      char filename[256], len;
      g_set_sounds = new SoundEffect * [sounds + 1];

      for ( i = 0; i < sounds; i++ ) {
        SDL_RWread( file, &len, sizeof(unsigned char), 1 );
        SDL_RWread( file, filename, sizeof(char), len );
        filename[len] = '\0';
        g_set_sounds[i] = new SoundEffect( (sndd + filename).c_str() );
      }
      g_set_sounds[i] = NULL;
    }

    g_utypes = new UnitType [units];
    for ( i = 0; i < units; i++ )
      if ( g_utypes[i].Load( file, g_set_sounds ) ) return -1;

    LoadTerrainTypes( file, terrains );
    LoadMapGfx( file );
    Icons->DisplayFormat();
    rc = 0;
  }

  SDL_RWclose( file );
  return rc;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::NextTurn
// DESCRIPTION: Begin the turn of a player. player is expected to
//              point to the new player already. This function puts up
//              a window to greet the current player.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void Game::NextTurn( void ) {
  char buf[10];
  NoteWindow *nw;
  player->Colors( ColLight, ColDark );   // set player color scheme

  // show turn information and greeting dialog
  sprintf( buf, "Turn %d", g_turn );

  nw = new NoteWindow( player->Name(), buf, WIN_FONT_BIG, view );
  nw->button->SetID( G_BUTTON_START_TURN );
  nw->button->SetHook( this );             // this gets called when
                                           // the button is pushed
                                           // with its ID and window
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::StartTurn
// DESCRIPTION: Start a turn, i.e. draw the map to the display.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
//   07-03-2001 - position cursor on unit at the beginning of each turn
//   22-04-2001 - display jumped if cursor didn't start in the upper
//                left corner of the map
//   30-04-2001 - check player passwords in email games
//   12-06-2001 - replay events from last turn
////////////////////////////////////////////////////////////////////////

void Game::StartTurn( void ) {
  // in email games set up or check player passwords
  static bool pw_checked = false;

  if ( g_flags & GI_PBEM ) {
    if ( !pw_checked ) {
      PasswordWindow *pwin;
      short butid;

      if ( player->Password() == NULL ) butid = G_BUTTON_NEW_PW;
      else butid = G_BUTTON_REQUEST_PW;

      pwin = new PasswordWindow( player->Name(), "Enter password",
                                 player->Password(), view );
      pwin->button->SetHook( this );
      pwin->button->SetID( butid );
      pw_checked = true;
      return;
    } else pw_checked = false;
  }


  if ( g_turn_phase == TURN_START ) {
    // replay
    if ( g_history ) {
      g_history->Replay();
      delete g_history;
    }

    // begin new turn
    if ( NextPlayer()->Type() == COMPUTER ) g_history = NULL;
    else g_history = new History();
  }

  player->SetMode( MODE_IDLE );
  CheckEvents();
  g_turn_phase = TURN_IN_PROGRESS;

  if ( player->Type() == COMPUTER ) {
    DisableMapDisplay();
    AI ai;
    ai.Play();
    EndTurn();
    EnableMapDisplay();
  } else {
    Reset();

    // set the cursor to one of the player's units
    Point startcursor = { 0, 0 };
    for ( Unit *u = static_cast<Unit *>( g_units.Head() );
          u; u = static_cast<Unit *>( u->Next() ) ) {
      if ( u->Owner() == player ) {
        startcursor = u->Position();
        break;
      }
    }

    view->DisableUpdates();
    CenterOnHex( startcursor.x, startcursor.y );
    SetCursor( startcursor.x, startcursor.y, false );
    view->EnableUpdates();
    view->Refresh();
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::CheckEvents
// DESCRIPTION: Check for pending game events.
// PARAMETERS : -
// RETURNS    : GUI status
//
// HISTORY
//   11-03-2001 - show mission briefing on the first turn
//   11-05-2001 - completely rewritten
//   20-07-2001 - don't handle mission briefings specially; should be
//                created as events of type EVENT_MESSAGE
//              - removed stage parameter
////////////////////////////////////////////////////////////////////////

GUI_Status Game::CheckEvents( void ) {
  Event *e = static_cast<Event *>( g_events.Head() ), *e2, *e3;
  MessageWindow *win = NULL;

  while ( e ) {
    e2 = static_cast<Event *>( e->Next() );
    short check = e->Check();
    if ( check > 0 ) {
      if ( (check == 2) && !win ) win = new MessageWindow( "Event", NULL, view );
      e3 = e->Execute( win, g_messages );
      if ( e3 ) g_events.AddTail( e3 );
      e->Remove();
      delete e;
      if ( check == 2 ) win->EventLoop();
    }
    e = e2;
  }
  if ( win ) view->CloseWindow( win );
  return GUI_OK;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::HaveWinner
// DESCRIPTION: Check whether one of the players has won completed his
//              mission. Notify players and quit the current game if so.
// PARAMETERS : -
// RETURNS    : true if mission is completed by any player AND both
//              players have been informed so, false otherwise
//
// HISTORY
////////////////////////////////////////////////////////////////////////

bool Game::HaveWinner( void ) {
  Player *winner = NULL;
  bool quit = false;
  bool both = false;

  if ( g_players[PLAYER_ONE]->Success( 0 ) >= 100 ) winner = g_players[PLAYER_ONE];
  if ( g_players[PLAYER_TWO]->Success( 0 ) >= 100 ) {
    if ( winner ) both = true;
    else winner = g_players[PLAYER_TWO];
  }

  if ( winner ) {
    unsigned char status, status_next;

    if ( both ) status = status_next = VIC_DRAW;
    else if ( winner == player ) {
      status = VIC_VICTORY;
      status_next = VIC_DEFEAT;
    } else {
      status = VIC_DEFEAT;
      status_next = VIC_VICTORY;
    }

    if ( NextPlayer()->Type() != HUMAN ) quit = true;
    else {
      g_events.AddHead( new Event( EVENT_VICTORY, ETRIGGER_TURN, Turn(), 0, 0,
                            status_next, 0, 0, -1, -1, NextPlayer() ) );
    }

    ShowDebriefing( status, quit );
  }
  return quit;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::EndTurn
// DESCRIPTION: End turn for the current player. Execute combat orders
//              and prepare everything for the next player.
// PARAMETERS : -
// RETURNS    : GUI_Status (GUI_QUIT if mission is complete)
//
// HISTORY
////////////////////////////////////////////////////////////////////////

GUI_Status Game::EndTurn( void ) {
  GUI_Status rc = GUI_OK;

  if ( unit ) DeselectUnit();
  player->SetMode( MODE_OFF );
  CursorOff();

  // calculate modifiers for combat
  Combat *combat = static_cast<Combat *>( g_combat.Head() );
  while ( combat ) {
    combat->CalcModifiers( this );
    combat = static_cast<Combat *>( combat->Next() );
  }

  // execute combat orders
  while ( !g_combat.IsEmpty() ) {
    Combat *combat = static_cast<Combat *>(g_combat.RemHead());
    CombatWindow *cwin = combat->Resolve( this, view );
    delete combat;

    if ( cwin ) {
      rc = cwin->EventLoop();
      view->CloseWindow( cwin );
      if ( rc == GUI_CLOSE ) rc = GUI_OK;
    }
  }

  // destroyed units may have triggered events...
  if ( rc == GUI_OK ) rc = CheckEvents();

  // check for mission completion
  if ( (rc == GUI_OK) && !HaveWinner() ) {
    ShowOff();
    Show();       // black display

    // set new player
    player = NextPlayer();
    if ( player->ID() == PLAYER_ONE ) g_turn++;

    // remove all destroyed units from the list,
    // restore movement points, reset status
    Unit *next, *u = static_cast<Unit *>(g_units.Head());
    while ( u ) {
      next = static_cast<Unit *>(u->Next());
      if ( !u->IsAlive() ) {
        u->Remove();
        delete u;
      } else {
        if ( u->Owner() == player ) u->RefreshMoves();
        else u->UnsetFlags( U_MOVED|U_ATTACKED|U_DONE|U_BUSY );
      }
      u = next;
    }

    // produce crystals in mines
    Building *b = static_cast<Building *>(g_buildings.Head());
    while ( b ) {
      if ( (b->Owner() == player) && b->IsMine() )
        b->SetCrystals( b->Crystals() + b->CrystalProduction() );
      b = static_cast<Building *>(b->Next());
    }

    // check if we're playing an email game. if so, save and exit
    if ( g_flags & GI_PBEM ) {
      string filebuf( get_save_dir() );

      filebuf.append( g_mission );
      filebuf.append( ".sav" );

      int err = Save( filebuf.c_str() );
      if ( err ) {
        NoteWindow *nw = new NoteWindow( "Error", "Game was not saved!", WIN_CLOSE_ESC, view );
        nw->button->SetID( GUI_RESTART );
      }
    } else {
      g_turn_phase = TURN_START;
      NextTurn();
    }
  }

  return rc;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::GetBuildingByID
// DESCRIPTION: Find the building corresponding to the given identifier.
// PARAMETERS : id - identifier of the building to be searched for
// RETURNS    : pointer to the building, or NULL if no building with
//              that ID exists
//
// HISTORY
////////////////////////////////////////////////////////////////////////

Building *Game::GetBuildingByID( unsigned short id ) const {
  Building *b = static_cast<Building *>( g_buildings.Head() );
  while ( b ) {
    if ( b->ID() == id ) return b;
    b = static_cast<Building *>( b->Next() );
  }
  return NULL;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::GetUnitByID
// DESCRIPTION: Find the unit corresponding to the given identifier.
// PARAMETERS : id - identifier of the unit to be searched for
// RETURNS    : pointer to the unit, or NULL if no unit with that ID
//              exists
//
// HISTORY
////////////////////////////////////////////////////////////////////////

Unit *Game::GetUnitByID( unsigned short id ) const {
  Unit *u = static_cast<Unit *>( g_units.Head() );
  while ( u ) {
    if ( u->ID() == id ) return u;
    u = static_cast<Unit *>( u->Next() );
  }
  return NULL;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::MoveUnit
// DESCRIPTION: Move a unit to another hex.
// PARAMETERS : u  - unit to be moved
//              hx - destination hex x
//              hy - destination hex y
// RETURNS    : the unit if it's still available, or NULL if it
//              cannot be selected this turn (read: deselect it)
//
// HISTORY
//   11-03-2001 - fixed crash introduced with 'deviate' in FindPath()
//   03-04-2001 - when trying to move to the hex the unit is already on,
//                simply deselect the unit (fixes units "moving back"
//                into their container to be considered U_MOVED)
//   03-06-2001 - when a U_SLOW unit has been moved, set U_DONE
////////////////////////////////////////////////////////////////////////

Unit *Game::MoveUnit( Unit *u, short hx, short hy ) {
  if ( (player->Type() == HUMAN) &&
       (m_path[XY2Index( hx, hy )] == -1) ) return u;
  if ( (u->Position().x == hx) && (u->Position().y == hy) ) {
    DeselectUnit();
    return NULL;
  }

  if ( FindPath( u, hx, hy, PATH_BEST, 0 ) == -1 ) {
    fprintf( stderr, "Internal error: FindPath() failed!\n" );
    return u;
  }

  SoundEffect *sfx = u->MoveSound();
  if ( sfx ) sfx->Play( SFX_LOOP );

  short step; 
  while ( (step = m_path[XY2Index(u->Position().x,u->Position().y)]) != -1)
    MoveUnit( u, (Direction)step );

  u->SetFlags( U_MOVED );
  u->SetMoves( 0 );

  if ( sfx ) sfx->Stop();

  if ( u->IsSlow() || !UnitTargets( u ) ) {
    u->SetFlags( U_DONE );
    DeselectUnit();
    CheckEvents();
    return NULL;
  }
  CreateMoveMap();
  Draw();
  Show();
  CheckEvents();
  return u;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::MoveUnit
// DESCRIPTION: Move a unit one hex in a given direction.
// PARAMETERS : u   - unit to be moved
//              dir - direction
// RETURNS    : 0 on success, -1 on error
//
// HISTORY
//   02-03-2001 - units now 'glide' from one hex to another instead of
//                jumping
//   12-06-2001 - for each move, record a "historical" event
//   21-09-2001 - record move before SetUnit(), otherwise when the unit
//                conquers a building, the owner will already change
//                _before_ the actual move in replay
////////////////////////////////////////////////////////////////////////

int Game::MoveUnit( Unit *u, Direction dir ) {
  const Point &pos = u->Position();
  Point posnew;
  if ( Dir2XY( pos.x, pos.y, dir, posnew ) ) return -1;

  short cost, moves = u->Moves();
  if ( u->IsAircraft() ) cost = MCOST_MIN;
  else cost = MoveCost( posnew.x, posnew.y );
  if ( cost > moves ) return -1;

  u->Face( dir );
  u->SetMoves( moves - cost );

  if ( !u->IsSheltered() ) {
    Point pixels;
    SetUnit( NULL, pos.x, pos.y );
    if ( g_display_map ) {
      Hex2Pixel( pos.x, pos.y, pixels );
      DrawHex( HexImage( pos.x, pos.y ), this, pixels.x, pixels.y, *this );
    }
  } else {
    Transport *t = static_cast<Transport *>( GetUnit( pos.x, pos.y ) );
    if ( t ) {
      if ( u->IsDummy() ) u->UnsetFlags( U_SHELTERED );
      else t->RemoveUnit( u );
    } else GetBuilding( pos.x, pos.y )->RemoveUnit( u );
  }

  if ( g_display_map && (HexOnScreen( pos.x, pos.y ) ||
                         HexOnScreen( posnew.x, posnew.y )) )
    MoveHex( u->Image(), pos.x, pos.y, posnew.x, posnew.y, ANIM_SPEED_UNIT );

  if ( u->IsDummy() ) {
    SetUnit( u, posnew.x, posnew.y );
    UpdateHex( posnew.x, posnew.y );
  } else {
    if ( g_history ) g_history->RecordMoveEvent( *u, dir );
    SetUnit( u, posnew.x, posnew.y );
  }
  return 0;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::CreateUnit
// DESCRIPTION: Create a new unit for the currently active player.
// PARAMETERS : type   - unit type
//              x      - horizontal position on map
//              y      - vertical position on map
//              player - proud owner of a freshman unit
// RETURNS    : unit or NULL on error
//
// HISTORY
////////////////////////////////////////////////////////////////////////

Unit *Game::CreateUnit( const UnitType *type, short x, short y, Player *player ) {
  // find an unused unit ID
  unsigned short id = 0;
  Unit *u = static_cast<Unit *>( g_units.Head() );
  while ( u ) {
    if ( u->ID() == id ) {
      id++;
      u = static_cast<Unit *>( g_units.Head() );
    } else u = static_cast<Unit *>( u->Next() );
  }

  if ( type->ut_flags & U_TRANSPORT )
    u = new Transport( type, player, id, x, y );
  else u = new Unit( type, player, id, x, y );
  if ( u ) {
    u->SetFlags( U_DONE );   // can't move on the first turn
    g_units.AddTail( u );
    SetUnit( u, x, y );
  }
  return u;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::UnitTargets
// DESCRIPTION: Get the number of targets the unit could shoot at.
// PARAMETERS : u - unit to check targets for
// RETURNS    : number of enemy units in range
//
// HISTORY
////////////////////////////////////////////////////////////////////////

unsigned short Game::UnitTargets( Unit *u ) const {
  Unit *tg = static_cast<Unit *>( g_units.Head() );
  unsigned short num = 0;
  while ( tg ) {
    if ( u->CanHit( tg ) ) num++;
    tg = static_cast<Unit *>( tg->Next() );
  }
  return num;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::CreateMoveMap
// DESCRIPTION: Check where the active unit can move. Each hex in
//              m_path where the unit can go this turn is marked with
//              1.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void Game::CreateMoveMap( void ) {
  List openlist;
  PathNode *pnode, *pnode2, *walk;
  short px, py, moves;

  if ( unit == NULL ) return;
  ClearPath();

  px = unit->Position().x;
  py = unit->Position().y;
  moves = unit->Moves();

  // insert starting node into openlist
  pnode = new PathNode;
  if ( !pnode ) return;

  pnode->x = px;
  pnode->y = py;
  pnode->cost = 0;
  m_path[XY2Index( px, py )] = 1;
  openlist.AddHead( pnode );

  while ( !openlist.IsEmpty() ) {
    // get the next node with minimum cost
    pnode = static_cast<PathNode *>( openlist.Head() );
    walk = static_cast<PathNode *>( pnode->Next() );
    while ( walk ) {
      if ( walk->cost < pnode->cost ) pnode = walk;
      walk = static_cast<PathNode *>( walk->Next() );
    }
    pnode->Remove();

    // take a look at all the hexes around
    for ( short dir = NORTH; dir <= NORTHWEST; dir++ ) {
      Point p;
      if ( !Dir2XY( pnode->x, pnode->y, (Direction)dir, p ) ) {
        short tindex = XY2Index( p.x, p.y );
        TerrainType *type = HexType( p.x, p.y );
        short newcost = pnode->cost;

        if ( unit->IsAircraft() ) newcost += MCOST_MIN;
        else newcost += type->tt_move;

        Unit *u = GetUnit( p.x, p.y );
        Building *b = GetBuilding( p.x, p.y );

        if ( u && (newcost <= moves) && !unit->IsSheltered() &&
                  u->IsTransport() &&
                  static_cast<Transport *>(u)->Allow( unit ) ) m_path[tindex] = 1;
        else if ( !u && (newcost <= moves) ) {
          if ( b ) {
            if ( !unit->IsSheltered() && b->Allow( unit ) ) m_path[tindex] = 1;

          // can the unit move over that kind of terrain?
          } else if ( type->tt_type & unit->Terrain() ) {
            // if we come through a narrow spot (enemy on one side and
            // enemy or impassable on the other), cut the path short here
            Point adj;
            bool narrow = false, foe_near = false;
            short invalid;

            if ( !Dir2XY( pnode->x, pnode->y, TurnLeft( (Direction)dir ), adj ) ) {
              if ( !(TerrainTypes(adj.x,adj.y) & unit->Terrain()) ) narrow = true;
              if ( (u = GetUnit(adj.x,adj.y)) && (u->Owner() != unit->Owner()) ) foe_near = true;
            } else narrow = true;

            invalid = Dir2XY( pnode->x, pnode->y, TurnRight( (Direction)dir ), adj );
            if ( (foe_near && (invalid
                        || !(TerrainTypes(adj.x, adj.y) & unit->Terrain())
                        || ((u = GetUnit(adj.x,adj.y)) && (u->Owner() != unit->Owner()))) ) ||
                 (narrow && !invalid && (u = GetUnit(adj.x,adj.y)) &&
                            (u->Owner() != unit->Owner())) ) m_path[tindex] = 1;
            else {

              pnode2 = new PathNode;
              if ( !pnode2 ) return;

              pnode2->x = p.x;
              pnode2->y = p.y;
              pnode2->cost = newcost;

              // if the new hex is not marked in path yet, mark it and put it in openlist
              if ( m_path[ tindex ] == -1 ) {
                m_path[ tindex ] = 1;
                openlist.AddTail( pnode2 );
              } else {
                // try to find the node in openlist and replace it if new cost is smaller
                walk = static_cast<PathNode *>( openlist.Head() );
                while ( walk ) {
                  if ( (walk->x == pnode2->x) && (walk->y == pnode2->y) ) {
                    // found the node. shall we replace it?
                    if ( walk->cost > pnode2->cost ) {
                      walk->Remove();
                      delete walk;
                      openlist.AddTail( pnode2 );
                    } else delete pnode2;
                    break;
                  } else walk = static_cast<PathNode *>( walk->Next() );
                }
              }
            }
          }
        }
      }
    }   // end for( dirs )
    delete pnode;
  }     // end while ( !openlist.IsEmpty() )

  // check for potential targets
  Unit *u = static_cast<Unit *>( g_units.Head() );
  while ( u ) {
    if ( unit->CanHit( u ) )
      m_path[ XY2Index( u->Position().x, u->Position().y ) ] = 1;
    u = static_cast<Unit *>( u->Next() );
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::SelectUnit
// DESCRIPTION: Select a unit to move/attack.
// PARAMETERS : u - unit to be selected
// RETURNS    : -
//
// HISTORY
//   24-10-2001 - play sound when selecting a unit
////////////////////////////////////////////////////////////////////////

void Game::SelectUnit( Unit *u ) {
  if ( u->IsReady() ) {
    if ( unit ) DeselectUnit( false );

    unit = u;
    player->SetMode( MODE_BUSY );

    if ( g_display_map ) {
      FogOn();
      CreateMoveMap();
      Draw();
      play_audio( SND_GAM_SELECT, 0 );
      Show();

      SetCursorImage( IMG_CURSOR_SELECT );
      SetCursor( u->Position().x, u->Position().y, true );
    }
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::DeselectUnit
// DESCRIPTION: Deselect the currently active unit.
// PARAMETERS : update - whether to update the display or not (default
//                       value is "true")
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void Game::DeselectUnit( bool display /* = true */ ) {
  player->SetMode( MODE_IDLE );

  if ( g_display_map ) {
    SetCursorImage( IMG_CURSOR_IDLE );

    FogOff();
    if ( display ) {
      Draw();
      Show();

      SetCursor( unit->Position().x, unit->Position().y, true );
    }
  }
  unit = NULL;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::HandleEvent
// DESCRIPTION: Handle key and mouse button events special to the map
//              window.
// PARAMETERS : event - event received by the event handler
// RETURNS    : GUI status
//
// HISTORY
//   03-04-2001 - middle mouse button can bring up building or transport
//                window
//   04-04-2001 - game menu accessible via keyboard
////////////////////////////////////////////////////////////////////////

GUI_Status Game::HandleEvent( const SDL_Event &event ) {
  GUI_Status rc = GUI_OK;

  // check for keyboard commands
  if ( event.type == SDL_KEYDOWN ) {

    switch ( event.key.keysym.sym ) {
    case SDLK_KP1: case SDLK_KP2: case SDLK_KP3:
    case SDLK_KP4: case SDLK_KP6: case SDLK_KP7:
    case SDLK_KP8: case SDLK_KP9:
    case SDLK_LEFT: case SDLK_RIGHT: case SDLK_UP: case SDLK_DOWN:
      MoveCommand( event.key.keysym.sym );
      break;
    case SDLK_SPACE:      // select the unit underneath the cursor
      SelectCommand();
      break;
    case SDLK_e:          // end turn
      rc = EndTurn();
      break;
    case SDLK_i: {        // display unit information
      Unit *u = GetUnit( cursor.x, cursor.y );
      if ( u ) UnitInfo( u );
      break; }
    case SDLK_m:          // open tactical map window
      new TacticalWindow( view );
      break;
    case SDLK_c: {        // show transport or building contents
      Building *b = GetBuilding( cursor.x, cursor.y );
      if ( b ) ContainerContent( NULL, b );
      else {
        Unit *u = GetUnit( cursor.x, cursor.y );
        if ( u && u->IsTransport() )
          ContainerContent( static_cast<Transport *>(u), NULL );
      }
      break; }
    case SDLK_g:         // game menu
      GameMenu();
      break;
    case SDLK_ESCAPE:
      if ( unit ) {
        DeselectUnit();
        break;
      }                  // fall through...
    case SDLK_q:         // quit
      Quit();
      break;
    default:
      break;
    }
  } else if ( event.type == SDL_MOUSEBUTTONDOWN ) {
    Point pos;
    if ( event.button.button == SDL_BUTTON_LEFT ) {
      if ( !Pixel2Hex( event.button.x - x, event.button.y - y, pos ) ) {
        if ( (pos.x == cursor.x) && (pos.y == cursor.y) ) SelectCommand();
        else SetCursor( pos.x, pos.y, true );
      }
    } else if ( !Pixel2Hex( event.button.x - x, event.button.y - y, pos ) ) {
      Unit *u = GetUnit( pos.x, pos.y );
      if ( event.button.button == SDL_BUTTON_RIGHT ) {
        if ( u ) UnitMenu( u );
        else GameMenu();
      } else {		// middle mouse button
        if ( u ) {
          if ( u->IsTransport() ) ContainerContent( static_cast<Transport *>(u), NULL );
        } else if ( GetBuilding( pos.x, pos.y ) )
          ContainerContent( NULL, GetBuilding( pos.x, pos.y ) );
      }
    } else GameMenu();
  }
 
  return rc;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::MoveCommand
// DESCRIPTION: Got a move command from the user. See what he wants to
//              do. Move the cursor or a selected unit.
// PARAMETERS : key - the key code used to give the order
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void Game::MoveCommand( int key ) {
  Direction dir;

  switch ( key ) {
  case SDLK_KP1:  dir = SOUTHWEST; break;
  case SDLK_DOWN:
  case SDLK_KP2:  dir = SOUTH;     break;
  case SDLK_KP3:  dir = SOUTHEAST; break;
  case SDLK_KP7:  dir = NORTHWEST; break;
  case SDLK_UP:
  case SDLK_KP8:  dir = NORTH;     break;
  case SDLK_KP9:  dir = NORTHEAST; break;
  case SDLK_LEFT:
  case SDLK_KP4:  dir = WEST;      break;
  case SDLK_RIGHT:
  case SDLK_KP6:  dir = EAST;      break;
  default: return;
  }

  MoveCursor( dir );
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::SelectCommand
// DESCRIPTION: Got a select command from the user. See what he wants to
//              do. Select/deselect a unit, enter a building, or attack
//              an enemy unit.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void Game::SelectCommand( void ) {
  Unit *u = GetUnit( cursor.x, cursor.y );

  if ( player->Mode() == MODE_BUSY ) {
    if ( u ) {
      if ( u->Owner() == player ) {
        if ( u->IsTransport() &&
             static_cast<Transport *>(u)->Allow( unit ) &&
             (m_path[XY2Index(cursor.x,cursor.y)] != -1) )
          MoveUnit( unit, cursor.x, cursor.y );    // try to move into transport
        else if ( unit != u ) SelectUnit( u );
        else DeselectUnit();
      } else if ( unit->CanHit( u ) ) {   // attack the unit
        RegisterCombat( unit, u );
        DeselectUnit();
      }
    } else MoveUnit( unit, cursor.x, cursor.y );  // try to move there
  } else if ( player->Mode() == MODE_IDLE ) {
    if ( u && (u->Owner() == player) ) SelectUnit( u );
    else {
      Building *b = GetBuilding( cursor.x, cursor.y );
      if ( b ) ContainerContent( NULL, b );
    }
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::ContainerContent
// DESCRIPTION: Open a window to display the content of a transport or
//              building.
// PARAMETERS : transport - unit to look into
//              building  - building to look into
// RETURNS    : -
//
// NOTE       : Only one of transport and building may be given. The
//              other argument must be NULL.
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void Game::ContainerContent( Transport *t, Building *b ) {
  UnitContainer *c;
  const char *name;

  if ( t ) {
    c = t;
    name = t->Name();
  } else {
    c = b;
    name = b->Name();
  }

  if ( unit ) DeselectUnit();

  if ( c->Owner() == player ) new ContainerWindow( t, b, view );
  else new NoteWindow( name, "Access denied!", WIN_CLOSE_ESC, view );
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::UnitInfo
// DESCRIPTION: Display a window with information about a unit. If the
//              current player is not authorised to peek at the unit
//              specifications (e.g. because it's a hostile unit) the
//              information request will fail.
// PARAMETERS : unit - unit to show information about
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void Game::UnitInfo( Unit *unit ) {
  if ( unit->Owner() == player )
    new UnitInfoWindow( unit->Type(), this, view );
  else new NoteWindow( unit->Name(), "Access denied!", WIN_CLOSE_ESC, view );
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::ShowLevelInfo
// DESCRIPTION: Display level information supplied by the creator, if
//              any. If no info was supplied, say so.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void Game::ShowLevelInfo( void ) const {
  const char *msg;

  if ( g_level_info != -1 ) msg = g_messages[g_level_info];
  else msg = "No level information available!";

  new NoteWindow( "Level Information", msg, WIN_CLOSE_ESC, view );
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::ShowBriefing
// DESCRIPTION: Display a window with the mission objectives for the
//              current player. If the mission creator did not supply a
//              briefing, pop up an error.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void Game::ShowBriefing( void ) const {
  if ( player->Briefing() != -1 )
    new MessageWindow( g_mission, g_messages[player->Briefing()], view );
  else new NoteWindow( g_mission, "No mission briefing available!", WIN_CLOSE_ESC, view );
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::ShowDebriefing
// DESCRIPTION: When the mission is over, display a message telling
//              the players whether they won or lost and optionally
//              return to the main menu.
// PARAMETERS : status  - victory, defeat, or draw (see VIC_* defines)
//              restart - whether to return to the main menu
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void Game::ShowDebriefing( unsigned char status, bool restart ) const {

  if ( player->Type() == HUMAN ) {
    const char *msg;
    if ( status == VIC_DRAW ) msg = "This game was a draw!";
    else if ( status == VIC_DEFEAT ) msg = "You have been defeated!";
    else msg = "Congratulations! You are the winner!";

    NoteWindow *note = new NoteWindow( "Mission Debriefing", msg, 0, view );
    if ( restart ) note->button->SetID( GUI_RESTART );
    else {
      note->EventLoop();
      view->CloseWindow( note );
    }
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::GameMenu
// DESCRIPTION: Pop up a MenuWindow with general game options like
//              "End Turn" or "Quit".
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
//   11-03-2001 - added Briefing menu item
//   29-03-2001 - added Abort and Level Info menu items
//   30-03-2001 - added Map menu item
//   11-05-2001 - for PBeM games disable saving
//   21-10-2001 - added video options entry
////////////////////////////////////////////////////////////////////////

void Game::GameMenu( void ) {
  const char *labels[] = { "eEnd Turn", "mMap...", "bBriefing...",
              MENU_ITEM_SEPARATOR, "sSave...", "lLevel Info...",
              MENU_ITEM_SEPARATOR, "vVideo Options...",
              MENU_ITEM_SEPARATOR, "aAbort...", "qQuit...", 0 };
  if ( g_flags & GI_PBEM ) labels[4] = "!Save...";

  new MenuWindow( PROGRAMNAME, labels, this, G_BUTTON_END_TURN, view );
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::UnitMenu
// DESCRIPTION: Pop up a MenuWindow with possible actions for a selected
//              unit.
// PARAMETERS : unit - unit to open window for
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void Game::UnitMenu( Unit *unit ) {

  if ( unit->IsTransport() && (unit->Owner() == player) ) {
    const char *labels[] = { "iInfo...", "cContent...", 0 };

    g_tmp_prv_unit = unit;
    new MenuWindow( unit->Name(), labels, this, G_BUTTON_UNIT_INFO, view );
  } else UnitInfo( unit );
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::BeginReplay
// DESCRIPTION: Prepare for viewing the last turn replay. We must store
//              the current list of units in a safe place and replace
//              them by their "historical" versions.
// PARAMETERS : hunits - list of historical units
//              backup - list to store current unit list in
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void Game::BeginReplay( List &hunits, List &backup ) {
  player->SetMode( MODE_REPLAY );

  Unit *u;
  while ( !g_units.IsEmpty() ) {
    u = static_cast<Unit *>( g_units.RemHead() );
    backup.AddTail( u );
    if ( !u->IsSheltered() ) SetUnit( NULL, u->Position().x, u->Position().y );
  }

  while ( !hunits.IsEmpty() ) {
    u = static_cast<Unit *>( hunits.RemHead() );
    g_units.AddTail( u );
    if ( !u->IsSheltered() ) SetUnit( u, u->Position().x, u->Position().y );
  }

  show = true;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::EndReplay
// DESCRIPTION: End replay mode and initiate the next turn. Remove the
//              historical units and put the real ones back in place.
// PARAMETERS : backup - list containing the real units
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void Game::EndReplay( List &backup ) {
  Unit *u;

  while ( !g_units.IsEmpty() ) {
    u = static_cast<Unit *>( g_units.RemHead() );
    if ( !u->IsSheltered() ) SetUnit( NULL, u->Position().x, u->Position().y );
    delete u;
  }

  while ( !backup.IsEmpty() ) {
    u = static_cast<Unit *>( backup.RemHead() );
    g_units.AddTail( u );
    if ( !u->IsSheltered() ) SetUnit( u, u->Position().x, u->Position().y );
  }
  player->SetMode( MODE_IDLE );
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::RegisterCombat
// DESCRIPTION: File a new combat. Record it for historical replay.
// PARAMETERS : att - attacking unit
//              def - defending unit
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void Game::RegisterCombat( Unit *att, Unit *def ) {
  g_combat.AddTail( new Combat( att, def ) );
  att->Attack( def );
  if ( g_history ) g_history->RecordAttackEvent( *att, *def );
}

////////////////////////////////////////////////////////////////////////
// NAME       : Game::Activate
// DESCRIPTION: The Activate() method gets called whenever a button
//              from the game menu or another associated window (e.g.
//              password confirmation) is activated.
// PARAMETERS : button - pointer to the button that called the function
//              win    - pointer to the window the button belongs to
// RETURNS    : GUI status
//
// HISTORY
//   04-04-2001 - confirm before aborting the current game
////////////////////////////////////////////////////////////////////////

GUI_Status Game::Activate( ButtonWidget *button, Window *win ) {
  GUI_Status rc = GUI_OK;

  switch ( button->ID() ) {
  case G_BUTTON_START_TURN:
    view->CloseWindow( win );
    StartTurn();
    break;

  case G_BUTTON_END_TURN:
    view->CloseWindow( win );
    rc = EndTurn();
    break;
  case G_BUTTON_MAP:
    view->CloseWindow( win );
    new TacticalWindow( view );
    break;
  case G_BUTTON_BRIEFING:
    view->CloseWindow( win );
    ShowBriefing();
    break;
  case G_BUTTON_SAVE:
    view->CloseWindow( win );
    Save( NULL );
    break;
  case G_BUTTON_LEV_INFO:
    view->CloseWindow( win );
    ShowLevelInfo();
    break;
  case G_BUTTON_VIDEO_OPTIONS:
    view->CloseWindow( win );
    new VideoOptionsWindow( view );
    break;
  case G_BUTTON_ABORT: {
    view->CloseWindow( win );
    RequestWindow *req = new RequestWindow( "Return to main menu?",
                                          "yYes", "nNo", 0, view );
    play_audio( SND_GUI_ASK, 0 );
    req->button1->SetID( GUI_RESTART );
    break; }
  case G_BUTTON_QUIT:
    view->CloseWindow( win );
    Quit();
    break;

  case G_BUTTON_UNIT_INFO:
    view->CloseWindow( win );
    UnitInfo( g_tmp_prv_unit );
    break;
  case G_BUTTON_UNIT_CONTENT:
    view->CloseWindow( win );
    ContainerContent( static_cast<Transport *>(g_tmp_prv_unit), NULL );
    break;

  case G_BUTTON_NEW_PW: {     // player defines his password
    PasswordWindow *pwwin = static_cast<PasswordWindow *>(win);
    const char *pw = pwwin->string->String();

    // only accept if player has entered a password
    if ( pw ) {
      CurrentPlayer()->SetPassword( pw );
      pwwin->NewPassword( CurrentPlayer()->Password() );
      pwwin->string->SetTitle( "Confirm password" );
      pwwin->string->SetString( NULL, false );
      pwwin->Draw();
      pwwin->Show();
      pwwin->button->SetID( G_BUTTON_CONFIRM_PW );
    }
    pwwin->string->SetFocus();
    break; }
  case G_BUTTON_CONFIRM_PW:     // player confirms previously given password
  case G_BUTTON_REQUEST_PW: {   // player is asked for password
    PasswordWindow *pwwin = static_cast<PasswordWindow *>(win);
    if ( pwwin->PasswordOk() ) {
      view->CloseWindow( win );
      StartTurn();
    } else {
      play_audio( SND_GUI_ERROR, 0 );
      if ( button->ID() == G_BUTTON_CONFIRM_PW ) {
        CurrentPlayer()->SetPassword( NULL );
        pwwin->button->SetID( G_BUTTON_NEW_PW );
        pwwin->string->SetTitle( "Enter password" );
        pwwin->Draw();
        pwwin->Show();
        pwwin->NewPassword( NULL );
      }
      pwwin->string->SetString( NULL, true );
      pwwin->string->SetFocus();
    }
    break; }
  }

  return rc;
}

