// Crimson Fields -- a game of tactical warfare
// Copyright (C) 2000, 2001 Jens Granseuer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

////////////////////////////////////////////////////////////////////////
// extwindow.cpp
//
// History:
//  03-12-2000 - created
//  02-04-2001 - added TacticalWindow class
//  23-06-2001 - added ProgressWindow class
///////////////////////////////////////////////////////////////////////

#include <string.h>

#include "extwindow.h"
#include "misc.h"
#include "game.h"
#include "globals.h"

////////////////////////////////////////////////////////////////////////
// NAME       : RequestWindow::RequestWindow
// DESCRIPTION: Display a window containing a short message and two
//              button widgets.
// PARAMETERS : msg  - message
//              txt1 - title of first widget; the first character
//                     of the title is not printed, but used as the
//                     keyboard shortcut for this button
//              txt0 - title of second widget; first character is
//                     also used for keyboard control
//              def  - default button (1 or 0)
//              view - pointer to the window's view
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

RequestWindow::RequestWindow( const char *msg, const char *txt1,
       const char *txt0, short def, View *view ) : Window( 0, view ) {
  if ( def != 1 ) def = 0;          // just to be on the safe side...

  // calculate window size and position
  short button_h = sfont->Height() + 8;
  short button_w = MAX( sfont->TextWidth( txt1 ), sfont->TextWidth( txt0 ) ) + 40;
  short left, top;
  unsigned short width = 0, height;
  Rect rect[2];

  // calculate message width and height
  short i = 0, line = 0, num_lines = 1;
  while ( msg[i] != '\0' ) {
    if ( msg[i] == '\n' ) {
      if ( line > width ) width = line;
      line = 0;
      num_lines++;
    } else line += sfont->CharWidth( msg[i] );
    i++;
  }
  if ( line > width ) width = line;

  // get window width and height
  width = MAX( button_w * 2, width ) + 40;
  height = num_lines * (sfont->Height() + 4) + button_h + 20;

  rect[1] = Rect( 5, height - 5 - button_h, button_w, button_h );
  rect[0] = Rect( width - button_w - 5, rect[1].y, button_w, button_h );

  int mx, my;
  SDL_GetMouseState( &mx, &my );
  left = mx - rect[def].x - button_w / 2;
  top = my - height + button_h / 2 + 5;
  if ( left < 0 ) left = 0;
  else if ( left + width >= view->Width() ) left = view->Width() - width;
  if ( top < 0 ) top = 0;
  else if ( top + height >= view->Height() ) top = view->Height() - height;

  // fix the window dimensions
  SetSize( left, top, width, height );

  // create the button widgets
  button1 = new ButtonWidget( GUI_CLOSE, rect[1].x, rect[1].y, rect[1].w, rect[1].h,
                txt1[0], (def == 1) ? WIDGET_DEFAULT : 0, &txt1[1], this );
  button0 = new ButtonWidget( GUI_CLOSE, rect[0].x, rect[0].y, rect[0].w, rect[0].h,
                txt0[0], (def == 0) ? WIDGET_DEFAULT : 0, &txt0[1], this );
  new TextWidget( 2, 5, 5, w - 10, rect[0].y - 10, msg, WIDGET_ALIGN_CENTER,
                  NULL, this );
  Draw();
  Show();
}


////////////////////////////////////////////////////////////////////////
// NAME       : NoteWindow::NoteWindow
// DESCRIPTION: Create a new note window. A NoteWindow is a window
//              containing a short message and a title and is
//              automatically centered on the screen.
// PARAMETERS : title - window title
//              note  - the message, always printed using small font
//              flags - window flags (see window.h for details)
//              view  - pointer to the window's view
// RETURNS    : -
//
// HISTORY
//   29-03-2001 - uses TextWidget instead of painting directly to the
//                surface and therefore supports multi-line messages
////////////////////////////////////////////////////////////////////////

NoteWindow::NoteWindow( const char *title, const char *note,
                        unsigned short flags, View *view ) :
            Window( WIN_CENTER|flags, view ) {
  this->title = title;

  // calculate window dimensions
  Font *usefont = (flags & WIN_FONT_BIG ? LargeFont() : SmallFont() );

  // get number of lines and maximum line length
  unsigned short lines = 1, len = 0, maxlen = 0;
  const char *str = note;

  while ( *str ) {
    if ( *str == '\n' ) {
      if ( len > maxlen ) maxlen = len;
      len = 0;
      lines++;
    } else len += usefont->CharWidth( *str );
    str++;
  }
  if ( len > maxlen ) maxlen = len;

  SetSize( MAX( usefont->TextWidth( title ), maxlen ) + 40,
           usefont->Height() * (lines + 1) + 50 + sfont->Height() );

  textbox = new TextWidget( 0, 5, usefont->Height() + 20, w - 10, (usefont->Height() + 4) * lines,
                            note, WIDGET_ALIGN_CENTER, NULL, this );
  button = new ButtonWidget( GUI_CLOSE, 1, h - sfont->Height() - 11, w - 2, sfont->Height() + 10,
                             'o', WIDGET_DEFAULT, "OK", this );
  Draw();
  Show();
}

////////////////////////////////////////////////////////////////////////
// NAME       : NoteWindow::Draw
// DESCRIPTION: Draw the window.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void NoteWindow::Draw( void ) {
  Font *font = (flags & WIN_FONT_BIG ? LargeFont() : SmallFont() );
  short xoff = (w - font->TextWidth( title )) / 2;

  Window::Draw();
  font->Write( title, this, xoff + 3, 13, ColDark );
  font->Write( title, this, xoff, 10, ColLight );
}


////////////////////////////////////////////////////////////////////////
// NAME       : MessageWindow::MessageWindow
// DESCRIPTION: Create a message window. It will contain a TextScroll
//              and a Button widget.
// PARAMETERS : title - window title (may be NULL)
//              msg   - the message to be displayed in the text widget
//                      (may be NULL)
//              view  - pointer to the window's view
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

MessageWindow::MessageWindow( const char *title, const char *msg, View *view ) :
            Window( WIN_CENTER|WIN_CLOSE_ESC, view ) {
  this->title = title;

  // calculate window dimensions
  SetSize( view->Width() / 2, view->Height() * 3 / 4 );

  button = new ButtonWidget( GUI_CLOSE, 1, h - sfont->Height() - 11, w - 2, sfont->Height() + 10,
                             'o', WIDGET_DEFAULT, "OK", this );

  short boxy = 5;
  if ( title ) boxy += lfont->Height() * 2;
  textscroll = new TextScrollWidget( 1, 5, boxy, w - 10, h - boxy - button->Height() - 15,
                                     msg, 0, NULL, this );
  Draw();
  Show();
}

////////////////////////////////////////////////////////////////////////
// NAME       : MessageWindow::Draw
// DESCRIPTION: Draw the window.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void MessageWindow::Draw( void ) {
  Window::Draw();
  if ( title ) {
    int xoff = (w - lfont->TextWidth( title )) / 2;
    int yoff = (5 + lfont->Height()) / 2;

    lfont->Write( title, this, xoff+3, yoff+3, ColDark );
    lfont->Write( title, this, xoff, yoff, ColLight );
  }
}


////////////////////////////////////////////////////////////////////////
// NAME       : PasswordWindow::PasswordWindow
// DESCRIPTION: Create a window containing a string widget for asking
//              the user for a password.
// PARAMETERS : title - window title
//              msg   - single-line message to be printed (may be NULL)
//              pass  - password to check the user input against; if
//                      this is NULL any input will be accepted
//              view  - pointer to the window's view
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

PasswordWindow::PasswordWindow( const char *title, const char *msg,
                        const char *pass, View *view ) :
            Window( WIN_CENTER, view ) {
  this->title = title;
  password = pass;

  // calculate window dimensions
  unsigned short width = MAX( sfont->TextWidth(title) + 20, sfont->Width() * 9 + 25 ),
                 height = sfont->Height() * 3 + 45;
  if ( msg ) {
    width = MAX( width, sfont->TextWidth(msg) + 40 );
    height += sfont->Height() + 10;
  }
  SetSize( width, height );

  string = new StringWidget( 0, (w - sfont->Width() * 9 - 4) / 2,
                             h - sfont->Height() * 2 - 30,
                             sfont->Width() * 9 + 4, sfont->Height() + 4, 'p',
                             NULL, 7, WIDGET_ALIGN_ABOVE|WIDGET_STR_PASSWORD,
                             msg, this );

  button = new ButtonWidget( GUI_CLOSE, 1, h - sfont->Height() - 9,
                             w - 2, sfont->Height() + 8, 'o', WIDGET_DEFAULT,
                             "OK", this );
  Draw();
  Show();
  string->SetFocus();
}

////////////////////////////////////////////////////////////////////////
// NAME       : PasswordWindow::Draw
// DESCRIPTION: Draw the password window.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void PasswordWindow::Draw( void ) {
  short xoff = (w - sfont->TextWidth( title )) / 2;

  Window::Draw();
  sfont->Write( title, this, xoff + 3, 10, ColDark );
  sfont->Write( title, this, xoff, 7, ColLight );
  DrawBox( Rect( 5, 15 + sfont->Height(),
                 w - 10, h - 2 * sfont->Height() - 30 ), BOX_RECESSED );
}

////////////////////////////////////////////////////////////////////////
// NAME       : PasswordWindow::PasswordOk
// DESCRIPTION: Check the user input against the given password string.
// PARAMETERS : -
// RETURNS    : TRUE on match, FALSE otherwise
//
// HISTORY
////////////////////////////////////////////////////////////////////////

bool PasswordWindow::PasswordOk( void ) const {
  if ( !password ) return true;
  else if ( !string->String() ) return false;
  return !strcmp( password, string->String() );
}

////////////////////////////////////////////////////////////////////////
// NAME       : PasswordWindow::NewPassword
// DESCRIPTION: Set new password to check the user input against.
// PARAMETERS : pass - new password; if this is NULL any user input will
//                     be accepted
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void PasswordWindow::NewPassword( const char *pass ) {
  password = pass;
}


////////////////////////////////////////////////////////////////////////
// NAME       : ProgressWindow::ProgressWindow
// DESCRIPTION: Create a window containing a progress bar display and
//              an 'abort' button.
// PARAMETERS : x     - left edge of window
//              y     - top edge of window
//              w     - window width
//              h     - window height
//              pmin  - value indicating an empty progress bar
//              pmax  - value indicating a filled progress bar
//              flags - window flags (see window.h for details)
//              abort - whether to create a 'Abort' button
//              view  - pointer to the window's view
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

ProgressWindow::ProgressWindow( short x, short y, unsigned short w,
                unsigned short h, short pmin, short pmax,
                unsigned short flags, bool abort, View *view ) :
                Window( x, y, w, h, flags, view ) {
  short bwidth = 0;

  if ( abort ) {
    bwidth = sfont->Width() * 5 + 20;
    button = new ButtonWidget( GUI_CLOSE, w - bwidth - 5, 5, bwidth, h - 10,
                               'a', WIDGET_DEFAULT, "Abort", this );
  } else button = NULL;

  progress = new ProgressWidget( GUI_OK, 5, 5, w - 10 - bwidth, h - 10,
                                 pmin, pmax, 0, NULL, this );
  progress->SetColor( ColLight );

  Draw();
  Show();
}

////////////////////////////////////////////////////////////////////////
// NAME       : ProgressWindow::Aborted
// DESCRIPTION: Check whether the user clicked the 'Abort' button.
// PARAMETERS : -
// RETURNS    : TRUE if button was activated, FALSE otherwise
//
// HISTORY
////////////////////////////////////////////////////////////////////////

bool ProgressWindow::Aborted( void ) {
  GUI_Status peek;
  bool abort = false;

  do { 
    SDL_Event event;
    peek = view->PeekEvent( event );
    if ( (peek != GUI_NONE) &&
         (HandleEvent( event ) == GUI_CLOSE) ) abort = true;
  } while ( peek != GUI_NONE );

  return abort;
}


////////////////////////////////////////////////////////////////////////
// NAME       : MenuWindow::MenuWindow
// DESCRIPTION: Create a new menu window.
// PARAMETERS : title   - menu title
//              labels  - label strings of the menu items; the first
//                        character of each label is not printed but
//                        used as the keyboard equivalent to activate
//                        the respective item; if the key is '!', the
//                        item will not be available for selection
//              hook    - if you give a hook here, it will be
//                        assigned to each menu item so that the hook
//                        will be activated when an item is selected
//              firstid - the menu item ID's will be numbered from
//                        firstid up
//              view    - pointer to the window's view
// RETURNS    : -
//
// HISTORY
//   27-07-2001 - use MenuButtonWidget instead of ButtonWidget
//   28-10-2001 - play sound when opening window
////////////////////////////////////////////////////////////////////////

MenuWindow::MenuWindow( const char *title, const char **labels,
                        ButtonHook *hook, short firstid, View *view ) :
            Window( WIN_CLOSE_ESC|WIN_CLOSE_UNFOCUS, view ) {
  this->title = title;
  this->labels = labels;

  hspacing = 4;
  barheight = hspacing * 2 + 2;		// height of the separator bar

  // calculate window dimensions
  unsigned short maxwidth = 0, width, ypos = 0, titleheight = 0;
  const char **ptr;
  for ( ptr = labels; *ptr; ptr++ ) {
    if ( *ptr != MENU_ITEM_SEPARATOR ) {
      ypos += sfont->Height() + hspacing;
      width = sfont->TextWidth( *ptr + 1 );              // first char of each label
      if ( width > maxwidth ) maxwidth = width;          // is keyboard shortcut
    } else ypos += barheight;
  }

  titleheight = sfont->Height() + barheight;
  width = sfont->TextWidth( title );
  if ( width > maxwidth ) maxwidth = width;

  int mx, my;
  SDL_GetMouseState( &mx, &my );

  Rect win( mx - (maxwidth - 20) / 2, my - titleheight - sfont->Height() / 2,
            maxwidth + 20, ypos + titleheight + 10 );
            
  win.Align( *view );
  SetSize( win.x, win.y, win.w, win.h );


  // create menu items; the first char in each label is the keyboard shortcut
  ypos = 5 + sfont->Height() + barheight;

  for ( ptr = labels; *ptr; ptr++ ) {
    MenuButtonWidget *btn;
    if ( *ptr != MENU_ITEM_SEPARATOR ) {
      btn = new MenuButtonWidget( firstid++, 5, ypos, w - 10, sfont->Height() + hspacing,
                    **ptr, WIDGET_STYLE_MENU|WIDGET_STYLE_NOBORDER|
                           (**ptr == '!' ? WIDGET_DISABLED : 0 ), *ptr + 1, this );
      btn->SetHook( hook );
      ypos += sfont->Height() + hspacing;
    } else ypos += barheight;
  }
  play_audio( SND_GUI_MENU_SHOW, 0 );
  Draw();
  Show();
}

////////////////////////////////////////////////////////////////////////
// NAME       : MenuWindow::Draw
// DESCRIPTION: Draw the window.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void MenuWindow::Draw( void ) {
  short posy = 5 + sfont->Height() + barheight;
  short width = sfont->TextWidth( title );

  Window::Draw();

  sfont->Write( title, this, (w - width) / 2, 5, ColLight );
  FillRect( 5, 5 + sfont->Height() + hspacing, w - 10, 1, Color(CF_COLOR_SHADOW) );
  FillRect( 5, 6 + sfont->Height() + hspacing, w - 10, 1, Color(CF_COLOR_HIGHLIGHT) );

  // draw item separators
  for ( const char **ptr = labels; *ptr; ptr++ ) {
    if ( *ptr == MENU_ITEM_SEPARATOR ) {
      FillRect( 5, posy + hspacing, w - 10, 1, Color(CF_COLOR_SHADOW) );
      FillRect( 5, posy + hspacing + 1, w - 10, 1, Color(CF_COLOR_HIGHLIGHT) );
      posy += barheight;
    } else posy += sfont->Height() + hspacing;
  }
}



////////////////////////////////////////////////////////////////////////
// NAME       : TacticalWindow::TacticalWindow
// DESCRIPTION: Create a window and show an overview map of the level.
// PARAMETERS : view - pointer to the window's view
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

TacticalWindow::TacticalWindow( View *view ) :
        Window( WIN_CLOSE_ESC|WIN_CENTER, view )  {

  unsigned short width = MAX( 22 + Gam->MapWidth(), sfont->Width() * 40 );
  magnify = MIN( (width - 20) / Gam->MapWidth(), (Gam->Height() * 3 / 4) / Gam->MapHeight() );

  // calculate window dimensions
  map.w = Gam->MapWidth() * magnify + 2;
  map.h = Gam->MapHeight() * magnify + magnify/2 + 2;
  map.x = (width - map.w) / 2;
  map.y = 10;

  SetSize( width, map.y * 2 + map.h + sfont->Height() * 3 + 30 );

  new ButtonWidget( GUI_CLOSE, 1, h - sfont->Height() - 9,
                    w - 2, sfont->Height() + 8, 'o',
                    WIDGET_DEFAULT, "OK", this );

  mapbuffer = new Surface();
  if ( mapbuffer->Create( map.w - 2, map.h - 2, DISPLAY_BPP, 0 ) ) return;
  DrawTacticalMap( mapbuffer, 0, 0, magnify );
  CalcViewport();

  Draw();
  Show();
}

////////////////////////////////////////////////////////////////////////
// NAME       : TacticalWindow::~TacticalWindow
// DESCRIPTION: Destroy the window and free its resources.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

TacticalWindow::~TacticalWindow( void ) {
  delete mapbuffer;
}

////////////////////////////////////////////////////////////////////////
// NAME       : TacticalWindow::HandleEvent
// DESCRIPTION: Handle system events. If the mouse is clicked inside the
//              map area, we want the display to jump to the requested
//              place on the map.
// PARAMETERS : event - event received by the event handler
// RETURNS    : GUI status
//
// HISTORY
//   04-04-2001 - added keyboard control of viewport position
//   22-04-2001 - fixed missing display update when viewport changed
////////////////////////////////////////////////////////////////////////

GUI_Status TacticalWindow::HandleEvent( const SDL_Event &event ) {
  GUI_Status rc = GUI_OK;

  if ( (event.type == SDL_MOUSEBUTTONDOWN) &&
       (event.button.button == SDL_BUTTON_LEFT) &&
       map.Contains( event.button.x - LeftEdge(),
                     event.button.y - TopEdge() ) ) {
    Point hex;
    hex.x = (event.button.x - LeftEdge() - map.x - 1) / magnify;
    hex.y = (event.button.y - TopEdge() - map.y - 1 - (hex.x & 1 ? 0 : magnify/2)) / magnify;
    if ( hex.y == Gam->MapHeight() ) hex.y--;

    view->DisableUpdates();
    if ( Gam->HexOnScreen( hex.x, hex.y ) ) Gam->SetCursor( hex.x, hex.y, true );
    else {
      Gam->CenterOnHex( hex.x, hex.y );
      Gam->SetCursor( hex.x, hex.y, true );
    }

    // reblit the overview map and adjust the viewport
    mapbuffer->Blit( this, Rect( 0, 0, mapbuffer->Width(), mapbuffer->Height() ),
                                       map.x + 1, map.y + 1 );
    CalcViewport();
    DrawViewport();
    view->EnableUpdates();
    view->Refresh();
  } else if ( (event.type == SDL_KEYDOWN) &&
           ((event.key.keysym.sym == SDLK_UP) || (event.key.keysym.sym == SDLK_DOWN) ||
            (event.key.keysym.sym == SDLK_LEFT) ||(event.key.keysym.sym == SDLK_RIGHT)) ) {
    short scrx = 0, scry = 0;

    switch ( event.key.keysym.sym ) {
    case SDLK_UP:
      scry = - Gam->Height() / 2;
      break;
    case SDLK_DOWN:
      scry = Gam->Height() / 2;
      break;
    case SDLK_LEFT:
      scrx = - Gam->Width() / 2;
      break;
    case SDLK_RIGHT:
      scrx = Gam->Width() / 2;
      break;
    default:
      break;
    }

    view->DisableUpdates();
    Gam->Scroll( scrx, scry );
    Gam->SetCursor( Gam->MinXHex(-1) + (Gam->MaxXHex(-1,0) - Gam->MinXHex(-1)) / 2,
                    Gam->MinYHex(-1) + (Gam->MaxYHex(-1,0) - Gam->MinYHex(-1)) / 2, true );

    // reblit the overview map and redraw the viewport
    mapbuffer->Blit( this, Rect( 0, 0, mapbuffer->Width(), mapbuffer->Height() ),
                     map.x + 1, map.y + 1 );
    CalcViewport();
    DrawViewport();
    view->EnableUpdates();
    view->Refresh();
  } else rc = Window::HandleEvent( event );
  return rc;
}

////////////////////////////////////////////////////////////////////////
// NAME       : TacticalWindow::Draw
// DESCRIPTION: Draw the window.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void TacticalWindow::Draw( void ) {
  Window::Draw();
  DrawBox( map, BOX_RECESSED );

  // blit mapbuffer to window surface
  mapbuffer->Blit( this, Rect( 0, 0, mapbuffer->Width(), mapbuffer->Height() ),
                         map.x + 1, map.y + 1 );
  DrawViewport();

  // show unit and building count
  Color col, colbg;
  unsigned short linewidth = sfont->Width() * 22, lx, l1y, l2y;
  char numbuf[4];

  unsigned short bs1 = 0, bs2 = 0, bstotal = 0;
  Building *b = static_cast<Building *>( Gam->BuildingsList()->Head() );
  while ( b ) {
    if ( b->Owner() ) {
      if ( b->Owner()->ID() == PLAYER_ONE ) bs1++;
      else if ( b->Owner()->ID() == PLAYER_TWO ) bs2++;
    }
    bstotal++;
    b = static_cast<Building *>( b->Next() );
  }


  lx = MIN( (w - linewidth - 10) / 2, map.x + 5 );
  l1y = map.y + map.h + 15;
  l2y = l1y + sfont->Height() + 3;
  DrawBox( Rect( lx - 5, l1y - 5, w - 2*lx + 10, sfont->Height() * 2 + 13 ),
           BOX_RECESSED );

  lx = (w - linewidth - 10) / 2;
  sfont->Write( "Units", this, lx, l1y );
  sfont->Write( "Buildings", this, lx, l2y );

  lx += sfont->Width() * 10;
  Gam->GetPlayer(PLAYER_ONE)->Colors( col, colbg );
  sfont->Write( itoa( Gam->GetPlayer(PLAYER_ONE)->Units(0), numbuf ), this,
                lx, l1y, col );
  sfont->Write( itoa( bs1, numbuf ), this, lx, l2y, col );

  lx += sfont->Width() * 4;
  Gam->GetPlayer(PLAYER_TWO)->Colors( col, colbg );
  sfont->Write( itoa( Gam->GetPlayer(PLAYER_TWO)->Units(0), numbuf ), this,
                lx, l1y, col );
  sfont->Write( itoa( bs2, numbuf ), this, lx, l2y, col );

  lx += sfont->Width() * 4;
  sfont->Write( itoa( Gam->UnitsList()->CountNodes(), numbuf ), this, lx, l1y );
  sfont->Write( itoa( bstotal, numbuf ), this, lx, l2y );
}

////////////////////////////////////////////////////////////////////////
// NAME       : TacticalWindow::DrawTacticalMap
// DESCRIPTION: Draw a small map giving an overview of the entire map
//              including units and buildings.
// PARAMETERS : dest    - destination surface
//              x       - starting position x on surface
//              y       - starting position y on surface
//              magnify - size of one hex in pixels (>= 2)
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void TacticalWindow::DrawTacticalMap( Surface *dest, short x, short y, unsigned char magnify ) const {
  Color color;
  unsigned short index = 0;
  short yoff;

  dest->FillRect( 0, 0, dest->Width(), dest->Height(), Color(CF_COLOR_BACKGROUND) );

  for ( int py = 0; py < Gam->MapHeight(); py++ ) {
    for ( int px = 0; px < Gam->MapWidth(); px++ ) {
      if ( px & 1 ) yoff = 0;
      else yoff = magnify / 2;

      color = Color( Gam->HexColor( index++ ) );
      dest->FillRect( x + magnify * px, y + magnify * py + yoff, magnify, magnify, color );
    }
  }

  Color pcols[2];
  Gam->GetPlayer(PLAYER_ONE)->Colors( pcols[PLAYER_ONE], color );
  Gam->GetPlayer(PLAYER_TWO)->Colors( pcols[PLAYER_TWO], color );

  Unit *u = static_cast<Unit *>( Gam->UnitsList()->Head() );
  while ( u ) {
    if ( !u->IsSheltered() ) {
      const Point &pos = u->Position();
      dest->FillRect( x + magnify * pos.x,
                      y + magnify * pos.y + ((pos.x & 1)^1) * (magnify / 2),
                      magnify, magnify, pcols[u->Owner()->ID()] );
    }
    u = static_cast<Unit *>( u->Next() );
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : TacticalWindow::CalcViewport
// DESCRIPTION: Determine which part of the map is currently displayed
//              on the map window and set the viewport rect accordingly.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void TacticalWindow::CalcViewport( void ) {
  viewport.x = Gam->MinXHex(-1) * magnify;
  viewport.y = Gam->MinYHex(-1) * magnify;
  viewport.w = (Gam->MaxXHex(-1,0) + 1) * magnify - viewport.x;
  viewport.h = (Gam->MaxYHex(-1,0) + 1) * magnify - viewport.y + (magnify/2);
}

////////////////////////////////////////////////////////////////////////
// NAME       : TacticalWindow::DrawViewport
// DESCRIPTION: Draw a border around the currently visible part of the
//              map.
// PARAMETERS : -
// RETURNS    : -
//
// HISTORY
//   17-08-2001 - use alpha shading to mark the viewport
////////////////////////////////////////////////////////////////////////

void TacticalWindow::DrawViewport( void ) {
  if ( (viewport.w < map.w - 2) || (viewport.h < map.h - 2) ) {
    // darken the entire map, the light up the visible part
    FillRectAlpha( map.x + 1, map.y + 1, map.w - 2, map.h - 2,
                   Color(CF_COLOR_BLACK), 96 );
    mapbuffer->Blit( this, viewport, viewport.x + map.x + 1, viewport.y + map.y + 1 );
  }
}

