// Crimson Fields -- a game of tactical warfare
// Copyright (C) 2000, 2001 Jens Granseuer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

////////////////////////////////////////////////////////////////////////
// building.cpp
//
// History:
//  13-12-2000 - created
////////////////////////////////////////////////////////////////////////

#include "SDL_endian.h"

#include "building.h"


////////////////////////////////////////////////////////////////////////
// NAME       : Building::Building
// DESCRIPTION: Load a building definition from a file.
// PARAMETERS : file    - data file descriptor
//              players - pointer to the array of players
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

Building::Building( SDL_RWops *file, Player **players ) : MapObject( MO_BUILDING ) {
  unsigned char pid;

  b_id = SDL_ReadLE16( file );
  b_pos.x = SDL_ReadLE16( file );
  b_pos.y = SDL_ReadLE16( file );
  b_flags = SDL_ReadLE16( file );
  b_crystals = SDL_ReadLE16( file );
  b_crystalstore = SDL_ReadLE16( file );
  b_uprod = SDL_ReadLE32( file );

  SDL_RWread( file, &b_cprod, sizeof(unsigned char), 1 );
  SDL_RWread( file, &pid, sizeof(unsigned char), 1 );
  SDL_RWread( file, b_name, sizeof(char), 20 );

  if ( pid == PLAYER_NONE ) b_player = NULL;
  else b_player = players[pid];
}

////////////////////////////////////////////////////////////////////////
// NAME       : Building::Save
// DESCRIPTION: Save the building to a file.
// PARAMETERS : file - save file descriptor
// RETURNS    : 0 on success, -1 on error
//
// HISTORY
////////////////////////////////////////////////////////////////////////

int Building::Save( SDL_RWops *file ) const {
  unsigned char pid;

  if ( b_player ) pid = b_player->ID();
  else pid = PLAYER_NONE;

  SDL_WriteLE16( file, b_id );
  SDL_WriteLE16( file, b_pos.x );
  SDL_WriteLE16( file, b_pos.y );
  SDL_WriteLE16( file, b_flags );
  SDL_WriteLE16( file, b_crystals );
  SDL_WriteLE16( file, b_crystalstore );
  SDL_WriteLE32( file, b_uprod );

  SDL_RWwrite( file, &b_cprod, sizeof(unsigned char), 1 );
  SDL_RWwrite( file, &pid, sizeof(unsigned char), 1 );
  SDL_RWwrite( file, b_name, sizeof(char), 20 );
  return 0;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Building::Allow
// DESCRIPTION: Check whether a unit is allowed to enter the building.
// PARAMETERS : unit - unit to check permission for
// RETURNS    : true if unit may enter, false if not
//
// HISTORY
////////////////////////////////////////////////////////////////////////

bool Building::Allow( const Unit *unit ) const {
  if ( (unit->Owner() == b_player) || unit->IsConquer() )
    return UnitContainer::Allow( unit );
  return false;
}

////////////////////////////////////////////////////////////////////////
// NAME       : Building::SetCrystals
// DESCRIPTION: Set crystals in stock to a specified value. This can
//              never be higher than b_crystalstore.
// PARAMETERS : crystals - requested amount of crystals
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void Building::SetCrystals( unsigned short crystals ) {
  b_crystals = MIN( crystals, b_crystalstore );
}

////////////////////////////////////////////////////////////////////////
// NAME       : Building::SetOwner
// DESCRIPTION: Set the controller of the building. This also makes all
//              units inside change sides.
// PARAMETERS : player - new controller
// RETURNS    : -
//
// HISTORY
////////////////////////////////////////////////////////////////////////

void Building::SetOwner( Player *player ) {
  b_player = player;

  // convert all units inside to their new master
  UCNode *n = static_cast<UCNode *>( uc_units.Head() );
  while ( n ) {
    n->uc_unit->SetOwner( player );
    n->uc_unit->SetFlags( U_DONE );
    n = static_cast<UCNode *>( n->Next() );
  }
}

