/*
   ISO 3166 Country Codes program
   --------------------------------------------------------------------
   Country Codes

   Copyright (C) 1999, 2000 Diego Javier Grigna <diego@grigna.com>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "common.h"

/* Function prototypes */
static void usage( void);
static void iso3166_print_heading( void);
static void iso3166_print_cc( int index);
static void iso3166_show_all( void);
static void iso3166_show_one( char *code);
static void domain_name_print( char *str);

/* Global variables */

/* Domains */
typedef struct {
       const char *str;        /* domain name   */
       const char *expl;       /* Explanation   */
} DOM;

static DOM topdom[] = {
{ "COM" , "Commercial entity"                    ,} ,
{ "EDU" , "Educational institutions"             ,} ,
{ "NET" , "Network providers"                    ,} ,
{ "ORG" , "Organizations"                        ,} ,
{ "INT" , "International treaties, or databases" ,} ,
{ "GOV" , "Government"                           ,} ,
{ "MIL" , "Military"                             ,} ,
{ "FIRM", "Businesses or firms"                  ,} ,
{ "SHOP", "Businesses offering goods to purchase",} ,
{ "WEB" , "World Wide Web entities"              ,} ,
{ "ARTS", "Cultural and entertainment entities"  ,} ,
{ "REC" , "Recreation/entertainment entities"    ,} ,
{ "INFO", "Information services"                 ,} ,
{ "NOM" , "Individual or personal nomenclature"  ,} ,
{ " "   , " "                                    ,} ,
};

/* Sub domains */
static DOM subdom[] = {
{ "AC"    , "Academic establishments and organizations",} ,
{ "CO"    , "Companies or commercial organization"     ,} ,
{ "CS"    , "Computer science"                         ,} ,
{ "GW"    , "Gateway"                                  ,} ,
{ "GOPHER", "Internet Gopher"                          ,} ,
{ "FTP"   , "File Transfer Protocol"                   ,} ,
{ "LTD"   , "Private limited"                          ,} ,
{ "MAIL"  , "Mail services SMTP"                       ,} ,
{ "NNTP"  , "Network News Transfer Protocol"           ,} ,
{ "NEWS"  , "Network News Transfer Protocol"           ,} ,
{ "NS"    , "Name server"                              ,} ,
{ "NS1"   , "Name server"                              ,} ,
{ "NS2"   , "Name server"                              ,} ,
{ "PLC"   , "Public limited"                           ,} ,
{ "POP"   , "Post Office Protocol"                     ,} ,
{ "POP3"  , "Post Office Protocol, version 3"          ,} ,
{ "PROXY" , "Proxy server"                             ,} ,
{ "SMTP"  , "Simple Mail Transfer Protocol"            ,} ,
{ "SSH"   , "Secure Shell service"                     ,} ,
{ "WEB"   , "World Wide Web HTTP"                      ,} ,
{ "WWW"   , "World Wide Web HTTP"                      ,} ,
{ "WWW2"  , "World Wide Web HTTP"                      ,} ,
{ " "     , " "                                        ,} ,
};

/* iso 3166 Country codes struct */
typedef struct {
       const char *country;       /* Country name      */
       const char codes2l[ 2];    /* 2 letters codes   */
       const char codes3l[ 3];    /* 3 letters codes   */
       unsigned short int number; /* Number of country */
} ISO3166CCS;

static ISO3166CCS isoccs[] = {
{ "Afghanistan, Islamic State of"               , "af", "afg",    4,} ,
{ "Albania"                                     , "al", "alb",    8,} ,
{ "Algeria"                                     , "dz", "dza",   12,} ,
{ "American Samoa"                              , "as", "asm",   16,} ,
{ "Andorra, Pricipality of"                     , "ad", "and",   20,} ,
{ "Angola, Republic of"                         , "ao", "ago",   24,} ,
{ "Anguilla"                                    , "ai", "aia",  660,} ,
{ "Antarctica"                                  , "aq", "ata",   10,} ,
{ "Antigua and Barbuda"                         , "ag", "atg",   28,} ,
{ "Argentina"                                   , "ar", "arg",   32,} ,
{ "Armenia"                                     , "am", "arm",   51,} ,
{ "Aruba"                                       , "aw", "abw",  533,} ,
{ "Australia"                                   , "au", "aus",   36,} ,
{ "Austria"                                     , "at", "aut",   40,} ,
{ "Azerbaijan"                                  , "az", "aze",   31,} ,
{ "Bahamas"                                     , "bs", "bhs",   44,} ,
{ "Bahrain"                                     , "bh", "bhr",   48,} ,
{ "Bailiwicks of Guernsey"                      , "gg", "   ",    0,} ,
{ "Bangladesh"                                  , "bd", "bgd",   50,} ,
{ "Barbados"                                    , "bb", "brb",   52,} ,
{ "Belarus"                                     , "by", "blr",  112,} ,
{ "Belgium"                                     , "be", "bel",   56,} ,
{ "Belize"                                      , "bz", "blz",   84,} ,
{ "Benin"                                       , "bj", "ben",  204,} ,
{ "Bermuda"                                     , "bm", "bmu",   60,} ,
{ "Bhutan"                                      , "bt", "btn",   64,} ,
{ "Bolivia"                                     , "bo", "bol",   68,} ,
{ "Bosnia and Herzegowina"                      , "ba", "bih",   70,} ,
{ "Botswana"                                    , "bw", "bwa",   72,} ,
{ "Bouvet Island"                               , "bv", "bvt",   74,} ,
{ "Brazil"                                      , "br", "bra",   76,} ,
{ "British Indian Ocean Territory"              , "io", "iot",   86,} ,
{ "Brunei Darussalam"                           , "bn", "brn",   96,} ,
{ "Bulgaria"                                    , "bg", "bgr",  100,} ,
{ "Burkina Faso"                                , "bf", "bfa",  854,} ,
{ "Burundi"                                     , "bi", "bdi",  108,} ,
{ "Cambodia, Kingdom of"                        , "kh", "khm",  116,} ,
{ "Cameroon"                                    , "cm", "cmr",  120,} ,
{ "Canada"                                      , "ca", "can",  124,} ,
{ "Cape Verde"                                  , "cv", "cpv",  132,} ,
{ "Cayman Islands"                              , "ky", "cym",  136,} ,
{ "Central African Republic"                    , "cf", "caf",  140,} ,
{ "Chad"                                        , "td", "tcd",  148,} ,
{ "Chile"                                       , "cl", "chl",  152,} ,
{ "China"                                       , "cn", "chn",  156,} ,
{ "Christmas Island"                            , "cx", "cxr",  162,} ,
{ "Cocos (Keeling) Islands"                     , "cc", "cck",  166,} ,
{ "Colombia"                                    , "co", "col",  170,} ,
{ "Comoros"                                     , "km", "com",  174,} ,
{ "Congo"                                       , "cg", "cog",  178,} ,
{ "Congo, The Democratic Republic of the"       , "cd", "cod",  180,} ,
{ "Cook Islands"                                , "ck", "cok",  184,} ,
{ "Costa Rica"                                  , "cr", "cri",  188,} ,
{ "Cote D'Ivoire (Ivory Coast)"                 , "ci", "civ",  384,} ,
{ "Croatia (Hrvatska)"                          , "hr", "hrv",  191,} ,
{ "Cuba"                                        , "cu", "cub",  192,} ,
{ "Cyprus"                                      , "cy", "cyp",  196,} ,
{ "Czech Republic"                              , "cz", "cze",  203,} ,
{ "Czechoslovakia (Officially deleted)"         , "cs", "   ",    0,} ,
{ "Denmark"                                     , "dk", "dnk",  208,} ,
{ "Djibouti"                                    , "dj", "dji",  262,} ,
{ "Dominica"                                    , "dm", "dma",  212,} ,
{ "Dominican Republic"                          , "do", "dom",  214,} ,
{ "East Timor"                                  , "tp", "tmp",  626,} ,
{ "Ecuador"                                     , "ec", "ecu",  218,} ,
{ "Egypt"                                       , "eg", "egy",  818,} ,
{ "El Salvador"                                 , "sv", "slv",  222,} ,
{ "Equatorial Guinea"                           , "gq", "gnq",  226,} ,
{ "Eritrea"                                     , "er", "eri",  232,} ,
{ "Estonia"                                     , "ee", "est",  233,} ,
{ "Ethiopia"                                    , "et", "eth",  231,} ,
{ "Falkland Islands (Malvinas)"                 , "fk", "flk",  238,} ,
{ "Faroe Islands"                               , "fo", "fro",  234,} ,
{ "Fiji"                                        , "fj", "fji",  242,} ,
{ "Finland"                                     , "fi", "fin",  246,} ,
{ "France"                                      , "fr", "fra",  250,} ,
{ "France, metropolitan"                        , "fx", "fxx",  249,} ,
{ "French Guiana"                               , "gf", "guf",  254,} ,
{ "French Polynesia"                            , "pf", "pyf",  258,} ,
{ "French Southern Territories"                 , "tf", "atf",  260,} ,
{ "Gabon"                                       , "ga", "gab",  266,} ,
{ "Gambia"                                      , "gm", "gmb",  270,} ,
{ "Georgia"                                     , "ge", "geo",  268,} ,
{ "Germany"                                     , "de", "deu",  276,} ,
{ "Ghana"                                       , "gh", "gha",  288,} ,
{ "Gibraltar"                                   , "gi", "gib",  292,} ,
{ "Greece"                                      , "gr", "grc",  300,} ,
{ "Greenland"                                   , "gl", "grl",  304,} ,
{ "Great Britain"                               , "gb", "gbr",  826,} ,
{ "Great Britain (iso 3166 code is gb)"         , "uk", "gbr",  826,} ,
{ "Grenada"                                     , "gd", "grd",  308,} ,
{ "Guadeloupe"                                  , "gp", "glp",  312,} ,
{ "Guam"                                        , "gu", "gum",  316,} ,
{ "Guatemala"                                   , "gt", "gtm",  320,} ,
{ "Guinea"                                      , "gn", "gin",  324,} ,
{ "Guinea-Bissau"                               , "gw", "gnb",  624,} ,
{ "Guyana"                                      , "gy", "guy",  328,} ,
{ "Haiti"                                       , "ht", "hti",  332,} ,
{ "Heard and MC Donald Islands"                 , "hm", "hmd",  334,} ,
{ "Holy See (Vatican City State)"               , "va", "vat",  336,} ,
{ "Honduras"                                    , "hn", "hnd",  340,} ,
{ "Hong kong"                                   , "hk", "hkg",  344,} ,
{ "Hungary"                                     , "hu", "hun",  348,} ,
{ "Iceland"                                     , "is", "isl",  352,} ,
{ "India"                                       , "in", "ind",  356,} ,
{ "Indonesia"                                   , "id", "idn",  360,} ,
{ "Iran, Islamic Republic of"                   , "ir", "irn",  364,} ,
{ "Iraq"                                        , "iq", "irq",  368,} ,
{ "Ireland"                                     , "ie", "irl",  372,} ,
{ "Isle of Man"                                 , "im", "   ",    0,} ,
{ "Israel"                                      , "il", "isr",  376,} ,
{ "Italy"                                       , "it", "ita",  380,} ,
{ "Jamaica"                                     , "jm", "jam",  388,} ,
{ "Japan"                                       , "jp", "jpn",  392,} ,
{ "Jersey"                                      , "je", "   ",    0,} ,
{ "Jordan"                                      , "jo", "jor",  400,} ,
{ "Kazakhstan"                                  , "kz", "kaz",  398,} ,
{ "Kenya"                                       , "ke", "ken",  404,} ,
{ "Kiribati"                                    , "ki", "kir",  296,} ,
{ "Korea, Democratic People's Republic of"      , "kp", "prk",  408,} ,
{ "Korea, Republic of"                          , "kr", "kor",  410,} ,
{ "Kuwait"                                      , "kw", "kwt",  414,} ,
{ "Kyrgyzstan (Kyrgyz Republic)"                , "kg", "kgz",  417,} ,
{ "Lao People's Democratic Republic"            , "la", "lao",  418,} ,
{ "Latvia"                                      , "lv", "lva",  428,} ,
{ "Lebanon"                                     , "lb", "lbn",  422,} ,
{ "Lesotho"                                     , "ls", "lso",  426,} ,
{ "Liberia"                                     , "lr", "lbr",  430,} ,
{ "Libyan Arab Jamahiriya"                      , "ly", "lby",  434,} ,
{ "Liechtenstein"                               , "li", "lie",  438,} ,
{ "Lithuania"                                   , "lt", "ltu",  440,} ,
{ "Luxembourg"                                  , "lu", "lux",  442,} ,
{ "Macau"                                       , "mo", "mac",  446,} ,
{ "Macedonia, the former Yugoslav Republic of"  , "mk", "mkd",  807,} ,
{ "Madagascar, Republic of"                     , "mg", "mdg",  450,} ,
{ "Malawi"                                      , "mw", "mwi",  454,} ,
{ "Malaysia"                                    , "my", "mys",  458,} ,
{ "Maldives"                                    , "mv", "mdv",  462,} ,
{ "Mali"                                        , "ml", "mli",  466,} ,
{ "Malta"                                       , "mt", "mlt",  470,} ,
{ "Marshall Islands"                            , "mh", "mhl",  584,} ,
{ "Martinique"                                  , "mq", "mtq",  474,} ,
{ "Mauritania"                                  , "mr", "mrt",  478,} ,
{ "Mauritius"                                   , "mu", "mus",  480,} ,
{ "Mayotte"                                     , "yt", "myt",  175,} ,
{ "Mexico"                                      , "mx", "mex",  484,} ,
{ "Micronesia, Federated States of"             , "fm", "fsm",  583,} ,
{ "Moldova, Republic of"                        , "md", "mda",  498,} ,
{ "Monaco"                                      , "mc", "mco",  492,} ,
{ "Mongolia"                                    , "mn", "mng",  496,} ,
{ "Montserrat"                                  , "ms", "msr",  500,} ,
{ "Morocco"                                     , "ma", "mar",  504,} ,
{ "Mozambique"                                  , "mz", "moz",  508,} ,
{ "Myanmar"                                     , "mm", "mmr",  104,} ,
{ "Namibia"                                     , "na", "nam",  516,} ,
{ "Nauru"                                       , "nr", "nru",  520,} ,
{ "Nepal"                                       , "np", "npl",  524,} ,
{ "Netherlands"                                 , "nl", "nld",  528,} ,
{ "Netherlands Antilles"                        , "an", "ant",  530,} ,
{ "Neutral Zone (Officially deleted)"           , "nt", "   ",    0,} ,
{ "New Caledonia"                               , "nc", "ncl",  540,} ,
{ "New Zealand"                                 , "nz", "nzl",  554,} ,
{ "Nicaragua"                                   , "ni", "nic",  558,} ,
{ "Niger"                                       , "ne", "ner",  562,} ,
{ "Nigeria"                                     , "ng", "nga",  566,} ,
{ "Niue"                                        , "nu", "niu",  570,} ,
{ "Norfolk Island"                              , "nf", "nfk",  574,} ,
{ "Northern Mariana Islands"                    , "mp", "mnp",  580,} ,
{ "Norway"                                      , "no", "nor",  578,} ,
{ "Oman"                                        , "om", "omn",  512,} ,
{ "Pakistan"                                    , "pk", "pak",  586,} ,
{ "Palau"                                       , "pw", "plw",  585,} ,
{ "Panama"                                      , "pa", "pan",  591,} ,
{ "Papua New Guinea"                            , "pg", "png",  598,} ,
{ "Paraguay"                                    , "py", "pry",  600,} ,
{ "Peru"                                        , "pe", "per",  604,} ,
{ "Philippines"                                 , "ph", "phl",  608,} ,
{ "Pitcairn"                                    , "pn", "pcn",  612,} ,
{ "Poland"                                      , "pl", "pol",  616,} ,
{ "Portugal"                                    , "pt", "prt",  620,} ,
{ "Puerto Rico"                                 , "pr", "pri",  630,} ,
{ "Qatar"                                       , "qa", "qat",  634,} ,
{ "Reunion"                                     , "re", "reu",  638,} ,
{ "Romania"                                     , "ro", "rom",  642,} ,
{ "Russian Federation"                          , "ru", "rus",  643,} ,
{ "Rwanda"                                      , "rw", "rwa",  646,} ,
{ "Saint Kitts and Nevis"                       , "kn", "kna",  659,} ,
{ "Saint Lucia"                                 , "lc", "lca",  662,} ,
{ "Saint Vincent and the Grenadines"            , "vc", "vct",  670,} ,
{ "Samoa"                                       , "ws", "wsm",  882,} ,
{ "San Marino"                                  , "sm", "smr",  674,} ,
{ "Sao Tome and Principe"                       , "st", "stp",  678,} ,
{ "Saudi Arabia"                                , "sa", "sau",  682,} ,
{ "Senegal"                                     , "sn", "sen",  686,} ,
{ "Seychelles"                                  , "sc", "syc",  690,} ,
{ "Sierra Leone"                                , "sl", "sle",  694,} ,
{ "Singapore"                                   , "sg", "sgp",  702,} ,
{ "Slovakia (Slovak Republic)"                  , "sk", "svk",  703,} ,
{ "Slovenia"                                    , "si", "svn",  705,} ,
{ "Solomon Islands"                             , "sb", "slb",   90,} ,
{ "Somalia"                                     , "so", "som",  706,} ,
{ "South Africa"                                , "za", "zaf",  710,} ,
{ "South Georgia and the South Sandwich Islands", "gs", "sgs",  239,} ,
{ "Soviet Union (See Russian Federation)"       , "su", "   ",    0,} ,
{ "Spain"                                       , "es", "esp",  724,} ,
{ "Sri Lanka"                                   , "lk", "lka",  144,} ,
{ "St. Helena"                                  , "sh", "shn",  654,} ,
{ "St. Pierre and Miquelon"                     , "pm", "spm",  666,} ,
{ "Sudan"                                       , "sd", "sdn",  736,} ,
{ "Suriname"                                    , "sr", "sur",  740,} ,
{ "Svalbard and Jan Mayen Islands"              , "sj", "sjm",  744,} ,
{ "Swaziland"                                   , "sz", "swz",  748,} ,
{ "Sweden"                                      , "se", "swe",  752,} ,
{ "Switzerland"                                 , "ch", "che",  756,} ,
{ "Syrian Arab Republic"                        , "sy", "syr",  760,} ,
{ "Taiwan, Province of China"                   , "tw", "twn",  158,} ,
{ "Tajikistan"                                  , "tj", "tjk",  762,} ,
{ "Tanzania, United Republic of"                , "tz", "tza",  834,} ,
{ "Thailand"                                    , "th", "tha",  764,} ,
{ "Togo"                                        , "tg", "tgo",  768,} ,
{ "Tokelau"                                     , "tk", "tkl",  772,} ,
{ "Tonga"                                       , "to", "ton",  776,} ,
{ "Trinidad and Tobago"                         , "tt", "tto",  780,} ,
{ "Tunisia"                                     , "tn", "tun",  788,} ,
{ "Turkey"                                      , "tr", "tur",  792,} ,
{ "Turkmenistan"                                , "tm", "tkm",  795,} ,
{ "Turks and Caicos Islands"                    , "tc", "tca",  796,} ,
{ "Tuvalu"                                      , "tv", "tuv",  798,} ,
{ "Uganda"                                      , "ug", "uga",  800,} ,
{ "Ukraine"                                     , "ua", "ukr",  804,} ,
{ "United Arab Emirates"                        , "ae", "are",  784,} ,
{ "United Kingdom"                              , "gb", "gbr",  826,} ,
{ "United Kingdom (iso 3166 code is gb)"        , "uk", "gbr",  826,} ,
{ "United States"                               , "us", "usa",  840,} ,
{ "United States minor outlying islands"        , "um", "umi",  581,} ,
{ "Uruguay"                                     , "uy", "ury",  858,} ,
{ "Uzbekistan"                                  , "uz", "uzb",  860,} ,
{ "Vanuatu"                                     , "vu", "vut",  548,} ,
{ "Venezuela"                                   , "ve", "ven",  862,} ,
{ "Viet Nam"                                    , "vn", "vnm",  704,} ,
{ "Virgin Islands (British)"                    , "vg", "vgb",   92,} ,
{ "Virgin Islands (U.S.)"                       , "vi", "vir",  850,} ,
{ "Wallis and Futuna Islands"                   , "wf", "wlf",  876,} ,
{ "Western Sahara"                              , "eh", "esh",  732,} ,
{ "Yemen"                                       , "ye", "yem",  887,} ,
{ "Yugoslavia"                                  , "yu", "yug",  891,} ,
{ "Zaire (see country number 180)"              , "zr", "   ",    0,} ,
{ "Zambia"                                      , "zm", "zmb",  894,} ,
{ "Zimbabwe"                                    , "zw", "zwe",  716,} ,
{ " "                                           , "  ", "   ",    0,} ,
};

/* Search types */
#define ISO_SEARCH_COUNTRY 0
#define ISO_SEARCH_2L      1
#define ISO_SEARCH_3L      2
#define ISO_SEARCH_NUMBER  3

int main( int argc, char *argv[])
{
 int domain_flag = 0;
 int ch;

 progname = cc_basename( strdup( argv[ 0]));

 if( argc < 2)
     usage();

 while(( ch = getopt( argc, argv, "ad:")) != EOF){
         switch( ch) {
                case 'a': iso3166_show_all();
                          break;
                case 'd': if( optarg == NULL)
                              usage();
                          domain_name_print( optarg);
                          domain_flag = 1;
                          break;
                default : usage();
         }
 }

 if( !domain_flag && optind >= argc)
     usage();

 if( !domain_flag)
     iso3166_print_heading();

 while( optind < argc)
        if( domain_flag)
            domain_name_print( argv[ optind++]);
        else
            iso3166_show_one( argv[ optind++]);

 putchar( '\n');

 return 0;
}

static void usage( void)
{
 fprintf( stderr, "\nCountry Codes Release %s - ", COCO_VERSION);
 fprintf( stderr, "ISO 3166 Country Codes\n\n");
 fprintf( stderr, "Copyright (C) 1999, 2000 Diego J. Grigna "
                  "(diego@grigna.com)\n");
 fprintf( stderr, "This program is free software, covered by the "
                  "GNU General Public License.\n\n");
 fprintf( stderr, "Usage:\n%s [-a] [[code]|[country]|[number]...] "
                  "[-d <domain name>]\n\n", progname);
 fprintf( stderr, "   -a   Show all codes.\n");
 fprintf( stderr, " [code]|[country]|[number]\n");
 fprintf( stderr, "        You could enter a 2 letter (or 3 letter) "
                  "iso 3166 code,\n");
 fprintf( stderr, "        a country name, or a country number.\n");
 fprintf( stderr, "   -d   <domain name>\n");
 fprintf( stderr, "        Show the domain name country (if possible) and "
                  "try to explain it.\n\n");
 exit( -1);
}

static void iso3166_print_heading( void)
{
 printf( "\nCountry                                          2 "
         "letter  3 letter  Number\n");
 printf( "%s\n", cc_hyp());
}

static void iso3166_print_cc( int index)
{
 printf( "%-52.52s %-2.2s       %-3.3s     %3d\n"
                   , isoccs[ index].country
                   , isoccs[ index].codes2l
                   , isoccs[ index].codes3l
                   , isoccs[ index].number
        );
}

static void iso3166_show_all( void)
{
 int i;

 iso3166_print_heading();

 i = 0;
 while( isoccs[ i].country[ 0] != ' ') {
        iso3166_print_cc( i);
        i++;
 }

 printf( "\nDomain      Explanation\n");
 printf( "%s\n", cc_hyp());

 i = 0;
 while( topdom[ i].str[ 0] != ' ') {
        printf( "%-12.12s%-60.60s\n", topdom[ i].str, topdom[ i].expl);
        i++;
 }

 printf( "\nSubdomain      Explanation\n");
 printf( "%s\n", cc_hyp());

 i = 0;
 while( subdom[ i].str[ 0] != ' ') {
        printf( "%-12.12s%-60.60s\n", subdom[ i].str, subdom[ i].expl);
        i++;
 }

 putchar( '\n');
 exit( 0);
}

static void iso3166_show_one( char *code)
{
 char buffer[ 256];
 unsigned short int number;
 int search_type = 0;
 int len         = 0;
 int i           = 0;
 int found       = 0;

 if( code == NULL)
     return;

 len = strlen( code);

 if( len > (sizeof( buffer) - 1)) {
     fprintf( stderr, "\n%s: %s Argument too long\n\n", progname, code);
     return;
 }

 strcpy( buffer, code);

 number = atoi( buffer);

 if( number > 0)
     search_type = ISO_SEARCH_NUMBER;
 else if( len == 2)
          search_type = ISO_SEARCH_2L;
      else if( len == 3)
               search_type = ISO_SEARCH_3L;
           else {
               cc_lowercase( buffer);
               search_type = ISO_SEARCH_COUNTRY;
           }

 while( isoccs[ i].country[ 0] != ' ') {
        switch( search_type) {
                case ISO_SEARCH_COUNTRY: buffer[ 0] = toupper( buffer[ 0]);
                                         if( strstr( isoccs[ i].country, buffer) != NULL) {
                                             iso3166_print_cc( i);
                                             found = 1;
                                         }
                                         break;
                case ISO_SEARCH_2L     : if( !strncasecmp( isoccs[ i].codes2l, buffer, 2)) {
                                             iso3166_print_cc( i);
                                             found = 1;
                                         }
                                         break;
                case ISO_SEARCH_3L     : if( !strncasecmp( isoccs[ i].codes3l, buffer, 3)) {
                                             iso3166_print_cc( i);
                                             found = 1;
                                         }
                                         break;
                case ISO_SEARCH_NUMBER : if( number == isoccs[ i].number) {
                                             iso3166_print_cc( i);
                                             found = 1;
                                         }
                                         break;
        }
        i++;
 }

 if( !found) {
     printf( "[%s] Not found ", code);
     if( index( code, '.') != NULL)
         printf( "(try -d %s)\n", code); 
     else
         printf( "\n"); 
 }

}

static void domain_name_print( char *str)
{
 char *dptr;
 char *wptr;
 char *buffer;
 int size;
 int i     = 0;
 int found = 0;
 int level = 0;
 int len   = 0;

 printf( "\nDomain name  : %s\n", str);

 size = strlen( str);

 buffer = ( char *) calloc( size + 2, sizeof( char));

 if( buffer == NULL) {
     fprintf( stderr, "\n%s: Not enough memory\n\n", progname);
     exit( 1);
 }

 buffer[ 0] = '.';
 strcat( buffer, str);

 dptr = rindex( buffer, '.');

 while( dptr != NULL) {
        found = 0;

        wptr = dptr + 1;

        len = strlen( wptr);

        if( level == 0) 
            printf( "Top domain   : %s \t" , wptr);
        else
            printf( "Sub domain #%d: %s \t", level, wptr);

        i = 0;
        while( !found && topdom[ i].str[ 0] != ' ') {
               if( len != strlen( topdom[ i].str)) {
                   i++;
                   continue;
               }
               if( !strncasecmp( topdom[ i].str, wptr, len)) {
                   printf( "(%s)", topdom[ i].expl);
                   found = 1;
               }
               i++;
        }

        i = 0;
        while( !found && subdom[ i].str[ 0] != ' ') {
               if( len != strlen( subdom[ i].str)) {
                   i++;
                   continue;
               }
               if( !strncasecmp( subdom[ i].str, wptr, len)) {
                   printf( "(%s)", subdom[ i].expl);
                   found = 1;
               }
               i++;
        }

        i = 0;

        if( level == 0)
            while( !found && isoccs[ i].country[ 0] != ' ') {
                   if( len != 2) {
                       i++;
                       continue;
                   }
                   if( !strncasecmp( isoccs[ i].codes2l, wptr, len)) {
                       printf( "(%s)", isoccs[ i].country);
                       found = 1;
                   }
                   i++;
            }

        if( !found)
            printf( "(Unknown)");

        putchar( '\n');

        level++;

        *dptr = 0;
        dptr = rindex( buffer, '.');
 } /* end while */

 putchar( '\n');

 free( buffer);
}

