;;; -*- Syntax: Ansi-Common-Lisp; Base: 10; Mode: lisp; Package: imho -*-
;;; $Id: session.lisp,v 1.11 2002/03/29 06:44:26 craig Exp $
;;;
;;; Copyright (c) 1999, 2000, 2001 onShore Development, Inc.
;;;
;;; See the file 'COPYING' in this directory for terms.

(in-package :imho)

(defmethod initialize-instance :after ((self http-session) &rest ignore)
  (declare (ignore ignore))
  (with-slots (session-application timeout)
    self
    (when session-application
      (setq timeout (application-session-timeout session-application)))))

;; ------------------------------------------------------------
;; method: session-element
;;
;; Find a html-element in a session from a supplied id.

(defmethod session-element ((session http-session) element-external-name)
  "Returns the ELEMENT in a session with the supplied external name."
  (gethash element-external-name (session-html-elements session)))

(defmethod set-session-element ((http-session http-session) element)
  (with-slots (session element-external-name)
    element
    (setf session http-session
	  (gethash element-external-name
                   (session-html-elements session)) element)))
  
(defsetf session-element (session) (element)
  "Adds an ELEMENT to an HTTP-SESSIONs set of managed ELEMENT instances."
  `(set-session-element ,session ,element))

(defun new-session-instance (class)
  "Always creates a new session instance of a class and hashes it properly."
  (with-slots (session-instances) *active-session*
    (let ((instance (make-instance class)))
      (setf (gethash class session-instances) instance
	    (session-element *active-session*) instance)
      instance)))

(defun session-instance (class)
  "Returns the session instance of a class, making a new one if needed."
  (with-slots (session-instances) *active-session*
    (let ((instance (gethash class session-instances)))
      (if (not instance)
          (setf instance (new-session-instance class))
	instance))))

(defun get-session-instance (class)
  "Get the session instance of a class, NIL if one does not exist."
  (with-slots (session-instances)
    *active-session*
    (gethash class session-instances)))

(defmethod page-for-session ((page symbol) &key value)
  "Returns the session instance of an ELEMENT class, initializing it's
ELEMENT-VALUE when the :VALUE keyword is given."
  (let ((element (session-instance page)))
    (when value
      (setf (element-value element) value))
    element))

(defun bounce-sessions ()
  (labels ((clear-application (name application)
             (cmsg "clearing sessions for ~A" name)
             (maphash #'clear-session (application-sessions application)))
           (clear-session (id session)
             (cmsg "clearing session ~A" id)
             (with-slots (session-instances)
               session
               (clrhash (slot-value session 'session-instances)))))
    (maphash #'clear-application *imho-active-apps*)))

;; ------------------------------------------------------------
;; method: begin-session

(defmethod begin-session ((app application) &rest initargs)
  (with-slots (session-class sessions)
    app
    (let* ((ssn-id (random-string :length 16))
	   (ssn-args (append (list session-class :id ssn-id :application app) initargs))
	   (ssn (apply #'make-instance ssn-args)))

      
      (start-session ssn)
      (setf (gethash ssn-id sessions) ssn)
      ssn)))

;; ------------------------------------------------------------
;; method: end-session
;;
;; FIX: Should disassemble html-elements, write out state, anything
;; else?  Should we provide timeout in the framework?

(defmethod end-session ((session http-session))
  (with-slots (session-application session-id)
    session
    (remhash session-id (application-sessions session-application)))
  (destroy-session session))
  
;; ------------------------------------------------------------
;; method: application-session
;;
;; Find a session in an application from a supplied id.

(defmethod application-session ((application application) session-id)
  "Return the session with the given session-id in the application."
  (gethash session-id (application-sessions application)))


(defun lookup-session (request)
  (let ((session (request-session request))
        (application (request-application request)))
    (or (and session (setf (session-timestamp session) (get-universal-time)) session)
	(let ((session (begin-session application)))
	  (setf (request-session request) session)
	  session))))
