#include "cp_types.h"
#include "cp_proto.h"

/* ======================  facedraworder ====================== */

/* Specify an order appropriate for drawing faces of a circle packing */

/* Main difficulty arises with non-simply connected complexes.
   Key idea is to define a fundamental domain, or "core", which
   is simply connected. Grow a "red" chain, a closed doubly 
   linked chain of faces (possibly passing through same face
   more than once) defining the positively oriented outer bdry 
   of the core. In the simply connected case, this is just the 
   chain of faces about the boundary. In non-simply connected 
   cases, the red chain will pick up cuts where it runs into 
   itself as it grows (and it will also incorporate any boundary 
   vertices it encounters). A general routine builds the red 
   chain and fdo (face drawing order); the fdo is thrown away 
   after storing that information in redfaces and f_data. */

/* The red chain starts as the star of faces about alpha, then 
   continually expands until it runs into itself (or the boundary). 
   Main difficulty is blue faces; for a blue face, the prev and 
   next faces in the red chain are the same, requiring extra care. 
   We try to avoid blue faces, but that is not possible in general. 
   Other problems can arise when part of the packing is attached 
   to the core through a narrow neck having just one edge. */

/* Terminology/strategy: "free" faces are simply those not 
   processed yet in building the red chain. Eventually,
   Faces should = "red" or "white". Red = simple closed chain 
   of bdry faces and ones along each side of "cuts" defining 
   core; rest are white. A 'blue' face (often temporary condition) 
   represents a kink in red chain (the faces of the 'prev' and 
   'next' entries of the chain are the same). */

/* Finally, end up with linked list for drawing all faces 
   (next_face element of f_data) and separate closed linked 
   list of red faces (next_red element of f_data). If the
   complex is NOT simply connected, then p->redfaces points
   to the doubly linked list of faces forming the final red
   chain. */

/* One can influence red chain by use of "poison" vertices.
   These are identified by vertex util_flag=-1 (set up and
   maintained by calling routine). For instance, if there is 
   a desired red chain of faces, then one can mark the outside 
   edge vertices as poison --- evolving red chain won't loop 
   over these. For other purposes, we allow quite arbitrary 
   collections of poison verts, but caution is in order; any 
   faces outside dividing curve of poison verts may not end up 
   anywhere in the face drawing order. */

/* fixup: these routines could be more efficient -- they're too 
   slow when working with large packings */

int facedraworder(struct p_data *p,int pflag)
     /* Compute order in which faces of complex are laid out 
(actually determines order in which circles are laid out), 
starting about alpha vert. If pflag is set, then 
some "poison" vertices have been labeled (util_flag=-1) -- see 
above (typically for influencing form of final red chain). */
{
  int i,f,alpha,debug=0,lastface;
  struct RedList *redlist=NULL;
  struct Vertlist *face_order,*fdo;
  f_data *faces;
  struct K_data *pK_ptr;

  pK_ptr=p->packK_ptr;
  faces=p->faces;
  /* initialize */
  free_redfaces(&(p->redfaces));
  alloc_edge_pair(p);
  face_order=fdo=(struct Vertlist *)
    calloc((size_t)1,sizeof(struct Vertlist));
  if (!pflag) for (i=1;i<=p->nodecount;i++)
    pK_ptr[i].util_flag=0;
  if (p->rwb_flags) free(p->rwb_flags);
  p->rwb_flags=(int *)calloc((size_t)(p->facecount+1),sizeof(int));

  /* normally start with alpha */
  alpha=p->alpha;
  if (pflag && pK_ptr[alpha].util_flag==-1) 
    {
      /* when pflag is set and alpha is poison, search for a
	 non-poison starting point, nghb's of alpha, else first
	 eligible interior */
      i=0;
      while (pK_ptr[pK_ptr[alpha].flower[i]].util_flag==-1 
	     && i<=pK_ptr[alpha].num) i++;
      if (i>pK_ptr[alpha].num)
	{
	  i=1;
	  while (i<=p->nodecount 
		 && (pK_ptr[i].bdry_flag || (pK_ptr[i].util_flag==-1)))
	    i++;
	  if (i==((p->nodecount)+1))  /* abandon poison verts */
	    for (i=1;i<=p->nodecount;i++) pK_ptr[i].util_flag=0;
	}
      else alpha=p->alpha=pK_ptr[alpha].flower[i];
    }
		
  /* Now compute red chain. */

  catalog_faces(p);
  redlist=build_redchain(p,alpha,face_order);

  if (!p->num_bdry_comp && p->euler==2 && !p->genus) /* sphere */
    {free(redlist);redlist=NULL;}

  if (redlist) redface_comb_info(p,redlist,p->edge_pair);

  /* set final order info in f_data -------- */
  lastface=1;
  fdo=face_order;
  if (p->f_util) free(p->f_util);
  p->f_util=(int *)calloc((size_t)(p->facecount+1),sizeof(int));
  if ( (f=final_order(p,&fdo,&lastface,redlist)) ) 
    /* have some missing faces */
    {
      if (f==-1) /* error, redo all by old-style drawing order */
	{
	  for (i=1;i<=p->nodecount;i++) 
	    pK_ptr[i].util_flag=0;
	  if (p->f_util) free(p->f_util);
	  p->f_util=(int *)calloc((size_t)(p->facecount+1),sizeof(int));
	  pK_ptr[alpha].util_flag=1;
	  f=*(face_org[alpha]+1);
	  for (i=0;i<3;i++) 
	    pK_ptr[faces[f].vert[i]].util_flag=1;
	  fdo=face_order; /* now, fall through */
	} 
      if (!wrapup_order(p,lastface)) /* old-style order */
	{
	  sprintf(msgbuf,"A drawing order error occurred.");
	  emsg();
	}
      if (p->f_util) {free(p->f_util);p->f_util=NULL;}
    }

  /* debug */
  if (debug)
    {
      record_redlist(redlist,p->facecount); 
      pford(p); 
    }

  if (redlist==NULL) /* e.g., sphere; throw away red chain */
    p->first_red_face=0;

  for (i=1;i<=p->facecount;i++)
    {
      if (!faces[i].next_face)
	faces[i].next_face=p->first_face;
      if (p->rwb_flags[i]>0 && !faces[i].next_red)
	faces[i].next_red=p->first_red_face;
    }

  destroy_catalog(p->nodecount);
  vert_free(&face_order);
  return 1;
} /* facedraworder */

int final_order(struct p_data *p,struct Vertlist **fdo,
		int *lastface,struct RedList *redlist)
     /* Strategy: Simply connected case, do in order without regard 
to face red/white color. Non-simply connected case, start with 
alpha, proceed to first red face, do whole red chain, come back for 
rest of interior. (Needed in nonsimply connected cases because some 
circles will be moved while drawing red faces.) Return 0 for success 
and all faces plotted, -1 if an error occurs which will require use 
of old method, 1 if success but some circles still not plotted. */
{
  int i,j,red_count=0,f,index,next_face,lastred;
  f_data *faces;
  struct Vertlist *temp;
  struct RedList *trace;
  struct K_data *pK_ptr;

  pK_ptr=p->packK_ptr;
  faces=p->faces;
  temp=*fdo;
  p->first_red_face=0;
  p->redfaces=NULL;
  p->first_face=(*fdo)->v;
  for (i=1;i<=p->nodecount;i++) pK_ptr[i].util_flag=0;
  for (i=1;i<=p->facecount;i++)
    faces[i].next_face=faces[i].next_red=0;

  /* take care of first face bookkeeping */
  for (j=0;j<3;j++) 
    pK_ptr[faces[(*fdo)->v].vert[j]].util_flag=1;
  p->f_util[(*fdo)->v]=1;
  if (!(*fdo)->next || !(*fdo)->next->next) return -1; /* error */

  /* ----------------- simply connected ------------------------- */

  if ((p->euler==1 || p->euler==2) && p->genus==0)
    {
      faces[(*fdo)->v].next_face=f=(*fdo)->next->v;
      index=faces[f].index_flag;
      faces[f].index_flag=nice_index(p,f,index);
      pK_ptr[faces[f].
	    vert[(faces[f].index_flag+2) % 3]].util_flag=1;
      p->f_util[f]=1;
      *lastface=f;
      /* now look for qualifying next_face for f to point to */
      *fdo=(*fdo)->next->next;
      while (*fdo) 
	/* get all rest in fdo, avoiding repeats 
	   (i.e., whipping out an original 'next_face') */
	{
	  while (*fdo && (faces[(*fdo)->v].next_face
			  || (*fdo)->v==f) )
	    *fdo=(*fdo)->next;
	  if (*fdo)
	    {
	      f=faces[f].next_face=(*fdo)->v;
	      index=faces[f].index_flag;
	      faces[f].index_flag=nice_index(p,f,index);
	      pK_ptr[faces[f].
		    vert[(faces[f].index_flag+2) % 3]].util_flag=1;
	      p->f_util[f]=1;
	      *lastface=f;
	      *fdo=(*fdo)->next;
	    }
	}

      /* set first_red_face and red face list in f_data */
      if (redlist!=NULL)
	{
	  while (temp->next && p->rwb_flags[temp->v]<=0) 
	    /* skip 'white' or 'free' faces */
	    temp=temp->next;
	  if (temp->next && position_redlist(&redlist,temp->v)) 
	    /* seem to have found one */
	    {
	      p->first_red_face=temp->v;
	      p->f_util[p->first_red_face]=1;
	      next_face=redlist->next->face;
	      faces[p->first_red_face].next_red
		= next_face;
	      p->f_util[next_face]=1;
	      trace=redlist->next;
	      while (trace!=redlist->prev && red_count<(2*p->facecount))
		{
		  red_count++;
		  next_face=trace->next->face;
		  /* skip over red faces already plotted */
		  while (trace!=redlist->prev 
			 && p->f_util[next_face])
		    {
		      trace=trace->next;
		      next_face=trace->next->face;
		    }
		  if (trace!=redlist->prev)
		    {
		      faces[trace->face].next_red
			=next_face;
		      p->f_util[next_face]=1;
		      trace=trace->next;
		    }
		}
	      if (red_count > p->facecount)
		{
		  sprintf(msgbuf,"Draw order problem with red chain.");
		  emsg();
		  return -1;
		}
	    }
	  else
	    {
	      sprintf(msgbuf,"Draw order problem with red chain.");
	      emsg();
	      return -1;
	    }
	}
      p->redfaces=redlist;
      /* check for remaining faces */
      for (i=1;i<=p->nodecount;i++) 
	if (!pK_ptr[i].util_flag)
	  return 1; /* some circles not plotted */
      return 0;
    }

  /* ------------------------ non-simply connected ----------------- */

  else
    {
      /* first, order the interiors; */

      f=*lastface=p->first_face;

      /* special handling to get started with first_face. */

      if (p->rwb_flags[f]<0) (*fdo)=(*fdo)->next;

      /* seek qualifying next_face for f; avoid repeats */
      while (*fdo && p->rwb_flags[(*fdo)->v]<0)
	{
	  while (*fdo && p->rwb_flags[(*fdo)->v]<0
		 && faces[(*fdo)->v].next_face)
	    *fdo=(*fdo)->next;
	  if (*fdo && p->rwb_flags[(*fdo)->v]<0)
	    {		   
	      f=faces[f].next_face=(*fdo)->v;
	      index=faces[f].index_flag;
	      faces[f].index_flag=nice_index(p,f,index);
	      pK_ptr[faces[f].
		    vert[(faces[f].index_flag+2) % 3]].util_flag=1;
	      *lastface=f;
	      *fdo=(*fdo)->next;
	    }
	}
      if (!(*fdo) || p->rwb_flags[(*fdo)->v]<0) 
	/* should not happen */
	{
	  faces[f].next_face=temp->v;
	  return -1;
	}
      /* Should be pointing to first red face (which could also be 
	 'first_face' itself); now, put in whole red chain, also arrange 
	 'next_red' data */
      if (f==p->first_face && p->rwb_flags[f]>=0) 
	/* first_face red? already in place */
	p->first_red_face=p->first_face;
      else
	{
	  f=faces[f].next_face=p->first_red_face=(*fdo)->v;
	  index=faces[f].index_flag;
	  faces[f].index_flag=nice_index(p,f,index);
	  pK_ptr[faces[f].
		vert[(faces[f].index_flag+2) % 3]].util_flag=1;
	}
      p->f_util[f]=1;
      *lastface=f;

      /* seek qualifying next_face for f, but now in red chain;
	 careful to avoid repeats */

      position_redlist(&redlist,p->first_red_face);
      p->redfaces=redlist;
      trace=redlist->next;
      while (trace!=redlist && red_count<(3*p->facecount))
	{
	  red_count++;
	  /* skip over faces with next_face already set */
	  while (trace!=redlist 
		 && (faces[trace->face].next_face
		     || trace->face==f)) /* careful not to point to self */
	    trace=trace->next;
	  if (trace!=redlist)
	    {
	      f=faces[f].next_face=faces[f].next_red=
		trace->face;
	      index=faces[f].index_flag;
	      faces[f].index_flag=nice_index(p,f,index);
	      pK_ptr[faces[f].
		    vert[(faces[f].index_flag+2) % 3]].util_flag=1;
	      p->f_util[f]=1;
	      *lastface=f;
	      trace=trace->next;
	    }
	}
      if (red_count > (3*p->facecount))
	{
	  sprintf(msgbuf,"Draw order problem with red chain.");
	  emsg();
	  return -1;
	}
      /* now at last red face */
      lastred=f;

      /* get rest of white by following fdo; avoid repeats, skip red. 
	 The last red face must point to the rest of the list for purposes
	 of comp_pack_centers. Last face indicated by next_face=0. */

      while (*fdo)
	{
	  while (*fdo && (p->rwb_flags[(*fdo)->v]>=0
			  || faces[(*fdo)->v].next_face) )
	    *fdo=(*fdo)->next;
	  if (*fdo)
	    {
	      if (lastred) 
		{
		  faces[lastred].next_face=(*fdo)->v;
		  lastred=0;
		}
	      f=faces[f].next_face=(*fdo)->v;
	      index=faces[f].index_flag;
	      faces[f].index_flag=nice_index(p,f,index);
	      pK_ptr[faces[f].
		    vert[(faces[f].index_flag+2) % 3]].util_flag=1;
	      p->f_util[f]=1;
	      *lastface=f;
	      *fdo=(*fdo)->next;
	    }
	}
 
      /* check: any circles missed? */
      for (i=1;i<=p->nodecount;i++) 
	if (!pK_ptr[i].util_flag) 
	  return 1;
      return 0;
    } /* end of non-simply connected case */
} /* final_order */

int wrapup_order(struct p_data *p,int lastface)
     /* Wrap up face order. p->f_util should have been set in
'final_order' to show faces handles alread (reset to zero if 
final_order hit a snag). Put in remaining faces; return 0 on error. */
{
  int k=1,hit=1,stop=0,F,i,index,count=0,n;
  f_data *faces;
  struct K_data *pK_ptr;

  pK_ptr=p->packK_ptr;
  faces=p->faces;
  F=p->facecount;
  for (i=1;i<=F;i++) if (p->f_util[i]) count++;
  while (count<F && !stop)
    {
      if (k > F)
	{
	  k=1;
	  if (!hit) stop=1; /* no new faces being added */
	  hit=0;
	}
      while (p->f_util[k] && k < F) k++;
      if (!(p->f_util[k]) && face_ready(p,k,&index))
	{
	  if ( (n=nghb_tri(p,lastface,k))<0 )
	    faces[k].index_flag=index;
	  else faces[k].index_flag=n;
	  faces[lastface].next_face=k;
	  p->f_util[k]=1;
	  pK_ptr[faces[k].
		vert[(index+2) % 3]].util_flag=1;
	  lastface=k;
	  count++;
	  hit=1;
	}
      k++;
    }
  faces[lastface].next_face=p->first_face;
  if (count<F) return 0; /* shouldn't ever happen */
  return 1;
} /* wrapup_order */

int face_ready(struct p_data *p,int face,int *index)
     /* return 1 if face ready to draw, set index of vert to be drawn first. */
{
  f_data *faces;
  struct K_data *pK_ptr;

  pK_ptr=p->packK_ptr;
  faces=p->faces+face;
  if (pK_ptr[faces->vert[0]].util_flag 
      && pK_ptr[faces->vert[1]].util_flag) 
    {
      *index=0;
      return 1;
    }
  if (pK_ptr[faces->vert[1]].util_flag 
      && pK_ptr[faces->vert[2]].util_flag) 
    {
      *index=1;
      return 1;	
    }
  if (pK_ptr[faces->vert[2]].util_flag 
      && pK_ptr[faces->vert[0]].util_flag) 
    {
      *index=2;
      return 1;
    }
  *index=0;
  return 0;
} /* face_ready */

int position_redlist(struct RedList **redlist,int face)
     /* position closed list at next occurance of face (skip starting 
ptr and test it at the last); return 0 if face not found. */
{
  struct RedList *current;

  current=(*redlist)->next;
  if (current->face==face)
    {
      *redlist=current;
      return 1;
    }
  current=current->next;
  while (current!=(*redlist)->next)
    {
      if (current->face==face)
	{
	  *redlist=current;
	  return 1;
	}
      current=current->next;
    }
  return 0;
} /* position_redlist */

int nice_index(struct p_data *p,int f,int index)
     /* set index_flag of face f so that it uses nghb'ing
white face using verts already drawn, if possible. "index" is first 
preference and default. util_flags are >=0 for verts already drawn. */
{
  int v,w,i,m,num,g,*face_ptr;
  f_data *faces;
  struct K_data *pK_ptr;

  pK_ptr=p->packK_ptr;
  faces=p->faces;
  for (i=0;i<3;i++)
    {
      v=faces[f].vert[(index+i)%3];
      w=faces[f].vert[(index+i+1)%3];
      if (pK_ptr[v].util_flag > 0 
	  && !pK_ptr[v].bdry_flag
	  && pK_ptr[w].util_flag > 0 
	  && !pK_ptr[w].bdry_flag)
	{
	  face_ptr=face_org[w];
	  num=*face_ptr;
	  face_ptr++;
	  m=0;
	  while (*(face_ptr+m)!=f && m<(num-1)) m++;
	  g=*(face_ptr+((m+1) % num));
	  if (p->rwb_flags[g]<0) return ((index+i)%3);
	  /* yes, face is white */
	}
    }
  for (i=0;i<3;i++) /* fallback: try for white neighbor */
    {
      v=faces[f].vert[(index+i)%3];
      w=faces[f].vert[(index+i+1)%3];
      if (pK_ptr[v].util_flag > 0
	  && pK_ptr[w].util_flag)
	{
	  face_ptr=face_org[w];
	  num=*face_ptr;
	  face_ptr++;
	  m=0;
	  while (*(face_ptr+m)!=f && m<(num-1)) m++;
	  g=*(face_ptr+((m+1) % num));
	  if (p->rwb_flags[g]<0) return ((index+i)%3);
	  /* yes, face is white */
	}
    }
  for (i=0;i<3;i++) /* fallback: use any two plotted ngbh's. */
    {
      v=faces[f].vert[(index+i)%3];
      w=faces[f].vert[(index+i+1)%3];
      if (pK_ptr[v].util_flag > 0
	  && pK_ptr[w].util_flag) return ((index+i)%3);
    }
  if (index<0 || index>2) return 0;
  return index;
} /* nice_index */
