package CipUX::Cfg::Client;

use strict;
use warnings;
use Carp;
use Class::Std;
use Data::Dumper;
use English qw( -no_match_vars );
use Getopt::Long;
use Log::Log4perl qw(get_logger :levels);
use Pod::Usage;
use YAML::Any;
use base qw(CipUX);

{

    use version; our $VERSION = qv('3.4.0.11');
    delete @ENV{qw(PATH IFS CDPATH ENV BASH_ENV)};

    # CONST
    Readonly::Scalar my $EMPTY_STRING => q{};

    # GLOBAL
    my %opt = ();

    my $cfg_space_dispatch_hr = {
        all               => \&all_cfg,
        cipux             => \&cipux_cfg,
        storage_structure => \&storage_structure_cfg,
        storage_access    => \&storage_access_cfg,
        storage           => \&storage_cfg,
        object_coupling   => \&object_coupling_cfg,
        object_object     => \&object_object_cfg,
        object_basis      => \&object_basis_cfg,
        object            => \&object_cfg,
        task_api          => \&task_api_cfg,
        task              => \&task_cfg,
    };
    my $cfg_format_dispatch_hr = {
        hash => \&as_hash,
        yaml => \&as_yaml,
    };

    # ENVIRONMENT
    Getopt::Long::Configure("bundling");

    GetOptions( \%opt, "debug|D", "hash", "yaml" )
        or pod2usage(
        -exitstatus => 2,
        -msg        => "Problems parsing command line options!\n"
        );

    sub run {

        my $self = shift;
        my $arg  = 'all';

        if ( not scalar @ARGV ) {
            push @ARGV, 'all';
        }

        my $format
            = ( exists $opt{hash} and defined $opt{hash} ) ? 'hash' : 'yaml';

        foreach my $scope (@ARGV) {
            my $get    = $cfg_space_dispatch_hr->{$scope};
            my $cfg_ar = $self->$get;
            my $as     = $cfg_format_dispatch_hr->{$format};
            foreach my $cfg_hr ( @{$cfg_ar} ) {
                $self->out("# === CipUX scope $scope, format $format ===\n");
                my $n = $cfg_hr->{name};
                my $v = $cfg_hr->{value_hr};
                $self->out( $self->$as( { name => $n, value_hr => $v } ) );
            }
        }
        return;
    }

    sub cipux_cfg {
        my $self = shift;

        my $cfg_hr = $self->cfg;

        return [ { name => 'cipux', value_hr => $cfg_hr } ];
    }

    sub storage_structure_cfg {

        eval { require CipUX::Storage };
        if ( not $EVAL_ERROR ) {

            my $storage        = CipUX::Storage->new();
            my $storage_cfg_hr = $storage->get_storage_cfg();
            my $out_cfg_hr     = {};
            $out_cfg_hr->{structure} = $storage_cfg_hr;

            return [
                { name => 'storage_structure', value_hr => $out_cfg_hr } ];
        }
        return [];
    }

    sub storage_access_cfg {

        eval { require CipUX::Storage };
        if ( not $EVAL_ERROR ) {

            my $storage       = CipUX::Storage->new();
            my $access_cfg_hr = $storage->get_access_cfg();

            return [
                { name => 'storage_access', value_hr => $access_cfg_hr } ];
        }

        return [];

    }

    sub storage_cfg {

        my ( $self, $arg_r ) = @_;

        # will not show access per default
        #$self->storage_access_cfg;
        return $self->storage_structure_cfg;

    }

    sub object_coupling_cfg {

        eval { require CipUX::Object::Action };
        if ( not $EVAL_ERROR ) {
            my $object          = CipUX::Object::Action->new();
            my $coupling_cfg_hr = $object->get_coupling_cfg();

            return [
                { name => 'object_coupling', value_hr => $coupling_cfg_hr } ];
        }

        return [];
    }

    sub object_object_cfg {

        eval { require CipUX::Object::Action };
        if ( not $EVAL_ERROR ) {
            my $object        = CipUX::Object::Action->new();
            my $object_cfg_hr = $object->get_object_cfg();

            return [
                { name => 'object_object', value_hr => $object_cfg_hr } ];
        }

        return [];
    }

    sub object_basis_cfg {

        eval { require CipUX::Object::Action };
        if ( not $EVAL_ERROR ) {
            my $object       = CipUX::Object::Action->new();
            my $basis_cfg_hr = $object->get_basis_cfg();

            return [ { name => 'object_basis', value_hr => $basis_cfg_hr } ];
        }

        return [];
    }

    sub object_cfg {
        my ( $self, $arg_r ) = @_;

        my @array = ();
        push @array, @{ $self->object_coupling_cfg };
        push @array, @{ $self->object_object_cfg };
        push @array, @{ $self->object_basis_cfg };

        return \@array;
    }

    sub task_api_cfg {

        eval { require CipUX::Task };
        if ( not $EVAL_ERROR ) {
            my $task            = CipUX::Task->new();
            my $task_api_cfg_hr = $task->get_task_api_cfg();
            my $out_cfg_hr      = {};
            $out_cfg_hr->{task_api} = $task_api_cfg_hr;

            return [ { name => 'task_api', value_hr => $out_cfg_hr } ];
        }

        return [];
    }

    sub task_cfg {
        my ( $self, $arg_r ) = @_;

        return $self->task_api_cfg;
    }

    sub all_cfg {
        my ( $self, $arg_r ) = @_;

        my @array = ();
        push @array, @{ $self->cipux_cfg };
        push @array, @{ $self->storage_cfg };
        push @array, @{ $self->object_cfg };
        push @array, @{ $self->task_cfg };

        return \@array;

    }

    sub as_yaml {
        my ( $self, $arg_r ) = @_;
        my $hr
            = ( exists $arg_r->{value_hr} and $self->h( $arg_r->{value_hr} ) )
            ? $arg_r->{value_hr}
            : $self->perr('value_hr');
        my $name
            = ( exists $arg_r->{name} and defined $arg_r->{name} )
            ? $self->l( $arg_r->{name} )
            : $self->perr('name');

        return YAML::Any::Dump($hr);
    }

    sub as_hash {
        my ( $self, $arg_r ) = @_;
        my $hr
            = ( exists $arg_r->{value_hr} and $self->h( $arg_r->{value_hr} ) )
            ? $arg_r->{value_hr}
            : $self->perr('value_hr');
        my $name
            = ( exists $arg_r->{name} and defined $arg_r->{name} )
            ? $self->l( $arg_r->{name} )
            : $self->perr('name');
        $Data::Dumper::Varname = $name;
        return Dumper($hr);
    }

}

1;

__END__

=pod

=for stopwords CipUX Kuelker perr VARNAME API exc STDERR config getopt homedir lf LDAP lp li latin UTF UTF-8 login STDOUT TODO cfg

=head1 NAME

CipUX::Cfg::Client - Common CipUX configuration client

=head1 VERSION

version 3.4.0.11

=head1 SYNOPSIS

  use CipUX::Cfg::Client;

  my $ ccfg = $CipUX::Cfg::Client->new;


=head1 DESCRIPTION

Provides CLI for displaying CipUX configuarations of differents scopes.

=head1 CONSTRUCTOR

=head2 new

Constructor

  my $trait = CipUX::Cfg::Client->new;

=head1 SUBROUTINES/METHODS

The following functions will be exported by CipUX::Cfg::Client.

=head2 run
=head2 cipux_cfg
=head2 storage_structure_cfg
=head2 storage_access_cfg
=head2 storage_cfg
=head2 object_coupling_cfg
=head2 object_object_cfg
=head2 object_basis_cfg
=head2 object_cfg
=head2 task_api_cfg
=head2 task_cfg
=head2 all_cfg
=head2 as_yaml
=head2 as_hash

=head1 DIAGNOSTICS

=head2 Problems parsing command line options!


=head1 CONFIGURATION AND ENVIRONMENT

Not applicable.

=head1 DEPENDENCIES

 CipUX
 Class::Std
 Data::Dumper
 English
 Getopt::Long
 Log::Log4perl
 Pod::Usage
 Readonly
 YAML::Any
 version

=head1 INCOMPATIBILITIES

Not known.

=head1 BUGS AND LIMITATIONS

Not known.

=head1 SEE ALSO

See the CipUX web page and the manual at L<http://www.cipux.org>

See the mailing list L<http://sympa.cipworx.org/wws/info/cipux-devel>

=head1 AUTHOR

Christian Kuelker  E<lt>christian.kuelker@cipworx.orgE<gt>

=head1 LICENSE AND COPYRIGHT

Copyright (C) 2010 by Christian Kuelker

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License as
published by the Free Software Foundation; either version 2, or (at
your option) any later version.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
02111-1307 USA

=cut

