;;;; srfi-18.scm - Simple thread unit - felix
;
; Copyright (c) 2000-2002, Felix L. Winkelmann
; All rights reserved.
;
; Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following
; conditions are met:
;
;   Redistributions of source code must retain the above copyright notice, this list of conditions and the following
;     disclaimer. 
;   Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following
;     disclaimer in the documentation and/or other materials provided with the distribution. 
;   Neither the name of the author nor the names of its contributors may be used to endorse or promote
;     products derived from this software without specific prior written permission. 
;
; THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS
; OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
; AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR
; CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
; CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
; SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
; THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
; OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
; POSSIBILITY OF SUCH DAMAGE.
;
; Send bugs, suggestions and ideas to: 
;
; felix@call-with-current-continuation.org
;
; Felix L. Winkelmann
; Steinweg 1A
; 37130 Gleichen, OT Weissenborn
; Germany


(declare
 (unit srfi-18)
 (interrupts-disabled)
 (standard-bindings)
 (extended-bindings)
 (no-bound-checks)
 (bound-to-procedure
  ##sys#add-to-ready-queue ##sys#remove-from-ready-queue
  ##sys#schedule ##sys#make-thread
  ##sys#update-thread-state-buffer ##sys#restore-thread-state
  ##sys#check-number ##sys#error ##sys#signal-hook ##sys#signal
  ##sys#current-exception-handler ##sys#abandon-mutexes ##sys#check-structure ##sys#structure? ##sys#make-mutex
  ##sys#delq ##sys#compute-time-limit ##sys#fudge) )

(cond-expand
 [unsafe
  (eval-when (compile)
    (define-macro (##sys#check-structure x y) '(##core#undefined))
    (define-macro (##sys#check-range x y z) '(##core#undefined))
    (define-macro (##sys#check-pair x) '(##core#undefined))
    (define-macro (##sys#check-list x) '(##core#undefined))
    (define-macro (##sys#check-symbol x) '(##core#undefined))
    (define-macro (##sys#check-string x) '(##core#undefined))
    (define-macro (##sys#check-char x) '(##core#undefined))
    (define-macro (##sys#check-exact x) '(##core#undefined))
    (define-macro (##sys#check-port x) '(##core#undefined))
    (define-macro (##sys#check-number x) '(##core#undefined))
    (define-macro (##sys#check-byte-vector x) '(##core#undefined)) ) ]
 [else] )

(register-feature! 'srfi-18)


;;; Helper routines:

(define (##sys#delq x lst)
  (let loop ([lst lst])
    (cond ((null? lst) lst)
	  ((eq? x (##sys#slot lst 0)) (##sys#slot lst 1))
	  (else (cons (##sys#slot lst 0) (loop (##sys#slot lst 1)))) ) ) )

(define ##sys#compute-time-limit
  (let ([truncate truncate])
    (lambda (tm)
      (and tm
	   (cond [(##sys#structure? tm 'time) (##sys#slot tm 1)]
		 [(number? tm) (fx+ (##sys#fudge 6) (inexact->exact (truncate (* tm 1000))))]
		 [else (##sys#signal-hook #:type-error "invalid timeout argument" tm)] ) ) ) ) )


;;; Time objects:

(define (current-time) 
  (##sys#make-structure 'time (##sys#fudge 6)) )

(define time->seconds
  (lambda (tm)
    (##sys#check-structure tm 'time)
    (/ (##sys#slot tm 1) 1000) ) )

(define (seconds->time n)
  (##sys#check-number n)
  (##sys#make-structure 'time (inexact->exact (truncate (* n 1000)))) )

(define (time? x) (##sys#structure? x 'time))


;;; Exception handling:

(define raise ##sys#signal)

(define (join-timeout-exception? x) 
  (and (##sys#structure? x 'condition)
       (memq 'join-timeout-exception (##sys#slot x 1)) ) )

(define (abandoned-mutex-exception? x)
  (and (##sys#structure? x 'condition)
       (memq 'abandoned-mutex-exception (##sys#slot x 1)) ) )

(define (terminated-thread-exception? x)
  (and (##sys#structure? x 'condition)
       (memq 'terminated-thread-exception (##sys#slot x 1)) ) )

(define (uncaught-exception? x)
  (and (##sys#structure? x 'condition)
       (memq 'uncaught-exception (##sys#slot x 1)) ) )

(define uncaught-exception-reason
  (let ([get-reason (condition-property-accessor 'uncaught-exception 'reason)])
    (lambda (ux) (get-reason ux)) ) )


;;; Threads:

(define make-thread
  (let ((gensym gensym))
    (lambda (thunk . name)
      (let ((thread
	     (##sys#make-thread
	      #f
	      'created
	      (if (pair? name) (##sys#slot name 0) (gensym 'thread))
	      (##sys#slot ##sys#current-thread 9) ) ) )
	(##sys#setslot 
	 thread 1 
	 (lambda () 
	   (let ((result (thunk)))
	     (##sys#setslot thread 2 result)
	     (##sys#setslot thread 3 'dead) 
	     (##sys#abandon-mutexes thread)
	     result) ) )
	thread) ) ) )

(define (thread? x) (##sys#structure? x 'thread))
(define (current-thread) ##sys#current-thread)

(define (thread-specific thread)
  (##sys#check-structure thread 'thread)
  (##sys#slot thread 10) )

(define (thread-specific-set! thread x)
  (##sys#check-structure thread 'thread)
  (##sys#setslot thread 10 x) )

(define (thread-quantum thread)
  (##sys#check-structure thread 'thread)
  (##sys#slot thread 9) )

(define (thread-quantum-set! thread q)
  (##sys#check-structure thread 'thread)
  (##sys#check-exact q)
  (##sys#setislot thread 9 (fxmax q 10)) )

(define (thread-name x)
  (##sys#check-structure x 'thread)
  (##sys#slot x 6) )

(define thread-start!
    (lambda (thread)
      (unless (eq? 'created (##sys#slot thread 3))
	(##sys#error "thread can not be started a second time" thread) )
      (##sys#check-structure thread 'thread)
      (##sys#setslot thread 3 'ready)
      (##sys#add-to-ready-queue thread) 
      thread) )

(define (thread-yield!)
  (##sys#call-with-current-continuation
   (lambda (return)
     (let ((ct ##sys#current-thread))
       (##sys#setslot ct 1 (lambda () (return (##core#undefined))))
       (##sys#schedule) ) ) ) )

(define thread-join!
    (lambda (thread . timeout)
      (##sys#check-structure thread 'thread)
      (let* ((limit (and (pair? timeout) (##sys#compute-time-limit (##sys#slot timeout 0))))
	     (rest (and (pair? timeout) (##sys#slot timeout 1)))
	     (tosupplied (and rest (pair? rest)))
	     (toval (and tosupplied (##sys#slot rest 0))) )
	(##sys#call-with-current-continuation
	 (lambda (return)
	   (let ([ct ##sys#current-thread])
	     (##sys#setslot ct 1 (lambda () (return (##sys#slot thread 2))))
	     (##sys#setslot ct 3 'blocked)
	     (##sys#setslot 
	      ct 4
	      (if limit
		  (lambda () 
		    (case (##sys#slot thread 3)
		      [(dead) #t]
		      [(terminated)
		       (##sys#setslot
			ct 1
			(lambda ()
			  (return 
			   (##sys#signal
			    (##sys#make-structure 'condition '(uncaught-exception) (list 'reason (##sys#slot thread 7)) ) ) ) ) )
		       #t]
		      [else
		       (and (fx> (##sys#fudge 6) limit)
			    (begin
			      (##sys#setslot 
			       ct 1 
			       (lambda ()
				 (return
				  (if tosupplied
				      toval
				      (##sys#signal (##sys#make-structure 'condition '(join-timeout-exception) '())) ) ) ) )
			      #t) ) ] ) )
		  (lambda () 
		    (case (##sys#slot thread 3)
		      [(dead) #t]
		      [(terminated)
		       (##sys#setslot
			ct 1
			(lambda ()
			  (return 
			   (##sys#signal
			    (##sys#make-structure 'condition '(uncaught-exception) (list 'reason (##sys#slot thread 7)) ) ) ) ) )
		       #t]
		      [else #f] ) ) ) )
	     (##sys#schedule) ) ) ) ) ) )

(define (thread-terminate! thread)
  (##sys#check-structure thread 'thread)
  (##sys#setslot thread 3 'terminated)
  (##sys#setislot thread 2 (##core#undefined))
  (##sys#setslot thread 7 (##sys#make-structure 'condition '(terminated-thread-exception) '()))
  (##sys#abandon-mutexes thread)
  (when (eq? thread ##sys#current-thread) (##sys#schedule)) )

(define (thread-suspend! thread)
  (##sys#check-structure thread 'thread)
  (##sys#setslot thread 3 'suspended)
  (when (eq? thread ##sys#current-thread)
    (##sys#call-with-current-continuation
     (lambda (return)
       (##sys#setslot thread 1 (lambda () (return (##core#undefined))))
       (##sys#schedule) ) ) ) )

(define (thread-resume! thread)
  (##sys#check-structure thread 'thread)
  (when (eq? (##sys#slot thread 3) 'suspended)
    (##sys#setslot thread 3 'ready)
    (##sys#add-to-ready-queue thread) ) )

(define thread-sleep!
  (lambda (tm)
    (unless tm (##sys#signal-hook #:tpe-error "invalid timeout argument" tm))
    (##sys#call-with-current-continuation
     (lambda (return)
       (let ([limit (##sys#compute-time-limit tm)]
	     [ct ##sys#current-thread] )
	 (##sys#setslot ct 1 (lambda () (return (##core#undefined))))
	 (##sys#setslot ct 3 'blocked)
	 (##sys#setslot ct 4 (lambda () (fx>= (##sys#fudge 6) limit)))
	 (##sys#schedule) ) ) ) ) )


;;; Mutexes:

(define (mutex? x) (##sys#structure? mutex 'mutex))

(define make-mutex
  (let ((gensym gensym))
    (lambda owner-and-id
      (let* ((owner (if (pair? owner-and-id) owner-and-id #f))
	     (ownert (if owner (##sys#slot owner 0) ##sys#current-thread))
	     (rest (and owner (##sys#slot owner 1)))
	     (id (if (and owner (pair? rest)) (##sys#slot rest 0) (gensym 'mutex)))
	     (m (##sys#make-mutex id ownert)) )
	(if ownert
	    (begin
	      (##sys#check-structure ownert 'thread)
	      (##sys#setslot ownert 8 (cons m (##sys#slot ownert 8))) ) ) 
	m) ) ) )

(define (mutex-name x)
  (##sys#check-structure x 'mutex) 
  (##sys#slot x 1) )

(define (mutex-owner mutex)
  (##sys#check-structure mutex 'mutex)
  (##sys#slot mutex 2) )

(define (mutex-specific mutex)
  (##sys#check-structure mutex 'mutex)
  (##sys#slot mutex 6) )

(define (mutex-specific-set! mutex x)
  (##sys#check-structure mutex 'mutex)
  (##sys#setslot mutex 6 x) )

(define (mutex-state mutex)
  (##sys#check-structure mutex 'mutex)
  (cond [(##sys#slot mutex 5) (or (##sys#slot mutex 2) 'not-owned)]
	[(##sys#slot mutex 4) 'abandoned]
	[else 'not-abandoned] ) )

(define mutex-lock! 
  (lambda (mutex . ms-and-t)
    (##sys#check-structure mutex 'mutex)
    (let ([limit (and (pair? ms-and-t) (##sys#compute-time-limit (car ms-and-t)))]
	  [thread (and (fx> (length ms-and-t) 1) (cadr ms-and-t))] )
      (when thread (##sys#check-structure thread 'thread))
      (##sys#call-with-current-continuation
       (lambda (return)
	 (let ([result
		(let ([ct ##sys#current-thread]
		      [success #t] )

		  (define (switch)
		    (##sys#setslot ct 1 (lambda () (return success)))
		    (##sys#setslot mutex 3 (##sys#append (##sys#slot mutex 3) (list ct)))
		    (##sys#schedule) )

		  (cond [(not (##sys#slot mutex 5)) #t]
			[limit
			 (##sys#setslot ct 3 'blocked)
			 (##sys#setslot 
			  ct 4 
			  (lambda ()
			    (and (fx>= (##sys#fudge 6) limit)
				 (begin 
				   (##sys#setslot mutex 3 (##sys#delq ct (##sys#slot mutex 3)))
				   (##sys#setslot mutex 2 thread)
				   (set! success #f)
				   #t) ) ) )
			 (switch) ]
			[else
			 (##sys#setslot ct 3 'sleeping)
			 (switch) ] ) ) ] )

	   (cond [(##sys#slot mutex 4)
		  (##sys#signal (##sys#make-structure 'condition '(abandoned-mutex-exception) '())) ]
		 [result
		  (let ([t (or thread ##sys#current-thread)])
		    (cond [(eq? 'terminated (##sys#slot t 3)) (##sys#setislot mutex 4 #t)]
			  [else
			   (##sys#setislot mutex 5 #t)
			   (##sys#setslot mutex 2 thread) ] ) )
		  result]
		 [else #f] ) ) ) ) ) ) )

(define mutex-unlock!
  (lambda (mutex . cvar-and-to)
    (##sys#check-structure mutex 'mutex)
    (let ([ct ##sys#current-thread]
	  [cvar (and (pair? cvar-and-to) (car cvar-and-to))]
	  [timeout (and (fx> (length cvar-and-to) 1) (cadr cvar-and-to))] )
      (when cvar (##sys#check-structure cvar 'condition-variable))
      (##sys#call-with-current-continuation
       (lambda (return)
	 (let ([waiting (##sys#slot mutex 3)]
	       [limit (and timeout (##sys#compute-time-limit timeout))] 
	       [result #t] )
	   (##sys#setislot mutex 4 #f)
	   (##sys#setislot mutex 5 #f)
	   (##sys#setslot ct 8 (##sys#delq mutex (##sys#slot ct 8)))
	   (##sys#setslot ct 1 (lambda () (return result)))
	   (when cvar
	     (##sys#setslot cvar 2 (##sys#append (##sys#slot cvar 2) (##sys#list ct)))
	     (cond [limit
		    (##sys#setslot ct 3 'blocked)
		    (##sys#setslot 
		     ct 4
		     (lambda () 
		       (and (fx> (##sys#fudge 6) limit)
			    (begin 
			      (##sys#setslot cvar 2 (##sys#delq ct (##sys#slot cvar 2)))
			      (set! result #f)
			      #t) ) ) ) ]
		   [else (##sys#setslot ct 3 'sleeping)] ) )
	   (unless (null? waiting)
	     (let* ([wt (##sys#slot waiting 0)]
		    [wts (##sys#slot wt 3)] )
	       (##sys#setslot mutex 3 (##sys#slot waiting 1))
	       (when (memq wts '(blocked sleeping))
		 (##sys#setslot mutex 2 wt)
		 (##sys#setslot wt 3 'ready)
		 (##sys#setslot wt 8 (cons mutex (##sys#slot wt 8)))
		 (when (eq? wts 'sleeping) (##sys#add-to-ready-queue wt) ) ) ) )
	   (##sys#schedule) ) ) ) ) ) )

;;; Condition variables:

(define make-condition-variable
  (let ([gensym gensym])
    (lambda name
      (##sys#make-structure
       'condition-variable 
       (if (pair? name)
	   (car name)
	   (gensym 'condition-variable) )
       '()				; list of waiting threads
       (##core#undefined) ) ) ) )	; specific

(define (condition-variable? x)
  (##sys#structure? x 'condition-variable) )

(define (condition-variable-specific cv)
  (##sys#check-structure cv 'condition-variable)
  (##sys#slot cv 3) )

(define (condition-variable-specific-set! cv x)
  (##sys#check-structure cv 'condition-variable)
  (##sys#setslot cv 3 x) )

(define (condition-variable-signal! cvar)
  (##sys#check-structure cvar 'condition-variable)
  (let ([ts (##sys#slot cvar 2)])
    (unless (null? ts)
      (let ([t0 (##sys#slot ts 0)])
	(##sys#setslot cvar 2 (##sys#slot ts 1))
	(when (memq (##sys#slot t0 3) '(blocked sleeping))
	  (##sys#setslot t0 3 'ready)
	  (##sys#add-to-ready-queue t0) ) ) ) ) )

(define (condition-variable-broadcast! cvar)
  (##sys#check-structure cvar 'condition-variable)
  (##sys#for-each
   (lambda (ti)
     (when (memq (##sys#slot ti 3) '(blocked sleeping))
       (##sys#setslot ti 3 'ready)
       (##sys#add-to-ready-queue ti) ) )
   (##sys#slot cvar 2) )
  (##sys#setislot cvar 2 '()) )
