;;;; library.scm - R5RS library for the CHICKEN compiler
;
; Copyright (c) 2000-2002, Felix L. Winkelmann
; All rights reserved.
;
; Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following
; conditions are met:
;
;   Redistributions of source code must retain the above copyright notice, this list of conditions and the following
;     disclaimer. 
;   Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following
;     disclaimer in the documentation and/or other materials provided with the distribution. 
;   Neither the name of the author nor the names of its contributors may be used to endorse or promote
;     products derived from this software without specific prior written permission. 
;
; THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS
; OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
; AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR
; CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
; CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
; SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
; THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
; OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
; POSSIBILITY OF SUCH DAMAGE.
;
; Send bugs, suggestions and ideas to: 
;
; felix@call-with-current-continuation.org
;
; Felix L. Winkelmann
; Steinweg 1A
; 37130 Gleichen, OT Weissenborn
; Germany


(declare
  (unit library)
  (interrupts-disabled)
  (standard-bindings)
  (extended-bindings)
  (no-bound-checks)
  (block-global ##sys#ready-queue-head ##sys#ready-queue-tail)
  (bound-to-procedure ##sys#check-char ##sys#check-exact ##sys#check-port ##sys#check-string
		      ##sys#for-each ##sys#map ##sys#setslot ##sys#allocate-vector ##sys#check-pair 
		      ##sys#not-a-proper-list-error ##sys#error ##sys#warn ##sys#signal-hook
		      ##sys#check-symbol ##sys#check-vector ##sys#floor ##sys#ceiling ##sys#truncate ##sys#round 
		      ##sys#check-number ##sys#cons-flonum
		      ##sys#flonum-fraction ##sys#make-port ##sys#fetch-and-check-port-arg ##sys#print 
		      ##sys#check-structure ##sys#make-structure
		      ##sys#macroexpand ##sys#gcd ##sys#lcm ##sys#ensure-heap-reserve ##sys#check-list 
		      ##sys#enable-interrupts ##sys#disable-interrupts ##sys#->feature-id
		      ##sys#call-custom-port-handler ##sys#fudge ##sys#user-read-hook ##sys#check-range ##sys#read
		      ##sys#string->symbol ##sys#symbol->string ##sys#dynamic-unwind
		      ##sys#error-handler ##sys#signal ##sys#abort
		      ##sys#reset-handler ##sys#exit-handler ##sys#dynamic-wind
		      ##sys#grow-vector ##sys#run-pending-finalizers
		      ##sys#add-to-ready-queue ##sys#remove-from-ready-queue
		      ##sys#schedule ##sys#make-thread ##sys#print-to-string
		      ##sys#update-thread-state-buffer ##sys#restore-thread-state-buffer ##sys#user-print-hook 
		      ##sys#current-exception-handler ##sys#default-exception-handler ##sys#abandon-mutexes ##sys#make-mutex
		      ##sys#register-entry-point ##sys#port-file
		      ##sys#dispatch-to-entry-point ##sys#intern-symbol
		      ##sys#append ##sys#list ##sys#cons ##sys#list->vector ##sys#list ##sys#apply ##sys#make-vector
		      ##sys#write-char ##sys#force-finalizers ##sys#cleanup-before-exit
		      ##sys#port-file-resolve ##sys#kill-threads-and-reset ##sys#default-read-info-hook ##sys#read-error)
  (foreign-declare"
#include <string.h>
#include <math.h>
#include <ctype.h>
#include <errno.h>

#ifdef HAVE_SYSEXITS_H
# include <sysexits.h>
#endif

#ifndef EX_SOFTWARE
# define EX_SOFTWARE    70
#endif

#define C_m_op_exp(n)         (C_temporary_flonum = exp(C_c_double(n)), C_SCHEME_UNDEFINED)
#define C_m_op_log(n)         (C_temporary_flonum = log(C_c_double(n)), C_SCHEME_UNDEFINED)
#define C_m_op_sin(n)         (C_temporary_flonum = sin(C_c_double(n)), C_SCHEME_UNDEFINED)
#define C_m_op_cos(n)         (C_temporary_flonum = cos(C_c_double(n)), C_SCHEME_UNDEFINED)
#define C_m_op_tan(n)         (C_temporary_flonum = tan(C_c_double(n)), C_SCHEME_UNDEFINED)
#define C_m_op_asin(n)        (C_temporary_flonum = asin(C_c_double(n)), C_SCHEME_UNDEFINED)
#define C_m_op_acos(n)        (C_temporary_flonum = acos(C_c_double(n)), C_SCHEME_UNDEFINED)
#define C_m_op_atan(n)        (C_temporary_flonum = atan(C_c_double(n)), C_SCHEME_UNDEFINED)
#define C_m_op_atan2(n1, n2)  (C_temporary_flonum = atan2(C_c_double(n1), C_c_double(n2)), C_SCHEME_UNDEFINED)
#define C_m_op_sqrt(n)        (C_temporary_flonum = sqrt(C_c_double(n)), C_SCHEME_UNDEFINED)

#define C_close_file(p)       (fclose((FILE *)(C_port_file(p))), C_SCHEME_UNDEFINED)
#define C_f64peek(b, i)       (C_temporary_flonum = ((double *)C_data_pointer(b))[ C_unfix(i) ], C_SCHEME_UNDEFINED)
#define C_fetch_c_strlen(b, i) C_fix(strlen((char *)C_block_item(b, C_unfix(i))))
#define C_peek_c_string(b, i, to, len) (C_memcpy(C_data_pointer(to), (char *)C_block_item(b, C_unfix(i)), C_unfix(len)), C_SCHEME_UNDEFINED)
") )


(include "parameters")


;;; System routines:

(define (exit . code) (apply (##sys#exit-handler) code))
(define (reset) ((##sys#reset-handler)))

(define (##sys#error msg . args)
  (apply ##sys#signal-hook #:error msg args) )

(define ##sys#warnings-enabled #t)

(define (##sys#warn msg . args)
  (when ##sys#warnings-enabled
    (apply ##sys#signal-hook #:warning msg args) ) )

(define error ##sys#error)

(define (set-gc-report! flag) (##core#inline "C_set_gc_report" flag))
(define ##sys#gc (##core#primitive "C_gc"))
(define gc ##sys#gc)
(define (##sys#setslot x i y) (##core#inline "C_i_setslot" x i y))
(define (##sys#setislot x i y) (##core#inline "C_i_set_i_slot" x i y))
(define ##sys#allocate-vector (##core#primitive "C_allocate_vector"))
(define argv (##core#primitive "C_get_argv"))
(define ##sys#make-structure (##core#primitive "C_make_structure"))
(define ##sys#ensure-heap-reserve (##core#primitive "C_ensure_heap_reserve"))
(define (##sys#fudge fudge-factor) (##core#inline "C_fudge" fudge-factor))
(define ##sys#call-host (##core#primitive "C_call_host"))
(define ##sys#host-data (##core#primitive "C_host_data"))
(define ##sys#set-host-data! (##core#primitive "C_set_host_data"))
(define ##sys#file-info (##core#primitive "C_file_info"))
(define ##sys#symbol-table-info (##core#primitive "C_get_symbol_table_info"))
(define ##sys#memory-info (##core#primitive "C_get_memory_info"))
(define (current-seconds) (##sys#fudge 2))
(define (current-milliseconds) (##sys#fudge 16))
(define ##sys#decode-seconds (##core#primitive "C_decode_seconds"))
(define getenv (##core#primitive "C_get_environment_variable"))
(define (##sys#start-timer) (##core#inline "C_start_timer"))
(define ##sys#stop-timer (##core#primitive "C_stop_timer"))
(define (##sys#immediate? x) (not (##core#inline "C_blockp" x)))
(define (##sys#message str) (##core#inline "C_message" str))
(define (##sys#byte x i) (##core#inline "C_subbyte" x i))
(define (##sys#setbyte x i n) (##core#inline "C_setbyte" x i n))
(define (##sys#check-structure x y) (##core#inline "C_i_check_structure" x y))
(define (##sys#check-byte-vector x) (##core#inline "C_i_check_bytevector" x))
(define (##sys#check-pair x) (##core#inline "C_i_check_pair" x))
(define (##sys#check-list x) (##core#inline "C_i_check_list" x))
(define (##sys#check-string x) (##core#inline "C_i_check_string" x))
(define (##sys#check-number x) (##core#inline "C_i_check_number" x))
(define (##sys#check-exact x) (##core#inline "C_i_check_exact" x))
(define (##sys#check-symbol x) (##core#inline "C_i_check_symbol" x))
(define (##sys#check-vector x) (##core#inline "C_i_check_vector" x))
(define (##sys#check-char x) (##core#inline "C_i_check_char" x))
(define ##sys#callback-continuation-stack '())
(define (void) (##core#undefined))
(define (end-of-file) (##sys#fudge 1))
(define (##sys#halt) (##core#inline "C_halt" #f))

(define ##sys#check-range 
  (lambda (i from to)
    (##sys#check-exact i)
    (if (or (not (fx>= i from))
	    (not (fx< i to)) ) 
	(##sys#error "index out of range" i from to) ) ) )

(cond-expand
 [unsafe
  (eval-when (compile)
    (define-macro (##sys#check-structure x y) '(##core#undefined))
    (define-macro (##sys#check-range x y z) '(##core#undefined))
    (define-macro (##sys#check-pair x) '(##core#undefined))
    (define-macro (##sys#check-list x) '(##core#undefined))
    (define-macro (##sys#check-symbol x) '(##core#undefined))
    (define-macro (##sys#check-string x) '(##core#undefined))
    (define-macro (##sys#check-char x) '(##core#undefined))
    (define-macro (##sys#check-exact x) '(##core#undefined))
    (define-macro (##sys#check-port x) '(##core#undefined))
    (define-macro (##sys#check-number x) '(##core#undefined))
    (define-macro (##sys#check-byte-vector x) '(##core#undefined)) ) ]
 [else] )

(define (force promise)
  (if (##sys#structure? promise 'promise)
      ((##sys#slot promise 1))
      promise) )

(define pathname-directory-separator
  (case (##sys#fudge 3)
    ((#\M) #\:)
    ((#\V #\D) #\\)
    (else #\/) ) )

(define pathname-extension-separator #\.)

(define (system cmd)
  (##sys#check-string cmd)
  (##core#inline "C_execute_shell_command" cmd) )


;;; Operations on booleans:

(define (not x) (##core#inline "C_i_not" x))
(define (boolean? x) (##core#inline "C_booleanp" x))


;;; Equivalence predicates:

(define (eq? x y) (##core#inline "C_eqp" x y))
(define (eqv? x y) (##core#inline "C_i_eqvp" x y))
(define (equal? x y) (##core#inline "C_i_equalp" x y))


;;; Pairs and lists:

(define (pair? x) (##core#inline "C_i_pairp" x))
(define (cons x y) (##core#inline_allocate ("C_a_i_cons" 3) x y))
(define (car x) (##core#inline "C_i_car" x))
(define (cdr x) (##core#inline "C_i_cdr" x))

(define (set-car! x y) (##core#inline "C_i_set_car" x y))
(define (set-cdr! x y) (##core#inline "C_i_set_cdr" x y))
(define (cadr x) (##core#inline "C_i_cadr" x))
(define (caddr x) (##core#inline "C_i_caddr" x))
(define (cadddr x) (##core#inline "C_i_cadddr" x))
(define (cddddr x) (##core#inline "C_i_cddddr" x))

(define (caar x) (car (car x)))
(define (cdar x) (cdr (car x)))
(define (cddr x) (cdr (cdr x)))
(define (caaar x) (car (car (car x))))
(define (caadr x) (car (##core#inline "C_i_cadr" x)))
(define (cadar x) (##core#inline "C_i_cadr" (car x)))
(define (cdaar x) (cdr (car (car x))))
(define (cdadr x) (cdr (##core#inline "C_i_cadr" x)))
(define (cddar x) (cdr (cdr (car x))))
(define (cdddr x) (cdr (cdr (cdr x))))
(define (caaaar x) (car (car (car (car x)))))
(define (caaadr x) (car (car (##core#inline "C_i_cadr" x))))
(define (caadar x) (car (##core#inline "C_i_cadr" (car x))))
(define (caaddr x) (car (##core#inline "C_i_caddr" x)))
(define (cadaar x) (##core#inline "C_i_cadr" (car (car x))))
(define (cadadr x) (##core#inline "C_i_cadr" (##core#inline "C_i_cadr" x)))
(define (caddar x) (##core#inline "C_i_caddr" (car x)))
(define (cdaaar x) (cdr (car (car (car x)))))
(define (cdaadr x) (cdr (car (##core#inline "C_i_cadr" x))))
(define (cdadar x) (cdr (##core#inline "C_i_cadr" (car x))))
(define (cdaddr x) (cdr (##core#inline "C_i_caddr" x)))
(define (cddaar x) (cdr (cdr (car (car x)))))
(define (cddadr x) (cdr (cdr (##core#inline "C_i_cadr" x))))
(define (cdddar x) (cdr (cdr (cdr (car x)))))

(define (null? x) (eq? x '()))
(define (list . lst) lst)
(define (length lst) (##core#inline "C_i_length" lst))
(define (list-tail lst i) (##core#inline "C_i_list_tail" lst i))
(define (list-ref lst i) (##core#inline "C_i_list_ref" lst i))

(define ##sys#not-a-proper-list-error
  (lambda (arg)
    (##sys#signal-hook #:type-error "argument is not a proper list" arg) ) )

(define append
  (lambda lsts
    (if (eq? lsts '())
	lsts
	(let loop ((lsts lsts))
	  (if (eq? (##sys#slot lsts 1) '())
	      (##sys#slot lsts 0)
	      (let copy ((node (##sys#slot lsts 0)))
		(cond-expand
		 [unsafe
		  (if (eq? node '()) 
		      (loop (##sys#slot lsts 1))
		      (cons (##sys#slot node 0) (copy (##sys#slot node 1))) ) ]
		 [else
		  (cond ((eq? node '()) (loop (##sys#slot lsts 1)))
			((pair? node)
			 (cons (##sys#slot node 0) (copy (##sys#slot node 1))) )
			(else (##sys#not-a-proper-list-error (##sys#slot lsts 0))) ) ] ) ) ) ) ) ) )

(define reverse 
  (lambda (lst0)
    (let loop ((lst lst0) (rest '()))
      (cond-expand
       [unsafe
	(if (eq? lst '()) 
	    rest
	    (loop (##sys#slot lst 1) (cons (##sys#slot lst 0) rest))  ) ]
       [else
	(cond ((eq? lst '()) rest)
	      ((pair? lst)
	       (loop (##sys#slot lst 1) (cons (##sys#slot lst 0) rest)) )
	      (else (##sys#not-a-proper-list-error lst0)) ) ] ) ) ) )

(define (memq x lst) (##core#inline "C_i_memq" x lst))
(define (memv x lst) (##core#inline "C_i_memv" x lst))
(define (member x lst) (##core#inline "C_i_member" x lst))
(define (assq x lst) (##core#inline "C_i_assq" x lst))
(define (assv x lst) (##core#inline "C_i_assv" x lst))
(define (assoc x lst) (##core#inline "C_i_assoc" x lst))

(define (list? x) (##core#inline "C_i_listp" x))


;;; Strings:

(define (string? x) (##core#inline "C_i_stringp" x))
(define (string-length s) (##core#inline "C_i_string_length" s))
(define (string-ref s i) (##core#inline "C_i_string_ref" s i))
(define (string-set! s i c) (##core#inline "C_i_string_set" s i c))

(define (make-string size . fill)
  (##sys#check-exact size)
  (##sys#allocate-vector
   size #t
   (if (null? fill)
       #\space
       (let ((c (car fill)))
	 (begin (##sys#check-char c) c) ) )
   #f) )

(define string->list 
  (lambda (s)
    (##sys#check-string s)
    (let ((len (##core#inline "C_block_size" s)))
      (let loop ((i 0))
	(if (fx>= i len)
	    '()
	    (cons (##core#inline "C_subchar" s i)
		  (loop (fx+ i 1)) ) ) ) ) ) )

(define list->string
  (let ([make-string make-string])
    (lambda (lst0)
      (let ([s #f])
	(let loop ([lst lst0] [i 0])
	  (cond [(eq? lst '()) (set! s (make-string i))]
		[(and (##core#inline "C_blockp" lst) (##core#inline "C_pairp" lst))
		 (loop (##sys#slot lst 1) (fx+ i 1))
		 (let ([c (##sys#slot lst 0)])
		   (##sys#check-char c)
		   (##core#inline "C_setsubchar" s i c) ) ]
		[else (##sys#not-a-proper-list-error lst0)] ) )
	s) ) ) )

(define (string-fill! s c)
  (##sys#check-string s)
  (##sys#check-char c)
  (##core#inline "C_set_memory" s c (##sys#size s))
  (##core#undefined) )

(define string-copy
  (let ([make-string make-string])
    (lambda (s)
      (##sys#check-string s)
      (let* ([len (##sys#size s)]
	     [s2 (make-string len)] )
	(##core#inline "C_copy_memory" s2 s len)
	s2) ) ) )

(define substring
  (let ([make-string make-string])
    (lambda (s start end)
      (##sys#check-string s)
      (##sys#check-exact start)
      (##sys#check-exact end)
      (let ((len (##core#inline "C_block_size" s)))
	(if (cond-expand
	     [(not unsafe)
	      (or (fx< start 0)
		  (fx> start len)
		  (fx< end 0)
		  (fx> end len)
		  (fx> start end) ) ]
	     [else #f] )
	    (##sys#error "substring-index out of bounds" start end)
	    (let* ((len2 (fx- end start))
		   (s2 (make-string len2)) )
	      (##core#inline "C_substring_copy" s s2 start end 0)
	      s2) ) ) ) ) )

(define (string=? x y)
  (cond-expand [unsafe (##core#inline "C_u_i_string_equal_p" x y)]
	       [else (##core#inline "C_i_string_equal_p" x y)] ) )

(define (string-ci=? x y) (##core#inline "C_i_string_ci_equal_p" x y))

(letrec ((compare 
	  (lambda (s1 s2 k)
	    (##sys#check-string s1)
	    (##sys#check-string s2)
	    (let ((len1 (##core#inline "C_block_size" s1))
		  (len2 (##core#inline "C_block_size" s2)) )
	      (k len1 len2
		 (##core#inline "C_string_compare"
			    s1
			    s2
			    (if (fx< len1 len2)
				len1
				len2) ) ) ) ) ) )
  (set! string<? (lambda (s1 s2)
		   (compare 
		    s1 s2
		    (lambda (len1 len2 cmp)
		      (or (fx< cmp 0)
			  (and (fx< len1 len2)
			       (eq? cmp 0) ) ) ) ) ) )
  (set! string>? (lambda (s1 s2)
		   (compare 
		    s1 s2
		    (lambda (len1 len2 cmp)
		      (or (fx> cmp 0)
			  (and (fx< len2 len1)
			       (eq? cmp 0) ) ) ) ) ) )
  (set! string<=? (lambda (s1 s2)
		    (compare 
		     s1 s2
		     (lambda (len1 len2 cmp)
		       (if (eq? cmp 0)
			   (fx>= len1 len2)
			   (fx< cmp 0) ) ) ) ) )
  (set! string>=? (lambda (s1 s2)
		    (compare 
		     s1 s2
		     (lambda (len1 len2 cmp)
		       (if (eq? cmp 0)
			   (fx<= len1 len2)
			   (fx> cmp 0) ) ) ) ) ) )

(letrec ((compare 
	  (lambda (s1 s2 k)
	    (##sys#check-string s1)
	    (##sys#check-string s2)
	    (let ((len1 (##core#inline "C_block_size" s1))
		  (len2 (##core#inline "C_block_size" s2)) )
	      (k len1 len2
		 (##core#inline "C_string_compare_case_insensitive"
				s1
				s2
				(if (fx< len1 len2)
				    len1
				    len2) ) ) ) ) ) )
  (set! string-ci<? (lambda (s1 s2)
		      (compare 
		       s1 s2
		       (lambda (len1 len2 cmp)
			 (or (fx< cmp 0)
			     (and (fx< len1 len2)
				  (eq? cmp 0) ) ) ) ) ) )
  (set! string-ci>? (lambda (s1 s2)
		      (compare 
		       s1 s2
		       (lambda (len1 len2 cmp)
			 (or (fx> cmp 0)
			     (and (fx< len2 len1)
				  (eq? cmp 0) ) ) ) ) ) )
  (set! string-ci<=? (lambda (s1 s2)
		       (compare 
			s1 s2
			(lambda (len1 len2 cmp)
			  (if (eq? cmp 0)
			      (fx>= len1 len2)
			      (fx< cmp 0) ) ) ) ) )
  (set! string-ci>=? (lambda (s1 s2)
		       (compare 
			s1 s2
			(lambda (len1 len2 cmp)
			  (if (eq? cmp 0)
			      (fx<= len1 len2)
			      (fx> cmp 0) ) ) ) ) ) )

(define string-append
  (let ([make-string make-string])
    (lambda all
      (let ([snew #f])
	(let loop ([strs all] [n 0])
	  (if (eq? strs '())
	      (set! snew (make-string n))
	      (let ([s (##sys#slot strs 0)])
		(##sys#check-string s)
		(let ([len (##sys#size s)])
		  (loop (##sys#slot strs 1) (fx+ n len))
		  (##core#inline "C_substring_copy" s snew 0 len n) ) ) ) )
	snew) ) ) )

(define string
  (let ([list->string list->string])
    (lambda chars (list->string chars)) ) )


;;; Numeric routines:

(define (fixnum? x) (##core#inline "C_fixnump" x))
(define (fx+ x y) (##core#inline "C_fixnum_plus" x y))
(define (fx- x y) (##core#inline "C_fixnum_difference" x y))
(define (fx* x y) (##core#inline "C_fixnum_times" x y))
(define (fx= x y) (eq? x y))
(define (fx> x y) (##core#inline "C_fixnum_greaterp" x y))
(define (fx< x y) (##core#inline "C_fixnum_lessp" x y))
(define (fx>= x y) (##core#inline "C_fixnum_greater_or_equal_p" x y))
(define (fx<= x y) (##core#inline "C_fixnum_less_or_equal_p" x y))
(define (fxmin x y) (##core#inline "C_i_fixnum_min" x y))
(define (fxmax x y) (##core#inline "C_i_fixnum_max" x y))
(define (fxneg x) (##core#inline "C_fixnum_negate" x))

(define fx/
  (lambda (x y)
    (cond-expand
     [unsafe (##core#inline "C_fixnum_divide" x y)]
     [else
      (if (eq? y 0)
	  (##sys#signal-hook #:arithmetic-error "division by zero" x y)
	  (##core#inline "C_fixnum_divide" x y) ) ] ) ) )

(define fxmod
  (lambda (x y)
    (cond-expand
     [unsafe (##core#inline "C_fixnum_modulo" x y)]
     [else
      (if (eq? y 0)
	  (##sys#signal-hook #:arithmetic-error "division by zero" x y)
	  (##core#inline "C_fixnum_modulo" x y) ) ] ) ) )

(define * (##core#primitive "C_times"))
(define - (##core#primitive "C_minus"))
(define + (##core#primitive "C_plus"))
(define / (##core#primitive "C_divide"))
(define = (##core#primitive "C_nequalp"))
(define > (##core#primitive "C_greaterp"))
(define < (##core#primitive "C_lessp"))
(define >= (##core#primitive "C_greater_or_equal_p"))
(define <= (##core#primitive "C_less_or_equal_p"))

(define add1 (lambda (n) (+ n 1)))
(define sub1 (lambda (n) (- n 1)))

(define ##sys#floor (##core#primitive "C_flonum_floor"))
(define ##sys#ceiling (##core#primitive "C_flonum_ceiling"))
(define ##sys#truncate (##core#primitive "C_flonum_truncate"))
(define ##sys#round (##core#primitive "C_flonum_round"))
(define quotient (##core#primitive "C_quotient"))
(define ##sys#cons-flonum (##core#primitive "C_cons_flonum"))
(define (number? x) (##core#inline "C_i_numberp" x))
(define complex? number?)
(define real? number?)
(define rational? number?)
(define ##sys#flonum-fraction (##core#primitive "C_flonum_fraction"))
(define (integer? x) (##core#inline "C_i_integerp" x))
(define (exact? x) (##core#inline "C_i_exactp" x))
(define (inexact? x) (##core#inline "C_i_inexactp" x))
(define expt (##core#primitive "C_expt"))
(define (##sys#fits-in-int? n) (##core#inline "C_fits_in_int_p" n))
(define (##sys#fits-in-unsigned-int? n) (##core#inline "C_fits_in_unsigned_int_p" n))
(define (##sys#flonum-in-fixnum-range? n) (##core#inline "C_flonum_in_fixnum_range_p" n))
(define (zero? n) (##core#inline "C_i_zerop" n))
(define (positive? n) (##core#inline "C_i_positivep" n))
(define (negative? n) (##core#inline "C_i_negativep" n))
(define (abs n) (##core#inline_allocate ("C_a_i_abs" 4) n))	; 4 => words-per-flonum

(define signum
  (lambda (n)
    (cond ((> n 0) 1)
	  ((< n 0) -1)
	  (else 0) ) ) )

(define exact->inexact (##core#primitive "C_exact_to_inexact"))
(define (inexact->exact n) (##core#inline "C_i_inexact_to_exact" n))

(define (floor x)
  (##sys#check-number x)
  (if (##core#inline "C_fixnump" x) 
      x
      (##sys#floor x) ) )

(define (ceiling x)
  (##sys#check-number x)
  (if (##core#inline "C_fixnump" x) 
      x
      (##sys#ceiling x) ) )

(define (truncate x)
  (##sys#check-number x)
  (if (##core#inline "C_fixnump" x) 
      x
      (##sys#truncate x) ) )

(define (round x)
  (##sys#check-number x)
  (if (##core#inline "C_fixnump" x) 
      x
      (##sys#round x) ) )

(define remainder 
  (lambda (x y) (- x (* (quotient x y) y))) )

(define modulo
  (let ([floor floor])
    (lambda (x y)
      (let ((div (/ x y)))
	(- x (* (if (integer? div)
		    div
		    (let* ([fd (floor div)]
			   [fdx (##core#inline "C_quickflonumtruncate" fd)] )
		      (if (= fd fdx)
			  fdx
			  fd) ) )
		y) ) ) ) ) )

(define (even? n) (##core#inline "C_i_evenp" n))
(define (odd? n) (##core#inline "C_i_oddp" n))

(let ([> >]
      [< <] )
  (letrec ([maxmin
	    (lambda (n1 ns pred)
	      (let loop ((nbest n1) (ns ns))
		(if (eq? ns '())
		    nbest
		    (let ([ni (##sys#slot ns 0)])
		      (loop (if (pred ni nbest)
				(if (and (##core#inline "C_blockp" nbest) 
					 (##core#inline "C_flonump" nbest) 
					 (not (##core#inline "C_blockp" ni)) )
				    (exact->inexact ni)
				    ni)
				nbest)
			    (##sys#slot ns 1) ) ) ) ) ) ] )
    (set! max (lambda (n1 . ns) (maxmin n1 ns >)))
    (set! min (lambda (n1 . ns) (maxmin n1 ns <))) ) )

(define (exp n)
  (##sys#check-number n)
  (##core#inline "C_m_op_exp" n)
  (##sys#cons-flonum) )

(define (log n)
  (##sys#check-number n)
  (##core#inline "C_m_op_log" n)
  (##sys#cons-flonum) )

(define (sin n)
  (##sys#check-number n)
  (##core#inline "C_m_op_sin" n)
  (##sys#cons-flonum) )

(define (cos n)
  (##sys#check-number n)
  (##core#inline "C_m_op_cos" n)
  (##sys#cons-flonum) )

(define (tan n)
  (##sys#check-number n)
  (##core#inline "C_m_op_tan" n)
  (##sys#cons-flonum) )

(define (asin n)
  (##sys#check-number n)
  (##core#inline "C_m_op_asin" n)
  (##sys#cons-flonum) )

(define (acos n)
  (##sys#check-number n)
  (##core#inline "C_m_op_acos" n)
  (##sys#cons-flonum) )

(define (atan n1 . n2)
  (##sys#check-number n1)
  (cond ((null? n2) (##core#inline "C_m_op_atan" n1))
	(else
	 (let ((n2 (car n2)))
	   (##sys#check-number n2)
	   (##core#inline "C_m_op_atan2" n1 n2) ) ) )
  (##sys#cons-flonum) )

(define (sqrt n)
  (##sys#check-number n)
  (##core#inline "C_m_op_sqrt" n)
  (##sys#cons-flonum) )

(define ##sys#gcd
  (let ((remainder remainder))
    (lambda (x y)
      (let loop ((x x) (y y))
	(if (zero? y)
	    (abs x)
	    (loop y (remainder x y)) ) ) ) ) )

(define gcd
  (lambda ns
    (if (eq? ns '())
	0
	(let loop ((ns ns))
	  (let ((head (##sys#slot ns 0))
		(next (##sys#slot ns 1)) )
	    (if (eq? next '())
		(abs head)
		(loop (cons (##sys#gcd head (##sys#slot next 0))
			    (##sys#slot next 1) ) ) ) ) ) ) ) )

(define ##sys#lcm
  (lambda (x y) (quotient (* x y) (##sys#gcd x y))) )

(define lcm
  (lambda ns
    (if (eq? ns '())
	1
	(let loop ((ns ns))
	  (let ((head (##sys#slot ns 0))
		(next (##sys#slot ns 1)) )
	    (if (eq? next '())
		(abs head)
		(loop (cons (##sys#lcm head (##sys#slot next 0))
			    (##sys#slot next 1) ) ) ) ) ) ) ) )

(define string->number (##core#primitive "C_string_to_number"))
(define number->string (##core#primitive "C_number_to_string"))


;;; Symbols:

(define ##sys#make-symbol (##core#primitive "C_make_symbol"))
(define (symbol? x) (##core#inline "C_i_symbolp" x))
(define ##sys#snafu '##sys#fnord)
(define ##sys#intern-symbol (##core#primitive "C_string_to_symbol"))

(define (##sys#string->symbol str)
  (##sys#check-string str)
  (##sys#intern-symbol str) )

(let ([string-append string-append]
      [substring substring] 
      [string-copy string-copy] )
  (define (split str len)
    (let ([b0 (##sys#byte str 0)])	; we fetch the byte, wether len is 0 or not
      (if (and (fx> len 0) (fx< b0 len) (fx<= b0 namespace-max-id-len))
	  (fx+ b0 1)
	  #f) ) )
  (set! ##sys#symbol->string
    (lambda (s)
      (let* ([str (##sys#slot s 1)]
	     [len (##sys#size str)]
	     [i (split str len)] )
	(if i (substring str i len) str) ) ) )
  (set! ##sys#symbol->qualified-string 
    (lambda (s)
      (let* ([str (##sys#slot s 1)]
	     [len (##sys#size str)] 
	     [i (split str len)] )
	(if i
	    (string-append "##" (substring str 1 i) "#" (substring str i len))
	    str) ) ) )
  (set! ##sys#qualified-symbol-prefix 
    (lambda (s)
      (let* ([str (##sys#slot s 1)]
	     [len (##sys#size str)]
	     [i (split str len)] )
	(and i (substring str 0 i)) ) ) ) )

(define ##sys#string->qualified-symbol
  (let ([string-append string-append])
    (lambda (prefix str)
      (##sys#string->symbol
       (if prefix
	   (string-append prefix str)
	   str) ) ) ) )

(define (symbol->string s)
  (##sys#check-symbol s)
  (##sys#symbol->string s) )

(define string->symbol
  (let ([string-copy string-copy])
    (lambda (str)
      (##sys#check-string str)
      (##sys#string->symbol (string-copy str)) ) ) )

(define string->uninterned-symbol
  (let ([string-copy string-copy])
    (lambda (str)
      (##sys#check-string str)
      (##sys#make-symbol (string-copy str)) ) ) )

(define gensym
  (let ([counter -1]
	[string-append string-append] )
    (lambda str-or-sym
      (let ([err (lambda (prefix) (##sys#signal-hook #:type-error "argument is not a string or symbol" prefix))])
	(set! counter (fx+ counter 1))
	(##sys#make-symbol
	 (string-append
	  (if (eq? str-or-sym '())
	      "g"
	      (let ([prefix (car str-or-sym)])
		(or (and (##core#inline "C_blockp" prefix)
			 (cond [(##core#inline "C_stringp" prefix) prefix]
			       [(##core#inline "C_symbolp" prefix) (##sys#symbol->string prefix)]
			       [else (err prefix)] ) )
		    (err prefix) ) ) )
	  (number->string counter) ) ) ) ) ) )


;;; Keywords:

(define (keyword? x)
  (and (symbol? x) (fx= 0 (##sys#byte (##sys#slot x 1) 0))) )

(define string->keyword
  (let ([string-append string-append]
	[string string] )
    (lambda (s)
      (##sys#check-string s)
      (##sys#string->symbol (string-append (string (integer->char 0)) s)) ) ) )

(define keyword->string
  (let ([keyword? keyword?])
    (lambda (kw)
      (if (keyword? kw)
	  (##sys#symbol->string kw)
	  (##sys#signal-hook #:type-error "bad argument type - not a keyword" kw) ) ) ) )

(define get-keyword 
    (lambda (key args0 . default)
      (##sys#check-symbol key)
      (##sys#check-list args0)
      (let loop ([args args0])
	(if (null? args)
	    (and (pair? default) ((car default)))
	    (cond-expand
	     [unsafe
	      (let ([x (##sys#slot args 0)]
		    [r (##sys#slot args 1)] )
		(cond [(eq? x key) (##sys#slot r 0)]
		      [(and (symbol? x) (fx= 0 (##sys#byte (##sys#slot x 1) 0)))
		       (loop (##sys#slot r 1)) ]
		      [else (loop r)] ) ) ]
	     [else
	      (if (pair? args)
		  (let ([x (##sys#slot args 0)]
			[r (##sys#slot args 1)] )
		    (cond [(and (symbol? x) (fx= 0 (##sys#byte (##sys#slot x 1) 0)))
			   (unless (pair? r) (##sys#error "invalid keyword argument list" args0))
			   (if (eq? key x)
			       (##sys#slot r 0)
			       (loop (##sys#slot r 1)) ) ]
			  [else (loop r)] ) )
		  (##sys#not-a-proper-list-error args0) ) ] ) ) ) ) )


;;; Vectors:

(define (vector? x) (##core#inline "C_i_vectorp" x))
(define (vector-length v) (##core#inline "C_i_vector_length" v))
(define (vector-ref v i) (##core#inline "C_i_vector_ref" v i))
(define (vector-set! v i x) (##core#inline "C_i_vector_set" v i x))

(define (##sys#make-vector size . fill)
  (##sys#check-exact size)
  (##sys#allocate-vector
   size #f
   (if (null? fill)
       (##core#undefined)
       (car fill) )
   #f) )

(define make-vector ##sys#make-vector)

(define list->vector
  (lambda (lst0)
    (let ((v #f))
      (let loop ((lst lst0) (i 0))
	(cond ((eq? lst '()) (set! v (##sys#make-vector i)))
	      ((and (##core#inline "C_blockp" lst) (##core#inline "C_pairp" lst))
	       (loop (##sys#slot lst 1) (fx+ i 1))
	       (##sys#setslot v i (##sys#slot lst 0)) )
	      (else (##sys#not-a-proper-list-error lst0)) ) )
      v) ) )

(define vector->list
  (lambda (v)
    (##sys#check-vector v)
    (let ((len (##core#inline "C_block_size" v)))
      (let loop ((i 0))
	(if (fx>= i len)
	    '()
	    (cons (##sys#slot v i)
		  (loop (fx+ i 1)) ) ) ) ) ) )

(define vector
  (lambda xs (##sys#list->vector xs)) )

(define (vector-fill! v x)
  (##sys#check-vector v)
  (let ((len (##core#inline "C_block_size" v)))
    (do ((i 0 (fx+ i 1)))
	((fx>= i len))
      (##sys#setslot v i x) ) ) )

(define vector-copy!
    (lambda (from to . n)
      (##sys#check-vector from)
      (##sys#check-vector to)
      (let* ([len-from (##sys#size from)]
	     [len-to (##sys#size to)] 
	     [n (if (pair? n) (##sys#slot n 0) len-to)] )
	(##sys#check-exact n)
	(cond-expand
	 [(not unsafe)
	  (when (fx> n len-to)
	    (##sys#error "destination vector too small" from to n) ) ]
	 [else] )
	(do ([i 0 (fx+ i 1)])
	    ((fx>= i n))
	  (##sys#setslot to i (##sys#slot from i)) ) ) ) )

(define ##sys#grow-vector 
  (lambda (v n init)
    (let ([v2 (##sys#make-vector n init)]
	  [len (##sys#size v)] )
      (do ([i 0 (fx+ i 1)])
	  ((fx>= i len) v2)
	(##sys#setslot v2 i (##sys#slot v i)) ) ) ) )
	

;;; Characters:

(define (char? x) (##core#inline "C_charp" x))

(define (char->integer c)
  (##sys#check-char c)
  (##core#inline "C_fix" (##core#inline "C_character_code" c)) )

(define (integer->char n)
  (##sys#check-exact n)
  (##core#inline "C_make_character" (##core#inline "C_unfix" n)) )

(define (char=? c1 c2)
  (##sys#check-char c1)
  (##sys#check-char c2)
  (eq? c1 c2) )

(define (char>? c1 c2)
  (##sys#check-char c1)
  (##sys#check-char c2)
  (fx> c1 c2) )

(define (char<? c1 c2)
  (##sys#check-char c1)
  (##sys#check-char c2)
  (fx< c1 c2) )

(define (char>=? c1 c2)
  (##sys#check-char c1)
  (##sys#check-char c2)
  (fx>= c1 c2) )

(define (char<=? c1 c2)
  (##sys#check-char c1)
  (##sys#check-char c2)
  (fx<= c1 c2) )

(define (char-upcase c)
  (##sys#check-char c)
  (##core#inline "C_make_character"
	     (##core#inline toupper (##core#inline "C_character_code" c)) ) )

(define (char-downcase c)
  (##sys#check-char c)
  (##core#inline "C_make_character"
	     (##core#inline tolower (##core#inline "C_character_code" c)) ) )

(let ((char-downcase char-downcase))
  (set! char-ci=? (lambda (x y) (eq? (char-downcase x) (char-downcase y))))
  (set! char-ci>? (lambda (x y) (fx> (char-downcase x) (char-downcase y))))
  (set! char-ci<? (lambda (x y) (fx< (char-downcase x) (char-downcase y))))
  (set! char-ci>=? (lambda (x y) (fx>= (char-downcase x) (char-downcase y))))
  (set! char-ci<=? (lambda (x y) (fx<= (char-downcase x) (char-downcase y)))) )

(define (char-upper-case? c)
  (##sys#check-char c)
  (##core#inline "C_u_i_char_upper_casep" c) )

(define (char-lower-case? c)
  (##sys#check-char c)
  (##core#inline "C_u_i_char_lower_casep" c) )

(define (char-numeric? c)
  (##sys#check-char c)
  (##core#inline "C_u_i_char_numericp" c) )

(define (char-whitespace? c)
  (##sys#check-char c)
  (##core#inline "C_u_i_char_whitespacep" c) )

(define (char-alphabetic? c)
  (##sys#check-char c)
  (##core#inline "C_u_i_char_alphabeticp" c) )

(define char-name
  (let ([chars-to-names (make-vector char-name-table-size '())]
	[names-to-chars '()] )

    (define (lookup-char c)
      (let* ([code (char->integer c)]
	     [key (##core#inline "C_fixnum_modulo" code char-name-table-size)] )
	(let loop ([b (##sys#slot chars-to-names key)])
	  (and (pair? b)
	       (let ([a (##sys#slot b 0)])
		 (if (eq? (##sys#slot a 0) c)
		     a
		     (loop (##sys#slot b 1)) ) ) ) ) ) )

    (lambda (x . y)
      (let ([chr (if (pair? y) (car y) #f)])
	(cond [(char? x)
	       (and-let* ([a (lookup-char x)])
		 (##sys#slot a 1) ) ]
	      [chr
	       (##sys#check-symbol x)
	       (##sys#check-char chr)
	       (let ([a (lookup-char chr)])
		 (if a 
		     (let ([b (assq x names-to-chars)])
		       (##sys#setslot a 1 x)
		       (if b
			   (##sys#setislot b 1 chr)
			   (set! names-to-chars (cons (cons x chr) names-to-chars)) ) )
		     (let ([key (##core#inline "C_fixnum_modulo" (char->integer chr) char-name-table-size)])
		       (set! names-to-chars (cons (cons x chr) names-to-chars))
		       (##sys#setslot chars-to-names key (cons (cons chr x) (##sys#slot chars-to-names key))) ) ) ) ]
	      [else
	       (##sys#check-symbol x)
	       (and-let* ([a (assq x names-to-chars)])
		 (##sys#slot a 1) ) ] ) ) ) ) )

(char-name 'space #\space)
(char-name 'tab #\tab)
(char-name 'linefeed #\linefeed)
(char-name 'newline #\newline)
(char-name 'return #\return)
(char-name 'page (integer->char 12))
(char-name 'backspace (integer->char 8))


;;; Procedures:

(define (procedure? x) (##core#inline "C_i_closurep" x))
(define apply (##core#primitive "C_apply"))
(define ##sys#call-with-current-continuation (##core#primitive "C_call_cc"))
(define values (##core#primitive "C_values"))
(define ##sys#call-with-values (##core#primitive "C_call_with_values"))
(define call-with-values ##sys#call-with-values)

(define (##sys#for-each p lst0)
  (let loop ((lst lst0))
    (cond-expand
     [unsafe
      (if (eq? lst '()) 
	  (##core#undefined)
	  (begin
	    (p (##sys#slot lst 0))
	    (loop (##sys#slot lst 1)) ) ) ]
     [else
      (cond ((eq? lst '()) (##core#undefined))
	    ((pair? lst)
	     (p (##sys#slot lst 0))
	     (loop (##sys#slot lst 1)) )
	    (else (##sys#not-a-proper-list-error lst0)) ) ] ) ) )

(define (##sys#map p lst0)
  (let loop ((lst lst0))
    (cond-expand
     [unsafe
      (if (eq? lst '()) 
	  lst
	  (cons (p (##sys#slot lst 0)) (loop (##sys#slot lst 1))) ) ]
     [else
      (cond ((eq? lst '()) lst)
	    ((pair? lst)
	     (cons (p (##sys#slot lst 0)) (loop (##sys#slot lst 1))) )
	    (else (##sys#not-a-proper-list-error lst0)) ) ] ) ) )

(let ([car car]
      [cdr cdr] )
  (letrec ((mapsafe
	    (lambda (p lsts start)
	      (if (eq? lsts '())
		  lsts
		  (let ((item (##sys#slot lsts 0)))
		    (cond ((eq? item '())
			   (cond-expand [unsafe (##core#undefined)]
					[else (check lsts start)] ) )
			  ((pair? item)
			   (cons (p item) (mapsafe p (##sys#slot lsts 1) #f)) )
			  (else (##sys#not-a-proper-list-error item)) ) ) ) ) )
	   (check 
	    (lambda (lsts start)
	      (if (or (not start)
		      (let loop ((lsts lsts))
			(and (not (eq? lsts '()))
			     (not (eq? (##sys#slot lsts 0) '()))
			     (loop (##sys#slot lsts 1)) ) ) )
		  (##sys#error "lists are not of same length" lsts) ) ) ) )
    (set! for-each
	  (lambda (fn lst1 . lsts)
	    (if (null? lsts)
		(##sys#for-each fn lst1)
		(let loop ((all (cons lst1 lsts)))
		  (let ((first (##sys#slot all 0)))
		    (cond ((pair? first)
			   (apply fn (mapsafe car all #t))
			   (loop (mapsafe cdr all #t)) )
			  (else (check all #t)) ) ) ) ) ) )
    (set! map
	  (lambda (fn lst1 . lsts)
	    (if (null? lsts)
		(##sys#map fn lst1)
		(let loop ((all (cons lst1 lsts)))
		  (let ((first (##sys#slot all 0)))
		    (cond ((pair? first)
			   (cons (apply fn (mapsafe car all #t))
				 (loop (mapsafe cdr all #t)) ) )
			  (else (check (##core#inline "C_i_cdr" all) #t)
				'() ) ) ) ) ) ) ) ) )


;;; dynamic-wind:
;
; (taken more or less directly from SLIB)
;
; This implementation is relatively costly: we have to shadow call/cc
; with a new version that unwinds suspended thunks, but for this to
; happen the return-values of the escaping procedure have to be saved
; temporarily in a list. Since call/cc is very efficient under this
; implementation, and because allocation of memory that is to be
; garbage soon has also quite a low overhead, the performance-penalty
; might be acceptable (ctak needs about 4 times longer).

(define ##sys#dynamic-winds '())

(define dynamic-wind
  (let ((values values))
    (lambda (before thunk after)
      (before)
      (set! ##sys#dynamic-winds (cons (cons before after) ##sys#dynamic-winds))
      (##sys#call-with-values thunk
	(lambda results
	  (set! ##sys#dynamic-winds (##sys#slot ##sys#dynamic-winds 1))
	  (after)
	  (apply values results) ) ) ) ) )

(define ##sys#dynamic-wind dynamic-wind)

(define call-with-current-continuation
  (lambda (proc)
    (let ((winds ##sys#dynamic-winds))
      (##sys#call-with-current-continuation
       (lambda (cont)
	 (proc
	  (lambda results
	    (##sys#dynamic-unwind winds (fx- (length ##sys#dynamic-winds) (length winds)))
	    (apply cont results) ) ) ) ) ) ) )

(define (##sys#dynamic-unwind winds n)
  (cond [(eq? ##sys#dynamic-winds winds)]
	[(fx< n 0)
	 (##sys#dynamic-unwind (##sys#slot winds 1) (fx+ n 1))
	 ((##sys#slot (##sys#slot winds 0) 0))
	 (set! ##sys#dynamic-winds winds) ]
	[else
	 (let ([after (##sys#slot (##sys#slot ##sys#dynamic-winds 0) 1)])
	   (set! ##sys#dynamic-winds (##sys#slot ##sys#dynamic-winds 1))
	   (after)
	   (##sys#dynamic-unwind winds (fx- n 1)) ) ] ) )


;;; Ports:

(define (port? x) (##core#inline "C_i_portp" x))

(define (input-port? x)
  (and (##core#inline "C_blockp" x)
       (##core#inline "C_portp" x)
       (not (##sys#slot x 1)) ) )

(define (output-port? x)
  (and (##core#inline "C_blockp" x)
       (##core#inline "C_portp" x)
       (##sys#slot x 1) ) )

(define ##sys#make-port (##core#primitive "C_make_port"))
(define ##sys#standard-input (##sys#make-port 0 6 #f #f))
(define ##sys#standard-output (##sys#make-port 1 6 #f #f))
(define ##sys#standard-error (##sys#make-port 2 6 #f #f))

(##sys#setslot ##sys#standard-input 3 "(stdin)")
(##sys#setslot ##sys#standard-input 4 0)
(##sys#setslot ##sys#standard-input 5 0)
(##sys#setslot ##sys#standard-output 1 #t)
(##sys#setslot ##sys#standard-output 3 "(stdout)")
(##sys#setslot ##sys#standard-output 4 0)
(##sys#setslot ##sys#standard-output 5 0)
(##sys#setslot ##sys#standard-error 1 #t)
(##sys#setslot ##sys#standard-error 3 "(stderr)")
(##sys#setslot ##sys#standard-error 4 0)
(##sys#setslot ##sys#standard-error 5 0)

(define ##sys#check-port
    (lambda (x)
      (if (or (not (##core#inline "C_blockp" x))
	      (not (##core#inline "C_portp" x)) )
	  (##sys#signal-hook #:type-error "argument is not a port" x) ) ) )

(define (##sys#fetch-and-check-port-arg parg default)
  (let ((p (if (eq? parg '())
	       default
	       (##sys#slot parg 0) ) ) )
    (##sys#check-port p)
    p) )

(define (##sys#call-custom-port-handler operation port args defhandler)
  (let* ([h (##sys#slot port 2)]
	 [r (and h (h operation port args))] )
    (if (and h (not (eq? r ##sys#snafu)))
	r
	(defhandler port) ) ) )

(define (current-output-port) ##sys#standard-output)
(define (current-input-port) ##sys#standard-input)
(define (current-error-port) ##sys#standard-error)
(define (##sys#tty-port? port) (##core#inline "C_tty_portp" port))

(define ##sys#port-file-resolve
    (lambda (port)
      (##sys#call-custom-port-handler #:resolve port '() (lambda (p) p)) ) )

(define ##sys#port-file
    (lambda (port)
      (let ([fp (##sys#peek-unsigned-integer (##sys#port-file-resolve port) 0)])
	(if (fx= fp 0)
	    (##sys#signal-hook #:type-error "not a file port" port)
	    fp) ) ) )

  (letrec ([open 
	    (lambda (name outp modes)
	      (##sys#check-string name)
	      (let ([bmode #f]
		    [iomode (if outp #\w #\r)] )
		(do ([modes modes (##sys#slot modes 1)])
		    ((null? modes))
		  (let ([o (##sys#slot modes 0)])
		    (case o
		      [(###binary) (set! bmode #t)]
		      [(###text) (set! bmode #f)]
		      [(###append) (set! iomode #\a)]
		      [else (##sys#error "invalid file option" o)] ) ) )
		(let ([port (##sys#make-port name 6 bmode iomode)])
		  (##sys#update-errno)
		  (unless port (##sys#signal-hook #:file-error "can not open file" name))
		  (##sys#setslot port 1 outp)
		  (##sys#setslot port 3 name)
		  (##sys#setslot port 4 0)
		  (##sys#setslot port 5 0)
		  port) ) ) ]
	   [close
	    (lambda (port mode)
	      (##sys#check-port port)
	      (unless (eq? 0 (##sys#slot port 1))
		(##sys#call-custom-port-handler
		 mode port '()
		 (lambda (p)
		   (let ([r (##core#inline "C_close_file" (##sys#port-file-resolve p))])
		     (##sys#update-errno)
		     r) ) )
		(##sys#setslot port 1 0) )
	      (##core#undefined) ) ] )
    (set! open-input-file (lambda (name . mode) (open name #f mode)))
    (set! open-output-file (lambda (name . mode) (open name #t mode)))
    (set! close-input-port (lambda (port) (close port '#:close-input-port)))
    (set! close-output-port (lambda (port) (close port '#:close-output-port))) )

(define call-with-input-file
  (let ([open-input-file open-input-file]
	[close-input-port close-input-port] )
    (lambda (name p . mode)
      (let ([f (apply open-input-file name mode)])
	(##sys#call-with-values
	 (lambda () (p f))
	 (lambda results
	   (close-input-port f)
	   (apply values results) ) ) ) ) ) )

(define call-with-output-file
  (let ([open-output-file open-output-file]
	[close-output-port close-output-port] )
    (lambda (name p . mode)
      (let ([f (apply open-output-file name mode)])
	(##sys#call-with-values
	 (lambda () (p f))
	 (lambda results
	   (close-output-port f)
	   (apply values results) ) ) ) ) ) )

(define with-input-from-file 
  (let ((open-input-file open-input-file)
	(close-input-port close-input-port) )
    (lambda (str thunk . mode)
      (let ((old ##sys#standard-input)
	    (file (apply open-input-file str mode)) )
	(set! ##sys#standard-input file)
	(##sys#call-with-values thunk
	  (lambda results
	    (close-input-port file)
	    (set! ##sys#standard-input old)
	    (apply values results) ) ) ) ) ) )

(define with-output-to-file 
  (let ((open-output-file open-output-file)
	(close-output-port close-output-port) ) 
    (lambda (str thunk . mode)
      (let ((old ##sys#standard-output)
	    (file (apply open-output-file str mode)) )
	(set! ##sys#standard-output file)
	(##sys#call-with-values thunk
	  (lambda results
	    (close-output-port file)
	    (set! ##sys#standard-output old)
	    (apply values results) ) ) ) ) ) )

(define (file-exists? name)
  (##sys#check-string name)
  (not (eq? #f (##sys#file-info name))) )

(define (flush-output . port)
  (##sys#call-custom-port-handler 
   #:flush-output
   (##sys#fetch-and-check-port-arg port ##sys#standard-output)
   '()
   (lambda (p) 
     (##core#inline "C_flush_output" (##sys#port-file-resolve p))) ) )

(define port-name
    (lambda (port)
      (##sys#check-port port)
      (if (fx>= (##sys#size port) 4)
	  (##sys#slot port 3)
	  (##sys#error "can not access name of port" port) ) ) )

(define port-position
    (lambda (port)
      (##sys#check-port port)
      (if (and (not (##sys#slot port 1)) (fx>= (##sys#size port) 6))
	  (values (##sys#slot port 4) (##sys#slot port 5))
	  (##sys#error "can not compute position of port" port) ) ) )

(define delete-file
    (lambda (filename)
      (##sys#check-string filename)
      (unless (eq? 0 (##core#inline "C_delete_file" (##sys#make-c-string filename)))
	(##sys#update-errno)
	(##sys#signal-hook #:file-error "can not delete file" filename) ) ) )

(define rename-file
    (lambda (old new)
      (##sys#check-string old)
      (##sys#check-string new)
      (unless (eq? 0 (##core#inline "C_rename_file" (##sys#make-c-string old) (##sys#make-c-string new)))
	(##sys#update-errno)
	(##sys#signal-hook #:file-error "can not rename file" old new) ) ) )


;;; Input:

(define (eof-object? x) (##core#inline "C_eofp" x))

(define (char-ready? . port)
  (##sys#call-custom-port-handler
   #:char-ready?
   (##sys#fetch-and-check-port-arg port ##sys#standard-input)
   '()
   (lambda (p) 
     (##core#inline "C_char_ready_p" (##sys#port-file-resolve p))) ) )

(define (read-char . port)
  (let* ([p (##sys#fetch-and-check-port-arg port ##sys#standard-input)]
	 [c (##sys#call-custom-port-handler
	     #:read-char p '() 
	     (lambda (p)
	       (##core#inline "C_read_char" (##sys#port-file-resolve p)) ) ) ] )
    (cond [(char=? c #\newline)
	   (##sys#setslot p 4 (fx+ (##sys#slot p 4) 1))
	   (##sys#setslot p 5 0) ]
	  [else (##sys#setslot p 5 (fx+ (##sys#slot p 5) 1))] )
    c) )

(define (peek-char . port)
  (##sys#call-custom-port-handler
   #:peek-char
   (##sys#fetch-and-check-port-arg port ##sys#standard-input)
   '()
   (lambda (p) 
     (##core#inline "C_peek_char" (##sys#port-file-resolve p))) ) )

(define (read . port)
  (##sys#read (##sys#fetch-and-check-port-arg port ##sys#standard-input) ##sys#default-read-info-hook) )

(define ##sys#read-line-counter 0)
(define ##sys#default-read-info-hook #f)
(define ##sys#read-error-with-line-number #f)
(define ##sys#current-namespace #f)
(define ##sys#default-namespace-prefix #f)
(define ##sys#enable-qualifiers #t)

(define ##sys#read-warning
  (let ([string-append string-append])
    (lambda (msg . args)
      (apply
       ##sys#warn
       (if ##sys#read-error-with-line-number
	   (string-append msg " in line " (number->string ##sys#read-line-counter))
	   msg)
       args) ) ) )

(define ##sys#read-error
  (let ([string-append string-append] )
    (lambda (msg . args)
      (apply
       ##sys#error 
       (if ##sys#read-error-with-line-number
	   (string-append msg " in line " (number->string ##sys#read-line-counter))
	   msg)
       args) ) ) )

(define ##sys#read
  (let ([list->string list->string]
	[read-char read-char]
	[peek-char peek-char]
	[reverse reverse]
	[list? list?]
	[string-append string-append]
	[make-string make-string]
	[substring substring]
	[container (lambda (c) (##sys#error "unexpected list terminator" c))] 
	[char-name char-name]
	[kwprefix (string (integer->char 0))] )
    (lambda (port infohandler)
      (let ([terminating-characters '(#\, #\; #\( #\) #\[ #\] #\{ #\} #\' #\")]
	    [inexact-flag #f] 
	    [rat-flag #f] )

	(define (info class data val)
	  (if infohandler
	      (infohandler class data val)
	      data) )

	(define (advance)
	  (set! ##sys#read-line-counter (fx+ ##sys#read-line-counter 1)) )

        (define (readrec)

          (define (r-spaces)
            (let loop ([c (peek-char port)])
	      (cond ((##core#inline "C_eofp" c) #f)
		    ((eq? #\; c)
		     (let skip ((c (read-char port)))
		       (if (and (not (##core#inline "C_eofp" c)) (not (eq? #\newline c)))
			   (skip (read-char port))
			   (begin
			     (advance)
			     (loop (peek-char port)) ) ) ) )
		    ((eq? c #\newline)
		     (advance)
		     (read-char port)
		     (loop (peek-char port)) )
		    ((char-whitespace? c)
		     (read-char port)
		     (loop (peek-char port)) ) ) ) )
          
          (define (r-string)
            (if (eq? (read-char port) #\")
		(let loop ((c (read-char port)) (lst '()))
		  (cond ((##core#inline "C_eofp" c) 
			 (##sys#read-error "unterminated string") )
			((eq? #\\ c)
			 (set! c (read-char port))
			 (case c
			   ((#\t) (set! c #\tab))
			   ((#\r) (set! c #\return))
			   ((#\n) (set! c #\newline)) )
			 (loop (read-char port) (cons c lst)) )
			((eq? #\" c) (list->string (reverse lst)))
			((eq? c #\newline) 
			 (advance)
			 (loop (read-char port) (cons c lst)) )
			(else (loop (read-char port) (cons c lst))) ) )
		(##sys#read-error "missing '\"'") ) )
          
          (define (r-list start end)
	    (if (eq? (read-char port) start)
		(let ([first #f]
		      [ln0 #f]
		      [outer-container container] )
		  (##sys#call-with-current-continuation
		   (lambda (return)
		     (set! container
		       (lambda (c)
			 (if (eq? c end)
			     (return #f)
			     (##sys#read-error "list-terminator mismatch" c end) ) ) )
		     (let loop ([last '()])
		       (r-spaces)
		       (unless first (set! ln0 ##sys#read-line-counter))
		       (let ([c (peek-char port)])
			 (cond ((##core#inline "C_eofp" c)
				(##sys#read-error "unterminated list") )
			       ((eq? c end)
				(read-char port) )
			       ((eq? c #\.)
				(read-char port)
				(let ([c2 (peek-char port)])
				  (cond [(char-whitespace? c2)
					 (r-spaces)
					 (##sys#setslot last 1 (readrec))
					 (r-spaces)
					 (unless (eq? (read-char port) end)
					   (##sys#read-error "missing ')'") ) ]
					[else
					 (let* ((tok (string-append "." (r-token)))
						(n (and (char-numeric? c2) (string->number tok)))
						(val (or n (resolve-symbol tok))) 
						(node (cons val '())) )
					   (if first 
					       (##sys#setslot last 1 node)
					       (set! first node) )
					   (loop node) ) ] ) ) )
			       (else
				(let ([node (cons (readrec) '())])
				  (if first
				      (##sys#setslot last 1 node)
				      (set! first node) )
				  (loop node) ) ) ) ) ) ) )
		  (set! container outer-container)
		  (if first
		      (info 'list-info first ln0)
		      '() ) )
		(##sys#read-error "missing token" start) ) )
          
          (define (r-vector)
            (##sys#list->vector (r-list #\( #\))) )
          
          (define (r-number radix)
	    (set! inexact-flag #f)
	    (set! rat-flag #f)
	    (let* ([tok (r-token)]
		   [val (string->number tok (or radix 10))] )
	      (cond [val
		     (when (inexact? val)
		       (cond [rat-flag (##sys#read-warning "can not represent exact fraction" tok)]
			     [(not inexact-flag) (##sys#read-warning "exact literal exceeds range" tok)] ) )
		     val]
		    [radix (##sys#read-error "illegal number syntax" tok)]
		    [else (resolve-symbol tok)] ) ) )

	  (define (r-number-with-exactness radix)
	    (cond [(char=? #\# (peek-char port))
		   (read-char port)
		   (let ([c2 (read-char port)])
		     (cond [(eof-object? c2) (##sys#read-error "unexpected end of numeric literal")]
			   [(char=? c2 #\i) (exact->inexact (r-number radix))]
			   [(char=? c2 #\e) (inexact->exact (r-number radix))]
			   [else (##sys#read-error "illegal number syntax - invalid exactness prefix" c2)] ) ) ]
		  [else (r-number radix)] ) )
          
	  (define (r-number-with-radix)
	    (cond [(char=? #\# (peek-char port))
		   (read-char port)
		   (let ([c2 (read-char port)])
		     (cond [(eof-object? c2) (##sys#read-error "unexpected end of numeric literal")]
			   [(char=? c2 #\x) (r-number 16)]
			   [(char=? c2 #\o) (r-number 8)]
			   [(char=? c2 #\b) (r-number 2)]
			   [else (##sys#read-error "illegal number syntax - invalid radix" c2)] ) ) ]
		  [else (r-number 10)] ) )
        
	  (define (r-token)
	    (let loop ([c (peek-char port)] [lst '()])
	      (cond [(or (eof-object? c)
			 (char-whitespace? c)
			 (memq c terminating-characters) )
		     (list->string (reverse lst)) ]
		    [else
		     (cond [(or (char=? c #\.) (char=? c #\#) (char=? c #\e) (char=? c #\E)) (set! inexact-flag #t)]
			   [(char=? c #\/) (set! rat-flag #t)] )
		     (read-char port)
		     (loop (peek-char port) 
			   (cons (if (##sys#case-sensitive)
				     c
				     (char-downcase c) )
				 lst) ) ] ) ) )

	  (define (r-next-token)
	    (r-spaces)
	    (r-token) )
          
          (define (r-symbol)
	    (let ((s (resolve-symbol
		      (if (char=? (peek-char port) #\|)
			  (r-xtoken)
			  (r-token) ) ) ) )
	      (info 'symbol-info s ##sys#read-line-counter) ) )

	  (define (r-xtoken)
	    (if (char=? #\| (read-char port))
		(let loop ((c (read-char port)) (lst '()))
		  (cond ((eof-object? c) (##sys#read-error "unexpected end of special symbol"))
			((char=? c #\\)
			 (let ((c (read-char port)))
			   (loop (read-char port) (cons c lst)) ) )
			((char=? c #\|)
			 (list->string (reverse lst)) )
			(else (loop (read-char port) (cons c lst))) ) )
		(##sys#read-error "missing \'|\'") ) )
          
          (define (r-char)
	    (let* ([c (peek-char port)]
		   [tk (r-token)] )
	      (cond [(char-name (##sys#string->symbol tk))]
		    [(fx> (string-length tk) 1) (##sys#read-error "unknown named character" tk)]
		    [(memq c terminating-characters) (read-char port)]
		    [else c] ) ) )

	  (define (r-comment)
	    (let loop ((i 0))
	      (let ((c (read-char port)))
		(case c
		  ((#\newline)
		   (advance)
		   (loop i) )
		  ((#\|) (if (eq? #\# (read-char port))
			     (if (not (eq? i 0))
				 (loop (fx- i 1)) )
			     (loop i) ) )
		  ((#\#) (loop (if (eq? #\| (read-char port))
				   (fx+ i 1)
				   i) ) )
		  (else (if (eof-object? c)
			    (##sys#read-error "unterminated block-comment")
			    (loop i) ) ) ) ) ) )

	  (define (r-namespace)
	    (set! ##sys#current-namespace (##sys#make-vector namespace-size '()))
	    (let* ([ns (r-next-token)]
		   [nslen (##sys#size ns)]
		   [p (make-string 1)] )
	      (when (fx> nslen namespace-max-id-len)
		(set! ns (substring ns 0 namespace-max-id-len))
		(set! nslen namespace-max-id-len) )
	      (##sys#setbyte p 0 (##sys#size ns))
	      (let ([prefix (string-append p ns)])
		(let loop ([toks '()])
		  (r-spaces)
		  (cond [(char=? #\} (peek-char port))
			 (read-char port)
			 (for-each
			  (lambda (tok)
			    (let ([i (##core#inline "C_fixnum_modulo" (##core#inline "C_hash_string" tok) namespace-size)])
			      (##sys#setslot 
			       ##sys#current-namespace i
			       (cons (cons tok (##sys#string->symbol (string-append prefix tok)))
				     (##sys#slot ##sys#current-namespace i) ) ) ) )
			  toks) ]
			[else (loop (cons (r-next-token) toks))] ) ) ) ) )

	  (define (r-ext-symbol)
	    (let* ([p (make-string 1)]
		   [tok (r-token)] 
		   [toklen (##sys#size tok)] )
	      (unless ##sys#enable-qualifiers 
		(##sys#read-error "qualified symbol syntax is not allowed" tok) )
	      (let loop ([i 0])
		(cond [(fx>= i toklen) (##sys#read-error "invalid qualified symbol syntax" tok)]
		      [(fx= (##sys#byte tok i) (char->integer #\#))
		       (when (fx> i namespace-max-id-len)
			 (set! tok (substring tok 0 namespace-max-id-len)) )
		       (##sys#setbyte p 0 i)
		       (##sys#string->symbol
			(string-append p (substring tok 0 i) (substring tok (fx+ i 1) toklen)) ) ]
		      [else (loop (fx+ i 1))] ) ) ) )

	  (define (resolve-symbol tok)
	    (cond [(string=? tok ".") (##sys#read-error "invalid use of '.'")]
		  [(not ##sys#current-namespace) (build-symbol tok)]
		  [else
		   (let ([i (##core#inline "C_fixnum_modulo" (##core#inline "C_hash_string" tok) namespace-size)])
		     (let loop ([bucket (##sys#slot ##sys#current-namespace i)])
		       (if (null? bucket)
			   (build-symbol tok)
			   (let ([e (##sys#slot bucket 0)])
			     (if (string=? tok (##sys#slot e 0))
				 (##sys#slot e 1)
				 (loop (##sys#slot bucket 1)) ) ) ) ) ) ] ) )

	  (define (build-symbol tok)
	    (##sys#string->symbol
	     (if ##sys#default-namespace-prefix
		 (string-append ##sys#default-namespace-prefix tok)
		 tok) ) )

          (r-spaces)
          (let ([c (peek-char port)])
            (case c
              ((#\')
               (read-char port)
               (list 'quote (readrec)) )
              ((#\`)
               (read-char port)
               (list 'quasiquote (readrec)) )
              ((#\,)
               (read-char port)
               (cond ((eq? (peek-char port) #\@)
                      (read-char port)
                      (list 'unquote-splicing (readrec)) )
                     (else (list 'unquote (readrec))) ) )
              ((#\#)
               (read-char port)
	       (let ((dchar (peek-char port)))
		 (case (char-downcase dchar)
		   ((#\x) (read-char port) (r-number-with-exactness 16))
		   ((#\o) (read-char port) (r-number-with-exactness 8))
		   ((#\b) (read-char port) (r-number-with-exactness 2))
		   ((#\i) (read-char port) (exact->inexact (r-number-with-radix)))
		   ((#\e) (read-char port) (inexact->exact (r-number-with-radix)))
		   ((#\() (r-vector))
		   ((#\\) (read-char port) (r-char))
		   ((#\|) (read-char port) (r-comment) (readrec))
		   ((#\{) (read-char port) (r-namespace) (readrec))
		   ((#\#) (read-char port) (r-ext-symbol))
		   ((#\;) (read-char port) (readrec) (readrec))
		   ((#\') (read-char port) (list 'syntax (readrec)))
		   ((#\:) 
		    (read-char port)
		    (##sys#string->symbol (string-append kwprefix (r-token))) )
		   (else (##sys#user-read-hook dchar port)) ) ) )
              ((#\() (r-list #\( #\)))
	      ((#\[) (r-list #\[ #\]))
              ((#\) #\]) 
	       (read-char port)
	       (container c) )
	      ((#\{ #\})
	       (read-char port)
	       (##sys#read-error "illegal character" c))
              ((#\") (r-string))
	      ((#\.) (r-number #f))
              ((#\-) (r-number #f))
              (else (cond [(eof-object? c) c]
                          [(char-numeric? c) (r-number #f)]
                          [else (r-symbol)] ) ) ) ) )
	(readrec) ) ) ) )


;;; Hook for user-defined read-syntax:
;
; - Redefine this to handle new read-syntaxes. If 'char' doesn't match
;   your character then call the previous handler.
; - Don't forget to read 'char', it's only peeked at this point.

(define ##sys#user-read-hook
  (let ([read-char read-char] )
    (lambda (char port)
      (case char
	;; I put it here, so the SRFI-4 unit can intercept '#f...'
	((#\f #\F) (read-char port) #f)
	((#\t #\T) (read-char port) #t)
	(else (##sys#read-error "invalid sharp-sign read syntax" char) ) ) ) ) )


;;; Output:

(define (##sys#write-char c . port)
  (##sys#check-char c)
  (let ([p (##sys#fetch-and-check-port-arg port ##sys#standard-output)])
    (##sys#call-custom-port-handler 
     #:write-char p c
     (lambda (p)
       (##core#inline "C_display_char" (##sys#port-file-resolve p) c))) ) )

(define write-char ##sys#write-char)

(define (newline . port)
  (let ([p (##sys#fetch-and-check-port-arg port ##sys#standard-output)])
    (##sys#call-custom-port-handler
     #:write-char p #\newline
     (lambda (p)
       (##core#inline "C_display_char" (##sys#port-file-resolve p) #\newline))) ) )

(define (write x . port)
  (##sys#print x #t (##sys#fetch-and-check-port-arg port ##sys#standard-output)) )

(define (display x . port)
  (##sys#print x #f (##sys#fetch-and-check-port-arg port ##sys#standard-output)) )

(define print
  (let ([write-char write-char])
    (lambda args
      (for-each (lambda (x) (##sys#print x #f ##sys#standard-output)) args)
      (write-char #\newline) ) ) )

(define (print* . args)
  (for-each (lambda (x) (##sys#print x #f ##sys#standard-output)) args) )

(define ##sys#current-print-length 0)
(define ##sys#print-length-limit #f)
(define ##sys#print-exit #f)
(define ##sys#print-qualifiers #f)

(define ##sys#print
  (let ([substring substring]
	[char-name char-name]
	[string-append string-append] )
    (lambda (x readable port)

      (define (outstr port str)
	(set! ##sys#current-print-length (fx+ ##sys#current-print-length (##sys#size str)))
	(when (and ##sys#print-length-limit (fx>= ##sys#current-print-length ##sys#print-length-limit))
	  (outstr0 port "...")
	  (##sys#print-exit #t) )
	(outstr0 port str) )
	       
      (define (outstr0 port str)
	(##sys#call-custom-port-handler
	 #:write-string port str
	 (lambda (p) 
	   (##core#inline "C_display_string" (##sys#port-file-resolve p) str)) ) )

      (define (outchr port chr)
	(set! ##sys#current-print-length (fx+ ##sys#current-print-length 1))
	(when (and ##sys#print-length-limit (fx>= ##sys#current-print-length ##sys#print-length-limit))
	  (outstr0 port "...")
	  (##sys#print-exit #t) )
	(##sys#call-custom-port-handler
	 #:write-char port chr
	 (lambda (p) 
	   (##core#inline "C_display_char" (##sys#port-file-resolve p) chr)) ) )

      (define (specialchar? chr)
	(let ([c (char->integer chr)])
	  (or (fx<= c 32)
	      (fx>= c 128)
	      (memq chr '(#\( #\) #\| #\, #\[ #\] #\{ #\} #\' #\" #\; #\\)) ) ) )

      (define (outreadablesym port str)
	(let ([len (##sys#size str)])
	  (outchr port #\|)
	  (let loop ([i 0])
	    (if (fx>= i len)
		(outchr port #\|)
		(let ([c (##core#inline "C_subchar" str i)])
		  (when (specialchar? c) (outchr port #\\))
		  (outchr port c)
		  (loop (fx+ i 1)) ) ) ) ) )

      (define (sym-is-readable? str)
	(let ([len (##sys#size str)])
	  (and (fx> len 0)
	       (not (string=? "." str))
	       (let loop ([i (fx- len 1)])
		 (or (fx= i 0)
		     (let ([c (##core#inline "C_subchar" str i)])
		       (and (or (not (##sys#case-sensitive)) (not (char-upper-case? c)))
			    (not (specialchar? c))
			    (loop (fx- i 1)) ) ) ) ) ) ) )

      (let out ([x x])
	(cond ((eq? x '()) (outstr port "()"))
	      ((eq? x #t) (outstr port "#t"))
	      ((eq? x #f) (outstr port "#f"))
	      ((##core#inline "C_eofp" x) (outstr port "#<eof>"))
	      ((##core#inline "C_undefinedp" x) (outstr port "#<unspecified>"))
	      ((##core#inline "C_charp" x)
	       (cond (readable
		      (outstr port "#\\")
		      (let ([cn (char-name x)])
			(if cn
			    (outstr port (##sys#slot cn 1))
			    (outchr port x) ) ) )
		     (else (outchr port x)) ) )
	      ((##core#inline "C_fixnump" x) (outstr port (number->string x)))
	      ((not (##core#inline "C_blockp" x)) (outstr port "#<unprintable object>"))
	      ((##core#inline "C_pointerp" x)
	       (outstr port "#<pointer ")
	       (outstr port (number->string (##sys#pointer->address x) 16))
	       (outchr port #\>) )
	      ((##core#inline "C_flonump" x) (outstr port (number->string x)))
	      ((##core#inline "C_stringp" x)
	       (cond (readable
		      (outchr port #\")
		      (do ((i 0 (fx+ i 1))
			   (c (##core#inline "C_block_size" x) (fx- c 1)) )
			  ((eq? c 0)
			   (outchr port #\") )
			(let ((chr (##core#inline "C_subbyte" x i)))
			  (case chr
			    ((34) (outstr port "\\\""))
			    ((92) (outstr port "\\\\"))
			    (else
			     (if (fx< chr 32)
				 (begin
				   (outchr port #\\)
				   (outchr
				    port
				    (case chr
				      ((9) #\t)
				      ((10) #\n)
				      ((13) #\r)
				      ((8) #\b)
				      (else (##core#inline "C_fix_to_char" chr)) ) ) )
				 (outchr port (##core#inline "C_fix_to_char" chr)) ) ) ) ) ) )
		     (else (outstr port x)) ) )
	      ((##core#inline "C_symbolp" x)
	       (cond [(fx= 0 (##sys#byte (##sys#slot x 1) 0))
		      (outstr port "#:")
		      (outstr port (##sys#symbol->string x)) ]
		     [else
		      (let ([str (if ##sys#print-qualifiers
				     (##sys#symbol->qualified-string x) 
				     (##sys#symbol->string x) ) ] )
			(if (or (not readable) (sym-is-readable? str))
			    (outstr port str)
			    (outreadablesym port str) ) ) ] ) )
	      ((##core#inline "C_pairp" x)
	       (outchr port #\()
	       (out (##sys#slot x 0))
	       (do ((x (##sys#slot x 1) (##sys#slot x 1)))
		   ((or (not (##core#inline "C_blockp" x)) (not (##core#inline "C_pairp" x)))
		    (if (not (eq? x '()))
			(begin
			  (outstr port " . ")
			  (out x) ) )
		    (outchr port #\)) )
		 (outchr port #\space)
		 (out (##sys#slot x 0)) ) )
	      ((##core#inline "C_bytevectorp" x)
	       (if (##core#inline "C_permanentp" x)
		   (outstr port "#<static bytevector>")
		   (outstr port "#<bytevector>") ) )
	      ((##core#inline "C_structurep" x) (##sys#user-print-hook x readable port))
	      ((##core#inline "C_closurep" x) (outstr port "#<procedure>"))
	      ((##core#inline "C_portp" x)
	       (outstr port "#<port ")
	       (outstr port (##sys#slot x 3))
	       (outchr port #\>) )
	      ((##core#inline "C_vectorp" x)
	       (let ((n (##core#inline "C_block_size" x)))
		 (cond ((eq? 0 n)
			(outstr port "#()") )
		       (else
			(outstr port "#(")
			(out (##sys#slot x 0))
			(do ((i 1 (fx+ i 1))
			     (c (fx- n 1) (fx- c 1)) )
			    ((eq? c 0)
			     (outchr port #\)) )
			  (outchr port #\space)
			  (out (##sys#slot x i)) ) ) ) ) )
	      (else (##sys#error "unprintable non-immediate object encountered")) ) ) ) ) )

(define ##sys#record-printers '())

(define (##sys#register-record-printer type proc)
  (let ([a (assq type ##sys#record-printers)])
    (if a 
	(set-cdr! a proc)
	(set! ##sys#record-printers (cons (cons type proc) ##sys#record-printers)) ) ) )

(define (##sys#user-print-hook x readable port)
  (let* ([type (##sys#slot x 0)]
	 [a (assq type ##sys#record-printers)] )
    (cond [a ((##sys#slot a 1) x port)]
	  [else
	   (##sys#print "#<" #f port)
	   (##sys#print (##sys#symbol->string type) #f port)
	   (case type
	     [(condition)
	      (##sys#print ": " #f port)
	      (##sys#print (##sys#slot x 1) #f port) ]
	     [(thread)
	      (##sys#print ": " #f port)
	      (##sys#print (##sys#slot x 6) #f port) ] )
	   (##sys#print #\> #f port) ] ) ) )

(define (##sys#with-print-length-limit limit thunk)
  (call-with-current-continuation
   (lambda (return)
     (fluid-let ((##sys#print-length-limit limit)
		 (##sys#print-exit return) 
		 (##sys#current-print-length 0) )
       (thunk) ) ) ) )


;;; Bitwise fixnum operations:

(define (bitwise-and x y)
  (##sys#check-exact x)
  (##sys#check-exact y)
  (##core#inline "C_fixnum_and" x y) )

(define (bitwise-ior x y)
  (##sys#check-exact x)
  (##sys#check-exact y)
  (##core#inline "C_fixnum_or" x y) )

(define (bitwise-xor x y)
  (##sys#check-exact x)
  (##sys#check-exact y)
  (##core#inline "C_fixnum_xor" x y) )

(define (bitwise-not x)
  (##sys#check-exact x)
  (##core#inline "C_fixnum_not" x) )

(define (arithmetic-shift x y)
  (##sys#check-exact x)
  (##sys#check-exact y)
  (if (fx< y 0)
      (##core#inline "C_fixnum_shift_right" x (##core#inline "C_fixnum_negate" y))
      (##core#inline "C_fixnum_shift_left" x y) ) )


;;; String ports:

(define open-input-string 
    (lambda (string)
      (##sys#check-string string)
      (let ((port (##sys#make-port #t 6 #f #f))
	    (len (##core#inline "C_block_size" string))
	    (position 0) )
	(##sys#setslot port 3 "(string)")
	(##sys#setslot port 4 0)
	(##sys#setslot port 5 0)
	(##sys#setslot 
	 port 2
	 (lambda (op p args)
	   (case op
	     ((#:close-input-port)
	      (set! position len) )
	     ((#:char-ready?)
	      (fx< position len) )
	     ((#:read-char)
	      (if (>= position len)
		  (##sys#fudge 1)
		  (let ((c (##core#inline "C_subchar" string position)))
		    (set! position (fx+ position 1))
		    c) ) )
	     ((#:peek-char)
	      (if (fx>= position len)
		  (##sys#fudge 1)
		  (##core#inline "C_subchar" string position) ) )
	     (else ##sys#snafu) ) ) )
	port) ) )

(define open-output-string
  (let ([make-string make-string]
	[string-append string-append] )
    (lambda ()
      (let ([output (make-string output-string-initial-size)]
	    [limit output-string-initial-size]
	    [position 0]
	    [port (##sys#make-port #t 8 #f #f)] )

	(define (check n)
	  (let ([limit2 (fx+ position n)])
	    (when (fx>= limit2 limit)
	      (when (fx>= limit2 maximal-string-length)
		(##sys#error "string buffer full" port) )
	      (let* ([limit3 (fxmin maximal-string-length (fx+ limit limit))]
		     [buf (make-string limit3)] )
		(##sys#copy-bytes output buf 0 0 position)
		(set! output buf)
		(##sys#setslot port 6 buf)
		(set! limit limit3)
		(check n) ) ) ) )

	(##sys#setslot port 1 #t)
	(##sys#setslot port 3 "(string)")
	(##sys#setslot port 4 0)
	(##sys#setslot port 5 0)
	(##sys#setslot port 6 output)
	(##sys#setslot port 7 position)
	(##sys#setslot
	 port 2
	 (lambda (op p args)
	   (set! position (##sys#slot p 7))
	   (case op
	     ((#:close-output-port) (##sys#setslot p 7 limit))
	     ((#:flush-output) #f)
	     ((#:write-char)
	      (check 1)
	      (##core#inline "C_setsubchar" output position args)
	      (##sys#setslot p 7 (fx+ position 1)) )
	     ((#:write-string)
	      (let ([len (##core#inline "C_block_size" args)])
		(check len)
		(do ((i 0 (fx+ i 1)))
		    ((fx>= i len) (##sys#setslot p 7 position))
		  (##core#inline "C_setsubchar" output position (##core#inline "C_subchar" args i))
		  (set! position (fx+ position 1)) ) ) )
	     (else ##sys#snafu) ) ) ) 
	port) ) ) )

(define get-output-string
  (let ((substring substring) )
    (lambda (port)
      (##sys#check-port port)
      (if (or (not (string=? (##sys#slot port 3) "(string)")) 
	      (not (##sys#slot port 1)) )
	  (##sys#signal-hook #:type-error "argument is not a string-output-port" port) 
	  (substring (##sys#slot port 6) 0 (##sys#slot port 7)) ) ) ) )

(define ##sys#print-to-string
  (let ([get-output-string get-output-string]
	[open-output-string open-output-string] )
    (lambda xs
      (let ([out (open-output-string)])
	(for-each (lambda (x) (##sys#print x #f out)) xs)
	(get-output-string out) ) ) ) )

(define print-to-string ##sys#print-to-string)


;;; Parameters:

(define ##sys#default-parameter-vector (##sys#make-vector default-parameter-vector-size))
(define ##sys#current-parameter-vector '#())

(define make-parameter
  (let ([count 0])
    (lambda (init . guard)
      (let* ([guard (if (pair? guard) (car guard) (lambda (x) x))]
	     [val (guard init)] 
	     [i count] )
	(set! count (fx+ count 1))
	(when (fx>= i (##sys#size ##sys#default-parameter-vector))
	  (set! ##sys#default-parameter-vector 
	    (##sys#grow-vector ##sys#default-parameter-vector (fx+ i 1) (##core#undefined)) ) )
	(##sys#setslot ##sys#default-parameter-vector i val)
	(lambda arg
	  (let ([n (##sys#size ##sys#current-parameter-vector)])
	    (cond [(pair? arg)
		   (when (fx>= i n)
		     (set! ##sys#current-parameter-vector
		       (##sys#grow-vector ##sys#current-parameter-vector (fx+ i 1) ##sys#snafu) ) )
		   (##sys#setslot ##sys#current-parameter-vector i (guard (car arg)))
		   (##core#undefined) ]
		  [(fx>= i n)
		   (##sys#slot ##sys#default-parameter-vector i) ]
		  [else
		   (let ([val (##sys#slot ##sys#current-parameter-vector i)])
		     (if (eq? val ##sys#snafu)
			 (##sys#slot ##sys#default-parameter-vector i) 
			 val) ) ] ) ) ) ) ) ) )


;;; Feature identifiers:

(define ##sys#features '(#:chicken #:srfi-22 #:srfi-23))

(define ##sys#->feature-id
  (let ([string->keyword string->keyword]
	[keyword? keyword?] )
    (lambda (x)
      (cond [(string? x) (string->keyword x)]
	    [(keyword? x) x]
	    [(symbol? x) (string->keyword (##sys#symbol->string x))]
	    [else (##sys#signal-hook #:type-error "bad argument type - not a valid feature identifer" x)] ) ) ) )

(define register-feature! 
  (lambda fs
    (set! ##sys#features (##sys#append (map ##sys#->feature-id fs) ##sys#features))
    (##core#undefined) ) )

(define (unregister-feature! . fs)
  (let ([fs (map ##sys#->feature-id fs)])
    (set! ##sys#features
      (let loop ([ffs ##sys#features])
	(if (null? ffs)
	    '()
	    (let ([f (##sys#slot ffs 0)]
		  [r (##sys#slot ffs 1)] )
	      (if (memq f fs)
		  (loop r)
		  (cons f (loop r)) ) ) ) ) )
    (##core#undefined) ) )

(define (features) ##sys#features)


;;; Interrupt handling:

(let ([count 0])
  (set! ##sys#enable-interrupts
    (lambda val
      (set! count (fx+ count (if (pair? val) (car val) 1)))
      (when (eq? count 0) (##core#inline "C_enable_interrupts")) ) )
  (set! ##sys#disable-interrupts
    (lambda ()
      (when (eq? count 0) (##core#inline "C_disable_interrupts"))
      (set! count (fx- count 1)) ) ) )

(define enable-interrupts ##sys#enable-interrupts)
(define disable-interrupts ##sys#disable-interrupts)

(define (##sys#user-interrupt-hook)
  (##sys#signal-hook #:user-interrupt #f) )


;;; Default handlers:

(define error-handler
  (make-parameter
   (let ([string-append string-append]
	 [open-output-string open-output-string]
	 [get-output-string get-output-string] )
     (lambda (msg . args)
       (error-handler (lambda args (##core#inline "C_halt" "error in error")))
       (set! ##sys#print-qualifiers #t)
       (cond ((##sys#fudge 4)
	      (##core#inline "C_display_string" ##sys#standard-error "Error: ")
	      (##sys#print msg #f ##sys#standard-error)
	      (##sys#for-each
	       (lambda (x)
		 (##core#inline "C_display_char" ##sys#standard-error #\newline)
		 (##sys#print x #t ##sys#standard-error) )
	       args)
	      (##core#inline "C_display_char" ##sys#standard-error #\newline)
	      (##core#inline "C_halt" #f) )
	     (else
	      (let ((out (open-output-string)))
		(##sys#print msg #f out)
		(##sys#print #\newline #f out)
		(##sys#for-each (lambda (x) (##sys#print x #t out) (##sys#print #\newline #f out)) args)
		(##core#inline "C_halt" (get-output-string out)) ) ) ) ) ) ) )

(define (##sys#kill-threads-and-reset)
  (let ([ct ##sys#current-thread]
	[pt ##sys#primordial-thread] )
    (##sys#for-each
     (lambda (ti)
       (unless (eq? ti pt)
	 (##sys#setslot ti 3 'terminated)
	 (##sys#setslot ti 2 (##core#undefined))
	 (##sys#setslot ti 7 (##sys#make-structure 'terminated-thread-exception))
	 (##sys#abandon-mutexes ti) ) )
     (cons ct ##sys#ready-queue-head) )
    (##sys#setslot pt 1 (lambda () ((##sys#reset-handler))))
    (##sys#setslot pt 3 'ready)
    (##sys#schedule) ) )

(define-foreign-variable _ex_software int "EX_SOFTWARE")

(define reset-handler 
  (make-parameter 
   (lambda ()
     ((##sys#exit-handler) _ex_software)) ) )

(define exit-handler
  (make-parameter
   (lambda code
     (##sys#cleanup-before-exit)
     (##core#inline
      "C_exit_runtime"
      (if (null? code)
	  0
	  (let ([code (car code)])
	    (##sys#check-exact code)
	    code) ) ) ) ) )

(define implicit-exit-handler
  (make-parameter
   (lambda ()
     (##sys#cleanup-before-exit)
     (when ##sys#entry-points-defined-flag
       (##sys#dispatch-to-entry-point) ) ) ) )

(define ##sys#error-handler error-handler)
(define ##sys#exit-handler exit-handler)
(define ##sys#reset-handler reset-handler)
(define ##sys#implicit-exit-handler implicit-exit-handler)

(define (##sys#cleanup-before-exit)
  (##sys#force-finalizers) )


;;; Condition handling:

(define ##sys#signal-hook 
  (lambda (mode msg . args)
    (case mode
      [(#:error #:type-error #:arithmetic-error #:file-error #:runtime-error)
       (##sys#abort (##sys#make-structure 'condition '(exn) (list 'message msg 'arguments args))) ]
      [(#:user-interrupt)
       (##sys#abort (##sys#make-structure 'condition '(user-interrupt))) ]
      [(#:warning)
       (##sys#print "Warning: " #f ##sys#standard-error)
       (##sys#print msg #f ##sys#standard-error)
       (##sys#write-char #\newline ##sys#standard-error)
       (for-each
	(lambda (x)
	  (##sys#print x #t ##sys#standard-error)
	  (##sys#write-char #\newline ##sys#standard-error) )
	args) ] 
      [else (##core#inline "C_halt" "invalid signal mode")] ) ) )

(define (##sys#abort x)
  (##sys#current-exception-handler x)
  (##sys#abort (make-property-condition 'exn 'message "exception handler returned")) )

(define (##sys#signal x)
  (##sys#current-exception-handler x) )

(define abort ##sys#abort)
(define signal ##sys#signal)

(define ##sys#current-exception-handler
  (lambda (c)
    (when (##sys#structure? c 'condition)
      (case (##sys#slot (##sys#slot c 1) 0)
	[(exn)
	 (let ([props (##sys#slot c 2)])
	   (apply
	    (##sys#error-handler)
	    (or (and-let* ([a (memq 'message props)])
		  (cadr a) )
		"<exn: has no `message' property>")
	    (or (and-let* ([a (memq 'arguments props)])
		  (cadr a) )
		'() ) )
	   (##sys#kill-threads-and-reset) ) ]
	[(user-interrupt)
	 (##sys#print "*** User interrupt ***\n" #f ##sys#standard-error)
	 (##sys#kill-threads-and-reset) ] 
	[(uncaught-exception)
	 ((##sys#error-handler)
	  "uncaught exception"
	  (cadr (memq 'reason (##sys#slot c 2))) )
	 (##sys#kill-threads-and-reset) ] ) )
    (##sys#abort
     (##sys#make-structure
      'condition 
      '(uncaught-exception) 
      (list 'reason c)) ) ) )

(define (with-exception-handler handler thunk)
  (let ([oldh ##sys#current-exception-handler])
    (##sys#dynamic-wind 
	(lambda () (set! ##sys#current-exception-handler handler))
	(lambda () (thunk))
	(lambda () (set! ##sys#current-exception-handler oldh)) ) ) )

(define (current-exception-handler) ##sys#current-exception-handler)

(define (make-property-condition kind . props)
  (##sys#make-structure 'condition (list kind) props) )

(define make-composite-condition
    (lambda (c1 . conds)
      (let ([conds (cons c1 conds)])
	(for-each (lambda (c) (##sys#check-structure c 'condition)) conds)
	(##sys#make-structure
	 'condition
	 (apply ##sys#append (map (lambda (c) (##sys#slot c 1)) conds))
	 (apply ##sys#append (map (lambda (c) (##sys#slot c 2)) conds)) ) ) ) )

(define (condition? x) (##sys#structure? x 'condition))

(define (condition-predicate kind)
  (lambda (c) 
    (##sys#check-structure c 'condition)
    (memv kind (##sys#slot c 1)) ) )

(define (condition-property-accessor kind prop)
  (lambda (c)
    (##sys#check-structure c 'condition)
    (and (memv kind (##sys#slot c 1))
	 (let ([a (memq prop (##sys#slot c 2))])
	   (if a
	       (cadr a)
	       (##sys#error "condition has no such property" prop) ) ) ) ) )


;;; Error hook (called by runtime-system):

(define ##sys#error-hook
  (lambda (code . args)
    (case code
      ((1) (apply ##sys#error "bad argument count" args))
      ((2) (apply ##sys#error "too few arguments" args))
      ((3) (apply ##sys#signal-hook #:type-error "bad argument type" args))
      ((4) (apply ##sys#error "unbound variable" args))
      ((5) (apply ##sys#error "parameter limit exceeded" args))
      ((6) (apply ##sys#signal-hook #:runtime-error "out of memory" args))
      ((7) (apply ##sys#signal-hook #:arithmetic-error "division by zero" args))
      ((8) (apply ##sys#error "out of range" args))
      ((9) (apply ##sys#signal-hook #:type-error "call of non-procedure" args))
      ((10) (apply ##sys#error "continuation can not receive multiple values" args))
      ((12) (apply ##sys#signal-hook #:runtime-error "recursion too deep" args))
      ((13) (apply ##sys#signal-hook #:type-error "inexact number can not be represented as an exact number" args))
      ((14) (apply ##sys#signal-hook #:type-error "argument is not a proper list" args))
      ((15) (apply ##sys#signal-hook #:type-error "bad argument type - not a fixnum" args))
      ((16) (apply ##sys#signal-hook #:type-error "bad argument type - not a number" args))
      ((17) (apply ##sys#signal-hook #:type-error "bad argument type - not a string" args))
      ((18) (apply ##sys#signal-hook #:type-error "bad argument type - not a pair" args))
      ((19) (apply ##sys#signal-hook #:type-error "bad argument type - not a list" args))
      ((20) (apply ##sys#signal-hook #:type-error "bad argument type - not a character" args))
      ((21) (apply ##sys#signal-hook #:type-error "bad argument type - not a vector" args))
      ((22) (apply ##sys#signal-hook #:type-error "bad argument type - not a symbol" args))
      ((23) (apply ##sys#signal-hook #:runtime-error "stack overflow" args))
      ((24) (apply ##sys#signal-hook #:type-error "bad argument type - not a structure of the required type" args))
      ((25) (apply ##sys#signal-hook #:type-error "bad argument type - not a bytevector" args))
      (else (apply ##sys#signal-hook #:runtime-error "unknown internal error" args)) ) ) )


;;; Miscellaneous low-level routines:

(define (##sys#structure? x s) (##core#inline "C_i_structurep" x s))
(define (##sys#generic-structure? x) (##core#inline "C_structurep" x))
(define (##sys#slot x i) (##sys#slot x i))
(define (##sys#size x) (##core#inline "C_block_size" x))
(define ##sys#make-pointer (##core#primitive "C_make_pointer"))
(define (##sys#pointer? x) (##core#inline "C_pointerp" x))
(define (##sys#set-pointer-address! ptr addr) (##core#inline "C_update_pointer" addr ptr))
(define (##sys#bytevector? x) (##core#inline "C_bytevectorp" x))
(define (##sys#string->pbytevector s) (##core#inline "C_string_to_pbytevector" s))
(define (##sys#permanent? x) (##core#inline "C_permanentp" x))
(define (##sys#block-address x) (##core#inline "C_block_address" x))

(define (##sys#null-pointer)
  (##core#inline "C_update_pointer" 0 (##sys#make-pointer)) )

(define (##sys#null-pointer? x)
  (eq? 0 (##sys#pointer->address x)) )

(define (##sys#address->pointer addr)
  (##core#inline "C_update_pointer" addr (##sys#make-pointer)) )

(define (##sys#pointer->address ptr)
  ;; *** '4' is platform dependent!
  (##core#inline_allocate ("C_a_int_to_num" 4) (##sys#slot ptr 0)) )

(define ##sys#make-c-string 
  (let ((string-append string-append)
	(string string) )
    (lambda (str)
      (string-append str (string (##core#inline "C_make_character" (##core#inline "C_unfix" 0)))) ) ) )

(define ##sys#peek-signed-integer (##core#primitive "C_peek_signed_integer"))
(define ##sys#peek-unsigned-integer (##core#primitive "C_peek_unsigned_integer"))
(define (##sys#peek-fixnum b i) (##core#inline "C_peek_fixnum" b i))

(define (##sys#vector->structure! vec) (##core#inline "C_vector_to_structure" vec))

(define (##sys#peek-double b i)
  (##core#inline "C_f64peek" b i)
  (##sys#cons-flonum) )

(define ##sys#peek-c-string
  (let ([make-string make-string])
    (lambda (b i)
      (and (not (##sys#null-pointer? b))
	   (let* ([len (##core#inline "C_fetch_c_strlen" b i)]
		  [str2 (make-string len)] )
	     (##core#inline "C_peek_c_string" b i str2 len)
	     str2) ) ) ) )

(define (##sys#poke-c-string b i s) 
  (##core#inline "C_poke_c_string" b i (##sys#make-c-string s)) )

(define (##sys#poke-integer b i n) (##core#inline "C_poke_integer" b i n))

(define (##sys#vector->closure! vec addr)
  (##core#inline "C_vector_to_closure" vec)
  (##core#inline "C_update_pointer" addr vec) )

(define (##sys#symbol-has-toplevel-binding? s)
  (not (eq? (##sys#slot s 0) (##sys#slot '##sys#arbitrary-unbound-symbol 0))) )

(define (##sys#copy-bytes from to offset1 offset2 bytes)
  (##core#inline 
   "C_substring_copy"
   from to
   offset1 (fx+ offset1 bytes)
   offset2) )

(define (##sys#copy-words from to offset1 offset2 words)
  (##core#inline 
   "C_subvector_copy"
   from to
   offset1 (fx+ offset1 words)
   offset2) )

(define (##sys#compare-bytes from to offset1 offset2 bytes)
  (##core#inline 
   "C_substring_compare"
   from to
   offset1 offset2 bytes) )

(define ##sys#zap-strings (foreign-lambda void "C_zap_strings" scheme-object))

(define (##sys#block-pointer x)
  (let ([ptr (##sys#make-pointer)])
    (##core#inline "C_pointer_to_block" ptr x)
    ptr) )


;;; Support routines for foreign-function calling:

(define ##sys#foreign-char-argument
    (lambda (x)
      (if (char? x) 
	  x
	  (##sys#error "argument to foreign function is not a character" x) ) ) )

(define ##sys#foreign-fixnum-argument
    (lambda (x)
      (if (and (number? x) (exact? x))
	  x
	  (##sys#error "argument to foreign function can not be coerced to a fixnum" x)) ) )

(define ##sys#foreign-flonum-argument
    (lambda (x)
      (if (number? x)
	  x
	  (##sys#error "argument to foreign function can not be coerced to a flonum" x) ) ) )

(define ##sys#foreign-block-argument
    (lambda (x)
      (if (##core#inline "C_blockp" x)
	  x
	  (##sys#error "argument to foreign function is not a block" x) ) ) )

(define ##sys#foreign-string-argument
    (lambda (x)
      (if (and (##core#inline "C_blockp" x) (##core#inline "C_stringp" x))
	  x
	  (##sys#error "argument to foreign function is not a string" x) ) ) )

(define ##sys#foreign-pointer-argument
    (lambda (x)
      (if (and (##core#inline "C_blockp" x) (##core#inline "C_pointerp" x))
	  x
	  (##sys#error "argument to foreign function is not a pointer" x) ) ) )

(define ##sys#foreign-integer-argument
    (lambda (x)
      (if (or (exact? x)
	      (and (integer? x)
		   (##core#inline "C_flonum_in_int_range_p" x) ) )
	  x
	  (##sys#error "argument to foreign function does not fit into an integer" x)) ) )

(define ##sys#foreign-unsigned-integer-argument
    (lambda (x)
      (if (or (exact? x)
	      (and (integer? x)
		   (##core#inline "C_flonum_in_uint_range_p" x) ) )
	  x
	  (##sys#error "argument to foreign function does not fit into an unsigned integer" x)) ) )

(define ##sys#host-data-string
  (let ([make-string make-string])
    (lambda (hd)
      (let* ([len (##core#inline "C_block_size" hd)]
	     [s2 (make-string len)] )
	(##core#inline "C_copy_memory" s2 hd len)
	s2) ) ) )


;;; Low-level threading interface:

(define ##sys#default-thread-quantum 10000)

(define (##sys#default-exception-handler arg)
  (let ([ct ##sys#current-thread])
    (##sys#setslot ct 7 arg)
    (##sys#setslot ct 3 'terminated)
    (##sys#setslot ct 8 '())
    (##sys#abandon-mutexes ct)
    (##sys#schedule) ) )

(define (##sys#make-thread thunk state name q)
  (##sys#make-structure
   'thread
   thunk				; thunk
   #f					; result
   state				; state
   (lambda () #t)			; resume-test
   (vector				; state buffer
    ##sys#dynamic-winds
    ##sys#standard-input
    ##sys#standard-output
    ##sys#standard-error
    ##sys#default-exception-handler
    (##sys#grow-vector ##sys#current-parameter-vector (##sys#size ##sys#current-parameter-vector) #f) )
   name					; name
   (##core#undefined)			; end-exception
   '()					; owned mutexes
   q					; quantum
   (##core#undefined) ) )			; specific

(define ##sys#primordial-thread (##sys#make-thread #f 'running 'primordial ##sys#default-thread-quantum))
(define ##sys#current-thread ##sys#primordial-thread)

(define (##sys#make-mutex id owner)
  (##sys#make-structure
   'mutex
   id					; name
   owner				; thread or #f
   '()					; list of waiting threads
   #f					; abandoned
   #f					; locked
   (##core#undefined) ) )		; specific

(define (##sys#abandon-mutexes thread)
  (##sys#for-each
   (lambda (m)
     (##sys#setslot m 2 #f)
     (##sys#setslot m 4 #t) 
     (##sys#setslot m 5 #f)
     (##sys#setslot m 3 '()) )
   (##sys#slot thread 8)) )

(define ##sys#ready-queue-head '())
(define ##sys#ready-queue-tail '())

(define (##sys#add-to-ready-queue thread)
  (let ((new-pair (cons thread '())))
    (cond ((eq? '() ##sys#ready-queue-head) 
	   (set! ##sys#ready-queue-head new-pair))
	  (else (##sys#setslot ##sys#ready-queue-tail 1 new-pair)) )
    (set! ##sys#ready-queue-tail new-pair) ) )

(define (##sys#remove-from-ready-queue)
  (let ((first-pair ##sys#ready-queue-head))
    (when (null? first-pair) (##core#inline "C_halt" "no threads left"))
    (let ((first-cdr (##sys#slot first-pair 1)))
      (set! ##sys#ready-queue-head first-cdr)
      (when (eq? '() first-cdr) (set! ##sys#ready-queue-tail '()))
      (##sys#slot first-pair 0) ) ) )

(define (##sys#update-thread-state-buffer thread)
  (let ([buf (##sys#slot thread 5)])
    (##sys#setslot buf 0 ##sys#dynamic-winds)
    (##sys#setslot buf 1 ##sys#standard-input)
    (##sys#setslot buf 2 ##sys#standard-output)
    (##sys#setslot buf 3 ##sys#standard-error)
    (##sys#setslot buf 4 ##sys#current-exception-handler)
    (##sys#setslot buf 5 ##sys#current-parameter-vector) ) )

(define (##sys#restore-thread-state-buffer thread)
  (let ([buf (##sys#slot thread 5)])
    (set! ##sys#dynamic-winds (##sys#slot buf 0))
    (set! ##sys#standard-input (##sys#slot buf 1))
    (set! ##sys#standard-output (##sys#slot buf 2))
    (set! ##sys#standard-error (##sys#slot buf 3)) 
    (set! ##sys#current-exception-handler (##sys#slot buf 4))
    (set! ##sys#current-parameter-vector (##sys#slot buf 5)) ) )

(define (##sys#schedule)

  (define (switch thread)
    (set! ##sys#current-thread thread)
    (##sys#setslot thread 3 'running)
    (##sys#restore-thread-state-buffer thread)
    (##core#inline "C_set_initial_timer_interrupt_period" (##sys#slot thread 9))
    ((##sys#slot thread 1)) )

  (let ([ct ##sys#current-thread])
    (##sys#update-thread-state-buffer ct)
    (case (##sys#slot ct 3)
      [(running)
       (##sys#setslot ct 3 'ready)
       (##sys#add-to-ready-queue ct) ]
      [(ready blocked)
       (##sys#add-to-ready-queue ct) ] )
    (let loop ()
      (let ([nt (##sys#remove-from-ready-queue)])
	(case (##sys#slot nt 3)
	  [(ready) (switch nt)]
	  [(blocked)
	   (if ((##sys#slot nt 4))
	       (switch nt)
	       (##sys#add-to-ready-queue nt) ) ] )
	(loop) ) ) ) )

(define (##sys#all-threads)
  (cons ##sys#current-thread ##sys#ready-queue-head) )


;;; Interrupt-handling:

(define ##sys#context-switch (##core#primitive "C_context_switch"))

(define (##sys#interrupt-hook reason state)
  (when (eq? reason 255)		; C_TIMER_INTERRUPT_NUMBER
    (unless (null? ##sys#ready-queue-head)
      (let ([ct ##sys#current-thread])
	(##sys#setslot 
	 ct 1 
	 (lambda () 
	   (if (fx> (##sys#slot ##sys#pending-finalizers 0) 0)
	       (##sys#run-pending-finalizers state)
	       (##sys#context-switch state) ) ) )
	(##sys#schedule) ) ) )		; expected not to return!
  (if (fx> (##sys#slot ##sys#pending-finalizers 0) 0)
      (##sys#run-pending-finalizers state)
      (##sys#context-switch state) ) )


;;; Platform configuration inquiry:

(define software-type
  (let ([sym (string->symbol ((##core#primitive "C_software_type")))])
    (lambda () sym) ) )

(define machine-type
  (let ([sym (string->symbol ((##core#primitive "C_machine_type")))])
    (lambda () sym) ) )


;;; Entry point dispatching:

(define ##sys#entry-points-defined-flag #f)

(define ##sys#undefined-entry-point
    (lambda (id buffer)
      (##sys#error "undefined entry point" id) ) )

(define ##sys#entry-point-dispatch-table (##sys#make-vector 8 ##sys#undefined-entry-point))

(define ##sys#register-entry-point
  (let ([vector-copy! vector-copy!])
    (lambda (id handler)
      (when (fx> id (##sys#size ##sys#entry-point-dispatch-table))
	(let ([epdt2 (##sys#make-vector (fx+ id 8) ##sys#undefined-entry-point)])
	  (vector-copy! ##sys#entry-point-dispatch-table edt2)
	  (set! ##sys#entry-point-dispatch-table edt2) ) )
      (set! ##sys#entry-points-defined-flag #t)
      (##sys#setslot ##sys#entry-point-dispatch-table id handler) ) ) )

(define ##sys#dispatch-to-entry-point
  (lambda ()
    (let loop ()
      (let ([idbuf (##sys#host-data 0)])
	(when idbuf
	  (let ([id (##sys#peek-unsigned-integer idbuf 0)]
		[data (##sys#host-data 1)] )
	    ((vector-ref ##sys#entry-point-dispatch-table id) id data)
	    (##sys#set-host-data! 1 data) ) )
	(##sys#call-host) 
	(loop) ) ) ) )


;;; Accessing "errno":

(define-foreign-variable ##sys#errno integer "errno")

(let ([rn 0])
  (set! ##sys#update-errno (lambda () (set! rn ##sys#errno)))
  (set! errno (lambda () rn)) )


;;; Special string quoting syntax:

(set! ##sys#user-read-hook
  (let ([old ##sys#user-read-hook]
	[read-char read-char]
	[peek-char peek-char] 
	[write-char write-char]
	[open-output-string open-output-string]
	[get-output-string get-output-string] 
	[reverse reverse]
	[read read]
	[display display] )
    (define (readln port)
      (let ([ln (open-output-string)])
	(do ([c (read-char port) (read-char port)])
	    ((or (eof-object? c) (char=? #\newline c))
	     (cond [(char? c)
		    (set! ##sys#read-line-counter (fx+ ##sys#read-line-counter 1))
		    (get-output-string ln) ]
		   [else c] ) )
	  (write-char c ln) ) ) )
    (define (err) (##sys#error "unexpected end of file - unterminated string literal"))
    (define (fetch str lst)
      (let ([s (get-output-string str)])
	(if (fx= 0 (##sys#size s))
	    lst
	    (cons s lst) ) ) )
    (lambda (char port)
      (cond [(not (char=? #\< char)) (old char port)]
	    [else
	     (read-char port)
	     (case (peek-char port)
	       [(#\<)
		(read-char port)
		(let ([str (open-output-string)]
		      [end (readln port)] 
		      [f #f] )
		  (do ([ln (readln port) (readln port)])
		      ((string=? end ln) (get-output-string str))
		    (when (eof-object? ln) (err))
		    (if f 
			(write-char #\newline str)
			(set! f #t) )
		    (display ln str) ) ) ]
	       [(#\#)
		(read-char port)
		(let ([lst '()]
		      [end (readln port)] 
		      [str (open-output-string)] )
		  (let loop ()
		    (let ([c (read-char port)])
		      (when (eof-object? c) (err))
		      (case c
			[(#\newline)
			 (let ([s (get-output-string str)])
			   (set! str (open-output-string))
			   (cond [(string=? end s) `(##sys#print-to-string ,@(reverse (cdr lst)))] ; drop last newline
				 [else
				  (set! lst (cons #\newline (cons s lst)))
				  (loop) ] ) ) ]
			[(#\#)
			 (let ([c (peek-char port)])
			   (case c
			     [(#\#)
			      (write-char (read-char port) str)
			      (loop) ]
			     [(#\{)
			      (read-char port)
			      (set! lst (cons (read port) (fetch str lst)))
			      (set! str (open-output-string))
			      (let loop2 ()
				(let ([c (read-char port)])
				  (cond [(eof-object? c) (err)]
					[(char=? #\} c) (loop)]
					[else (loop2)] ) ) ) ]
			     [else
			      (set! lst (cons (read port) (fetch str lst)))
			      (set! str (open-output-string))
			      (loop) ] ) ) ]
			[else
			 (write-char c str)
			 (loop) ] ) ) ) ) ]
	       [else (##sys#error "unreadable object")] ) ] ) ) ) )


;;; Script invocation:

(define ##sys#script-main
  (let ([argv argv]
	[list-ref list-ref] )
    (lambda (prgi n)
      (let* ([av (argv)]
	     [args (list-tail av n)] )
	(unless (##sys#symbol-has-toplevel-binding? 'main)
	  (##sys#error "script has no `main' procedure") )
	((##sys#exit-handler) 
	 (let ([r (main args)])
	   (if (fixnum? r)
	       r
	       _ex_software) ) ) ) ) ) )


;;; Case-sensitivity parameter:

(define ##sys#case-sensitive (make-parameter #f))
(define case-sensitive ##sys#case-sensitive)


;;; Finalization:

(define-foreign-variable _max_pending_finalizers int "C_MAX_PENDING_FINALIZERS")

(define ##sys#pending-finalizers 
  (##sys#make-vector (fx+ (fx* 2 _max_pending_finalizers) 1) (##core#undefined)) )

(##sys#setslot ##sys#pending-finalizers 0 0)

(define set-finalizer! (##core#primitive "C_register_finalizer"))

(define ##sys#run-pending-finalizers
  (let ([vector-fill! vector-fill!])
    (lambda (state)
      (let* ([n (##sys#size ##sys#pending-finalizers)]
	     [c (##sys#slot ##sys#pending-finalizers 0)]
	     [pf (##sys#make-vector n)] )
	(##core#inline "C_subvector_copy" ##sys#pending-finalizers pf 0 n 0)
	(vector-fill! ##sys#pending-finalizers (##core#undefined))
	(##sys#setslot ##sys#pending-finalizers 0 0)
	(do ([i 0 (fx+ i 1)])
	    ((fx>= i c))
	  (let ([i2 (fx+ 1 (fx* i 2))])
	    ((##sys#slot pf (fx+ i2 1)) (##sys#slot pf i2)) ) )
	(when state (##sys#context-switch state) ) ) ) ) )

(define (##sys#force-finalizers)
  (let loop ()
    (##sys#gc)
    (when (fx> (##sys#slot ##sys#pending-finalizers 0) 0)
      (##sys#run-pending-finalizers #f)
      (loop) ) ) )


;;; Auxilliary definitions for safe use in quasiquoted forms and evaluated code:

(define ##sys#list->vector list->vector)
(define ##sys#list list)
(define ##sys#cons cons)
(define ##sys#append append)
(define ##sys#vector vector)
(define ##sys#apply apply)
(define ##sys#values values)
