/**
 * @file disk.h
 * Extension for accessing a disk drive on the remote host
 * @author Marko Mkel (msmakela@nic.funet.fi)
 */

/*
 * Copyright  1994-1996 Marko Mkel and Olaf Seibert
 * Copyright  2001 Marko Mkel
 * Original Linux and Commodore 64/128/Vic-20 version by Marko Mkel
 * Ported to the PET and the Amiga series by Olaf Seibert
 * Restructured by Marko Mkel
 * 
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 * 
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 * 
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/** install the disk extension
 * @param comm		the communication primitives
 * @param hostinfo	information on the remote host
 * @param device	the device number of the disk drive
 * @return		zero on success, nonzero on error
 */
int
disk_install (const struct comm* comm,
	      const struct hostinfo* hostinfo,
	      unsigned device);

/** remove the disk extension
 * @param comm		the communication primitives
 * @return		zero on success, nonzero on error
 */
int
disk_remove (const struct comm* comm);

/** copy a disk from the remote host
 * @param comm		the communication primitives
 * @param unit		the disk unit (drive 0 or 1)
 * @param interleave	the interleave factor (number of sectors to skip)
 * @param track		number of the track to start reading from
 * @param track_end	number of the last track, plus 1
 * @param file		output file
 * @param buf		a buffer of at least 1256 bytes
 * @return		zero on success, nonzero on error
 */
int
disk_read (const struct comm* comm,
	   unsigned unit,
	   unsigned interleave,
	   unsigned track,
	   unsigned track_end,
	   FILE* file, char* buf);

/** copy a disk to the remote host
 * @param comm		the communication primitives
 * @param unit		the disk unit (drive 0 or 1)
 * @param interleave	the interleave factor (number of sectors to skip)
 * @param track		number of the track to start reading from
 * @param track_end	number of the last track, plus 1
 * @param file		output file
 * @param buf		a buffer of at least 1260 bytes
 * @return		zero on success, nonzero on error
 */
int
disk_write (const struct comm* comm,
	    unsigned unit,
	    unsigned interleave,
	    unsigned track,
	    unsigned track_end,
	    FILE* file, char* buf);

/** copy data from a disk drive's address space to a file
 * @param comm		the communication primitives
 * @param file		output file
 * @param start		start address (inclusive)
 * @param end		end address (exclusive)
 * @param buf		a buffer of at least 260 bytes
 * @return		zero on success, nonzero on error
 */
int
disk_mread (const struct comm* comm, FILE* file,
	    unsigned start, unsigned end,
	    char* buf);

/** copy data from a disk drive's controller address space to a file
 * @param comm		the communication primitives
 * @param file		output file
 * @param start		start address (inclusive)
 * @param end		end address (exclusive)
 * @param buf		a buffer of at least 260 bytes
 * @return		zero on success, nonzero on error
 */
int
disk_cread (const struct comm* comm, FILE* file,
	    unsigned start, unsigned end,
	    char* buf);

/** copy data from a file to a disk drive's address space
 * @param comm		the communication primitives
 * @param file		input file
 * @param start		start address (inclusive)
 * @param buf		a buffer of at least 42 bytes
 * @return		zero on success, nonzero on error
 */
int
disk_mwrite (const struct comm* comm, FILE* file,
	     unsigned start,
	     char* buf);
