# Loom, a plugin for bzr to assist in developing focused patches.
# Copyright (C) 2006 Canonical Limited.
# 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 2 as published
# by the Free Software Foundation.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
# 


"""Tests of the loom Tree related routines."""


import bzrlib
from bzrlib import (
    errors,
    merge as _mod_merge,
)

from bzrlib.plugins.loom.branch import EMPTY_REVISION
from bzrlib.plugins.loom.tests import TestCaseWithLoom
import bzrlib.plugins.loom.tree
from bzrlib.revision import NULL_REVISION


class TestTreeDecorator(TestCaseWithLoom):
    """Tests of the LoomTreeDecorator class."""
    
    def get_loom_with_two_threads(self):
        tree = self.get_tree_with_loom('source')
        tree.branch.new_thread('bottom')
        tree.branch.new_thread('top')
        tree.branch.nick = 'bottom'
        return bzrlib.plugins.loom.tree.LoomTreeDecorator(tree)

    def test_down_thread(self):
        tree = self.get_tree_with_loom('source')
        tree.branch.new_thread('bottom')
        tree.branch.new_thread('top')
        tree.branch.nick = 'top'
        loom_tree = bzrlib.plugins.loom.tree.LoomTreeDecorator(tree)
        loom_tree.down_thread()
        self.assertEqual('bottom', tree.branch.nick)

    def _add_thread(self, tree, name):
        """Create a new thread with a commit and return the commit id."""
        tree.branch.new_thread(name)
        tree.branch.nick = name
        return tree.commit(name)

    def test_down_named_thread(self):
        tree = self.get_tree_with_loom('source')
        loom_tree = bzrlib.plugins.loom.tree.LoomTreeDecorator(tree)
        bottom_id = self._add_thread(tree, 'bottom')
        self._add_thread(tree, 'middle')
        self._add_thread(tree, 'top')
        self.assertNotEqual(bottom_id, tree.last_revision())
        loom_tree.down_thread('bottom')
        self.assertEqual('bottom', tree.branch.nick)
        self.assertEqual([bottom_id], tree.get_parent_ids())

    def test_up_thread(self):
        loom_tree = self.get_loom_with_two_threads()
        tree = loom_tree.tree
        loom_tree.up_thread()
        self.assertEqual('top', tree.branch.nick)
        self.assertEqual([], tree.get_parent_ids())

    def test_up_to_no_commits(self):
        tree = self.get_tree_with_loom('tree')
        tree.branch.new_thread('bottom')
        tree.branch.new_thread('top')
        tree.branch.nick = 'bottom'
        bottom_rev1 = tree.commit('bottom_commit')
        tree_loom_tree = bzrlib.plugins.loom.tree.LoomTreeDecorator(tree)
        tree_loom_tree.up_thread()
        self.assertEqual('top', tree.branch.nick)
        self.assertEqual([bottom_rev1], tree.get_parent_ids())

    def test_up_already_merged(self):
        """up-thread into a thread that already has this thread is a no-op."""
        tree = self.get_tree_with_loom('tree')
        tree.branch.new_thread('bottom')
        tree.branch.nick = 'bottom'
        bottom_rev1 = tree.commit('bottom_commit')
        tree.branch.new_thread('top', 'bottom')
        tree.branch.nick = 'top'
        top_rev1 = tree.commit('top_commit', allow_pointless=True)
        tree_loom_tree = bzrlib.plugins.loom.tree.LoomTreeDecorator(tree)
        tree_loom_tree.down_thread()
        # check the test will be valid
        self.assertEqual([None, bottom_rev1, top_rev1],
            tree.branch.repository.get_ancestry([top_rev1]))
        self.assertEqual([bottom_rev1], tree.get_parent_ids())
        tree_loom_tree.up_thread()
        self.assertEqual('top', tree.branch.nick)
        self.assertEqual([top_rev1], tree.get_parent_ids())

    def test_up_not_merged(self):
        """up-thread from a thread with new work."""
        tree = self.get_tree_with_loom('tree')
        tree.branch.new_thread('bottom')
        tree.branch.nick = 'bottom'
        bottom_rev1 = tree.commit('bottom_commit')
        tree.branch.new_thread('top', 'bottom')
        tree.branch.nick = 'top'
        top_rev1 = tree.commit('top_commit', allow_pointless=True)
        tree_loom_tree = bzrlib.plugins.loom.tree.LoomTreeDecorator(tree)
        tree_loom_tree.down_thread()
        # check the test will be valid
        self.assertEqual([None, bottom_rev1, top_rev1],
            tree.branch.repository.get_ancestry([top_rev1]))
        self.assertEqual([bottom_rev1], tree.get_parent_ids())
        bottom_rev2 = tree.commit('bottom_two', allow_pointless=True)
        tree_loom_tree.up_thread()
        self.assertEqual('top', tree.branch.nick)
        self.assertEqual([top_rev1, bottom_rev2], tree.get_parent_ids())

    def test_up_thread_at_top_with_lower_commit(self):
        loom_tree = self.get_loom_with_two_threads()
        self.build_tree_contents([('source/a', 'a')])
        loom_tree.tree.commit('add a')
        loom_tree.up_thread()
        e = self.assertRaises(errors.BzrCommandError, loom_tree.up_thread)
        self.assertEqual('Cannot move up from the highest thread.', str(e))

    def test_up_thread_merge_type(self):
        loom_tree = self.get_loom_with_two_threads()
        self.build_tree_contents([('source/a', 'a')])
        loom_tree.tree.add('a')
        loom_tree.tree.commit('add a')
        loom_tree.up_thread()
        self.build_tree_contents([('source/a', 'b')])
        loom_tree.tree.commit('content to b')
        loom_tree.down_thread()
        self.build_tree_contents([('source/a', 'c')])
        loom_tree.tree.commit('content to c')
        loom_tree.up_thread(_mod_merge.WeaveMerger)
        self.failIfExists('source/a.BASE')

    def test_revert_loom(self):
        tree = self.get_tree_with_loom(',')
        # ensure we have some stuff to revert
        tree.branch.new_thread('foo')
        tree.branch.new_thread('bar')
        tree.branch.nick = 'bar'
        tree.commit('change something', allow_pointless=True)
        loom_tree = bzrlib.plugins.loom.tree.LoomTreeDecorator(tree)
        loom_tree.revert_loom()
        # the tree should be reverted
        self.assertEqual(NULL_REVISION, tree.last_revision())
        # the current loom should be reverted 
        # (we assume this means branch.revert_loom was called())
        self.assertEqual([], tree.branch.get_loom_state().get_threads())

    def test_revert_thread(self):
        tree = self.get_tree_with_loom(',')
        # ensure we have some stuff to revert
        tree.branch.new_thread('foo')
        tree.branch.new_thread('bar')
        tree.branch.nick = 'bar'
        tree.commit('change something', allow_pointless=True)
        loom_tree = bzrlib.plugins.loom.tree.LoomTreeDecorator(tree)
        loom_tree.revert_loom(thread='bar')
        # the tree should be reverted
        self.assertEqual(NULL_REVISION, tree.last_revision())
        # the current loom should be reverted 
        # (we assume this means branch.revert_loom was called())
        self.assertEqual(
            [('foo', EMPTY_REVISION, [])],
            tree.branch.get_loom_state().get_threads())
        
    def test_revert_thread_different_thread(self):
        tree = self.get_tree_with_loom(',')
        # ensure we have some stuff to revert
        tree.branch.new_thread('foo')
        tree.branch.new_thread('bar')
        tree.branch.nick = 'bar'
        tree.commit('change something', allow_pointless=True)
        loom_tree = bzrlib.plugins.loom.tree.LoomTreeDecorator(tree)
        loom_tree.revert_loom(thread='foo')
        # the tree should not be reverted
        self.assertNotEqual(NULL_REVISION, tree.last_revision())
        # the bottom thread should be reverted
        # (we assume this means branch.revert_thread was 
        # called())
        self.assertEqual([('bar', tree.last_revision(), [])],
            tree.branch.get_loom_state().get_threads())
