/***************************************************************************
 *            burn-dialog.c
 *
 *  mer jun 29 13:05:45 2005
 *  Copyright  2005  Philippe Rouquier
 *  brasero-app@wanadoo.fr
 ***************************************************************************/

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdlib.h>
#include <string.h>

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <glib.h>
#include <glib/gi18n-lib.h>
#include <glib/gstdio.h>

#include <gdk/gdk.h>

#include <gtk/gtkwidget.h>
#include <gtk/gtkdialog.h>
#include <gtk/gtkmessagedialog.h>
#include <gtk/gtkbox.h>
#include <gtk/gtktogglebutton.h>
#include <gtk/gtkcheckbutton.h>
#include <gtk/gtkstock.h>
#include <gtk/gtklabel.h>

#ifdef HAVE_LIBNOTIFY
#include <libnotify/notify.h>
#endif

#include "utils.h"
#include "tray.h"
#include "burn-basics.h"
#include "burn-dialog.h"
#include "recorder-selection.h"
#include "progress.h"
#include "brasero-ncb.h"
extern gint debug;

static void brasero_burn_dialog_class_init (BraseroBurnDialogClass *klass);
static void brasero_burn_dialog_init (BraseroBurnDialog *obj);
static void brasero_burn_dialog_finalize (GObject *object);
static void brasero_burn_dialog_destroy (GtkObject *object);

static gboolean
brasero_burn_dialog_delete (GtkWidget *widget,
			    GdkEventAny *event);
static void
brasero_burn_dialog_cancel_clicked_cb (GtkWidget *button,
				       BraseroBurnDialog *dialog);

static void
brasero_burn_dialog_tray_cancel_cb (BraseroTrayIcon *tray,
				    BraseroBurnDialog *dialog);
static void
brasero_burn_dialog_tray_show_dialog_cb (BraseroTrayIcon *tray,
					 gboolean show,
					 GtkWidget *dialog);
static void
brasero_burn_dialog_tray_close_after_cb (BraseroTrayIcon *tray,
					 gboolean close,
					 BraseroBurnDialog *dialog);

struct BraseroBurnDialogPrivate {
	BraseroBurn *burn;
	NautilusBurnDrive *drive;
	BraseroTrackSourceType track_type;

	GtkWidget *waiting_disc_dialog;

	GtkWidget *close_check;
	GtkWidget *progress;
	GtkWidget *header;
	GtkWidget *image;
	GtkWidget *tray;

	GMainLoop *loop;
};


static GObjectClass *parent_class = NULL;

GType
brasero_burn_dialog_get_type ()
{
	static GType type = 0;

	if (type == 0) {
		static const GTypeInfo our_info = {
			sizeof (BraseroBurnDialogClass),
			NULL,
			NULL,
			(GClassInitFunc) brasero_burn_dialog_class_init,
			NULL,
			NULL,
			sizeof (BraseroBurnDialog),
			0,
			(GInstanceInitFunc) brasero_burn_dialog_init,
		};

		type = g_type_register_static (GTK_TYPE_DIALOG,
					       "BraseroBurnDialog",
					       &our_info, 0);
	}

	return type;
}

static void
brasero_burn_dialog_class_init (BraseroBurnDialogClass * klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);
	GtkObjectClass *gtk_object_class = GTK_OBJECT_CLASS (klass);
	GtkWidgetClass *widget_class = GTK_WIDGET_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);
	object_class->finalize = brasero_burn_dialog_finalize;
	gtk_object_class->destroy = brasero_burn_dialog_destroy;
	widget_class->delete_event = brasero_burn_dialog_delete;
}

static char *
brasero_burn_dialog_get_media_type_string (BraseroBurn *burn,
					   BraseroMediaType type,
					   gboolean insert)
{
	char *message = NULL;

	if (type & BRASERO_MEDIA_WITH_DATA) {
		if (!insert) {
			if (type & BRASERO_MEDIA_REWRITABLE)
				message = g_strdup (_("replace the disc with a rewritable disc holding data."));
			else
				message = g_strdup (_("replace the disc with a disc holding data."));
		}
		else {
			if (type & BRASERO_MEDIA_REWRITABLE)
				message = g_strdup (_("insert a rewritable disc holding data."));
			else
				message = g_strdup (_("insert a disc holding data."));
		}
	}
	else if (type & BRASERO_MEDIA_WRITABLE) {
		gint64 isosize = 0;
	
		brasero_burn_status (burn,
				     NULL,
				     &isosize,
				     NULL,
				     NULL,
				     NULL);

		if ((type & BRASERO_MEDIA_TYPE_CD) && !(type & BRASERO_MEDIA_TYPE_DVD)) {
			if (!insert) {
				if (isosize)
					message = g_strdup_printf (_("replace the disc with a recordable CD with a least %i Mb free."), 
								   (int) (isosize / 1048576));
				else
					message = g_strdup_printf (_("replace the disc with a recordable CD."));
			}
			else {
				if (isosize)
					message = g_strdup_printf (_("insert a recordable CD with a least %i Mb free."), 
								   (int) (isosize / 1048576));
				else
					message = g_strdup_printf (_("insert a recordable CD."));
			}
		}
		else if (!(type & BRASERO_MEDIA_TYPE_CD) && (type & BRASERO_MEDIA_TYPE_DVD)) {
			if (!insert) {
				if (isosize)
					message = g_strdup_printf (_("replace the disc with a recordable DVD with a least %i Mb free."), 
								   (int) (isosize / 1048576));
				else
					message = g_strdup_printf (_("replace the disc with a recordable DVD."));
			}
			else {
				if (isosize)
					message = g_strdup_printf (_("insert a recordable DVD with a least %i Mb free."), 
								   (int) (isosize / 1048576));
				else
					message = g_strdup_printf (_("insert a recordable DVD."));
			}
		}
		else if (!insert) {
			if (isosize)
				message = g_strdup_printf (_("replace the disc with a recordable CD or DVD with a least %i Mb free."), 
							   (int) (isosize / 1048576));
			else
				message = g_strdup_printf (_("replace the disc with a recordable CD or DVD."));
		}
		else {
			if (isosize)
				message = g_strdup_printf (_("insert a recordable CD or DVD with a least %i Mb free."), 
							   (int) (isosize / 1048576));
			else
				message = g_strdup_printf (_("insert a recordable CD or DVD."));
		}
	}

	return message;
}

static BraseroBurnResult
brasero_burn_dialog_insert_disc_cb (BraseroBurn *burn,
				    BraseroBurnError error,
				    BraseroMediaType type,
				    BraseroBurnDialog *dialog)
{
	gint result;
	GtkWindow *window;
	GtkWidget *message;
	gboolean hide = FALSE;
	char *main_message = NULL, *secondary_message = NULL;

	if (!GTK_WIDGET_VISIBLE (dialog)) {
		gtk_widget_show (GTK_WIDGET (dialog));
		hide = TRUE;
	}

	if (error == BRASERO_BURN_ERROR_MEDIA_BUSY) {
		main_message = g_strdup (_("The disc in the drive is busy:"));
		secondary_message = g_strdup (_("make sure another application is not using it and click OK"));
	} 
	else if (error == BRASERO_BURN_ERROR_MEDIA_NONE) {
		main_message = g_strdup (_("There is no disc in the drive:"));
		secondary_message = brasero_burn_dialog_get_media_type_string (burn, type, TRUE);
	}
	else if (error == BRASERO_BURN_ERROR_MEDIA_NOT_REWRITABLE) {
		main_message = g_strdup (_("The disc in the drive is not rewritable:"));
		secondary_message = brasero_burn_dialog_get_media_type_string (burn, type, FALSE);
	}
	else if (error == BRASERO_BURN_ERROR_MEDIA_BLANK) {
		main_message = g_strdup (_("The disc in the drive is empty:"));
		secondary_message = brasero_burn_dialog_get_media_type_string (burn, type, FALSE);
	}
	else if (error == BRASERO_BURN_ERROR_MEDIA_NOT_WRITABLE) {
		main_message = g_strdup (_("The disc in the drive is not writable:"));
		secondary_message = brasero_burn_dialog_get_media_type_string (burn, type, FALSE);
	}
	else if (error == BRASERO_BURN_ERROR_DVD_NOT_SUPPORTED) {
		main_message = g_strdup (_("The disc in the drive is a DVD:"));
		secondary_message = brasero_burn_dialog_get_media_type_string (burn, type, FALSE);
	}
	else if (error == BRASERO_BURN_ERROR_CD_NOT_SUPPORTED) {
		main_message = g_strdup (_("The disc in the drive is a CD:"));
		secondary_message = brasero_burn_dialog_get_media_type_string (burn, type, FALSE);
	}
	else if (error == BRASERO_BURN_ERROR_MEDIA_SPACE) {
		main_message = g_strdup (_("The disc in the drive is not big enough:"));
		secondary_message = brasero_burn_dialog_get_media_type_string (burn, type, FALSE);
	}
	else if (error == BRASERO_BURN_ERROR_NONE) {
		main_message = g_strdup_printf ("<b><big>%s</big></b>",
						brasero_burn_dialog_get_media_type_string (burn, type, FALSE));
	}
	else if (error == BRASERO_BURN_ERROR_RELOAD_MEDIA) {
		main_message = g_strdup (_("The disc in the drive needs to be reloaded:"));
		secondary_message = g_strdup (_("eject the disc and reload it."));
	}

	window = GTK_WINDOW (dialog);

	if (secondary_message) {
		message = gtk_message_dialog_new (window,
						  GTK_DIALOG_DESTROY_WITH_PARENT|
						  GTK_DIALOG_MODAL,
						  GTK_MESSAGE_QUESTION,
						  GTK_BUTTONS_CANCEL,
						  main_message);

		if (secondary_message) {
			gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (message),
								  secondary_message);
			g_free (secondary_message);
		}
	}
	else
		message = gtk_message_dialog_new_with_markup (window,
							      GTK_DIALOG_DESTROY_WITH_PARENT|
							      GTK_DIALOG_MODAL,
							      GTK_MESSAGE_QUESTION,
							      GTK_BUTTONS_CANCEL,
							      main_message);

	g_free (main_message);

	if (error == BRASERO_BURN_ERROR_MEDIA_NONE)
		gtk_window_set_title (GTK_WINDOW (message), _("Waiting for disc insertion"));
	else
		gtk_window_set_title (GTK_WINDOW (message), _("Waiting for disc replacement"));

	dialog->priv->waiting_disc_dialog = message;

	result = gtk_dialog_run (GTK_DIALOG (message));
	gtk_widget_destroy (message);

	dialog->priv->waiting_disc_dialog = NULL;

	if (hide)
		gtk_widget_hide (GTK_WIDGET (dialog));

	if (result != GTK_RESPONSE_OK)
		return BRASERO_BURN_CANCEL;

	return BRASERO_BURN_OK;
}

static BraseroBurnResult
brasero_burn_dialog_data_loss_cb (BraseroBurn *burn,
				  GtkDialog *dialog)
{
	gint result;
	GtkWindow *window;
	GtkWidget *button;
	GtkWidget *message;
	gboolean hide = FALSE;

	if (!GTK_WIDGET_VISIBLE (dialog)) {
		gtk_widget_show (GTK_WIDGET (dialog));
		hide = TRUE;
	}

	window = GTK_WINDOW (dialog);
	message = gtk_message_dialog_new (window,
					  GTK_DIALOG_DESTROY_WITH_PARENT|
					  GTK_DIALOG_MODAL,
					  GTK_MESSAGE_QUESTION,
					  GTK_BUTTONS_NONE,
					  _("The disc in the drive holds data:"));

	gtk_window_set_title (GTK_WINDOW (message), _("Erasing data"));

	gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (message),
						  _("Do you want to erase (you will lose all data previously written)\nor replace it ?"));

	button = brasero_utils_make_button (_("Replace the disc"), GTK_STOCK_REFRESH);
	gtk_widget_show_all (button);
	gtk_dialog_add_action_widget (GTK_DIALOG (message),
				      button, GTK_RESPONSE_ACCEPT);

	gtk_dialog_add_buttons (GTK_DIALOG (message),
				GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
				GTK_STOCK_CLEAR, GTK_RESPONSE_OK,
				NULL);

	result = gtk_dialog_run (GTK_DIALOG (message));
	gtk_widget_destroy (message);

	if (hide)
		gtk_widget_hide (GTK_WIDGET (dialog));

	if (result == GTK_RESPONSE_ACCEPT)
		return BRASERO_BURN_NEED_RELOAD;

	if (result != GTK_RESPONSE_OK)
		return BRASERO_BURN_CANCEL;

	return BRASERO_BURN_OK;
}

static BraseroBurnResult
brasero_burn_dialog_rewritable_cb (BraseroBurn *burn,
				   GtkDialog *dialog)
{
	gint result;
	GtkWindow *window;
	GtkWidget *button;
	GtkWidget *message;
	gboolean hide = FALSE;

	if (!GTK_WIDGET_VISIBLE (dialog)) {
		gtk_widget_show (GTK_WIDGET (dialog));
		hide = TRUE;
	}

	window = GTK_WINDOW (dialog);
	message = gtk_message_dialog_new (window,
					  GTK_DIALOG_DESTROY_WITH_PARENT|
					  GTK_DIALOG_MODAL,
					  GTK_MESSAGE_QUESTION,
					  GTK_BUTTONS_NONE,
					  _("Recording audio tracks on a rewritable disc is not advised:"));

	gtk_window_set_title (GTK_WINDOW (message), _("Rewritable disc"));

	gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (message),
						  _("you might not be able to listen to it with stereos.\nDo you want to continue anyway?"));

	button = brasero_utils_make_button (_("Replace the disc"), GTK_STOCK_REFRESH);
	gtk_widget_show_all (button);
	gtk_dialog_add_action_widget (GTK_DIALOG (message),
				      button, GTK_RESPONSE_ACCEPT);

	gtk_dialog_add_buttons (GTK_DIALOG (message),
				GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
				NULL);

	button = brasero_utils_make_button (_("Continue"), GTK_STOCK_OK);
	gtk_widget_show_all (button);
	gtk_dialog_add_action_widget (GTK_DIALOG (message),
				      button, GTK_RESPONSE_OK);

	result = gtk_dialog_run (GTK_DIALOG (message));
	gtk_widget_destroy (message);

	if (hide)
		gtk_widget_hide (GTK_WIDGET (dialog));

	if (result == GTK_RESPONSE_ACCEPT)
		return BRASERO_BURN_NEED_RELOAD;

	if (result != GTK_RESPONSE_OK)
		return BRASERO_BURN_CANCEL;

	return BRASERO_BURN_OK;
}

static BraseroBurnResult
brasero_burn_dialog_disable_joliet_cb (BraseroBurn *burn,
				       GtkDialog *dialog)
{
	gint result;
	GtkWindow *window;
	GtkWidget *button;
	GtkWidget *message;
	gboolean hide = FALSE;

	if (!GTK_WIDGET_VISIBLE (dialog)) {
		gtk_widget_show (GTK_WIDGET (dialog));
		hide = TRUE;
	}

	window = GTK_WINDOW (dialog);
	message = gtk_message_dialog_new (window,
					  GTK_DIALOG_DESTROY_WITH_PARENT|
					  GTK_DIALOG_MODAL,
					  GTK_MESSAGE_QUESTION,
					  GTK_BUTTONS_NONE,
					  _("Some files don't have a suitable name for a Windows-compatible CD:"));

	gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (message),
						  _("Do you want to continue with Windows compatibility disabled?"));

	gtk_window_set_title (GTK_WINDOW (message), _("Windows compatibility"));
	gtk_dialog_add_buttons (GTK_DIALOG (message),
				GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
				NULL);

	button = brasero_utils_make_button (_("Continue"), GTK_STOCK_OK);
	gtk_widget_show_all (button);
	gtk_dialog_add_action_widget (GTK_DIALOG (message),
				      button, GTK_RESPONSE_OK);

	result = gtk_dialog_run (GTK_DIALOG (message));
	gtk_widget_destroy (message);

	if (hide)
		gtk_widget_hide (GTK_WIDGET (dialog));

	if (result != GTK_RESPONSE_OK)
		return BRASERO_BURN_CANCEL;

	return BRASERO_BURN_OK;
}

static void
brasero_burn_dialog_progress_changed_cb (BraseroBurn *burn, 
					 gdouble overall_progress,
					 gdouble task_progress,
					 glong remaining,
					 BraseroBurnDialog *dialog)
{
	NautilusBurnMediaType type;
	gint64 isosize = -1;
	gint64 written = -1;
	gint mb_isosize = -1;
	gint mb_written = -1;
	gint64 rate = -1;
	gint fifo = -1;

	brasero_burn_status (dialog->priv->burn,
			     &type,
			     &isosize,
			     &written,
			     &rate,
			     &fifo);

	if (isosize > 0)
		mb_isosize = (gint) (isosize / 1048576);
	if (written >= 0)
		mb_written = (gint) (written / 1048576);

	brasero_burn_progress_set_status (BRASERO_BURN_PROGRESS (dialog->priv->progress),
					  (type > NAUTILUS_BURN_MEDIA_TYPE_CDRW),
					  overall_progress,
					  task_progress,
					  remaining,
					  mb_isosize,
					  mb_written,
					  rate,
					  fifo);

	brasero_tray_icon_set_progress (BRASERO_TRAYICON (dialog->priv->tray),
					overall_progress,
					remaining);
}

static void
brasero_burn_dialog_action_changed_cb (BraseroBurn *burn, 
				       BraseroBurnAction action,
				       BraseroBurnDialog *dialog)
{
	gchar *string = NULL;

	brasero_burn_get_action_string (dialog->priv->burn, action, &string);
	brasero_burn_progress_set_action (BRASERO_BURN_PROGRESS (dialog->priv->progress),
					  action,
					  string);
	g_free (string);

	brasero_tray_icon_set_action (BRASERO_TRAYICON (dialog->priv->tray),
				      action);
}

static void
brasero_burn_dialog_init (BraseroBurnDialog * obj)
{
	GtkWidget *box;
	GtkWidget *vbox;
	GtkWidget *cancel;
	GtkWidget *alignment;

	obj->priv = g_new0 (BraseroBurnDialogPrivate, 1);
	gtk_window_set_default_size (GTK_WINDOW (obj), 500, 0);

	gtk_dialog_set_has_separator (GTK_DIALOG (obj), FALSE);
	gtk_window_set_skip_taskbar_hint (GTK_WINDOW (obj), FALSE);
	gtk_window_set_skip_pager_hint (GTK_WINDOW (obj), FALSE);
	gtk_window_set_type_hint (GTK_WINDOW (obj), GDK_WINDOW_TYPE_HINT_NORMAL);
	gtk_window_set_position (GTK_WINDOW (obj),GTK_WIN_POS_CENTER);

	obj->priv->tray = brasero_tray_icon_new ();
	g_signal_connect (obj->priv->tray,
			  "cancel",
			  G_CALLBACK (brasero_burn_dialog_tray_cancel_cb),
			  obj);
	g_signal_connect (obj->priv->tray,
			  "show-dialog",
			  G_CALLBACK (brasero_burn_dialog_tray_show_dialog_cb),
			  obj);
	g_signal_connect (obj->priv->tray,
			  "close-after",
			  G_CALLBACK (brasero_burn_dialog_tray_close_after_cb),
			  obj);

	alignment = gtk_alignment_new (0.5, 0.5, 1.0, 1.0);
	gtk_alignment_set_padding (GTK_ALIGNMENT (alignment), 6, 8, 6, 6);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (obj)->vbox),
			    alignment,
			    TRUE,
			    TRUE,
			    0);

	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (alignment), vbox);

	box = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), box, TRUE, TRUE, 0);

	obj->priv->header = gtk_label_new (NULL);
	gtk_misc_set_alignment (GTK_MISC (obj->priv->header), 0.0, 0.5);
	gtk_misc_set_padding (GTK_MISC (obj->priv->header), 0, 18);
	gtk_box_pack_start (GTK_BOX (box), obj->priv->header, TRUE, TRUE, 0);

	obj->priv->image = gtk_image_new ();
	gtk_box_pack_start (GTK_BOX (box), obj->priv->image, FALSE, FALSE, 0);

	obj->priv->progress = brasero_burn_progress_new ();
	gtk_box_pack_start (GTK_BOX (vbox),
			    obj->priv->progress,
			    FALSE,
			    FALSE,
			    0);

	obj->priv->close_check = gtk_check_button_new_with_label (_("Close the application when the disc is successfully burnt"));
	gtk_box_pack_end (GTK_BOX (obj->priv->progress),
			  obj->priv->close_check,
			  FALSE,
			  FALSE,
			  0);

	/* Create tracks */
	obj->priv->burn = brasero_burn_new ();
	g_signal_connect (obj->priv->burn,
			  "insert-media",
			  G_CALLBACK (brasero_burn_dialog_insert_disc_cb),
			  obj);
	g_signal_connect (obj->priv->burn,
			  "warn-data-loss",
			  G_CALLBACK (brasero_burn_dialog_data_loss_cb),
			  obj);
	g_signal_connect (obj->priv->burn,
			  "warn-rewritable",
			  G_CALLBACK (brasero_burn_dialog_rewritable_cb),
			  obj);
	g_signal_connect (obj->priv->burn,
			  "disable-joliet",
			  G_CALLBACK (brasero_burn_dialog_disable_joliet_cb),
			  obj);
	g_signal_connect (obj->priv->burn,
			  "progress-changed",
			  G_CALLBACK (brasero_burn_dialog_progress_changed_cb),
			  obj);
	g_signal_connect (obj->priv->burn,
			  "action-changed",
			  G_CALLBACK (brasero_burn_dialog_action_changed_cb),
			  obj);

	/* buttons */
	cancel = gtk_button_new_from_stock (GTK_STOCK_CANCEL);
	g_signal_connect (G_OBJECT (cancel),
			  "clicked",
			  G_CALLBACK (brasero_burn_dialog_cancel_clicked_cb),
			  obj);
	gtk_dialog_add_action_widget (GTK_DIALOG (obj), cancel, GTK_RESPONSE_CANCEL);
}

static void
brasero_burn_dialog_destroy (GtkObject * object)
{
	BraseroBurnDialog *cobj;

	cobj = BRASERO_BURN_DIALOG (object);

	if (cobj->priv->burn) {
		g_object_unref (cobj->priv->burn);
		cobj->priv->burn = NULL;
	}

	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		GTK_OBJECT_CLASS (parent_class)->destroy (object);
}

static void
brasero_burn_dialog_finalize (GObject * object)
{
	BraseroBurnDialog *cobj;

	cobj = BRASERO_BURN_DIALOG (object);

	if (cobj->priv->burn) {
		g_object_unref (cobj->priv->burn);
		cobj->priv->burn = NULL;
	}

	if (cobj->priv->tray) {
		gtk_widget_destroy (cobj->priv->tray);
		cobj->priv->tray = NULL;
	}

	if (cobj->priv->loop) {
		g_main_loop_quit (cobj->priv->loop);
		g_main_loop_unref (cobj->priv->loop);
		cobj->priv->loop = NULL;
	}

	g_free (cobj->priv);
	G_OBJECT_CLASS (parent_class)->finalize (object);
}

GtkWidget *
brasero_burn_dialog_new (void)
{
	BraseroBurnDialog *obj;

	obj = BRASERO_BURN_DIALOG (g_object_new (BRASERO_TYPE_BURN_DIALOG, NULL));

	return GTK_WIDGET (obj);
}

static void
brasero_burn_dialog_update_info (BraseroBurnDialog *dialog)
{
	char *title = NULL;
	char *header = NULL;
	GdkPixbuf *pixbuf = NULL;
	NautilusBurnDrive *drive;
	NautilusBurnMediaType media_type;

	char *types [] = { 	NULL,
				NULL,
				NULL,
				"gnome-dev-cdrom",
				"gnome-dev-disc-cdr",
				"gnome-dev-disc-cdrw",
				"gnome-dev-disc-dvdrom",
				"gnome-dev-disc-dvdr",
				"gnome-dev-disc-dvdrw",
				"gnome-dev-disc-dvdram",
				"gnome-dev-disc-dvdr-plus",
				"gnome-dev-disc-dvdrw", /* FIXME */
				"gnome-dev-disc-dvdr-plus" /* FIXME */,
				NULL };

	drive = dialog->priv->drive;
	if (NCB_DRIVE_GET_TYPE (drive) == NAUTILUS_BURN_DRIVE_TYPE_FILE) {
		/* we are creating an image to the hard drive */
		pixbuf = brasero_utils_get_icon_for_mime ("application/x-cd-image", 48);

		/* FIXME: we don't want to break strings */
		header = g_strdup_printf ("<big><b>%s</b></big>", _("Creating image"));
		title = g_strdup (_("Creating image"));
		goto end;
	}

	media_type = nautilus_burn_drive_get_media_type (drive);

	if (media_type > NAUTILUS_BURN_MEDIA_TYPE_CDRW) {
		if (dialog->priv->track_type == BRASERO_TRACK_SOURCE_DATA
		||  dialog->priv->track_type == BRASERO_TRACK_SOURCE_ISO
		||  dialog->priv->track_type == BRASERO_TRACK_SOURCE_ISO_JOLIET
		||  dialog->priv->track_type == BRASERO_TRACK_SOURCE_GRAFTS) {
			title = g_strdup (_("Burning DVD"));
			header = g_strdup (_("<big><b>Burning data DVD</b></big>"));
		}
		else if (dialog->priv->track_type == BRASERO_TRACK_SOURCE_DISC) {
			title = g_strdup (_("Copying DVD"));
			header = g_strdup (_("<big><b>Copying data DVD</b></big>"));
		}
	}
	else if (dialog->priv->track_type == BRASERO_TRACK_SOURCE_AUDIO
	      ||  dialog->priv->track_type == BRASERO_TRACK_SOURCE_SONG) {
		title = g_strdup (_("Burning CD"));
		header = g_strdup_printf (_("<big><b>Burning audio CD</b></big>"));
		pixbuf = brasero_utils_get_icon ("gnome-dev-cdrom-audio", 48);
	}
	else if (dialog->priv->track_type == BRASERO_TRACK_SOURCE_DATA
	      ||  dialog->priv->track_type == BRASERO_TRACK_SOURCE_ISO
	      ||  dialog->priv->track_type == BRASERO_TRACK_SOURCE_ISO_JOLIET
	      ||  dialog->priv->track_type == BRASERO_TRACK_SOURCE_GRAFTS) {
		title = g_strdup (_("Burning CD"));
		header = g_strdup_printf (_("<big><b>Burning data CD</b></big>"));
	}
	else if (dialog->priv->track_type == BRASERO_TRACK_SOURCE_DISC) {
		title = g_strdup (_("Copying CD"));
		header = g_strdup(_("<big><b>Copying CD</b></big>"));
	}
	else if (dialog->priv->track_type == BRASERO_TRACK_SOURCE_CUE
	      ||  dialog->priv->track_type == BRASERO_TRACK_SOURCE_RAW) {
		title = g_strdup (_("Burning CD"));
		header = g_strdup (_("<big><b>Burning disc image</b></big>"));
	}

	if (!pixbuf) {
		pixbuf = brasero_utils_get_icon (types [media_type], 48);
		if (!pixbuf)
			pixbuf = brasero_utils_get_icon ("gnome-dev-removable", 48);
	}

end:

	gtk_window_set_title (GTK_WINDOW (dialog), title);
	g_free (title);

	gtk_image_set_from_pixbuf (GTK_IMAGE (dialog->priv->image), pixbuf);
	g_object_unref (pixbuf);

	gtk_label_set_markup (GTK_LABEL (dialog->priv->header), header);
	g_free (header);
}

static void
brasero_burn_dialog_media_added_cb (NautilusBurnDrive *drive,
				    BraseroBurnDialog *dialog)
{
	brasero_burn_dialog_update_info (dialog);

	/* we might have a dialog waiting for the 
	 * insertion of a disc if so close it */
	if (dialog->priv->waiting_disc_dialog) {
		gtk_dialog_response (GTK_DIALOG (dialog->priv->waiting_disc_dialog),
				     GTK_RESPONSE_OK);
	}
}

static void
brasero_burn_dialog_media_removed_cb (NautilusBurnDrive *drive,
				      BraseroBurnDialog *dialog)
{
	GdkPixbuf *pixbuf;

	pixbuf = brasero_utils_get_icon ("gnome-dev-removable", 48);
	gtk_image_set_from_pixbuf (GTK_IMAGE (dialog->priv->image), pixbuf);
	g_object_unref (pixbuf);
}

#ifdef HAVE_LIBNOTIFY

static void
brasero_burn_dialog_notify_daemon_close (NotifyNotification *notification,
					 BraseroBurnDialog *dialog)
{
	if (dialog->priv->loop
	&&  g_main_loop_is_running (dialog->priv->loop))
		g_main_loop_quit (dialog->priv->loop);
}

gboolean
brasero_burn_dialog_notify_daemon (BraseroBurnDialog *dialog,
				   const char *primary,
				   const char *secondary)
{
	NotifyNotification *notification;
	gboolean result;

	notification = notify_notification_new (primary,
						secondary,
						GTK_STOCK_CDROM,
						dialog->priv->tray);
	if (!notification)
		return FALSE;

	g_signal_connect (notification,
			  "closed",
			  G_CALLBACK (brasero_burn_dialog_notify_daemon_close),
			  dialog);

	notify_notification_set_timeout (notification, 10000);
	notify_notification_set_urgency (notification, NOTIFY_URGENCY_NORMAL);
	result = notify_notification_show (notification, NULL);

	/* now we wait for the notification to disappear or for the user to
	 * click on the icon in the tray */
	dialog->priv->loop = g_main_loop_new (NULL, FALSE);
	g_main_loop_run (dialog->priv->loop);

	if (dialog->priv->loop) {
		g_main_loop_unref (dialog->priv->loop);
		dialog->priv->loop = NULL;
	}

	return result;
}

#endif

static gboolean
brasero_burn_dialog_success_timeout (GtkDialog *dialog)
{
	gtk_dialog_response (dialog, GTK_RESPONSE_CANCEL);
	return FALSE;
}

static void
brasero_burn_dialog_notify_dialog (BraseroBurnDialog *dialog,
				   const gchar *primary,
				   const gchar *secondary,
				   GtkMessageType type,
				   gboolean timeout)
{
	gint id = 0;
	GtkWidget *message;
	GtkResponseType response;

	if (!GTK_WIDGET_VISIBLE (dialog))
		gtk_widget_show (GTK_WIDGET (dialog));

	message = gtk_message_dialog_new (GTK_WINDOW (dialog),
					  GTK_DIALOG_DESTROY_WITH_PARENT |
					  GTK_DIALOG_MODAL,
					  type,
					  GTK_BUTTONS_CLOSE,
					  primary);

	gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (message),
						  "%s.",
						  secondary);

	if (timeout
	&&  gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dialog->priv->close_check)))
		id = g_timeout_add (10000,
				    (GSourceFunc) brasero_burn_dialog_success_timeout,
				    message);

	response = gtk_dialog_run (GTK_DIALOG (message));
	gtk_widget_destroy (message);

	if (id && response != GTK_RESPONSE_CANCEL)
		g_source_remove (id);
}

static void
brasero_burn_dialog_notify_error (BraseroBurnDialog *dialog,
				  GError *error)
{
	char *secondary;

	if (error) {
		secondary =  g_strdup (error->message);
		g_error_free (error);
	}
	else
		secondary = g_strdup (_("An unknown error occured. Check your disc"));

	brasero_burn_dialog_notify_dialog (dialog,
					   _("Error while burning:"),
					   secondary,
					   GTK_MESSAGE_ERROR,
					   FALSE);

	g_free (secondary);
}

static void
brasero_burn_dialog_notify_success (BraseroBurnDialog *dialog,
				    const BraseroTrackSource *track,
				    gboolean to_disc)
{
	char *primary = NULL;
	char *secondary = NULL;

	switch (track->type) {
	case BRASERO_TRACK_SOURCE_SONG:
		primary = g_strdup (_("Audio disc successfully burnt:"));
		secondary = g_strdup_printf (_("\"%s\" is now ready for use"), track->contents.songs.album);
		break;
	case BRASERO_TRACK_SOURCE_DISC:
		if (to_disc) {
			primary = g_strdup (_("Disc successfully copied to disc:"));
			secondary = g_strdup_printf (_("disc is now ready for use"));
		}
		else {
			primary = g_strdup (_("Disc successfully copied:"));
			secondary = g_strdup_printf (_("Image is now ready for use"));
		}
		break;
	case BRASERO_TRACK_SOURCE_ISO:
	case BRASERO_TRACK_SOURCE_RAW:
	case BRASERO_TRACK_SOURCE_ISO_JOLIET:
	case BRASERO_TRACK_SOURCE_CUE:
		primary = g_strdup (_("Image successfully burnt to disc:"));
		secondary = g_strdup_printf (_("disc is now ready for use"));
		break;
	default:
		if (to_disc) {
			primary = g_strdup (_("Data disc successfully burnt:"));
			secondary = g_strdup_printf (_("\"%s\" is now ready for use"), track->contents.data.label);
		}
		else {
			primary = g_strdup (_("Image successfully created on disc:"));
			secondary = g_strdup_printf (_("\"%s\" is now ready for use"), track->contents.data.label);
		}
		break;
	}

#ifdef HAVE_LIBNOTIFY

	if (GTK_WIDGET_VISIBLE (dialog))
		brasero_burn_dialog_notify_dialog (dialog,
						   primary,
						   secondary,
						   GTK_MESSAGE_INFO,
						   TRUE);
	else if (!brasero_burn_dialog_notify_daemon (dialog,
						     primary,
						     secondary))

#endif

		brasero_burn_dialog_notify_dialog (dialog,
						   primary,
						   secondary,
						   GTK_MESSAGE_INFO,
						   TRUE);
	g_free (primary);
	g_free (secondary);
}

gboolean
brasero_burn_dialog_run (BraseroBurnDialog *dialog,
			 NautilusBurnDrive *drive,
			 gint speed,
			 const char *output,
			 const BraseroTrackSource *track,
			 BraseroBurnFlag flags)
{
	int added_id;
	int removed_id;
	GdkCursor *cursor;
	GError *error = NULL;
	gboolean close_dialog;
	BraseroBurnResult result;

	gtk_widget_show_all (dialog->priv->tray);

	dialog->priv->track_type = track->type;

	if (debug)
		flags |= BRASERO_BURN_FLAG_DEBUG;

	if (NCB_DRIVE_GET_TYPE (drive) == NAUTILUS_BURN_DRIVE_TYPE_FILE
	&&  track->type == BRASERO_TRACK_SOURCE_DISC)
		dialog->priv->drive = track->contents.drive.disc;
	else
		dialog->priv->drive = drive;

	g_object_ref (dialog->priv->drive);

	nautilus_burn_drive_ref (drive);
	added_id = g_signal_connect (drive,
				     "media-added",
				     G_CALLBACK (brasero_burn_dialog_media_added_cb),
				     dialog);
	removed_id = g_signal_connect (drive,
				       "media-removed",
				       G_CALLBACK (brasero_burn_dialog_media_removed_cb),
				       dialog);

	brasero_burn_dialog_update_info (dialog);		

	cursor = gdk_cursor_new (GDK_WATCH);
	gdk_window_set_cursor (GTK_WIDGET (dialog)->window, cursor);
	gdk_cursor_unref (cursor);

	result = brasero_burn_record (dialog->priv->burn,
				      flags,
				      drive,
				      speed,
				      track,
				      output,
				      &error);
	if (added_id) {
		g_signal_handler_disconnect (drive, added_id);
		added_id = 0;
	}

	if (removed_id) {
		g_signal_handler_disconnect (drive, removed_id);
		removed_id = 0;
	}

	g_object_unref (dialog->priv->burn);
	dialog->priv->burn = NULL;

	g_object_unref (dialog->priv->drive);
	dialog->priv->drive = NULL;

	/* tell the user the result of the operation */
	if (result == BRASERO_BURN_CANCEL) {
		/* nothing to do */
		close_dialog = FALSE;
	}
	else if (error || result != BRASERO_BURN_OK) {
		close_dialog = FALSE;
		brasero_burn_dialog_notify_error (dialog,
						  error);
	}
	else {
		brasero_burn_dialog_notify_success (dialog,
						    track,
						    (NCB_DRIVE_GET_TYPE (drive) != NAUTILUS_BURN_DRIVE_TYPE_FILE));

		close_dialog = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dialog->priv->close_check));
	}

	nautilus_burn_drive_unref (drive);
	return close_dialog;
}

static gboolean
brasero_burn_dialog_cancel_dialog (GtkWidget *toplevel)
{
	gint result;
	GtkWidget *button;
	GtkWidget *message;

	message = gtk_message_dialog_new (GTK_WINDOW (toplevel),
					  GTK_DIALOG_DESTROY_WITH_PARENT |
					  GTK_DIALOG_MODAL,
					  GTK_MESSAGE_QUESTION,
					  GTK_BUTTONS_NONE,
					  _("Do you really want to quit?"));

	gtk_window_set_title (GTK_WINDOW (message), _("Confirm"));

	gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG
						  (message),
						  _("Interrupting the process may make disc unusable."));

	gtk_dialog_add_buttons (GTK_DIALOG (message),
				GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
				NULL);

	button = brasero_utils_make_button (_("Continue"), GTK_STOCK_OK);
	gtk_widget_show_all (button);
	gtk_dialog_add_action_widget (GTK_DIALOG (message),
				      button, GTK_RESPONSE_OK);

	result = gtk_dialog_run (GTK_DIALOG (message));
	gtk_widget_destroy (message);

	return (result != GTK_RESPONSE_OK);
}

static gboolean
brasero_burn_dialog_cancel (BraseroBurnDialog *dialog)
{
	if (dialog->priv->burn) {
		BraseroBurnResult result;

		result = brasero_burn_cancel (dialog->priv->burn, TRUE);

		if (result == BRASERO_BURN_DANGEROUS) {
			if (brasero_burn_dialog_cancel_dialog (GTK_WIDGET (dialog)))
				brasero_burn_cancel (dialog->priv->burn, FALSE);
			else
				return FALSE;
		}
	}
	else
		gtk_widget_destroy (GTK_WIDGET (dialog));

	return TRUE;
}

static gboolean
brasero_burn_dialog_delete (GtkWidget *widget, 
			    GdkEventAny *event)
{
	BraseroBurnDialog *dialog;

	dialog = BRASERO_BURN_DIALOG (widget);

	brasero_tray_icon_set_show_dialog (BRASERO_TRAYICON (dialog->priv->tray), FALSE);
 	return TRUE;
}

static void
brasero_burn_dialog_cancel_clicked_cb (GtkWidget *button,
				       BraseroBurnDialog *dialog)
{
	brasero_burn_dialog_cancel (dialog);
}

static void
brasero_burn_dialog_tray_cancel_cb (BraseroTrayIcon *tray,
				    BraseroBurnDialog *dialog)
{
	brasero_burn_dialog_cancel (dialog);
}

static void
brasero_burn_dialog_tray_show_dialog_cb (BraseroTrayIcon *tray,
					 gboolean show,
					 GtkWidget *dialog)
{
	/* this is when the dialog is minimized and a notification is shown
	 * after a successful burning */
	if (BRASERO_BURN_DIALOG (dialog)->priv->loop
	&&  g_main_loop_is_running (BRASERO_BURN_DIALOG (dialog)->priv->loop)) {
		g_main_loop_quit (BRASERO_BURN_DIALOG (dialog)->priv->loop);
		return;
	}

	/* we prevent to show the burn dialog once the success dialog has been 
	 * shown to avoid the following strange behavior:
	 * Steps:
	 * - start burning
	 * - move to another workspace (ie, virtual desktop)
	 * - when the burning finishes, double-click the notification icon
	 * - you'll be unable to dismiss the dialogues normally and their behaviour will
	 *   be generally strange */
	if (!BRASERO_BURN_DIALOG (dialog)->priv->burn)
		return;

	if (show)
		gtk_widget_show (dialog);
	else
		gtk_widget_hide (dialog);
}

static void
brasero_burn_dialog_tray_close_after_cb (BraseroTrayIcon *tray,
					 gboolean close,
					 BraseroBurnDialog *dialog)
{
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dialog->priv->close_check), close);
}
