//  BMPx - The Dumb Music Player
//  Copyright (C) 2005 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License Version 2
//  as published by the Free Software Foundation.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non-GPL compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#include <cstdlib>
#include <cstring>
#include <cmath>
#include <algorithm>

#include <gtkmm.h>
#include <glibmm/i18n.h>
#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include <gdk/gdkx.h>
#include <X11/Xlib.h>

#include "main.hh"
#include "paths.hh"
#include "util.hh"
#include "ui_toolbox.hh"

namespace
{
  gdouble
  mm_to_inch (gdouble mm)
  {
    return mm / 25.40;
  }

  bool _has_alpha = false;
  GtkStyle *_style = 0;
  Glib::RefPtr<Gdk::Colormap> _colormap;
}

namespace Bmp
{
  namespace Util
  {
#ifdef ENABLE_NLS
    char *
    menu_translate (char const* message,
                    gpointer    data)
    {
      char *translation = const_cast<char *> (Q_(message));

      if (!translation)
        {
          g_warning ("Bad translation for message: %s", message);
          translation = const_cast<char *> (message);
        }

      return translation;
    }
#endif

    void
    window_set_busy (GtkWidget * widget)
    {
        GdkCursor *cursor = gdk_cursor_new_from_name (gdk_display_get_default (), "watch");
        if (!cursor) cursor = gdk_cursor_new_for_display (gdk_display_get_default (), GDK_WATCH);
        gdk_window_set_cursor (GTK_WIDGET(widget)->window, cursor);
    }

    void
    window_set_idle (GtkWidget * widget)
    {
        gdk_window_set_cursor (GTK_WIDGET(widget)->window, NULL);
    }

    GtkWidget *
    ui_manager_get_popup (GtkUIManager * ui_manager,
                          char const*    path)
    {
      GtkWidget *menu_item;
      GtkWidget *submenu;

      menu_item = gtk_ui_manager_get_widget (ui_manager, path);

      if (GTK_IS_MENU_ITEM (menu_item))
          submenu = gtk_menu_item_get_submenu (GTK_MENU_ITEM (menu_item));
      else
          submenu = NULL;

      return submenu;
    }

    void
    window_set_icon_list (GtkWidget * window,
                          char const* icon_name)
    {
      static const char* sizes[] = {
        "16", "32", "48", "64", "128"
      };

      GdkPixbuf *pixbufs[5];
      GError *error = NULL;
      GList *icon_list = NULL;
      const char *icon_theme;
      char *aux0, *aux1;
      int n;

      icon_theme = mcs->key_get<std::string>("bmp", "icon-theme"). c_str ();

      for (n = 0; n < 5; n++)
        {
        aux0 = g_strconcat("icon_", icon_name, "_", sizes[n], ".png", NULL);
        aux1 = g_build_filename (BMP_THEME_ICON_DIR, icon_theme, aux0, NULL);

        pixbufs[n] = gdk_pixbuf_new_from_file (aux1, &error);

        if (error != NULL)
          {
          g_message (error->message);
          g_error_free (error);
          error = NULL;
          }

        icon_list = g_list_append (icon_list, pixbufs[n]);
        g_free (aux1);
        g_free (aux0);
        }

      gtk_window_set_icon_list (GTK_WINDOW (window), icon_list);
      g_list_free (icon_list);

      for (n = 0; n < 5; n++)
        g_object_unref (pixbufs[n]);
    }

    void
    window_set_busy (Gtk::Window & window)
    {
        window.get_window()->set_cursor (Gdk::Cursor (Gdk::Display::get_default(), "watch"));
    }

    void
    window_set_idle (Gtk::Window & window)
    {
        window.get_window()->set_cursor ();
    }

    void
    color_to_rgba (Gdk::Color const& color,
                   double &          r,
                   double &          g,
                   double &          b,
                   double &          a)
    {
      r = color.get_red ()   / 65535.0;
      g = color.get_green () / 65535.0;
      b = color.get_blue ()  / 65535.0;
      a = 1.0;
    }

    double
    screen_get_resolution (Glib::RefPtr<Gdk::Screen> const& screen)
    {
      // NOTE: this assumes the width and height resolutions are
      // equal. This is only true for video modes with aspect ratios that
      // match the physical screen area ratio e.g. 1024x768 for 4:3
      // screens - descender

      return std::ceil (screen_get_y_resolution (screen));
    }

    double
    screen_get_x_resolution (Glib::RefPtr<Gdk::Screen> const& screen)
    {
      return static_cast<double> (screen->get_width ()) / mm_to_inch (screen->get_height_mm ());
    }

    double
    screen_get_y_resolution (Glib::RefPtr<Gdk::Screen> const& screen)
    {
      return static_cast<double> (screen->get_height ()) / mm_to_inch (screen->get_height_mm ());
    }

    Cairo::RefPtr<Cairo::Context>
    create_gdk_cairo_context (Glib::RefPtr<Gdk::Drawable> const& drawable)
    {
        return Cairo::RefPtr<Cairo::Context> (new Cairo::Context (::gdk_cairo_create (drawable->gobj ()), true));
    }

    void
    set_cairo_source_pixbuf (Cairo::RefPtr<Cairo::Context> &  cr,
                             Glib::RefPtr<Gdk::Pixbuf> const& pixbuf,
                             double                           x,
                             double                           y)
    {
        ::gdk_cairo_set_source_pixbuf (cr->cobj (), pixbuf->gobj (), 0, 0);
    }

    void
    set_cairo_source_color (Cairo::RefPtr<Cairo::Context> & cr,
                            Gdk::Color const&               color,
                            double                          alpha)
    {
      cr->set_source_rgba (color.get_red_p(), color.get_green_p(), color.get_blue_p(), alpha);
    }

    void 
    style_save (GtkWidget *widget)
    {
      _style = gtk_style_copy (widget->style); 
      GtkStyle *src = widget->style;
      
      g_slist_foreach (_style->icon_factories, (GFunc) g_object_unref, NULL);
      g_slist_free (_style->icon_factories);
      _style->icon_factories = g_slist_copy (src->icon_factories);
      g_slist_foreach (_style->icon_factories, (GFunc) g_object_ref, NULL);

      g_object_ref (_style);
    }

    GtkStyle*
    style_get ()
    {
        return _style;
    }

    // NOTE: gtkmm has no binding for gdk_screen_get_rgba_colormap()
    Glib::RefPtr<Gdk::Colormap>
    get_rgba_colormap ()
    {
        return _colormap;
    }

    void
    check_alpha ()
    {
      _colormap = Glib::RefPtr<Gdk::Colormap>(0);

      if (mcs->key_get<bool>("bmp","force-rgba-disable"))
        {
          _has_alpha = false;
          return;
        }

      GdkColormap *colormap = gdk_screen_get_rgba_colormap (gdk_screen_get_default());
      if (colormap)
        {
          _colormap = Glib::wrap (colormap, true);
          _has_alpha = true;
        }
    }

    bool
    has_alpha ()
    {
      return _has_alpha;
    }

  } // !Util::
} // !Bmp::
