//  BMPx - The Dumb Music Player
//  Copyright (C) 2005-2006 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non GPL-compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif //HAVE_CONFIG_H

#include <revision.h>
#include <build.h>

#include <utility>
#include <iostream>

#include <glibmm.h>
#include <glib/gi18n.h>
#include <gtkmm.h>
#include <libglademm.h>

#include <boost/format.hpp>
#include <boost/algorithm/string.hpp>

#ifdef HAVE_ALSA
#  define ALSA_PCM_NEW_HW_PARAMS_API
#  define ALSA_PCM_NEW_SW_PARAMS_API

#  include <alsa/global.h>
#  include <alsa/asoundlib.h>
#  include <alsa/pcm_plugin.h>
#  include <alsa/control.h>
#endif //HAVE_ALSA

#include <ne_basic.h>
#include <ne_compress.h>
#include <ne_socket.h>
#include <ne_utils.h>
#include <ne_props.h>
#include <ne_session.h>
#include <ne_request.h>
#include <ne_uri.h>

#include <mcs/mcs.h>

#include "main.hh"
#include "paths.hh"
#include "network.hh"

#include "stock.hh"
#include "actions.hh"

#ifdef HAVE_HAL
#  include <libhal.h>
#  include <libhal-storage.h>
#  include "x_hal.hh"
#endif

#include "x_library.hh"
#include "x_play.hh"
#include "x_mcsbind.hh"

#include "ui_toolbox.hh"
#include "preferences-ui.hh"

namespace Bmp
{
  namespace
  {
    static boost::format f_gbyte (_("%.2Lf GiB"));
    static boost::format f_mbyte (_("%.2Lf MiB"));

    const double MBYTE = 1048576;

    std::string
    get_size_string (uint64_t size_)
    {
      double size = size_/MBYTE;
      if (size > 1024) 
        {
          return (f_gbyte % (size / 1024.)).str();
        }
      else
        {
          return (f_mbyte % size).str();
        }
    }

    bool
    test_element (const char *name)
    {
      GstElement *element = 0;
      element = gst_element_factory_make (name, "test0");
      bool exists = GST_IS_ELEMENT (element);
      if (GST_IS_ELEMENT (element))
        {
          gst_object_unref (element);
          element = 0;
        }
      return exists;
    }

    struct Category
    {
        const char *icon_path;
        const char *name;
        int         page;
    };

    Category categories[] =
    {
        {BMP_IMAGE_DIR_PREFS G_DIR_SEPARATOR_S "playback.png",    N_("Audio"),          0},
        {BMP_IMAGE_DIR_PREFS G_DIR_SEPARATOR_S "library.png",     N_("Library"),        1},
        {BMP_IMAGE_DIR_PREFS G_DIR_SEPARATOR_S "misc.png",        N_("Miscellaneous"),  2},
        {BMP_IMAGE_DIR_PREFS G_DIR_SEPARATOR_S "podcasts.png",    N_("Podcasts"),       3},

#ifdef HAVE_OFA
        {BMP_IMAGE_DIR_PREFS G_DIR_SEPARATOR_S "musicbrainz.png", N_("MusicBrainz"),    4},
#endif //HAVE_OFA

    };

    struct AudioSystem
    {
        const char        * description;
        const char        * name;
        gint                tab;
        Bmp::Audio::Sink    sink;
    };

    AudioSystem audio_systems[] =
    {
#ifdef HAVE_ALSA
        {"ALSA (Advanced Linux Sound Architecture)",
         "alsasink",
         1,
         Audio::SINK_ALSA},
#endif
        {"OSS (Open Sound System)",
         "osssink",
         2,
         Audio::SINK_OSS},
#ifdef HAVE_SUN
        {"Sun/Solaris Audio",
         "sunaudiosink",
         3,
         Audio::SINK_SUNAUDIO},
#endif
        {"ESD (Enlightenment Sound Daemon)",
         "esdsink",
         4,
         Audio::SINK_ESD},
#ifdef HAVE_HAL
        {"HAL-Based Device Detection",
         "halaudiosink",
         5,
         Audio::SINK_HAL},
#endif
    };
  } // <anonymous> namespace

  class Preferences::CategoryView
      : public Gtk::TreeView
  {
    public:

        CategoryView (BaseObjectType                        *cobject,
                      const Glib::RefPtr<Gnome::Glade::Xml> &xml);

        int
        get_selected_page ();

    private:

        // Category columns record
        class ColumnRecord
            : public Gtk::TreeModel::ColumnRecord
        {
        public:

            Gtk::TreeModelColumn<Glib::RefPtr<Gdk::Pixbuf> > icon;
            Gtk::TreeModelColumn<Glib::ustring>              name;
            Gtk::TreeModelColumn<int>                        page;

            ColumnRecord ()
            {
                add (icon);
                add (name);
                add (page);
            }
        };

        Glib::RefPtr<Gtk::ListStore> list_store;
        ColumnRecord                 columns;
  };

  //-- Preferences::HALView implementation -------------------------------------

#ifdef HAVE_HAL

  int
  Preferences::HALView::store_sort_func ( Gtk::TreeModel::iterator const& iter_a,
                                          Gtk::TreeModel::iterator const& iter_b)
  {
    Library::HAL::Volume const& vol_a ((*iter_a)[cvolume.volume]);
    Library::HAL::Volume const& vol_b ((*iter_a)[cvolume.volume]);

    return vol_a.volume_udi.compare (vol_b.volume_udi);
  }

  void
  Preferences::HALView::hal_volume_add (Library::HAL::Volume volume)
  {
    MVolumes::iterator v_iter = m_volumes.find ( VolumeKey (volume.volume_udi, volume.device_udi) );

    using namespace Gtk;
    using namespace Glib;

    if (v_iter != m_volumes.end())    
    {
      TreeModel::iterator m_iter = m_list_store->get_iter (v_iter->second.get_path());
      (*m_iter)[cvolume.mounted] = true;
      m_list_store->row_changed (v_iter->second.get_path(), m_iter);
    }
    else
    if (library->volume_exists ( volume.volume_udi,
                                 volume.device_udi ))
    {
      TreeModel::iterator m_iter = m_list_store->append();

      (*m_iter)[cvolume.volume]   = volume; 
      (*m_iter)[cvolume.mounted]  = hal->volume_is_mounted  (volume);
      (*m_iter)[cvolume.devicon]  = hal->get_volume_icon    (volume)->scale_simple (24, 24, Gdk::INTERP_BILINEAR);  
      (*m_iter)[cvolume.voltype]  = hal->get_volume_type    (volume); 
      (*m_iter)[cvolume.volname]  = volume.label;

      m_volumes[VolumeKey (volume.volume_udi, volume.device_udi)] = TreeModel::RowReference
          (m_list_store, m_list_store->get_path (m_iter));
    }

    update_volume_details ();
  }

  void
  Preferences::HALView::hal_volume_del (Library::HAL::Volume volume)
  {
    MVolumes::iterator v_iter = m_volumes.find ( VolumeKey (volume.volume_udi, volume.device_udi) );

    using namespace Gtk;
    using namespace Glib;

    if (v_iter != m_volumes.end())    
    {
      TreeModel::iterator m_iter = m_list_store->get_iter (v_iter->second.get_path());
      (*m_iter)[cvolume.mounted] = false;
      m_list_store->row_changed (v_iter->second.get_path(), m_iter);
    }
    else
    {
      //FIXME: Error...
    }

    update_volume_details ();
  }

  // Model Celldata Functions  

  void
  Preferences::HALView::cell_data_func_mounted
                            (Gtk::CellRenderer *  cell_, Gtk::TreeModel::iterator const& m_iter)
  {
    Gtk::CellRendererPixbuf * cell = dynamic_cast<Gtk::CellRendererPixbuf *>(cell_);
    bool mounted ((*m_iter)[cvolume.mounted]);

    if (mounted)
      {
        cell->property_pixbuf() = render_icon (Gtk::StockID (BMP_STOCK_YES), Gtk::ICON_SIZE_MENU);
      }
    else
      {
        cell->property_pixbuf() = render_icon (Gtk::StockID (BMP_STOCK_NO),  Gtk::ICON_SIZE_MENU);
      }
  }

  void
  Preferences::HALView::cell_data_func_mountpath
                            (Gtk::CellRenderer *  cell_, Gtk::TreeModel::iterator const& m_iter)
  {
    Gtk::CellRendererText * cell = dynamic_cast<Gtk::CellRendererText *>(cell_);
    Library::HAL::Volume const& volume ((*m_iter)[cvolume.volume]);
    bool mounted ((*m_iter)[cvolume.mounted]);

    if (mounted)
      cell->property_text() = volume.mount_point;
    else
      cell->property_text() = ""; 
  }

  void
  Preferences::HALView::cell_data_func_device_udi
                            (Gtk::CellRenderer *  cell_, Gtk::TreeModel::iterator const& m_iter)
  {
    Gtk::CellRendererText * cell = dynamic_cast<Gtk::CellRendererText *>(cell_);
    Library::HAL::Volume const& volume ((*m_iter)[cvolume.volume]);
    cell->property_text() = volume.device_udi;
  }

  void
  Preferences::HALView::cell_data_func_volume_udi
                            (Gtk::CellRenderer *  cell_, Gtk::TreeModel::iterator const& m_iter)
  {
    Gtk::CellRendererText * cell = dynamic_cast<Gtk::CellRendererText *>(cell_);
    Library::HAL::Volume const& volume ((*m_iter)[cvolume.volume]);
    cell->property_text() = volume.volume_udi;
  }

  void
  Preferences::HALView::update_volume_details ()
  {
    using namespace Gtk;
    using namespace Glib;

    bool selected (get_selection()->count_selected_rows ());

    if (!selected)
      {
        for (unsigned int n = 0; n < N_LABELS; ++n)
          m_label[n]->set_text ("");

        m_device_icon->clear (); 
        m_ref_xml->get_widget
          ("preferences-library-vbox-hal-details")->set_sensitive (false);
      }
    else
      {
        TreeModel::iterator m_iter (get_selection()->get_selected());

        Library::HAL::Volume const& volume ((*m_iter)[cvolume.volume]);
        bool mounted ((*m_iter)[cvolume.mounted]);

        static boost::format format_llu ("%llu");

        m_label[L_VOLUME_UDI]->set_text (volume.volume_udi);
        m_label[L_DEVICE_UDI]->set_text (volume.device_udi);

        if (mounted)
          m_label[L_MOUNT_PATH]->set_text (volume.mount_point);
        else
          m_label[L_MOUNT_PATH]->set_text (_("(Not Mounted)"));

        m_label[L_DEVICE_SERIAL]->set_text (volume.drive_serial);
        m_label[L_VOLUME_NAME]->set_text (volume.label);

        if (mounted)
          m_label[L_DEVICE_FILE]->set_text (volume.device_file);
        else
          m_label[L_DEVICE_FILE]->set_text ("");

        m_label[L_DRIVE_BUS]->set_text (hal->get_volume_drive_bus (volume));
        m_label[L_DRIVE_TYPE]->set_text (hal->get_volume_type (volume));

        m_label[L_SIZE]->set_text (get_size_string (volume.size));
        m_label[L_DRIVE_SIZE]->set_text (get_size_string (volume.drive_size));

        m_label[L_MOUNT_TIME]->set_text (Util::get_timestr (volume.mount_time, 0));

        m_device_icon->set (hal->get_volume_icon (volume)); 

        m_ref_xml->get_widget
          ("preferences-library-vbox-hal-details")->set_sensitive (true);
      }
  }

  Preferences::HALView::HALView (BaseObjectType                       * obj,
                                 Glib::RefPtr<Gnome::Glade::Xml> const& xml)
      : Gtk::TreeView (obj),
        m_ref_xml     (xml)
  {
    using namespace Glib;
    using namespace Gtk;

    if (hal->is_initialized())
      {
        get_selection()->set_mode (Gtk::SELECTION_SINGLE);

        const char * label_names[] =
        {
          "hal-volume-udi",
          "hal-device-udi",
          "hal-mount-path",
          "hal-device-serial",
          "hal-volume-name",
          "hal-device-file",
          "hal-drive-bus",
          "hal-drive-type",
          "hal-size",
          "hal-drive-size",
          "hal-mount-time"
        };

        for (unsigned int n = 0; n < N_LABELS; ++n)
        {
          m_ref_xml->get_widget (label_names[n], m_label[n]);
        }

        m_ref_xml->get_widget ("hal-device-icon", m_device_icon);

        set_headers_clickable (false);
        set_headers_visible   (true);

        Gtk::CellRendererText   *cell_text    = 0;
        Gtk::CellRendererPixbuf *cell_pixbuf  = 0; 

        cell_pixbuf = Gtk::manage (new Gtk::CellRendererPixbuf ());
        cell_pixbuf->property_yalign () = 0.5;
        append_column ("", *cell_pixbuf);
        get_column (0)->set_resizable (false);
        get_column (0)->add_attribute (*cell_pixbuf, "pixbuf", 0);

        cell_pixbuf = Gtk::manage (new Gtk::CellRendererPixbuf ());
        cell_pixbuf->property_yalign () = 0.5;
        append_column ("", *cell_pixbuf);
        get_column (1)->set_resizable (false);
        get_column (1)->set_cell_data_func (*cell_pixbuf,
          sigc::mem_fun(this, &Bmp::Preferences::HALView::cell_data_func_mounted));

        cell_text = Gtk::manage (new Gtk::CellRendererText ());
        append_column (_("Type"), *cell_text);
        get_column (2)->set_resizable (false);
        get_column (2)->add_attribute (*cell_text, "text", 1);

        cell_text = Gtk::manage (new Gtk::CellRendererText ());
        append_column (_("Name"), *cell_text);
        get_column (3)->set_resizable (false);
        get_column (3)->add_attribute (*cell_text, "text", 2);

        cell_text = Gtk::manage (new Gtk::CellRendererText ());
        append_column (_("Mount Path"), *cell_text);
        get_column (4)->set_resizable (false);
        get_column (4)->set_cell_data_func (*cell_text,
          sigc::mem_fun(this, &Bmp::Preferences::HALView::cell_data_func_mountpath));

        cell_text = Gtk::manage (new Gtk::CellRendererText ());
        append_column (_("Volume UDI"), *cell_text);
        get_column (5)->set_resizable (false);
        get_column (5)->set_cell_data_func (*cell_text,
          sigc::mem_fun(this, &Bmp::Preferences::HALView::cell_data_func_volume_udi));

        cell_text = Gtk::manage (new Gtk::CellRendererText ());
        append_column (_("Device UDI"), *cell_text);
        get_column (6)->set_resizable (false);
        get_column (6)->set_cell_data_func (*cell_text,
          sigc::mem_fun (this, &Bmp::Preferences::HALView::cell_data_func_device_udi));

        m_list_store = Gtk::ListStore::create (cvolume);

        m_list_store->set_default_sort_func
          (sigc::mem_fun (this, &Bmp::Preferences::HALView::store_sort_func));

        m_list_store->set_sort_column 
          (Gtk::TreeSortable::DEFAULT_SORT_COLUMN_ID, Gtk::SORT_ASCENDING);

        rescan_devices ();

        set_model (m_list_store);

        hal->signal_volume_added().connect
          (sigc::mem_fun (this, &Bmp::Preferences::HALView::hal_volume_add));

        hal->signal_volume_removed().connect
          (sigc::mem_fun (this, &Bmp::Preferences::HALView::hal_volume_del));

        m_device_icon->clear (); 

        m_ref_xml->get_widget
          ("preferences-library-vbox-hal-details")->set_sensitive (false);

        get_selection()->signal_changed().connect
          (sigc::mem_fun (this, &Bmp::Preferences::HALView::update_volume_details));
      }
    else
      {
        set_headers_visible (false);
        set_sensitive (false);
      }
  }
  
  void
  Preferences::HALView::rescan_devices ()
  {
    using namespace Bmp::DB;
    using namespace Gtk;

    set_sensitive (false);
    m_list_store->clear ();

    Library::HAL::VVolumes vvolumes; 
    hal->volumes (vvolumes);

    for (Library::HAL::VVolumes::const_iterator v  = vvolumes.begin ();
                                                v != vvolumes.end (); ++v)
    {
      if (!library->volume_exists ( v->volume_udi,
                                    v->device_udi ))
        continue;

      TreeModel::iterator m_iter = m_list_store->append();

      (*m_iter)[cvolume.volume]   = (*v); 
      (*m_iter)[cvolume.mounted]  = hal->volume_is_mounted (*v);
      (*m_iter)[cvolume.devicon]  = hal->get_volume_icon (*v)->scale_simple (24, 24, Gdk::INTERP_BILINEAR); 
      (*m_iter)[cvolume.voltype]  = hal->get_volume_type (*v); 
      (*m_iter)[cvolume.volname]  = v->label;

      m_volumes[VolumeKey (v->volume_udi, v->device_udi)] =
        TreeModel::RowReference (m_list_store, m_list_store->get_path (m_iter));
    }
    set_sensitive (true);
  }

#endif

  //-- Preferences::CategoryView implementation --------------------------------

  Preferences::CategoryView::CategoryView (BaseObjectType                       * obj,
                                           Glib::RefPtr<Gnome::Glade::Xml> const& xml) : Gtk::TreeView (obj)
  {
      set_headers_clickable (false);
      set_headers_visible (false);

      Gtk::CellRendererPixbuf *cell_pixbuf = Gtk::manage (new Gtk::CellRendererPixbuf ());
      cell_pixbuf->property_ypad () = 4;
      cell_pixbuf->property_xpad () = 2;

      append_column ("Icon", *cell_pixbuf);
      get_column (0)->add_attribute (*cell_pixbuf, "pixbuf", 0);
      get_column (0)->set_resizable (false);

      Gtk::CellRendererText *cell = Gtk::manage (new Gtk::CellRendererText ());

      append_column ("Description", *cell);
      get_column (1)->add_attribute (*cell, "text", 1);
      get_column (1)->set_resizable (false);
      cell->property_xalign () = 0.0;

      list_store = Gtk::ListStore::create (columns);

      for (unsigned int i = 0; i < G_N_ELEMENTS (categories); i++)
        {
          Gtk::TreeModel::iterator iter = list_store->append ();
          (*iter)[columns.icon] = Gdk::Pixbuf::create_from_file (categories[i].icon_path);
          (*iter)[columns.name] = _(categories[i].name);
          (*iter)[columns.page] = categories[i].page;
        }
      set_model (list_store);
  }

  int
  Preferences::CategoryView::get_selected_page ()
  {
      Gtk::TreeModel::iterator iter = get_selection ()->get_selected ();
      return (*iter)[columns.page];
  }

  //-- Preferences implementation ----------------------------------------------
  Preferences *
  Preferences::create ()
  {
      const std::string path = BMP_GLADE_DIR G_DIR_SEPARATOR_S "preferences-ui.glade";
      Glib::RefPtr<Gnome::Glade::Xml> glade_xml = Gnome::Glade::Xml::create (path);

      Preferences *preferences = 0;
      glade_xml->get_widget_derived ("preferences", preferences);
      return preferences;
  }

#ifdef HAVE_HAL
  void
  Preferences::rescan_devices ()
  {
      m_volumes_view->rescan_devices (); 
  }
#endif //HAVE_HAL

  Preferences::Preferences (BaseObjectType                       * obj,
                            Glib::RefPtr<Gnome::Glade::Xml> const& xml)
      : Gtk::Window (obj),
        m_ref_xml   (xml)
  {
      // acquire consistently needed widgets
      m_ref_xml->get_widget ("category_notebook", m_category_notebook);
      m_ref_xml->get_widget_derived ("category_view", m_category_view);

#ifdef HAVE_HAL
      m_ref_xml->get_widget_derived ("prefs-view-volumes", m_volumes_view);
#else
      m_ref_xml->get_widget ("preferences-library-vbox-hal")->hide ();
#endif //HAVE_HAL

      Glib::RefPtr<Gtk::TreeSelection> selection = m_category_view->get_selection ();
      selection->signal_changed ().connect (sigc::mem_fun (this, &Preferences::on_category_changed));

      dynamic_cast<Gtk::Button *>(m_ref_xml->get_widget ("close"))->
          signal_clicked().connect (sigc::mem_fun (this, &Preferences::hide));

      // Version String 
      boost::format version_format ("<span size='small' color='#606060'>BMPx %s:%s-R%s (%s / %s)</span>");
      std::string version = (version_format % PACKAGE_VERSION % RV_LAST_CHANGED_DATE % RV_REVISION % BUILD_DATE % BUILD_BUILDUSER).str ();
      dynamic_cast<Gtk::Label *>(m_ref_xml->get_widget ("l_version"))->set_markup (version);

      // Misc Images 

      dynamic_cast<Gtk::Image *>(m_ref_xml->get_widget ("i-audio-caps"))->
          set (Gtk::StockID (BMP_STOCK_AUDIO), Gtk::ICON_SIZE_SMALL_TOOLBAR);
      dynamic_cast<Gtk::Image *>(m_ref_xml->get_widget ("i-audio-settings"))->
          set (Gtk::StockID (BMP_STOCK_AUDIO), Gtk::ICON_SIZE_SMALL_TOOLBAR);

      // Other Misc Widgets 
      m_ref_xml->get_widget ("cbox_audio_system", cbox_audio_system);

#ifdef HAVE_ALSA
      m_ref_xml->get_widget ("cbox_alsa_card", cbox_alsa_card);
      m_ref_xml->get_widget ("cbox_alsa_device", cbox_alsa_device);
      m_ref_xml->get_widget ("alsa_buffer_time", alsa_buffer_time);
#endif

      m_ref_xml->get_widget ("cbe_oss_device", cbe_oss_device);
      m_ref_xml->get_widget ("oss_buffer_time", oss_buffer_time);

      m_ref_xml->get_widget ("esd_host", esd_host);
      m_ref_xml->get_widget ("esd_buffer_time", esd_buffer_time);

#ifdef HAVE_SUNDAUDIO
      m_ref_xml->get_widget ("cbe_sun_device", cbe_sun_device);
      m_ref_xml->get_widget ("sun_buffer_time", sun_buffer_time);
#endif

      m_ref_xml->get_widget ("notebook_audio_system", notebook_audio_system);

      m_ref_xml->get_widget ("audio-system-apply-change", audio_system_apply);
      m_ref_xml->get_widget ("audio-system-changed-warning", audio_system_changed_warning);

#ifndef HAVE_HAL
      Gtk::ComboBoxEntry *cbe_cdrom;
      m_ref_xml->get_widget ("cbe_cdrom", cbe_cdrom);
      cbe_cdrom->show_all ();
      mcs_bind->bind_cbox_entry (cbe_cdrom, "audio", "cdrom-device");
      m_ref_xml->get_widget ("alignment189")->show_all ();
      m_ref_xml->get_widget ("alignment202")->show_all ();
#endif

      // Setup Audio 
      setup_audio ();

      // Setup Podcast Stuff
      if (!Glib::file_test (mcs->key_get <std::string> ("podcasts", "download-dir"), Glib::FILE_TEST_EXISTS))
        mcs->key_set  <std::string> 
                      ("podcasts","download-dir",
                      std::string(g_get_home_dir()));

      mcs_bind->bind_filechooser    ( dynamic_cast  
                                      <Gtk::FileChooser *>
                                      (m_ref_xml->get_widget ("podcast-download-filechooser")),
                                                              "podcasts", "download-dir");

      mcs_bind->bind_toggle_button  ( dynamic_cast
                                      <Gtk::ToggleButton *>
                                      (m_ref_xml->get_widget ("podcasts-update-on-startup")),
                                                              "podcasts", "update-on-startup");

#ifdef HAVE_OFA
      // MusicBrainz 
      mcs_bind->bind_entry (dynamic_cast<Gtk::Entry *>(m_ref_xml->get_widget ("musicbrainz-username")) , "musicbrainz", "username");
      mcs_bind->bind_entry (dynamic_cast<Gtk::Entry *>(m_ref_xml->get_widget ("musicbrainz-password")) , "musicbrainz", "password");
#endif //HAVE_OFA

      // Togglebuttons
      const char* toggle_buttons[] =
      {
          "ui-esc-trayconify",
          "display-notifications",
          "send-statistics",
          "physically-delete-files",
          "force-rgba-disable",
          "strip-mpeg-files"
      };

      for (unsigned int n = 0; n < G_N_ELEMENTS (toggle_buttons); n++)
      {
          Gtk::ToggleButton *button = 0;
          m_ref_xml->get_widget (toggle_buttons[n], button);
          if (button) 
            mcs_bind->bind_toggle_button (button, "bmp", toggle_buttons[n]);
          else
            g_log (G_LOG_DOMAIN, G_LOG_LEVEL_WARNING, "Widget '%s' not found in preferences glade xml UI", toggle_buttons[n]);
      }
  }

  void
  Preferences::on_category_changed ()
  {
      if (m_category_view->get_selection ()->count_selected_rows () == 0)
          return;

      m_category_notebook->set_current_page (m_category_view->get_selected_page ());
  }

  // Audio preferences
  void
  Preferences::on_cbox_audio_system_changed     ()
  {
      Gtk::TreeModel::iterator iter = cbox_audio_system->get_active ();
      notebook_audio_system->set_current_page ((*iter)[audio_system_columns.tab]);
  }

  void
  Preferences::audio_system_apply_set_sensitive ()
  {
      audio_system_apply->set_sensitive (true);
      audio_system_changed_warning->set_sensitive (true);
  }

#ifdef HAVE_ALSA

  void
  Preferences::on_cbox_alsa_card_changed ()
  {
      using namespace Gtk;

      cbox_alsa_device->clear();
      TreeModel::iterator const& c_iter (cbox_alsa_card->get_active ());
      AlsaCard const& card ((*c_iter)[alsa_card_columns.card]);

      int row = cbox_alsa_card->get_active_row_number ();

      if((row == 0) || card.devices.empty())
        {
          cbox_alsa_device->set_sensitive (false);
          return;
        }

      TreeModel::iterator m_iter;
      for (AlsaDevices::const_iterator d = card.devices.begin () ; d != card.devices.end() ; ++d)
        {
          m_iter= list_store_alsa_device->append ();
          (*m_iter)[alsa_device_columns.name]   = d->description;
          (*m_iter)[alsa_device_columns.device] = *d;
        }
      cbox_alsa_device->set_sensitive (true);
      cbox_alsa_device->set_active (0);
  }

#endif //HAVE_ALSA

  void
  Preferences::on_audio_system_apply ()
  {
    using namespace Gtk;

    audio_system_apply->set_sensitive (false);
    audio_system_changed_warning->set_sensitive (false);

    TreeModel::iterator iter = cbox_audio_system->get_active ();

    Audio::Sink sink    = (*iter)[audio_system_columns.sink];
    std::string name    = (*iter)[audio_system_columns.name];

    std::string device;

    play->request_status (PLAYSTATUS_STOPPED);
    mcs->key_set <std::string> ("audio", "sink", name);

    switch (sink)
    {
#ifdef HAVE_ALSA
      case Audio::SINK_ALSA:
      {
        AlsaDevice alsa_device;
        int active = cbox_alsa_card->get_active_row_number ();

        if (active > 0)
          {
            TreeModel::iterator iter = cbox_alsa_device->get_active ();
            alsa_device = (*iter)[alsa_device_columns.device];
            device = alsa_device.dev;
            mcs->key_set <std::string> ("audio", "device-alsa", device);
          }
        else
          {
            mcs->key_set <std::string> ("audio", "device-alsa", "default");
          }
        break;
      }
#endif //HAVE_ALSA

#ifdef HAVE_SUN
      case Audio::SINK_SUNAUDIO:
      {
        device = cbe_sun_device->get_entry()->get_text();
        mcs->key_set <std::string> ("audio", "device-sun", device);
        break;
      }
#endif //HAVE_SUN

      case Audio::SINK_OSS:
      {
        device = cbe_oss_device->get_entry()->get_text();
        mcs->key_set <std::string> ("audio", "device-oss", device);
        break;
      }

      case Audio::SINK_ESD:
      {
        device = esd_host->get_text();
        mcs->key_set <std::string> ("audio", "device-esd", device);
        break;
      }

      default:
        return;
    }
    play->reset();
  }

#ifdef HAVE_ALSA

  Bmp::Preferences::AlsaCards
  Preferences::get_alsa_cards ()
  {
      Bmp::Preferences::AlsaCards cards;
      int card = -1;
      int snd_err = snd_card_next (&card);

      if (snd_err)
        {
          g_critical ("%s: %s", G_STRLOC, snd_strerror (snd_err));
          return cards;
        }

      while (card > -1)
        {
          snd_pcm_info_t * pcm_info;
          snd_ctl_t      * ctl;
          int              pcm_device;

          Bmp::Preferences::AlsaCard alsa_card;

          char * name;
          snd_card_get_name (card, &name);

          std::string dev ((boost::format ("hw:%d") % card).str());

          snd_err = snd_ctl_open (&ctl, dev.c_str(), 0);
          if (snd_err)
            break;

          alsa_card.dev = dev;
          alsa_card.description = name;
          alsa_card.card_id = card;

          pcm_device = -1;

          while (42)
            {
              Bmp::Preferences::AlsaDevice alsa_device;

              if (!ctl)
                break;

              snd_err = snd_ctl_pcm_next_device (ctl, &pcm_device);
              if (snd_err)
                {
                  g_warning ("%s: %s", G_STRLOC, snd_strerror (snd_err));
                  break;
                }

              // FIXME: Do we really need this here still?
              if (pcm_device < 0)
                break;

              snd_pcm_info_malloc (&pcm_info);
              snd_pcm_info_set_device (pcm_info, pcm_device);
              snd_pcm_info_set_subdevice (pcm_info, 0);
              snd_pcm_info_set_stream (pcm_info, SND_PCM_STREAM_PLAYBACK);

              snd_err = snd_ctl_pcm_info (ctl, pcm_info);

              if (snd_err)
                break;

              alsa_device.dev         = (boost::format ("hw:%d,%d") % card % pcm_device).str ();
              alsa_device.description = (boost::format ("%d: %s") % pcm_device % snd_pcm_info_get_name (pcm_info)).str ();
              alsa_device.card_id     = card;
              alsa_device.device_id   = pcm_device;

              alsa_card.devices.push_back (alsa_device);

              snd_pcm_info_free (pcm_info);
            }

          cards.push_back (alsa_card);

          if (ctl)
            snd_ctl_close (ctl);

          snd_card_next (&card);

          if (card < 0 )
            break;
      }
      return cards;
  }

#endif

  void
  Preferences::setup_audio ()
  {
      using namespace Gtk;

      CellRendererText * cell;
      std::string        sink;
      int                current = -1;

      list_store_audio_systems = ListStore::create (audio_system_columns);

      sink = mcs->key_get<std::string>("audio", "sink");

      for (unsigned int n = 0; n < G_N_ELEMENTS (audio_systems); n++)
        {
          TreeModel::iterator m_iter;

          if (!test_element (audio_systems[n].name))
            continue;
  
          m_sinks.insert (audio_systems[n].name);

          m_iter = list_store_audio_systems->append ();

          (*m_iter)[audio_system_columns.description]   = audio_systems[n].description;
          (*m_iter)[audio_system_columns.name]          = audio_systems[n].name;
          (*m_iter)[audio_system_columns.tab]           = audio_systems[n].tab;
          (*m_iter)[audio_system_columns.sink]          = audio_systems[n].sink;

          if (sink == audio_systems[n].name) current = n;
        }

      cell = manage (new CellRendererText() );
      cbox_audio_system->clear ();
      cbox_audio_system->pack_start (*cell);
      cbox_audio_system->add_attribute (*cell, "text", 0);
      cbox_audio_system->set_model (list_store_audio_systems);
      cbox_audio_system->set_active (current);

      notebook_audio_system->set_current_page ((current == -1) ? 0 : audio_systems[current].tab);

#ifdef HAVE_ALSA

      if (m_sinks.find ("alsasink") != m_sinks.end ())
        {
          list_store_alsa_cards = ListStore::create (alsa_card_columns);

          AlsaCards cards = Bmp::Preferences::get_alsa_cards ();
          TreeModel::iterator m_iter;

          m_iter = list_store_alsa_cards->append ();
          (*m_iter)[alsa_card_columns.name] = "System Default";

          for (AlsaCards::iterator i = cards.begin (); i != cards.end (); ++i)
            {
              m_iter = list_store_alsa_cards->append ();

              std::cerr << " Adding ALSA Card '" << i->description << "'" << std::endl;

              (*m_iter)[alsa_card_columns.name] = i->description;
              (*m_iter)[alsa_card_columns.card] = *i;
            }

          cell = manage (new CellRendererText () );
          cbox_alsa_card->clear ();
          cbox_alsa_card->pack_start (*cell);
          cbox_alsa_card->add_attribute (*cell, "text", 0);
          cbox_alsa_card->set_model (list_store_alsa_cards);

          cell = manage (new CellRendererText () );
          cbox_alsa_device->clear ();
          cbox_alsa_device->pack_start (*cell);
          cbox_alsa_device->add_attribute (*cell, "text", 0);
          list_store_alsa_device = ListStore::create (alsa_device_columns);
          cbox_alsa_device->set_model (list_store_alsa_device);

          cbox_alsa_card->signal_changed ().connect
            (sigc::mem_fun (this, &Preferences::on_cbox_alsa_card_changed));

          if (sink == "alsasink")
            {
              std::string device;
              device = mcs->key_get<std::string>("audio", "device-alsa");

              if (device == "default")
                {
                  cbox_alsa_card->set_active (0);
                }
              else
                {
                  using namespace boost::algorithm;
                  std::vector<std::string> subs;
                  split (subs, device, is_any_of (":,"));

                  if (subs.size() && subs[0] != "hw")
                    {
                      g_message ("Invalid identifier '%s", subs[0].c_str ());
                    }
                  else
                  if (subs.size() && subs[0] == "hw")
                    {
                      int card   = atoi (subs[1].c_str ());
                      int device = atoi (subs[2].c_str ());

                      cbox_alsa_card->set_active (card);
                      cbox_alsa_device->set_active (device);
                    }
                }
            }

          mcs_bind->bind_spin_button (alsa_buffer_time, "audio", "alsa-buffer-time");

          alsa_buffer_time->signal_value_changed().connect
            (sigc::mem_fun (this, &Preferences::audio_system_apply_set_sensitive));

          cbox_alsa_device->signal_changed().connect
            (sigc::mem_fun (this, &Preferences::audio_system_apply_set_sensitive));

          cbox_alsa_card->signal_changed().connect
            (sigc::mem_fun (this, &Preferences::audio_system_apply_set_sensitive));
      }

#endif //HAVE_ALSA

      if (m_sinks.find ("osssink") != m_sinks.end ())
        {
          mcs_bind->bind_cbox_entry (cbe_oss_device, "audio", "device-oss");
          mcs_bind->bind_spin_button (oss_buffer_time, "audio", "oss-buffer-time");

          if (sink == "osssink")
            {
              cbe_oss_device->get_entry()->set_text (mcs->key_get<std::string>("audio", "device-oss"));
            }

          cbe_oss_device->signal_changed().connect
            (sigc::mem_fun (this, &Preferences::audio_system_apply_set_sensitive));

          oss_buffer_time->signal_value_changed().connect
            (sigc::mem_fun (this, &Preferences::audio_system_apply_set_sensitive));
        }

#ifdef HAVE_SUN

      if (m_sinks.find ("sunaudiosink") != m_sinks.end ())
        {
          mcs_bind->bind_cbox_entry (cbe_sun_device, "audio", "device-sun");
          mcs_bind->bind_spin_button (sun_buffer_time, "audio", "sun-buffer-time");

          sun_buffer_time->signal_value_changed ().connect
            (sigc::mem_fun (this, &Preferences::audio_system_apply_set_sensitive));

          cbe_sun_device->signal_changed ().connect
            (sigc::mem_fun (this, &Preferences::audio_system_apply_set_sensitive));
        }

#endif //HAVE_SUN

      if (m_sinks.find ("esdsink") != m_sinks.end ())
        {
          mcs_bind->bind_entry (esd_host, "audio", "device-esd");
          mcs_bind->bind_spin_button (esd_buffer_time, "audio", "esd-buffer-time");

          esd_buffer_time->signal_value_changed ().connect
            (sigc::mem_fun (this, &Preferences::audio_system_apply_set_sensitive));

          esd_host->signal_changed ().connect
            (sigc::mem_fun(this, &Preferences::audio_system_apply_set_sensitive));
        }

      audio_system_apply->signal_clicked().connect
        (sigc::mem_fun (this, &Preferences::on_audio_system_apply));

      cbox_audio_system->signal_changed().connect
        (sigc::mem_fun(this, &Preferences::on_cbox_audio_system_changed));

      cbox_audio_system->signal_changed().connect
        (sigc::mem_fun (this, &Preferences::audio_system_apply_set_sensitive));

      // Setup Support Status Indicators

      bool has_cdda = test_element ("cdparanoiasrc"); 
      bool has_mmsx = test_element ("mmssrc");
      bool has_http = true; // built-in bmpx-neonhttpsrc

      if (has_cdda)
        {
          m_ref_xml->get_widget ("expander-cdda")->show_all ();
        }

      dynamic_cast<Gtk::Image *>(m_ref_xml->get_widget ("img_status_cdda"))->set
         (has_cdda                       ?
          StockID(BMP_STOCK_PLUGIN) :
          StockID(BMP_STOCK_PLUGIN_DISABLED),
          ICON_SIZE_SMALL_TOOLBAR);

      dynamic_cast<Gtk::Image *>(m_ref_xml->get_widget ("img_status_http"))->set
         (has_http                       ?
          StockID(BMP_STOCK_PLUGIN) :
          StockID(BMP_STOCK_PLUGIN_DISABLED),
          ICON_SIZE_SMALL_TOOLBAR);

      dynamic_cast<Gtk::Image *>(m_ref_xml->get_widget ("img_status_mms"))->set
         (has_mmsx                       ?
          StockID(BMP_STOCK_PLUGIN) :
          StockID(BMP_STOCK_PLUGIN_DISABLED),
          ICON_SIZE_SMALL_TOOLBAR);

      struct SupportCheckTuple
      {
        char *widget;
        char *element;
      };

      const SupportCheckTuple support_check[] = 
      {
        {"img_status_mpc",    "musepackdec"},
        {"img_status_flac",   "flacdec"},
        {"img_status_m4a",    "faad"},
        {"img_status_wma",    "ffdec_wmav2"},
        {"img_status_sid",    "siddec"},
        {"img_status_wav",    "wavparse"},
        {"img_status_mod",    "modplug"},
        {"img_status_spc",    "spcdec"},
      };

      // Check for either mad or flump3dec 
      dynamic_cast<Gtk::Image *>(m_ref_xml->get_widget ("img_status_mp3"))->set
          (((test_element ("mad") || test_element ("flump3dec")) && test_element("id3demux") && test_element ("apedemux"))
            ? StockID(BMP_STOCK_PLUGIN) : StockID(BMP_STOCK_PLUGIN_DISABLED), ICON_SIZE_SMALL_TOOLBAR);
           
      dynamic_cast<Gtk::Image *>(m_ref_xml->get_widget ("img_status_ogg"))->set
        ((test_element("oggdemux") && test_element("vorbisdec")) ? 
          StockID(BMP_STOCK_PLUGIN) : 
          StockID(BMP_STOCK_PLUGIN_DISABLED), ICON_SIZE_SMALL_TOOLBAR);

      dynamic_cast<Gtk::Image *>(m_ref_xml->get_widget ("img_status_vorbisenc"))->set
        ((test_element("oggmux") && test_element("vorbisenc") && has_cdda) ? 
          StockID(BMP_STOCK_PLUGIN) : 
          StockID(BMP_STOCK_PLUGIN_DISABLED), ICON_SIZE_SMALL_TOOLBAR);

      dynamic_cast<Gtk::Image *>(m_ref_xml->get_widget ("img_status_mp3enc"))->set
        ((test_element("lame") && has_cdda) ? 
          StockID(BMP_STOCK_PLUGIN) : 
          StockID(BMP_STOCK_PLUGIN_DISABLED), ICON_SIZE_SMALL_TOOLBAR);

      dynamic_cast<Gtk::Image *>(m_ref_xml->get_widget ("img_status_flacenc"))->set
        ((test_element("flacenc") && has_cdda) ? 
          StockID(BMP_STOCK_PLUGIN) : 
          StockID(BMP_STOCK_PLUGIN_DISABLED), ICON_SIZE_SMALL_TOOLBAR);

      for (unsigned int n = 0; n < G_N_ELEMENTS (support_check); ++n)
      {
          dynamic_cast<Gtk::Image *>(m_ref_xml->get_widget (support_check[n].widget))->set
                (test_element (support_check[n].element) ?
                 StockID(BMP_STOCK_PLUGIN) : 
                 StockID(BMP_STOCK_PLUGIN_DISABLED), ICON_SIZE_SMALL_TOOLBAR);
      }

      audio_system_apply->set_sensitive (false);
      audio_system_changed_warning->set_sensitive (false);
  }
} // namespace Bmp

