//  BMPx - The Dumb Music Player
//  Copyright (C) 2005 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non-GPL compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifndef BMP_PODCAST_TOOLS_HH 
#define BMP_PODCAST_TOOLS_HH

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif //HAVE_CONFIG_H

#include <vector>

#include <glibmm.h>
#include <glibmm/markup.h>

#include <gtkmm.h>

#include <boost/format.hpp>
#include <boost/optional.hpp>

#include "util.hh"
#include "uuid.h"

namespace Bmp
{
  namespace PodCastTools
  {
    struct PodCastItem
    {
      // Item General Attributes
      oustring      title;
      oustring      description;
      oustring      link;
      oustring      copyright;
      oustring      pub_date;

      // Enclosure (Stream)
      oustring      enclosure_url;
      oustring      enclosure_type;
      unsigned int  enclosure_length;

      // GUID
      bool          guid_permalink;
      oustring      guid_value;

      // Our own data
      bool          listened_to;
      bool          downloaded;
      std::string   localfilename;
    };
    typedef std::vector<PodCastItem> VPodCastItems;

    struct PodCastOverlayItem
    {
      bool listened_to;
      bool downloaded;
      std::string localfilename;

      PodCastOverlayItem ()
      {
        listened_to = false;
        downloaded  = false;
        localfilename = std::string ();
      }
    };
    typedef std::map<Glib::ustring, PodCastOverlayItem> MOverlayItems;

    struct PodCast 
    {
      // PodCast
      oustring  title;
      oustring  description;
      oustring  link;
      oustring  last_build_date;
      oustring  pub_date;
      oustring  generator;

      // Image (Optional)
      oustring      image_url;
      oustring      image_title;
      oustring      image_link;
      oustring      image_description;

      VPodCastItems items;

      // Our own data
      std::string   cast_uuid;
      Glib::ustring uri;
      time_t        last_poll_time;
      MOverlayItems overlay_items;
    };

    typedef std::vector<Glib::ustring> PodCastList;
    typedef std::map<Glib::ustring, PodCast> MPodCasts;

    void overlay_from_item (PodCastItem const& item, PodCastOverlayItem & overlay_item);

#include "exception.hh"

    EXCEPTION(NetworkError)
    EXCEPTION(ParsingError)
    EXCEPTION(InvalidUriError)
    EXCEPTION(PodCastExistsError)
    EXCEPTION(InvalidPodcastListError)
    EXCEPTION(PodCastNotLoadedError)
    EXCEPTION(PodCastInvalidError)

    //////////////////////////////////////////////////////////////////////////////////////////////////////////

    class PodCastOverlayParser : public Glib::Markup::Parser
    {
      public:

        PodCastOverlayParser (MOverlayItems & overlays); 
        virtual ~PodCastOverlayParser ();

        bool check_sanity ();

      protected:

        virtual void
        on_start_element  (Glib::Markup::ParseContext& context,
                           Glib::ustring const& elementname,
                           const AttributeMap& attributes);
        virtual void
        on_end_element    (Glib::Markup::ParseContext& context,
                           Glib::ustring const& elementname);
        virtual void
        on_text	          (Glib::Markup::ParseContext& context,
                           Glib::ustring const& text);
        virtual void
        on_passtrough     (Glib::Markup::ParseContext& context,
                           Glib::ustring const& text);
        virtual void
        on_error	        (Glib::Markup::ParseContext& context,
                           const Glib::MarkupError& error);
      private:

        MOverlayItems & m_overlays;

        enum Element
        {
          E_NONE	              = 0, 
          E_BMP_CAST_OVERLAY    = 1 << 0,
          E_CAST_ITEM           = 1 << 1,
        };

        int state;
    };

    //////////////////////////////////////////////////////////////////////////////////////////////////////////

    class PodCastParser : public Glib::Markup::Parser
    {
      public:

        PodCastParser (PodCast & cast);
        virtual ~PodCastParser ();

        bool check_sanity ();

      protected:

        virtual void
        on_start_element  (Glib::Markup::ParseContext& context,
                           Glib::ustring const& elementname,
                           const AttributeMap& attributes);
        virtual void
        on_end_element    (Glib::Markup::ParseContext& context,
                           Glib::ustring const& elementname);
        virtual void
        on_text	          (Glib::Markup::ParseContext& context,
                           Glib::ustring const& text);
        virtual void
        on_passtrough     (Glib::Markup::ParseContext& context,
                           Glib::ustring const& text);
        virtual void
        on_error	        (Glib::Markup::ParseContext& context,
                           const Glib::MarkupError& error);
      private:

        PodCast & m_podcast;

        enum Element
        {
          E_NONE	              = 0, 
          E_CHANNEL             = 1 << 0,
          E_TITLE               = 1 << 1,
          E_DESCRIPTION         = 1 << 2,
          E_LINK                = 1 << 3,
          E_LAST_BUILD_DATE     = 1 << 4,
          E_PUB_DATE            = 1 << 5,
          E_GENERATOR           = 1 << 6,
          E_ITEM                = 1 << 7,
          E_ENCLOSURE           = 1 << 8,
          E_GUID                = 1 << 10,
          E_IMAGE               = 1 << 11,
          E_URL                 = 1 << 12,
          E_COPYRIGHT           = 1 << 13,
        };

        PodCastItem item; 

        int state;
    };

    //////////////////////////////////////////////////////////////////////////////////////////////////////////

    class PodCastManager;
    class OPMLParser : public Glib::Markup::Parser
    {
      public:

        OPMLParser (MPodCasts & casts, const PodCastManager * manager); 
        virtual ~OPMLParser ();

        bool check_sanity ();

      protected:

        virtual void
        on_start_element  (Glib::Markup::ParseContext & context,
                           Glib::ustring const& elementname,
                           AttributeMap const& attributes);
        virtual void
        on_end_element    (Glib::Markup::ParseContext & context,
                           Glib::ustring const& elementname);
        virtual void
        on_text	          (Glib::Markup::ParseContext & context,
                           Glib::ustring const& text);
        virtual void
        on_passtrough     (Glib::Markup::ParseContext & context,
                           Glib::ustring const& text);
        virtual void
        on_error	        (Glib::Markup::ParseContext & context,
                           Glib::MarkupError const& error);
      private:

        MPodCasts            & m_casts;
        const PodCastManager * m_manager;

        enum Element
        {
          E_NONE	              = 0, 
          E_OPML                = 1 << 0,
          E_HEAD                = 1 << 1,
          E_BODY                = 1 << 2,
          E_OUTLINE             = 1 << 3
        };

        int state;
    };

    //////////////////////////////////////////////////////////////////////////////////////////////////////////

    class PodCastManager
    {
      public:

        PodCastManager ();
        ~PodCastManager ();

        void pod_cast_get_list  (PodCastList & list);

        PodCast const& pod_cast_fetch  (Glib::ustring const&        uri,
                                        Glib::RefPtr<Gdk::Pixbuf> & pixbuf,
                                        bool                        load_cover = true) const;

        PodCast const& pod_cast_fetch  (Glib::ustring const& uri) const;

        void pod_cast_cache  (Glib::ustring const&  uri,
                              bool                  update = false,
                              std::string const&    uuid_ = std::string());

        void pod_cast_delete (Glib::ustring const& uri);

        void pod_cast_update (Glib::ustring const& uri);

        void pod_cast_item_change (Glib::ustring const& uri,
                                   Glib::ustring const& guid_value,
                                   PodCastOverlayItem const& overlay_item);

      private:

        friend class OPMLParser;

        void pod_cast_load (std::string const& uuid, PodCast & cast) const;

        void pod_cast_overlay_save (PodCast const& cast);

        void save_opml ();
        void save_overlays ();

        void save_state ();

        MPodCasts m_casts;
    };
  }
}
#endif
