//  BMPx - The Dumb Music Player
//  Copyright (C) 2005 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License Version 2 
//  as published by the Free Software Foundation.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non-GPL compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif //HAVE_CONFIG_H

#include <glibmm.h>

#include <src/util_string.hh>

// Musicbrainz
#include <musicbrainz/musicbrainz.h>

#include <glib/gi18n.h>
#include <glib/gstdio.h>
#include <gtkmm.h>

#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>

#include <iostream>
#include <sstream>

#include <boost/shared_ptr.hpp>
#include <boost/format.hpp>
#include <boost/optional.hpp>
#include <boost/algorithm/string.hpp>

#include <sqlite3_bmp.h>

#include "util_file.hh"
#include "uri++.hh"

#include "database.hh"
#include "library.hh"

// TagLib
#include <taglib/taglib.h>
#include <taglib/fileref.h>
#include <taglib/audioproperties.h>
#include <taglib/tfile.h>
#include <taglib/tag.h>
#include <taglib/id3v2tag.h>
#include <taglib/id3v2framefactory.h>
#include <taglib/textidentificationframe.h>
#include <taglib/uniquefileidentifierframe.h>

// GStreamer
#include <gst/gst.h>
#include <gst/gstelement.h>
#include "audio_typefind.hh"

#include "main.hh"
#include "paths.hh"
#include "debug.hh"

#include "x_library.hh"

#ifdef HAVE_HAL
#  include "x_hal.hh"
#endif //HAVE_HAL

#define TAG_READ_TUPLE(_data) ((TagReadTuple*)(_data))

#define MLIB_VERSION "100"

namespace
{
  // Attributes for the Albums meta table 
  ::Bmp::DB::DatumDefine album_attributes[] = 
  {
      {
          N_("Tracks"),
          N_("Tracks"),
          N_("Track Count"),
          "tracks",
          ::Bmp::DB::VALUE_TYPE_INT,
      },
  };

  // Attributes for a Track row

#ifdef HAVE_HAL
  ::Bmp::DB::DatumDefine track_attributes[31] =
#else
  ::Bmp::DB::DatumDefine track_attributes[27] =
#endif
  {
      {
          N_("Location"),
          N_("Locations"),
          N_("The location of the track (file, internet stream, etc.)"),
          "location",
          ::Bmp::DB::VALUE_TYPE_STRING,
      },

      {
          N_("Artist"),
          N_("Artists"),
          N_("The artist/peformer of the track"),
          "artist",
          ::Bmp::DB::VALUE_TYPE_STRING,
      },

      {
          N_("Album"),
          N_("Albums"),
          N_("The album of the track"),
          "album",
          ::Bmp::DB::VALUE_TYPE_STRING,
      },

      {
          N_("Title"),
          N_("Titles"),
          N_("The title of the current track"),
          "title",
          ::Bmp::DB::VALUE_TYPE_STRING,
      },

      {
          N_("Track"),
          N_("Tracks"),
          N_("The track number of the track"),
          "tracknumber",
          ::Bmp::DB::VALUE_TYPE_INT,
      },

      {
          N_("Time"),
          N_("Times"),
          N_("The length of the track"),
          "time",
          ::Bmp::DB::VALUE_TYPE_INT,
      },

      {
          N_("Genre"),
          N_("Genres"),
          N_("The genre (kind of music) of the track"),
          "genre",
          ::Bmp::DB::VALUE_TYPE_STRING,
      },

      {
          N_("Comment"),
          N_("Comments"),
          N_("Comments added to the track"),
          "comment",
          ::Bmp::DB::VALUE_TYPE_STRING,
      },

      {
          N_("Rating"),
          N_("Ratings"),
          N_("The rating of the track"),
          "rating",
          ::Bmp::DB::VALUE_TYPE_INT,
      },

      {
          N_("Date"),
          N_("Dates"),
          N_("The date (year) of the track"),
          "date",
          ::Bmp::DB::VALUE_TYPE_INT,
      },

      {
          N_("MTIME"),
          N_("MTIMEs"),
          N_("The modification time of the track"),
          "mtime",
          ::Bmp::DB::VALUE_TYPE_INT,
      },

      {
          N_("Bitrate"),
          N_("Bitrates"),
          N_("The bitrate of the track"),
          "bitrate",
          ::Bmp::DB::VALUE_TYPE_INT,
      },

      {
          N_("Samplerate"),
          N_("Samplerates"),
          N_("The samplerate of the track"),
          "samplerate",
          ::Bmp::DB::VALUE_TYPE_INT,
      },

      {
          N_("Play count"),
          N_("Play counts"),
          N_("The number of times this track was played"),
          "count",
          ::Bmp::DB::VALUE_TYPE_INT,
      },

      {
          N_("Last played date"),
          N_("Last played dates"),
          N_("The last date of when this track was played"),
          "playdate",
          ::Bmp::DB::VALUE_TYPE_INT,
      },

      {
          ("AMAZON_ASIN"),
          ("ASINs"),
          N_("Amazon Standard Index Number"),
          "asin",
          ::Bmp::DB::VALUE_TYPE_STRING,
      },

#ifdef HAVE_HAL
      {
          ("Volume UDI"),
          ("Volume UDIs"),
          N_("HAL Volume UDI this track resides on"),
          "volume_udi",
          ::Bmp::DB::VALUE_TYPE_STRING,
      },

      {
          ("Device UDI"),
          ("Device UDIs"),
          N_("HAL Device UDI this track resides on"),
          "device_udi",
          ::Bmp::DB::VALUE_TYPE_STRING,
      },

      {
          N_("Volume Relative Path"),
          N_("Volume Relative Paths"),
          N_("Path of the track relative to it's volume's root"),
          "volume_relative_path",
          ::Bmp::DB::VALUE_TYPE_STRING,
      },

      {
          N_("Volume Relative Base Dir"),
          N_("Volume Relative Base Dir"),
          N_("The base dir of the file relative to the volume's root"),
          "volume_relative_base_path",
          ::Bmp::DB::VALUE_TYPE_STRING,
      },

#endif //HAVE_HAL

      {
          ("MusicBrainz Album Artist"),
          ("MusicBrainz Album Artists"),
          N_("The MusicBrainz Release Artist"),
          "mb_albumartist",
          ::Bmp::DB::VALUE_TYPE_STRING,
      },

      {
          ("MusicBrainz Album Artist Id"),
          ("MusicBrainz Album Artist Ids"),
          N_("The MusicBrainz Release Artist ID (The ID of the Artist this release is accredited to)"),
          "mb_albumartist_id",
          ::Bmp::DB::VALUE_TYPE_STRING,
      },

      {
          ("MusicBrainz Album Artist Sortname"),
          ("MusicBrainz Album Artist Sortnames"),
          N_("The MusicBrainz Release Sort Name"),
          "mb_albumartistsortname",
          ::Bmp::DB::VALUE_TYPE_STRING,
      },

      {
          ("MusicBrainz Track ID"),
          ("MusicBrainz Track IDs"),
          N_("The MusicBrainz Track ID"),
          "mb_track_id",
          ::Bmp::DB::VALUE_TYPE_STRING,
      },

      {
          ("MusicBrainz Album Id"),
          ("MusicBrainz Album Ids"),
          N_("The MusicBrainz Release ID"),
          "mb_album_id",
          ::Bmp::DB::VALUE_TYPE_STRING,
      },

      {
          ("MusicBrainz Artist Id"),
          ("MusicBrainz Artist Ids"),
          N_("The MusicBrainz Artist ID"),
          "mb_artist_id",
          ::Bmp::DB::VALUE_TYPE_STRING,
      },

      {
          ("MusicBrainz Artist Sortname"),
          ("MusicBrainz Artist Sortnames"),
          N_("The MusicBrainz Artist Sort Name"),
          "mb_artistsortname",
          ::Bmp::DB::VALUE_TYPE_STRING,
      },

      {
          ("MusicBrainz Release Date"),
          ("MusicBrainz Release Dates"),
          N_("The MusicBrainz Artist Sort Name"),
          "mb_release_date",
          ::Bmp::DB::VALUE_TYPE_STRING,
      },

      {
          ("MusicIP PUID"),
          ("MusicIP PUIDs"),
          N_("PUID For the given file"),
          "puid",
          ::Bmp::DB::VALUE_TYPE_STRING,
      },

      {
          N_("New Item"),
          N_("New Items"),
          N_("Whether this item was just addded and/or yet not approved"),
          "new_item",
          ::Bmp::DB::VALUE_TYPE_BOOL,
      },

      {
          N_("THASH"),
          N_("THASHes"),
          N_("Binary Data Tag Hash"),
          "hash",
          ::Bmp::DB::VALUE_TYPE_STRING,
      },
  };
}

namespace Bmp
{
  namespace Library
  {
    namespace
    {
      inline bool
      is_module (std::string const &basename)
      {
        return Bmp::Util::str_has_suffix_nocase (basename.c_str (), G_MODULE_SUFFIX);
      } 

      struct TagReadTuple
      {
          ::Bmp::DB::Row * row;
          bool eos;
      };

      //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

      void
      map_insert (Bmp::DB::ValueMap& map, Bmp::Library::Datum datum)
      {
        map.insert (make_pair (string (track_attributes[datum].id),
                                       track_attributes[datum].type));
      }

      void
      row_set (Bmp::DB::Row & row, Bmp::Library::Datum datum, Bmp::DB::Variant variant)
      {
        row.insert (std::make_pair (track_attributes[datum].id, variant));
      }

      TagLib::ID3v2::UserTextIdentificationFrame*
      find_utif (TagLib::ID3v2::Tag *tag, const TagLib::String &description)
      {
          TagLib::ID3v2::FrameList l = tag->frameList("TXXX");
          for(TagLib::ID3v2::FrameList::Iterator it = l.begin(); it != l.end(); ++it) {
            TagLib::ID3v2::UserTextIdentificationFrame *f = dynamic_cast<TagLib::ID3v2::UserTextIdentificationFrame *>(*it);
            if(f && f->description() == description)
              return f;
          }
          return 0;
      }

      int
      get_duration_from_sectors (int sectors)
      {
          const int bytes_per_sector   = 2352;
          const int bytes_per_seconds  = (44100 / 8) / 16 / 2;

          return (sectors * bytes_per_sector / bytes_per_seconds)/1000;
      }

      void
      tag_foreach (const GstTagList * list,
                   const gchar      * tag,
                   TagReadTuple     * tuple)
      {
        int count = gst_tag_list_get_tag_size (list, tag);
        for (int i = 0; i < count; ++i)
        {
          const GValue *value = gst_tag_list_get_value_index (list, tag, i);

          if (G_VALUE_HOLDS (value, G_TYPE_INT))
            {
              tuple->row->insert
                (std::make_pair (std::string (tag), uint64_t(g_value_get_int(value))));
            }
          else
          if (G_VALUE_HOLDS (value, G_TYPE_UINT))
            {
              tuple->row->insert
                (std::make_pair (std::string (tag), uint64_t(g_value_get_uint(value))));
            }
          else
          if (G_VALUE_HOLDS (value, G_TYPE_STRING))
            {
              const char *_v_C = g_value_get_string (value);
              if (*_v_C)
                {
                  std::string v = _v_C; 
                  tuple->row->insert (std::make_pair (std::string (tag), v)); 
                }
            }
        }
      }

      gboolean
      metadata_bus_handler (GstBus       *bus,
                            GstMessage   *message,
                            TagReadTuple *tuple)
      {
          switch (GST_MESSAGE_TYPE (message))
          {
            case GST_MESSAGE_EOS:
                {
                  tuple->eos = true;
                  break;
                }

            case GST_MESSAGE_TAG:
                {
                  GstTagList *tag_list = 0;
                  gst_message_parse_tag (message, &tag_list);
                  if (tag_list)
                    {
                      gst_tag_list_foreach (tag_list, (GstTagForeachFunc) tag_foreach, tuple);
                      gst_tag_list_free (tag_list);
                    }
                  break;
                }

            case GST_MESSAGE_ERROR:
                {
                  GError *error = 0;
                  gst_message_parse_error (message, &error, 0);
                  g_print ("%s ERROR: %s\n", G_STRLOC, error->message);
                  g_error_free (error);
                  tuple->eos = true;
                  break;
                }

            default: break;
          }
          return FALSE;
      }

      void
      metadata_event_loop (TagReadTuple *tuple,
                           GstElement   *element,
                           gboolean      block)
      {
          GstBus *bus = gst_element_get_bus (element);
          g_return_if_fail (bus != 0);

          bool done = false;
          while (!done && !tuple->eos)
          {
              GstMessage *message;

              if (block)
                  message = gst_bus_poll (bus, GST_MESSAGE_ANY, -1);
              else
                  message = gst_bus_pop  (bus);

              if (message == 0)
              {
                  gst_object_unref (bus);
                  return;
              }

              done = metadata_bus_handler (bus, message, tuple);
              gst_message_unref (message);
          }
          gst_object_unref (bus);
      }
    } // unnamed namespace 

    void 
    Library::metadata_set_taglib (Glib::ustring const& location, UTrack & track)
    {
        using namespace std;

        string filename = Glib::filename_from_uri (location); 

        string type;
        if (!Audio::typefind (filename, type))
            throw MetadataWriteError ((boost::format ("Unknown file type: %s") % filename).str());
               
        if (((type == "application/x-id3") || (type == "audio/mpeg"))
              && (m_plugins.find ("mp3") != m_plugins.end()))
          {
            if (!m_plugins.find ("mp3")->second.set (filename, track))
              throw MetadataWriteError ((boost::format ("Unable to write metadata for file %s") % filename).str());
            return;
          }
        else 
        if ((type == "application/ogg") 
              && (m_plugins.find ("ogg") != m_plugins.end()))
          {
            if (!m_plugins.find ("ogg")->second.set (filename, track))
              throw MetadataWriteError ((boost::format ("Unable to write metadata for file %s") % filename).str());
            return;
          }
        else
        if ((type == "audio/x-m4a")
            && (m_plugins.find ("mp4") != m_plugins.end()))
          {
            if (!m_plugins.find ("mp4")->second.set (filename, track))
              throw MetadataWriteError ((boost::format ("Unable to write metadata for file %s") % filename).str());
            return;
          }
        else
        if ((type == "video/x-ms-asf")
              && (m_plugins.find ("wma") != m_plugins.end()))
          {
            if (!m_plugins.find ("wma")->second.set (filename, track))
              throw MetadataWriteError ((boost::format ("Unable to write metadata for file %s") % filename).str());
            return;
          }
        else
        if ((type == "audio/x-flac")
              && (m_plugins.find ("flac") != m_plugins.end()))
          {
            if (!m_plugins.find ("flac")->second.set (filename, track))
              throw MetadataWriteError ((boost::format ("Unable to write metadata for file %s") % filename).str());
            return;
          }
        else
        if ((type == "application/x-apetag")
              && (m_plugins.find ("mpc") != m_plugins.end()))
        {
            if (!m_plugins.find ("mpc")->second.set (filename, track))
              throw MetadataWriteError ((boost::format ("Unable to write metadata for file %s") % filename).str());
            return;
        }

      throw MetadataWriteError ((boost::format ("Unable to write metadata for file %s "
                                  "(Reason Unknown, Presumably unsupported format)") % filename).str());
    }

    void 
    Library::metadata_get_taglib (Glib::ustring const& uri, DB::Row & row) const
    {
      using namespace TagLib;
      using namespace boost;
      using namespace std;
      using namespace Glib;

      string filename;

      try {
          filename = filename_from_uri (uri);
        }
      catch (ConvertError& cxe)
        {
          throw NoMetadataTaglibError ((boost::format ("Unable to convert URI to filename for uri: %s") % uri.c_str()).str());
        } 

      if (!File::isReadable(filename.c_str()))
        throw NoMetadataTaglibError ((boost::format ("File is not readable: %s") % filename).str());

      TagLib::FileRef fileref(filename.c_str());
      if (fileref.isNull())
        throw NoMetadataTaglibError ((boost::format ("Unable to create a TagLib FileRef for this file: %s") % filename).str());
      if (!fileref.file()->isValid()) 
        throw NoMetadataTaglibError ((boost::format ("TagLib::FileRef is not valid: %s") % filename).str());


      File * track = fileref.file ();
      if (!track) 
        throw NoMetadataTaglibError ((boost::format ("Unable to read metadata with TagLib: %s") % filename).str());

      string type;
      if (Audio::typefind (filename, type))
      {
        if (((type == "application/x-id3") || (type == "audio/mpeg"))
              && (m_plugins.find ("mp3") != m_plugins.end()))
          {
            if (!m_plugins.find ("mp3")->second.get (fileref.file(), row, filename))
              throw NoMetadataTaglibError ((boost::format ("No Metadata From File: %s") % filename).str());
          }
        else
        if ((type == "video/x-ms-asf")
              && (m_plugins.find ("wma") != m_plugins.end()))
          {
            if (!m_plugins.find ("wma")->second.get (fileref.file(), row, filename))
              throw NoMetadataTaglibError ((boost::format ("No Metadata From File: %s") % filename).str());
          }
        else
        if ((type == "audio/x-m4a")
              && (m_plugins.find ("mp4") != m_plugins.end()))
          {
            if (!m_plugins.find ("mp4")->second.get (fileref.file(), row, filename))
              throw NoMetadataTaglibError ((boost::format ("No Metadata From File: %s") % filename).str());
          }
        else
        if ((type == "application/ogg")
              && (m_plugins.find ("ogg") != m_plugins.end()))
          {
            if (!m_plugins.find ("ogg")->second.get (fileref.file(), row, filename))
              throw NoMetadataTaglibError ((boost::format ("No Metadata From File: %s") % filename).str());
          }
        else
        if ((type == "audio/x-flac")
              && (m_plugins.find ("flac") != m_plugins.end()))
          {
            if (!m_plugins.find ("flac")->second.get (fileref.file(), row, filename))
              throw NoMetadataTaglibError ((boost::format ("No Metadata From File: %s") % filename).str());
          }
        else
        if ((type == "application/x-apetag")
              && (m_plugins.find ("mpc") != m_plugins.end()))
          {
            if (!m_plugins.find ("mpc")->second.get (fileref.file(), row, filename))
              throw NoMetadataTaglibError ((boost::format ("No Metadata From File: %s") % filename).str());
          }
      }

      string value;

      row_set (row, DATUM_TRACK, uint64_t(track->tag()->track()));
      row_set (row, DATUM_DATE, uint64_t(track->tag()->year()));

#ifndef HAVE_HAL
      row_set (row, DATUM_LOCATION, ustring (uri));
#endif //!HAVE_HAL

      value = track->tag()->artist().to8Bit (true);
      if (!value.empty())
          row_set (row, DATUM_ARTIST, ustring (value));

      value = track->tag()->album().to8Bit (true);
      if (!value.empty())
          row_set (row, DATUM_ALBUM, ustring (value));

      value = track->tag()->genre().to8Bit (true);
      if (!value.empty())
          row_set (row, DATUM_GENRE, ustring (value));

      value = track->tag()->comment().to8Bit (true);
      if (!value.empty())
          row_set (row, DATUM_COMMENT, ustring (value));

      row_set (row, DATUM_RATING, uint64_t(0)); 
      row_set (row, DATUM_COUNT, uint64_t(0)); 

      if (track->audioProperties())
        {
          row_set (row, DATUM_BITRATE, uint64_t(track->audioProperties()->bitrate()));
          row_set (row, DATUM_SAMPLERATE, uint64_t(track->audioProperties()->sampleRate()));
          row_set (row, DATUM_TIME, uint64_t(track->audioProperties()->length()));
        }
      else
        {
          row_set (row, DATUM_BITRATE, uint64_t(0));
          row_set (row, DATUM_SAMPLERATE, uint64_t(0));
          row_set (row, DATUM_TIME, uint64_t(0));
        }

      if  ((!track->tag()->title().toCString(true)) || (!strlen(track->tag()->title().toCString(true))))
          row_set (row, DATUM_TITLE, ustring(path_get_basename(filename)));
      else
        {
          value = track->tag()->title().to8Bit(true);
          if (!value.empty())
            row_set (row, DATUM_TITLE, ustring (value));
        }
    }

    void 
    Library::metadata_get_gst  (Glib::ustring const&  uri,
                                DB::Row            &  row) const
    {
      using namespace std;
      using namespace Glib;

      string filename;
      try {
          filename = filename_from_uri (uri);
        }
      catch (ConvertError& cxe)
        {
          throw NoMetadataGstError ((boost::format ("Unable to convert URI to filename for uri: %s") % uri.c_str()).str());
        } 

      boost::shared_ptr<TagReadTuple> ttuple;

      ttuple = boost::shared_ptr<TagReadTuple> (new TagReadTuple);
      ttuple->eos = false;
      ttuple->row = &row;

#ifndef HAVE_HAL
      ttuple->row->insert (std::make_pair (track_attributes[DATUM_LOCATION].id, std::string (uri)));
#endif //!HAVE_HAL

      GstElement *element   = gst_element_factory_make ("playbin",  "playbin0");
      GstElement *fakesink  = gst_element_factory_make ("fakesink", "fakesink0");
      GstElement *videosink = gst_element_factory_make ("fakesink", "fakesink1");

      g_object_set (G_OBJECT (element), "uri", uri.c_str(), "audio-sink", fakesink, "video-sink", videosink, NULL);

      GstStateChangeReturn state_ret = gst_element_set_state (element, GST_STATE_PAUSED);
      int change_timeout = 5;

      while (state_ret == GST_STATE_CHANGE_ASYNC && change_timeout > 0)
        {
          GstState state;
          state_ret = gst_element_get_state (GST_ELEMENT (element), &state, 0, 1 * GST_SECOND);
          change_timeout--;
        }

      if (state_ret != GST_STATE_CHANGE_FAILURE && state_ret != GST_STATE_CHANGE_ASYNC)
        {
          // Post application specific message so we'll know when to stop the message loop
          GstBus *bus;
          bus = gst_element_get_bus (GST_ELEMENT (element));
          if (bus)
            {
              gst_bus_post (bus, gst_message_new_application (GST_OBJECT (element), 0));
              gst_object_unref (bus);
            }
          // Poll the bus for messages
          metadata_event_loop (ttuple.get(), GST_ELEMENT (element), FALSE);
        }

      GstFormat  format = GST_FORMAT_TIME;
      GstQuery  *query  = gst_query_new_duration (format);

      if (gst_element_query (GST_ELEMENT(element), query))
        {
          gst_element_get_state (GST_ELEMENT(element), 0, 0, 100 * GST_MSECOND);
          gint64 length_in_nanoseconds;
          gst_query_parse_duration (query, &format, &length_in_nanoseconds);
          uint64_t length = length_in_nanoseconds / GST_SECOND;
          ttuple->row->insert (std::make_pair (track_attributes[DATUM_TIME].id, DB::Variant (length)));
        }
      gst_query_unref (query);
      gst_element_set_state (GST_ELEMENT (element), GST_STATE_NULL);
      gst_object_unref (GST_ELEMENT (element));
    }

    int Library::add (const Glib::ustring &uri)
    {
      using namespace Bmp::DB;
      using namespace Glib; 

      Row row;

      try {
          metadata_get_taglib (uri, row);
        }

      catch (NoMetadataTaglibError& cxe)
        {
          g_log (G_LOG_DOMAIN, G_LOG_LEVEL_WARNING, "Taglib Metadata Reading Error: %s", cxe.what());
          metadata_get_gst (uri, row);
        }

      catch (NoMetadataGstError& cxe)
        {
          g_log (G_LOG_DOMAIN, G_LOG_LEVEL_WARNING, "Gstreamer Metadata Reading Error: %s", cxe.what());
          throw NoMetadataError();
        }

      if (!row.empty ())
        {

          // Validate all string values to be valid UTF-8
          for (Row::const_iterator i = row.begin() ; i != row.end() ; ++i)
            {
              if ((i->second.which() == VALUE_TYPE_STRING) && !(Glib::ustring (boost::get<std::string>(i->second)).validate()))
                return 0;
            }

#ifdef HAVE_HAL

          if (hal->is_initialized()) try{
                HAL::Volume volume = hal->get_volume_for_uri (uri);
                row_set (row, DATUM_HAL_VOLUME_UDI, ustring (volume.volume_udi));
                row_set (row, DATUM_HAL_DEVICE_UDI, ustring (volume.device_udi));
                ustring volume_relative_path = filename_from_uri (uri).substr (volume.mount_point.length());
                ustring volume_relative_base_dir = Glib::path_get_dirname (volume_relative_path); 
                row_set (row, DATUM_VOLUME_RELATIVE_PATH, ustring (volume_relative_path));
                row_set (row, DATUM_VOLUME_RELATIVE_BASE_DIR, ustring (volume_relative_base_dir));
            }
          catch (HAL::Exception& cxe) 
            {
                g_log (G_LOG_DOMAIN, G_LOG_LEVEL_WARNING, "HAL Error: %s", cxe.what());
                throw NoHALInformationError();
            }

#endif //HAVE_HAL

          Track track (row);
          if (!(track.artist && track.album && track.title))
              throw NoMetadataError();

          if (track.artist)
            {
              std::string s (track.artist.get().c_str());
              if (!s.size()) 
                throw NoMetadataError();
            }

          if (track.album)
            {
              std::string s (track.album.get().c_str());
              if (!s.size())
                throw NoMetadataError();
            }

          if (track.title)
            {
              std::string s (track.title.get().c_str());
              if (!s.size())
                throw NoMetadataError();
            }

          /////////////////////////////////////// Hash Based Replacing 

          if (track.hash)
          {
              if (db_main->attr_exists ("main", track_attributes[DATUM_HASH].id, track.hash.get()))
              {
                DB::VAttributes a;
                a.push_back (Attribute (DB::EXACT, track_attributes[DATUM_HASH].id, track.hash.get())); 

                DB::VRows rows;
                query (a, rows); 

                if ((!rows.empty()))
                {
                  Track t (rows[0]);
                  if (t.location.get() != uri)
                    {
#ifdef HAVE_HAL
                      db_main->sqlite_exec_simple
                      (sql_uprintf ("UPDATE main SET %q='%q', %q='%q' WHERE %q='%q';",

                                    track_attributes[DATUM_VOLUME_RELATIVE_PATH].id, 
                                    track.volume_relative_path.get().c_str(),

                                    track_attributes[DATUM_HAL_VOLUME_UDI].id, 
                                    track.volume_udi.get().c_str(),

                                    track_attributes[DATUM_HASH].id, 
                                    track.hash.get().c_str()));
#else
                      db_main->sqlite_exec_simple
                      (sql_uprintf ("UPDATE main SET %q='%q' WHERE %q='%q';",
                                     track_attributes[DATUM_LOCATION].id, 
                                     track.location.get().c_str(),
                                     track_attributes[DATUM_HASH].id, 
                                     track.hash.get().c_str()));
#endif
                      signal_row_updated_.emit (track.location.get(), (UTrack (track)));   
                    }
                  return 0;
                }
              }
            }

          /////////////////////////////////////// Static adding

          try {
              row_set (row, DATUM_NEW_ITEM, true); 
              db_main->add ("main", row);
              m_new_items_modified = true;
              return 1;
            }
          catch (Bmp::DB::SqlError & cxe)
            {
              VAttributes a;
              VRows vector;

#ifndef HAVE_HAL
              a.push_back (Attribute (EXACT, track_attributes[DATUM_LOCATION].id, uri)); 
#else
              a.push_back (Attribute (EXACT, track_attributes[DATUM_VOLUME_RELATIVE_PATH].id, track.volume_relative_path.get()));
              a.push_back (Attribute (EXACT, track_attributes[DATUM_HAL_VOLUME_UDI].id, track.volume_udi.get()));
#endif //HAVE_HAL

              query (a, vector);

              if (!vector.empty())
                {
                  try {
                      db_main->del ("main", a);

                      row_set (row, DATUM_NEW_ITEM, false); 
                      db_main->add ("main", row);

                      signal_row_updated_.emit (track.location.get(), (UTrack (track)));   
                      return 1;
                    }
                  catch (Bmp::DB::SqlError& cxe)
                    {
                      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_WARNING, "%s: SQL Error: %s", G_STRFUNC, cxe.what());
                      throw DatabaseError (cxe.what());
                    }
                }
            }
        }
      return 0;
    }

    void
    Library::del (DB::VAttributes const& a)
    {
      signal_processing_start_.emit ();
      db_main->del("main", a);
      signal_processing_end_.emit ();
    }

    void
    Library::query (DB::VAttributes const&  a,
                    DB::VRows            &  v) const
    {
      db_main->get ("main", a, v);
    }

    bool Library::album_exists (Album const& album)
    {
      using namespace Bmp::DB;

      VRows v;
      db_main->get( "albums", album_to_attrs (album), v );
      return (!v.empty() && (v.size() == 1));
    }

    bool Library::new_exist ()
    {
      if (!m_new_items_modified)
        return m_has_new_items;

      m_has_new_items = (db_main->sqlite_exec_simple ( "SELECT DISTINCT new_item FROM main WHERE new_item>0;" ) > 0);
      m_new_items_modified = false;
      return m_has_new_items;
    }

#ifdef HAVE_HAL

    bool Library::volume_exists (std::string const& volume_udi, std::string const& device_udi) const
    {
      static boost::format f_volume_query ("SELECT DISTINCT %s, %s FROM %s WHERE %s='%s' AND %s='%s';");

      std::string query ((f_volume_query  % MetadatumId(DATUM_HAL_VOLUME_UDI)
                                          % MetadatumId(DATUM_HAL_DEVICE_UDI)
                                          % "main"  
                                          % MetadatumId(DATUM_HAL_VOLUME_UDI)
                                          % volume_udi
                                          % MetadatumId(DATUM_HAL_DEVICE_UDI)
                                          % device_udi).str());

      return (db_main->sqlite_exec_simple (query) > 0);
    }

#endif //HAVE_HAL

    bool Library::album_has_new_items (Album const& album) const
    {
      using namespace Glib; 
      using namespace Bmp::DB;

      ustring query ("SELECT count(*) FROM main WHERE "); 

      append_attributes( query, album_to_attrs (album, false) );
      query.append( " AND new_item > 0;" );

      return (db_main->sqlite_exec_simple (query) > 0);
    }

    Album
    Library::album_refetch (Album const& album) const
    {
      using namespace Bmp::DB;

      VRows v;
      db_main->get( "albums", album_to_attrs (album, false), v );
      return Bmp::Library::Album(v[0]);
    }

    void
    Library::all_albums (DB::VRows & vector) const
    {
      using namespace Bmp::DB;

      VAttributes a;
      db_main->get ("albums", a, vector);
    }

    void
    Library::new_albums (DB::VRows & vector) const
    {
      using namespace Bmp::DB;

      VAttributes attrs;
      attrs.push_back (Attribute (EXACT,
                                  MetadatumId (DATUM_NEW_ITEM),
                                  true)); 
      db_main->get ("albums", attrs, vector);
    }

    GHashTable*
    Library::get_hashtable (Glib::ustring const& uri) const
    {
      using namespace Glib;
      using namespace std;

      GHashTable *table;
      GValue     *value;
      DB::Row     row;

      get_metadata_row (uri, row);

      table = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, g_free);
      for (unsigned int n = 0; n < G_N_ELEMENTS (track_attributes); n++)
      {
        DB::Row::const_iterator i = row.find (track_attributes[n].id);
        if (i == row.end())
          {
            g_warning (G_STRLOC ": DB::Attribute '%s' not found in Row (this is currently normal for cdda:/// URIs)", track_attributes[n].id);
            continue;
          }

        value = g_new0 (GValue,1);
        switch (Bmp::DB::ValueType(i->second.which()))
          {
          case Bmp::DB::VALUE_TYPE_STRING:
              g_value_init (value, G_TYPE_STRING);
              g_value_set_string (value, (boost::get<std::string>(i->second)).c_str());
              break;

          case Bmp::DB::VALUE_TYPE_INT:
              g_value_init (value, G_TYPE_INT);
              g_value_set_int (value, (boost::get<uint64_t>(i->second)));
              break;

          case Bmp::DB::VALUE_TYPE_BOOL:
              g_value_init (value, G_TYPE_BOOLEAN);
              g_value_set_boolean (value, (boost::get<bool>(i->second)));
              break;

          case Bmp::DB::VALUE_TYPE_REAL:
              g_value_init (value, G_TYPE_DOUBLE);
              g_value_set_double (value, (boost::get<double>(i->second)));
              break;
        }
        g_hash_table_insert (table, g_strdup(track_attributes[n].id), value);
      }
      return table;
    }

    void Library::get_metadata_row (Glib::ustring const& uri, Bmp::DB::Row& row) const
    {
      using namespace Bmp::DB;
      using namespace std;
      using namespace Glib;

      VAttributes a;
      VRows vector;

#ifndef HAVE_HAL

      a.push_back (Attribute (EXACT, track_attributes[DATUM_LOCATION].id, uri)); 
      query (a, vector);

#else

      HAL::Volume volume = hal->get_volume_for_uri (uri);
      ustring volume_relative_path = filename_from_uri (uri).substr (volume.mount_point.length());
      a.push_back (Attribute (EXACT, track_attributes[DATUM_HAL_VOLUME_UDI].id, volume.volume_udi)); 
      a.push_back (Attribute (EXACT, track_attributes[DATUM_VOLUME_RELATIVE_PATH].id, volume_relative_path)); 
      query (a, vector);

#endif //HAVE_HAL

      if (vector.empty())
        {
          try {
              metadata_get_taglib (uri, row);
            }

          catch (NoMetadataTaglibError & cxe)
            {
              g_log (G_LOG_DOMAIN, G_LOG_LEVEL_WARNING, "Taglib Metadata Reading Error: %s", cxe.what());
              metadata_get_gst (uri, row);
            }

          catch (NoMetadataGstError & cxe)
            {
              g_log (G_LOG_DOMAIN, G_LOG_LEVEL_WARNING, "Taglib Metadata Reading Error: %s", cxe.what());
              throw NoMetadataError();
            }

#ifdef HAVE_HAL
          if (hal->is_initialized()) try {
                HAL::Volume volume = hal->get_volume_for_uri (uri);
                row_set (row, DATUM_HAL_VOLUME_UDI, ustring (volume.volume_udi));
                row_set (row, DATUM_HAL_DEVICE_UDI, ustring (volume.device_udi));
                ustring volume_relative_path = filename_from_uri (uri).substr (volume.mount_point.length());
                ustring volume_relative_base_dir = path_get_dirname (volume_relative_path); 
                row_set (row, DATUM_VOLUME_RELATIVE_PATH, ustring (volume_relative_path));
                row_set (row, DATUM_VOLUME_RELATIVE_BASE_DIR, ustring (volume_relative_base_dir));
              }
          catch (HAL::Exception& cxe) 
              {
                g_log (G_LOG_DOMAIN, G_LOG_LEVEL_WARNING, "HAL Error: %s", cxe.what());
                throw NoHALInformationError();
              }
#endif
        }
      else
        {
          row = vector[0]; 
        } 
    }

    void
    Library::get  (Glib::ustring const& uri,
                   Track &              track) const
    {
      Bmp::DB::Row row;
      Bmp::URI u (uri);
      Bmp::URI::Protocol p = u.get_protocol ();

      get_metadata_row (uri, row);

      if (p == Bmp::URI::PROTOCOL_HTTP || 
          p == Bmp::URI::PROTOCOL_MMS || 
          p == Bmp::URI::PROTOCOL_MMSU || 
          p == Bmp::URI::PROTOCOL_MMST ||
          p == Bmp::URI::PROTOCOL_CDDA)
        {

#ifdef HAVE_HAL

          track = Track (row, false);

#else

          track = Track (row);

#endif //HAVE_HAL

          return;
        }
        track = Track (row);
    }

    void
    Library::new_drop_all ()
    {
      db_main->sqlite_exec_simple (sql_uprintf("DELETE FROM main WHERE new_item > 0;"));  
      m_new_items_modified = true;
    }


    void
    Library::new_approve ()
    {
      db_main->sqlite_exec_simple (sql_uprintf("UPDATE main SET new_item=0;")); 
      m_new_items_modified = true;
    }


    void
    Library::new_approve (Album const& album)
    {
      Glib::ustring query( "UPDATE main SET new_item=0 WHERE " ); 
      append_attributes( query, album_to_attrs(album, false) );
      query.append( ";" );

      db_main->sqlite_exec_simple(query);

      m_new_items_modified = true;
    }

    void
    Library::recreate_db ()
    {
        using namespace Bmp::DB;
        using namespace std;
        using namespace Glib;

        delete db_main;
        db_main = new Database ("library", BMP_PATH_DATA_DIR, DB_TRUNCATE);

        ValueMap meta;
        meta.insert (make_pair ("version", VALUE_TYPE_STRING));
        db_main->create_table ("meta", "version", meta);

        ustring sql;

        sql .append ("INSERT INTO meta (version) VALUES ('")
            .append (string (MLIB_VERSION))
            .append ("');");

        db_main->sqlite_exec_simple (sql.c_str());
    }

    void Library::mod (UTrackV & v, bool retag, bool db_update, bool relocate)
    {
      signal_processing_start_.emit ();
      signal_modify_start_.emit (v.size ());

      for (UTrackV::iterator i = v.begin () ; i != v.end (); ++i)
        {
          using namespace Bmp::DB;
          using namespace Glib; 
          using namespace std; 

          UTrack & track (*i);
          DB::VAttributesSet a;

          if (db_update)
            {
              if (track.title)
                a.push_back (DB::AttributeSet (track_attributes[DATUM_TITLE].id,
                              track.title.get().raw())); 

              if (track.artist)
                a.push_back (DB::AttributeSet (track_attributes[DATUM_ARTIST].id,
                              track.artist.get().raw())); 

              if (track.album)
                a.push_back (DB::AttributeSet (track_attributes[DATUM_ALBUM].id,
                              track.album.get().raw())); 

              if (track.genre)
                a.push_back (DB::AttributeSet (track_attributes[DATUM_GENRE].id,
                              track.genre.get().raw())); 

              if (track.comment)
                a.push_back (DB::AttributeSet (track_attributes[DATUM_COMMENT].id,
                              track.comment.get().raw())); 

              if (track.tracknumber)
                a.push_back (DB::AttributeSet (track_attributes[DATUM_TRACK].id,
                              uint64_t(track.tracknumber.get()))); 

              if (track.rating)
                 a.push_back (DB::AttributeSet (track_attributes[DATUM_RATING].id,
                              uint64_t(track.rating.get()))); 

              if (track.count)
                a.push_back (DB::AttributeSet (track_attributes[DATUM_COUNT].id,
                              uint64_t(track.count.get()))); 

              if (track.asin)
                a.push_back (DB::AttributeSet (track_attributes[DATUM_ASIN].id,
                              track.asin.get().raw())); 

              if (track.puid)
                a.push_back (DB::AttributeSet (track_attributes[DATUM_MUSICIP_PUID].id,
                              track.puid.get().raw())); 

              if (track.mb_album_artist)
                a.push_back (DB::AttributeSet (track_attributes[DATUM_MB_ALBUM_ARTIST].id,
                              track.mb_album_artist.get().raw())); 

              if (track.mb_album_artist_id)
                a.push_back (DB::AttributeSet (track_attributes[DATUM_MB_ALBUM_ARTIST_ID].id,
                              track.mb_album_artist_id.get().raw())); 

              if (track.mb_album_artist_sort_name)
                a.push_back (DB::AttributeSet (track_attributes[DATUM_MB_ALBUM_ARTIST_SORTNAME].id,
                              track.mb_album_artist_sort_name.get().raw())); 

              if (track.mb_track_id)
                a.push_back (DB::AttributeSet (track_attributes[DATUM_MB_TRACK_ID].id,
                              track.mb_track_id.get().raw())); 

              if (track.mb_album_id)
                a.push_back (DB::AttributeSet (track_attributes[DATUM_MB_ALBUM_ID].id,
                              track.mb_album_id.get().raw())); 

              if (track.mb_artist_id)
                a.push_back (DB::AttributeSet (track_attributes[DATUM_MB_ARTIST_ID].id,
                              track.mb_artist_id.get().raw())); 

              if (track.mb_artist_sort_name)
                a.push_back (DB::AttributeSet (track_attributes[DATUM_MB_ARTIST_SORTNAME].id,
                              track.mb_artist_sort_name.get().raw())); 

              if (track.mb_release_date)
                {
                    a.push_back (DB::AttributeSet (track_attributes[DATUM_MB_RELEASE_DATE].id,
                                  track.mb_release_date.get().raw())); 

                    int y = 0, m = 0, d = 0;
                    sscanf (track.mb_release_date.get().c_str(), "%04d-%02d-%02d", &y, &m, &d);
                    a.push_back (DB::AttributeSet (track_attributes[DATUM_DATE].id, uint64_t(y))); 
                                 
                }

              if (track.date)
                {
                  a.push_back (DB::AttributeSet (track_attributes[DATUM_DATE].id,
                                uint64_t(track.date.get()))); 
                }

              if (track.new_item)
                {
                  a.push_back (DB::AttributeSet (track_attributes[DATUM_NEW_ITEM].id,
                                bool(track.new_item.get())));

                  if (track.new_item.get())
                    m_new_items_modified = true;
                }
            }
      
          boost::optional<ustring> modify_location;

#ifdef HAVE_HAL
          if ((track.location_new) && hal->is_initialized() && relocate)
          {
            HAL::Volume volume = hal->get_volume_for_uri (track.location_new.get());
            string filename_A = filename_from_uri (track.location_cur.get());
            string filename_B = filename_from_uri (track.location_new.get());
            try
              {
                Util::copy_file (filename_A, filename_B);
                if (Util::compare_files (filename_A, filename_B))
                {
                  g_unlink (filename_A.c_str());

                  ustring volume_relative_path (filename_B.substr (volume.mount_point.length()));
                  a.push_back (DB::AttributeSet (track_attributes[DATUM_VOLUME_RELATIVE_PATH].id, volume_relative_path));

                  a.push_back (DB::AttributeSet (track_attributes[DATUM_HAL_VOLUME_UDI].id, volume.volume_udi));
                  a.push_back (DB::AttributeSet (track_attributes[DATUM_HAL_DEVICE_UDI].id, volume.device_udi));
                  modify_location = filename_to_uri (filename_B);
                }
              }
            catch (std::exception & cxe)
              {
                g_critical ("%s: Unable to copy file to destination. Reason: '%s'. "
                            "Source: '%s'. Destination: '%s'",
                            G_STRLOC, cxe.what(), filename_A.c_str(), filename_B.c_str());
                continue;
              }
          }
#else
          if (track.location_new && relocate) 
          {
            string filename_A = filename_from_uri (track.location_cur.get());
            string filename_B = filename_from_uri (track.location_new.get());
            try{
                Util::copy_file (filename_A, filename_B);
                if (Util::compare_files (filename_A, filename_B))
                  {
                    g_unlink (filename_A.c_str());
                    modify_location = filename_to_uri(filename_B);
                    a.push_back (DB::AttributeSet (track_attributes[DATUM_LOCATION].id, modify_location));
                  }
                else
                  {
                    g_unlink (filename_B.c_str());
                  }
              }
            catch (std::exception& cxe)
              {
                g_critical ("%s: Unable to copy file to destination. Reason: '%s'. "
                            "Source: '%s'. Destination: '%s'",
                            G_STRLOC, cxe.what(), filename_A.c_str(), filename_B.c_str());
                continue;
              }
          }
#endif //HAVE_HAL

          try {

              if (retag)
                {
                  metadata_set_taglib (modify_location  ? modify_location.get() 
                                                        : track.location_cur.get(), track);

                  a.push_back (DB::AttributeSet (track_attributes[DATUM_HASH].id, track.hash.get().raw())); 
                }

#ifdef HAVE_HAL
              ustring where; 

              where.append (sql_uprintf (" WHERE %q='%q' AND %q='%q' ",
                            track_attributes[DATUM_HAL_VOLUME_UDI].id,
                            track.volume_udi.get().c_str(),
                            track_attributes[DATUM_VOLUME_RELATIVE_PATH].id,
                            track.volume_relative_path.get().c_str()));

              db_main->set ("main", where, a);
#else
              db_main->set ("main", track_attributes[DATUM_LOCATION].id, track.location_cur.get(), a);
#endif //HAVE_HAL

              signal_row_updated_.emit (track.location_cur.get(), track);   
            }
          catch (Bmp::Library::Exception & cxe) {}
          catch (Bmp::Library::Library::Exception & cxe) {}
          catch (Bmp::DB::SqlError & cxe) {}
          catch (...) {}

        signal_modify_step_.emit (); 
      }   

      signal_modify_end_.emit ();
      signal_processing_end_.emit ();
    }

    void 
    Library::project (std::string const& project, DB::VRows & vector, DB::VAttributes const& attributes) const
    {
      db_main->project ("albums", project, vector, attributes);
    }

    //////////////////////////////////////////////////////////////////////////////////////////////////////

    Library::Library ()

      : m_has_new_items       (false),
        m_new_items_modified  (true)

    {
      using namespace Bmp::DB;
      using namespace std;
      using namespace Glib;

      Util::dir_for_each_entry (BMP_TAGLIB_PLUGIN_DIR,
                                sigc::mem_fun (this, &Bmp::Library::Library::load_taglib_plugin));  

      //-- Create 'main' database

#ifndef HAVE_HAL

      for (unsigned int n = 0; n < G_N_ELEMENTS(track_attributes); n++)
        {
          map_main.insert (std::make_pair (std::string (track_attributes[n].id), track_attributes[n].type));
        }

#else

      for (unsigned int n = 1; n < G_N_ELEMENTS(track_attributes); n++)
        {
          map_main.insert (std::make_pair (std::string (track_attributes[n].id), track_attributes[n].type));
        }

#endif //HAVE_HAL

      try {
        db_main = new Database ("library", BMP_PATH_DATA_DIR, DB_OPEN);
      } catch (Bmp::DB::DbInitError& cxe) { abort (); }

      ValueMap meta;
      meta.insert (make_pair ("version", VALUE_TYPE_STRING));
      db_main->create_table  ("meta", "version", meta);

      VAttributes a;
      a.push_back (DB::Attribute (DB::EXACT, std::string("version"), std::string(MLIB_VERSION))); 

      VRows v;
      try {
          db_main->get ("meta", a, v);
          if (v.empty() || ((boost::get<std::string>((*(v[0].find ("version"))).second) != ustring (MLIB_VERSION)))) recreate_db ();
        }
      catch (Bmp::DB::SqlError& cxe) { recreate_db (); }

#ifndef HAVE_HAL
      db_main->create_table  ("main", string (track_attributes[DATUM_LOCATION].id), map_main);
#else //!HAVE_HAL

      try {
          ustring sql;
          sql.append ("CREATE TABLE IF NOT EXISTS main (");
          for (unsigned int n = 1; n < G_N_ELEMENTS(track_attributes); n++)
          {
            sql.append ("'");
            sql.append (track_attributes[n].id);
            sql.append ("' ");
            
            if (track_attributes[n].type == VALUE_TYPE_STRING)
                sql.append (" TEXT ");
            else
            if (track_attributes[n].type == VALUE_TYPE_INT)
                sql.append (" INTEGER DEFAULT 0 ");
            else
            if (track_attributes[n].type == VALUE_TYPE_BOOL)
                sql.append (" BOOL DEFAULT 0 ");
            sql.append (", ");
          }

          sql.append (" PRIMARY KEY ('");
          sql.append (track_attributes[DATUM_HAL_VOLUME_UDI].id);
          sql.append ("', '"); 
          sql.append (track_attributes[DATUM_VOLUME_RELATIVE_PATH].id);
          sql.append ("') ON CONFLICT REPLACE);"); 
          db_main->sqlite_exec_simple (sql.c_str());
        }
      catch (Bmp::DB::SqlError& cxe)
        {
          abort (); //FIXME: 
        }

      db_main->insert_map_for_table ("main", map_main);

#endif //HAVE_HAL

      //Create albums view
      map_insert (map_albums, DATUM_DATE);
      map_insert (map_albums, DATUM_ARTIST);
      map_insert (map_albums, DATUM_ALBUM);
      map_insert (map_albums, DATUM_ASIN);
      map_insert (map_albums, DATUM_NEW_ITEM);
      map_insert (map_albums, DATUM_MB_ALBUM_ID);
      map_insert (map_albums, DATUM_MB_ALBUM_ARTIST_ID);
      map_insert (map_albums, DATUM_MB_ALBUM_ARTIST_SORTNAME);
      map_albums.insert (make_pair (string (album_attributes[0].id), album_attributes[0].type));
                                           
      db_main->create_view_manual ("main", "albums",
          sql_uprintf ("SELECT DISTINCT %q, ifnull(%q, %q) AS %q, %q, %q, %q, %q, %q, %q, count(%q) AS tracks "
                        "FROM main GROUP BY %q, ifnull(%q, %q), %q, %q, %q, %q, %q, ORDER BY %q;",

                       track_attributes[DATUM_ALBUM].id,

                       track_attributes[DATUM_MB_ALBUM_ARTIST_SORTNAME].id, // ifnull (%q, %q)
                       track_attributes[DATUM_ARTIST].id,

                       track_attributes[DATUM_ARTIST].id, // AS %q

                       track_attributes[DATUM_DATE].id,
                       track_attributes[DATUM_ASIN].id,
                       track_attributes[DATUM_NEW_ITEM].id,
                       track_attributes[DATUM_MB_ALBUM_ID].id,
                       track_attributes[DATUM_MB_ALBUM_ARTIST_ID].id,
                       track_attributes[DATUM_MB_ALBUM_ARTIST_SORTNAME].id,

#ifdef HAVE_HAL
                       track_attributes[DATUM_VOLUME_RELATIVE_PATH].id, // count (%q)
#else
                       track_attributes[DATUM_LOCATION].id, // count (%q)
#endif //HAVE_HAL

                       track_attributes[DATUM_ALBUM].id,

                       track_attributes[DATUM_MB_ALBUM_ARTIST_SORTNAME].id, // ifnull (%q, %q) 
                       track_attributes[DATUM_ARTIST].id,

                       track_attributes[DATUM_DATE].id,
                       track_attributes[DATUM_ASIN].id,
                       track_attributes[DATUM_NEW_ITEM].id,
                       track_attributes[DATUM_MB_ALBUM_ID].id,
                       track_attributes[DATUM_MB_ALBUM_ARTIST_ID].id,

                       track_attributes[DATUM_ALBUM].id), // ORDER BY

                       map_albums);

      db_main->sqlite_exec_simple ("ANALYZE;");
    }

    Library::~Library ()
    {
        delete db_main;
    }

    // Generic "processing" progress #FIXME: What is this good for?  --deadchip
    Library::SignalProcessing&
    Library::signal_processing_start ()
    { 
      return signal_processing_start_;
    }

    Library::SignalProcessing&
    Library::signal_processing_end ()
    { 
      return signal_processing_end_;
    }

    Library::SignalRowUpdated&
    Library::signal_row_updated ()
    {
      return signal_row_updated_;
    }

    // Modification progress
    Library::ModifyStart&
    Library::signal_modify_start()
    { 
      return signal_modify_start_;
    }

    Library::ModifyStep&
    Library::signal_modify_step()
    { 
      return signal_modify_step_;
    }

    Library::ModifyEnd&
    Library::signal_modify_end()
    {
      return signal_modify_end_;
    }

    ///// Album
 
    std::string 
    Album::create_key (Album const& a)
    {
      return ((a.sortname ? a.sortname.get() 
                          : (a.artist ? a.artist .get()  : ""))
                          + (a.album  ? a.album  .get()  : "")).casefold_collate_key();
    }

    Album::TR
    Album::create_tpl (Album const& a)
    {
      return Album::TR  (   a.artist,
                            a.album,
                            a.asin,
                            a.release_id,
                            a.artist_id,
                            a.sortname,
                            a.date,
                            a.tracks       );
    }
 
    Album::Album (const Bmp::DB::Row& row)
    {
      using namespace Bmp::DB;
      using namespace Glib; 

      Row::const_iterator i;
      Row::const_iterator end = row.end();

      i = row.find (track_attributes[DATUM_ARTIST].id);
      if (i != end && !(boost::get<std::string>(i->second).empty()))
          artist = boost::get<std::string> (i->second);

      i = row.find (track_attributes[DATUM_ALBUM].id);
      if (i != end && !(boost::get<std::string>(i->second).empty()))
          album = boost::get<std::string> (i->second);

      i = row.find (track_attributes[DATUM_ASIN].id);
      if (i != end && !(boost::get<std::string>(i->second).empty()))
          asin = boost::get<std::string> (i->second);

      i = row.find (track_attributes[DATUM_MB_ALBUM_ID].id);
      if (i != end && !(boost::get<std::string>(i->second).empty()))
          release_id = boost::get<std::string> (i->second);

      i = row.find (track_attributes[DATUM_MB_ALBUM_ARTIST_ID].id);
      if (i != end && !(boost::get<std::string>(i->second).empty()))
          artist_id = boost::get<std::string> (i->second);

      i = row.find (track_attributes[DATUM_MB_ALBUM_ARTIST_SORTNAME].id);
      if (i != end && !(boost::get<std::string>(i->second).empty()))
          sortname = boost::get<std::string> (i->second);

      i = row.find (track_attributes[DATUM_DATE].id);
          date = boost::get<uint64_t>(i->second);

      i = row.find (track_attributes[DATUM_NEW_ITEM].id);
          new_item = boost::get<bool> (i->second);

      i = row.find (album_attributes[0].id);
          tracks = boost::get<uint64_t> (i->second);

      key = Album::create_key (*this);
      tpl = Album::create_tpl (*this);
    }

    // Track 
#ifdef HAVE_HAL
    Track::Track (const Bmp::DB::Row& row, bool use_hal)
#else
    Track::Track (const Bmp::DB::Row& row)
#endif //HAVE_HAL
    {
      using namespace Glib; 
      using namespace Bmp::DB;

      Row::const_iterator i, end = row.end ();

      i = row.find (track_attributes[DATUM_ARTIST].id);
      if ((i != end) && !(boost::get<std::string>(i->second).empty()))
          artist = boost::get<std::string> (i->second);

      i = row.find (track_attributes[DATUM_ALBUM].id);
      if ((i != end) && !(boost::get<std::string>(i->second).empty()))
          album = boost::get<std::string> (i->second);

      i = row.find (track_attributes[DATUM_TITLE].id);
      if ((i != end) && !(boost::get<std::string>(i->second).empty()))
          title = boost::get<std::string> (i->second);

      i = row.find (track_attributes[DATUM_GENRE].id);
      if ((i != end) && !(boost::get<std::string>(i->second).empty()))
          genre = boost::get<std::string> (i->second);

      i = row.find (track_attributes[DATUM_COMMENT].id);
      if ((i != end) && !(boost::get<std::string>(i->second).empty()))
          comment = boost::get<std::string> (i->second);

      i = row.find (track_attributes[DATUM_TRACK].id);
      if (i != end)
            tracknumber = boost::get<uint64_t> (i->second);

      i = row.find (track_attributes[DATUM_TIME].id);
      if (i != end)
            duration = boost::get<uint64_t> (i->second);

      i = row.find (track_attributes[DATUM_RATING].id);
      if (i != end)
            rating = boost::get<uint64_t> (i->second);

      i = row.find (track_attributes[DATUM_COUNT].id);
      if (i != end)
            count = boost::get<uint64_t> (i->second);

      i = row.find (track_attributes[DATUM_DATE].id);
      if (i != end)
            date = boost::get<uint64_t> (i->second);

      i = row.find (track_attributes[DATUM_MTIME].id);
      if (i != end)
            mtime = boost::get<uint64_t> (i->second);

      i = row.find (track_attributes[DATUM_BITRATE].id);
      if (i != end)
            bitrate = boost::get<uint64_t> (i->second);

      i = row.find (track_attributes[DATUM_SAMPLERATE].id);
      if (i != end)
            samplerate = boost::get<uint64_t> (i->second);

      i = row.find (track_attributes[DATUM_ASIN].id);
      if ((i != end) && !(boost::get<std::string>(i->second).empty()))
          asin = boost::get<std::string> (i->second);

#ifdef HAVE_HAL

      i = row.find (track_attributes[DATUM_HAL_VOLUME_UDI].id);
      if ((i != end) && !(boost::get<std::string>(i->second).empty()))
          volume_udi = boost::get<std::string> (i->second);

      i = row.find (track_attributes[DATUM_HAL_DEVICE_UDI].id);
      if ((i != end) && !(boost::get<std::string>(i->second).empty()))
          device_udi = boost::get<std::string> (i->second);

      i = row.find (track_attributes[DATUM_VOLUME_RELATIVE_PATH].id);
      if ((i != end) && !(boost::get<std::string>(i->second).empty()))
          volume_relative_path = boost::get<std::string> (i->second);
      
      if (use_hal)
        {
          try {
            std::string mount_point = hal->get_mount_point_for_volume (volume_udi.get().c_str(), device_udi.get().c_str());
            location = Glib::filename_to_uri
                (Glib::build_filename (mount_point.c_str(), volume_relative_path.get().c_str()));
            }
          catch (...) {}
        }
      else
        {
          i = row.find (track_attributes[DATUM_LOCATION].id);
          if ((i != end) && !(boost::get<std::string>(i->second).empty()))
              location = boost::get<std::string> (i->second);
        }

#else // !HAVE_HAL

      i = row.find (track_attributes[DATUM_LOCATION].id);
      if ((i != end) && !(boost::get<std::string>(i->second).empty()))
          location = boost::get<std::string> (i->second);

#endif //HAVE_HAL

      i = row.find (track_attributes[DATUM_MB_ARTIST_SORTNAME].id);
      if ((i != end) && !(boost::get<std::string>(i->second).empty()))
          mb_album_artist_sort_name = boost::get<std::string> (i->second);

      i = row.find (track_attributes[DATUM_MB_ALBUM_ARTIST_ID].id);
      if ((i != end) && !(boost::get<std::string>(i->second).empty()))
          mb_album_artist_id = boost::get<std::string> (i->second);

      i = row.find (track_attributes[DATUM_MB_ALBUM_ARTIST_SORTNAME].id);
      if ((i != end) && !(boost::get<std::string>(i->second).empty()))
          mb_album_artist_sort_name = boost::get<std::string> (i->second);

      i = row.find (track_attributes[DATUM_MB_TRACK_ID].id);
      if ((i != end) && !(boost::get<std::string>(i->second).empty()))
          mb_track_id = boost::get<std::string> (i->second);

      i = row.find (track_attributes[DATUM_MB_ALBUM_ID].id);
      if ((i != end) && !(boost::get<std::string>(i->second).empty()))
          mb_album_id = boost::get<std::string> (i->second);

      i = row.find (track_attributes[DATUM_MB_ARTIST_ID].id);
      if ((i != end) && !(boost::get<std::string>(i->second).empty()))
          mb_artist_id = boost::get<std::string> (i->second);

      i = row.find (track_attributes[DATUM_MUSICIP_PUID].id);
      if ((i != end) && !(boost::get<std::string>(i->second).empty()))
          puid = boost::get<std::string> (i->second);

      i = row.find (track_attributes[DATUM_HASH].id);
      if ((i != end) && !(boost::get<std::string>(i->second).empty()))
          hash = boost::get<std::string> (i->second);

      i = row.find (track_attributes[DATUM_NEW_ITEM].id);
      if (i != end)
        new_item = boost::get<bool> (i->second);

    }

    bool operator< (Album const& a, Album const &b)
    {
      return (a.artist      < b.artist)     || 
             (a.album       < b.album)      || 
             (a.asin        < b.asin)       || 
             (a.release_id  < b.release_id) || 
             (a.artist_id   < b.artist_id)  || 
             (a.sortname    < b.sortname)   || 
             (a.date        < b.date);
    } 

    bool operator== (Album const& a, Album const &b)
    {
      int truth = 0;
      int mask  = 0;

      enum ComparisonMask
      {
        NONE          = 0,
        M_ARTIST      = 1 << 0,
        M_ALBUM       = 1 << 1,
        M_RELEASE_ID  = 1 << 2,
        M_ARTIST_ID   = 1 << 3,
        M_ASIN        = 1 << 4,
        M_SORTNAME    = 1 << 5,
        M_DATE        = 1 << 6
      };

      if (a.asin || b.asin)
        {
          mask |= M_ASIN;

          if (a.asin == b.asin)
            truth |= M_ASIN;
          else
            return false;
        }

      if ((a.artist || b.artist) && !(a.sortname || b.sortname))
        {
          mask |= M_ARTIST;

          if (a.artist == b.artist)
            truth |= M_ARTIST;
          else
            return false;
        }

      if (a.album || b.album)
        {
          mask |= M_ALBUM;

          if (a.album == b.album)
            truth |= M_ALBUM;
          else
            return false;
        }

      if (a.release_id || b.release_id)
        {
          mask |= M_RELEASE_ID;

          if (a.release_id == b.release_id)
            truth |= M_RELEASE_ID;
          else
            return false;
        }

      if (a.artist_id || b.artist_id)
        {
          mask |= M_ARTIST_ID;

          if (a.artist_id == b.artist_id)
            truth |= M_ARTIST_ID;
          else
            return false;
        }

      if (a.sortname || b.sortname)
        {
          mask |= M_SORTNAME;

          if (a.sortname == b.sortname)
            truth |= M_SORTNAME;
          else
            return false;
        }

      if (a.date || b.date)
        {
          mask |= M_DATE;

          if (a.date == b.date)
            truth |= M_DATE;
          else
            return false;
        }

      return (mask == truth); 
    } 

    Bmp::DB::VAttributes 
    album_to_attrs (Bmp::Library::Album const& album, bool include_tracks)
    {
      using namespace Bmp::DB;

      Bmp::DB::VAttributes attrs; 

      if (album.release_id && album.artist_id)
      {
        attrs.push_back (Attribute (album.release_id ? EXACT : IS_NULL,
                         MetadatumId (DATUM_MB_ALBUM_ID),
                         album.release_id ? album.release_id.get() : ""));

        attrs.push_back (Attribute (album.artist_id ? EXACT : IS_NULL,
                         MetadatumId (DATUM_MB_ALBUM_ARTIST_ID),
                         album.artist_id ? album.artist_id.get() : ""));
      }
      else
      {
          if (!album.sortname)
          {
            attrs.push_back (Attribute (album.artist ? EXACT : IS_NULL,
                             MetadatumId (DATUM_ARTIST),
                             album.artist ? album.artist.get() : ""));
          }

          attrs.push_back (Attribute (album.album ? EXACT : IS_NULL,
                           MetadatumId (DATUM_ALBUM),
                           album.album ? album.album.get() : ""));

          attrs.push_back (Attribute (album.sortname ? EXACT : IS_NULL,
                           MetadatumId (DATUM_MB_ALBUM_ARTIST_SORTNAME),
                           album.sortname ? album.sortname.get() : ""));

          attrs.push_back (Attribute (album.asin ? EXACT : IS_NULL,
                           MetadatumId (DATUM_ASIN),
                           album.asin ? album.asin.get() : ""));

          attrs.push_back (Attribute (album.artist_id ? EXACT : IS_NULL,
                           MetadatumId (DATUM_MB_ALBUM_ARTIST_ID),
                           album.artist_id ? album.artist_id.get() : ""));

          attrs.push_back (Attribute (EXACT,
                           MetadatumId (DATUM_DATE),
                           uint64_t(album.date.get())));

          if (include_tracks)
          {
            attrs.push_back (Attribute (EXACT, "tracks", uint64_t(album.tracks.get())));
          }
      }

      return attrs;
    }

    bool
    Library::load_taglib_plugin (std::string const& path)
    {
      using namespace boost::algorithm;

      enum {
        LIB_BASENAME1,
        LIB_BASENAME2,
        LIB_PLUGNAME,
        LIB_SUFFIX
      };

      const std::string type = "taglib_plugin";

      std::string basename (Glib::path_get_basename (path));
      std::string pathname (Glib::path_get_dirname  (path));

      if (!is_module (basename))
        return false;

      std::vector<std::string> subs;
      split (subs, basename, is_any_of ("_."));
      std::string name  = type + std::string("_") + subs[LIB_PLUGNAME];
      std::string mpath = Glib::Module::build_path (BMP_TAGLIB_PLUGIN_DIR, name);

      Glib::Module module (mpath, Glib::ModuleFlags (0)); 

      if (module)
        {
          module.make_resident ();

          g_log (G_LOG_DOMAIN, G_LOG_LEVEL_INFO, "(II) Taglib plugin load SUCCESS '%s'", mpath.c_str ());

          void * _plugin_has_accessors = 0; //dummy

          if (module.get_symbol ("_plugin_has_accessors", _plugin_has_accessors))
          {
            TaglibPlugin plugin;

            g_module_symbol (module.gobj(), "_get", (gpointer*)(&plugin.get)); 
            g_module_symbol (module.gobj(), "_set", (gpointer*)(&plugin.set)); 
            m_plugins.insert( std::make_pair( subs[LIB_PLUGNAME], plugin ));

            g_log (G_LOG_DOMAIN, G_LOG_LEVEL_INFO, "     >> Registered %s", subs[LIB_PLUGNAME].c_str());
          }
        }
      else
        {
          g_critical ("(WW) Taglib plugin load FAILURE '%s': %s", mpath.c_str (), module.get_last_error().c_str());
        }

      return false;
    }

    void
    metadata_set_common (UTrack const& track, TagLib::Tag * tag)
    {
      using namespace TagLib;

      if (!tag)
        return;

      if (track.title)
          tag->setTitle   (String (track.title.get().c_str(), String::UTF8)); 

      if (track.artist)
          tag->setArtist  (String (track.artist.get().c_str(), String::UTF8));

      if (track.album)
          tag->setAlbum   (String (track.album.get().c_str(), String::UTF8));

      if (track.comment)
          tag->setComment (String (track.comment.get().c_str(), String::UTF8));

      if (track.genre)
          tag->setGenre   (String (track.genre.get().c_str(), String::UTF8));

      if (track.date)
          tag->setYear    (track.date.get());

      if (track.tracknumber)
          tag->setTrack   (track.tracknumber.get());
    }

    void
    metadata_get_id3v2 (TagLib::ID3v2::Tag * tag, DB::Row & row, std::string const& name) 
    {
        using namespace Glib;
        using namespace TagLib;
        using namespace boost;
        using namespace std;

        using boost::algorithm::split;
        using boost::algorithm::find_nth;
        using boost::iterator_range;

        const Datum extra_tags_mpeg[] =
        {
          DATUM_MB_ALBUM_ARTIST,
          DATUM_MB_ALBUM_ARTIST_ID,
          DATUM_MB_ALBUM_ID,
          DATUM_MB_ARTIST_ID,
          DATUM_MB_ARTIST_SORTNAME,
          DATUM_MB_ALBUM_ARTIST_SORTNAME,
          DATUM_ASIN,
          DATUM_MUSICIP_PUID,
        };

        ID3v2::UserTextIdentificationFrame *frame = 0;
        for (unsigned int n = 0; n < G_N_ELEMENTS(extra_tags_mpeg); ++n)
        {
            frame = find_utif (tag, String (track_attributes[extra_tags_mpeg[n]].id, String::UTF8));
            if (frame)
              {
                string str = frame->toString().to8Bit(true);

                iterator_range<std::string::iterator> match = find_nth (str, "] ", 0);
                if (match.empty())
                  continue;

                string substr (match.end(), str.end());
                if (!substr.length())
                  continue;

                row_set (row, extra_tags_mpeg[n], ustring (substr));
              }
        }

        // Check for UFID
        using TagLib::ID3v2::FrameList;
        FrameList const& map = tag->frameListMap()["UFID"];
        if (!map.isEmpty())
        {
            for (FrameList::ConstIterator iter = map.begin(), end = map.end (); iter != end; ++iter)
            {
                ID3v2::UniqueFileIdentifierFrame *ufid = reinterpret_cast<ID3v2::UniqueFileIdentifierFrame*>(*iter);
                if (ufid->owner() == "http://musicbrainz.org")
                {
                    ByteVector vec = ufid->identifier();
                    vec.append ('\0');
                    row_set (row, DATUM_MB_TRACK_ID, ustring (vec.data()));
                    break;
                }
            }

        // Check for date
        using TagLib::ID3v2::FrameList;
        const char *_frames[] = { "TDRL", "TDRC" , "TDOR", "XDOR" };
        for (unsigned int n = 0; n < G_N_ELEMENTS (_frames); ++n)
        {
          FrameList const& map = tag->frameListMap()[_frames[n]];
          if (!map.isEmpty())
              {
                std::string str = map.front()->toString().to8Bit();
                row_set (row, DATUM_MB_RELEASE_DATE, ustring(str)); 
              }
            }
        }

        // Check for sortname
        if (row.find (track_attributes[DATUM_MB_ALBUM_ARTIST_SORTNAME].id) == row.end())
        {
          using TagLib::ID3v2::FrameList;
          const char *id3v2frame = 0;
          if (!tag->frameListMap()["XSOP"].isEmpty()) id3v2frame = "XSOP";
          else if (!tag->frameListMap()["TSOP"].isEmpty()) id3v2frame = "TSOP";

          if (id3v2frame)
          {
            FrameList const& map = tag->frameListMap()[id3v2frame];
            if (!map.isEmpty())
            {
                std::string str = map.front()->toString().to8Bit();
                row_set (row, DATUM_MB_ALBUM_ARTIST_SORTNAME, ustring(str)); 
            }
          }
        }
  
        static boost::format fsizefmt ("%llu");

        struct stat tagstat;
        stat (name.c_str(), &tagstat);

        TagLib::ByteVector tagData = tag->render();
        std::string hash = Util::md5_hex ((char*)(tagData.data()), tagData.size())+(fsizefmt % (unsigned long long int)(tagstat.st_size)).str();
        row_set (row, DATUM_HASH, ustring(hash));
    }
  } // namespace Library

  std::string
  MetadatumId (Bmp::Library::Datum datum)
  {
    return track_attributes[datum].id;
  }

  Bmp::DB::DatumDefine
  MetadatumDefine (Bmp::Library::Datum datum)
  {
    return track_attributes[datum];
  }

} // namespace Bmp
