/***************************************************************************
 *   Copyright (C) 2003 by Sbastien Laot                                 *
 *   sebastien.laout@tuxfamily.org                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <qlayout.h>
#include <qlabel.h>
#include <qlineedit.h>
#include <qcheckbox.h>
#include <kcolorbutton.h>
#include <kiconloader.h>
#include <qiconset.h>
#include <kicondialog.h>
#include <klocale.h>
#include <qpushbutton.h>
#include <qradiobutton.h>
#include <kpushbutton.h>
#include <qvbuttongroup.h>
#include <qcombobox.h>
#include <kmessagebox.h>
#include <knuminput.h>

#include "basket.h"
#include "basketproperties.h"
#include "global.h"
#include "container.h"
#include "onclickaction.h"
#include "settings.h"

/** BasketPropertiesWidget */

BasketPropertiesWidget::BasketPropertiesWidget(Basket *basket, QWidget *parent, const char *name)
 : QWidget(parent, name)
{
	QHBoxLayout *hLay;
	QLabel      *label;

	m_basket = basket;
	QVBoxLayout *vLay = new QVBoxLayout(this, 0, 6);
	QGridLayout *gl   = new QGridLayout(vLay, 5, 3);
	gl->addItem(new QSpacerItem(0, 0, QSizePolicy::Expanding), 1, 2);

	m_name = new QLineEdit(m_basket->name(), this);
	label  = new QLabel(m_name, i18n("&Name:"), this);
	gl->addWidget(         label,  0, 0);
	gl->addMultiCellWidget(m_name, 0, 0, 1, 2);

	hLay   = new QHBoxLayout(0L, 0, 6);
	m_icon = new KIconButton(this);
	label  = new QLabel(m_icon, i18n("&Icon:"), this);
	m_icon->setIconType (KIcon::Small, KIcon::Any);
	QPushButton *button = new QPushButton(i18n("Cl&ear"), this); // Create before to know size here:
	connect( button, SIGNAL(pressed()), this, SLOT(emptyIcon()) );
	// BEGIN Icon button
	m_icon->setIcon(basket->icon());
	int minSize = button->sizeHint().height();
	// Make the icon button at least the same heigh than the other buttons for a better alignment (nicer to the eyes):
	if (m_icon->sizeHint().height() < minSize)
		m_icon->setFixedSize(minSize, minSize);
	else
		m_icon->setFixedSize(m_icon->sizeHint().height(), m_icon->sizeHint().height()); // Make it square
	if ( basket->icon().isEmpty() )
		m_icon->resetIcon();
	// END
	hLay->addWidget(m_icon);
	hLay->addWidget(button);
	gl->addWidget(label, 1, 0);
	gl->addItem(  hLay,  1, 1);

	m_color = new KColorButton(basket->color(), this);
	label   = new QLabel(m_color, i18n("&Background color:"), this);
	gl->addWidget(label,   2, 0);
	gl->addWidget(m_color, 2, 1);

	m_altColor = new KColorButton(basket->altColor(), this);
	label      = new QLabel(m_altColor, i18n("Alte&rnate background color:"), this);
	gl->addWidget(label,      3, 0);
	gl->addWidget(m_altColor, 3, 1);

	hLay = new QHBoxLayout(0L, 0, 6);
	m_hAlign = new QComboBox(this);
	label = new QLabel(m_hAlign, i18n("Items ali&gnment:"), this);
	m_hAlign->insertItem(i18n("Left"));
	m_hAlign->insertItem(i18n("Center"));
	m_hAlign->insertItem(i18n("Right"));
	m_hAlign->setCurrentItem(basket->hAlign());
	m_vAlign = new QComboBox(this);
	m_vAlign->insertItem(i18n("Top"));
	m_vAlign->insertItem(i18n("Middle"));
	m_vAlign->insertItem(i18n("Bottom"));
	m_vAlign->setCurrentItem(basket->vAlign());
	hLay->addWidget(m_hAlign);
	hLay->addWidget(m_vAlign);
	gl->addWidget(label, 4, 0);
	gl->addItem(  hLay,  4, 1);

	m_showCheckBoxes = new QCheckBox(i18n("Show items chec&k boxes"), this);
	m_showCheckBoxes->setChecked(basket->showCheckBoxes());
	vLay->addWidget(m_showCheckBoxes);

	vLay->addStretch();

	// Temporary useless option:
	m_vAlign->hide();

	m_name->setFocus();
}

BasketPropertiesWidget::~BasketPropertiesWidget()
{
}

void BasketPropertiesWidget::saveChanges()
{
	m_basket->setName(           m_name->text()                                   );
	m_basket->setIcon(           m_icon->icon()                                   );
	m_basket->setColors(         m_color->color(), m_altColor->color()            );
	m_basket->setAlign(          m_hAlign->currentItem(), m_vAlign->currentItem() );
	m_basket->setShowCheckBoxes( m_showCheckBoxes->isChecked()                    );
}

void BasketPropertiesWidget::emptyIcon()
{
	m_icon->resetIcon();
}

/** BasketPropertiesAddItemWidget */

BasketPropertiesAddItemWidget::BasketPropertiesAddItemWidget(Basket *basket, QWidget *parent, const char *name)
 : QWidget(parent, name)
{
	m_basket = basket;

	QVBoxLayout *vLay = new QVBoxLayout(this, 0, 6);

	QHBoxLayout *hLay = new QHBoxLayout(0L, 0, 6);
	m_insertAtEnd = new QComboBox(this);
	QLabel *label = new QLabel(m_insertAtEnd, i18n("&New items place:"), this);
	m_insertAtEnd->insertItem(i18n(/*"At begin (on top)"*/"On top"));
	m_insertAtEnd->insertItem(i18n(/*"At end (on bottom)"*/"On bottom"));
	m_insertAtEnd->setCurrentItem((int)basket->insertAtEnd());
	connect( m_insertAtEnd, SIGNAL(activated(int)), this, SLOT(insertSideSelected(int)) );
	hLay->addWidget(label);
	hLay->addWidget(m_insertAtEnd);
	hLay->addStretch();
	vLay->addLayout(hLay);

	m_insertAtCursorPos = new QCheckBox(i18n("Allow ins&ert items at cursor position"), this);
	m_insertAtCursorPos->setChecked(basket->insertAtCursorPos());
	vLay->addWidget(m_insertAtCursorPos);

	m_ViewFileContent = new QCheckBox*[Basket::TOTAL];

	QVButtonGroup *localFiles = new QVButtonGroup(i18n("View Content of Added Files for the Following Types"), this);
	m_ViewFileContent[Basket::FileText]  = new QCheckBox( i18n("&Text"),               localFiles );
	m_ViewFileContent[Basket::FileHTML]  = new QCheckBox( i18n("&HTML page"),          localFiles );
	m_ViewFileContent[Basket::FileImage] = new QCheckBox( i18n("&Image or animation"), localFiles );
	m_ViewFileContent[Basket::FileSound] = new QCheckBox( i18n("So&und"),              localFiles );
	vLay->addWidget(localFiles);
	vLay->addStretch();

	for (int i = 0; i < Basket::TOTAL; ++i)
		m_ViewFileContent[i]->setChecked( basket->viewFileContent((Basket::ViewFileContent)i)  );
}

BasketPropertiesAddItemWidget::~BasketPropertiesAddItemWidget()
{
}

void BasketPropertiesAddItemWidget::insertSideSelected(int index)
{
	emit insertSideChanged(index == 0);
}

void BasketPropertiesAddItemWidget::saveChanges()
{
	m_basket->setInsertAtEnd(       m_insertAtEnd->currentItem() == 1 );
	m_basket->setInsertAtCursorPos( m_insertAtCursorPos->isChecked()  );

	for (int i = 0; i < Basket::TOTAL; ++i)
		m_basket->setViewFileContent((Basket::ViewFileContent)i, m_ViewFileContent[i]->isChecked());
}

/** BasketPropertiesStackWidget */

BasketPropertiesStackWidget::BasketPropertiesStackWidget(Basket *basket, QWidget *parent, const char *name)
 : QWidget(parent, name)
{
	m_basket = basket;

	QVBoxLayout *vLay = new QVBoxLayout(this, 0, 6);
	m_isAStack = new QCheckBox(i18n("&Use this basket as a stack:"), this);
	m_isAStack->setChecked(m_basket->isAStack());
	vLay->addWidget(m_isAStack);

	m_stackOptions = new QWidget(this);
	QHBoxLayout *hLay = new QHBoxLayout(m_stackOptions, 0, 6);
	QVBoxLayout *sLay = new QVBoxLayout(0, 0, 6);
	hLay->addSpacing(20);
	hLay->addLayout(sLay);

	m_takeOnSameSide = new QVButtonGroup(i18n("Take Items (when Dragged From Systray Icon)"), m_stackOptions);
	new QRadioButton( "...same...",     m_takeOnSameSide ); // Text will be set by insertSideChanged(bool onTop)
	new QRadioButton( "...opposite...", m_takeOnSameSide );
	m_takeOnSameSide->setButton( ! m_basket->stackTakeOnSameSide() );
	insertSideChanged( ! m_basket->insertAtEnd() );
	sLay->addWidget(m_takeOnSameSide);

	m_afterDrag = new QVButtonGroup(i18n("After Dragging off an Item"), m_stackOptions);
	new QRadioButton( i18n("Do &nothing"),                         m_afterDrag );
	new QRadioButton( i18n("&Move the item at the opposite side"), m_afterDrag );
	new QRadioButton( i18n("&Delete the item"),                    m_afterDrag );
	m_afterDrag->setButton(m_basket->stackAfterDrag());
	sLay->addWidget(m_afterDrag);

	connect( m_isAStack, SIGNAL(toggled(bool)), m_stackOptions, SLOT(setEnabled(bool)) );
	m_stackOptions->setEnabled(m_isAStack->isChecked());

	vLay->addWidget(m_stackOptions);
	vLay->addStretch();
}

BasketPropertiesStackWidget::~BasketPropertiesStackWidget()
{
}

void BasketPropertiesStackWidget::insertSideChanged(bool onTop) // TODO: bool atEnd (and in other places too)
{
	QButton *same     = m_takeOnSameSide->find(/*id=*/0);
	QButton *opposite = m_takeOnSameSide->find(/*id=*/1);

	QString sameText     = i18n("%1=\"on top/bottom\"", "On the sam&e side they are inserted, ie. %1");
	QString oppositeText = i18n("%1=\"on top/bottom\"", "On the opposi&te side they are inserted, ie. %1");

	same->setText(     sameText.arg(     onTop ? i18n("on top")    : i18n("on bottom") ) );
	opposite->setText( oppositeText.arg( onTop ? i18n("on bottom") : i18n("on top")    ) );
}

void BasketPropertiesStackWidget::saveChanges()
{
	m_basket->setIsAStack(              m_isAStack->isChecked()        );
	m_basket->setStackTakeOnSameSide( ! m_takeOnSameSide->selectedId() );
	m_basket->setStackAfterDrag(        m_afterDrag->selectedId()      );
}

/** BasketPropertiesClipboardWidget */

BasketPropertiesClipboardWidget::BasketPropertiesClipboardWidget(Basket *basket, QWidget *parent, const char *name)
 : QWidget(parent, name)
{
	m_basket = basket;

	QVBoxLayout *layout = new QVBoxLayout(this, 0, 6);

	m_which = new QVButtonGroup(i18n("The Clipboard Manager Mirror"), this);
	new QRadioButton( i18n("The cl&ipboard"),                   m_which );
	new QRadioButton( i18n("The s&election"),                   m_which );
	new QRadioButton( i18n("The clipboard an&d the selection"), m_which );
	m_which->setButton(m_basket->clipboardWhich());
	layout->addWidget(m_which);

	m_maxItems = new KIntNumInput(m_basket->clipboardMaxItems(), this);
	m_maxItems->setLabel(i18n("Maximum number o&f items:"), KNumInput::AlignVCenter);
	m_maxItems->setRange(0, 999, 1, false);
	layout->addWidget(m_maxItems);

	layout->addStretch();
}

BasketPropertiesClipboardWidget::~BasketPropertiesClipboardWidget()
{
}

void BasketPropertiesClipboardWidget::saveChanges()
{
	m_basket->setClipboardWhich(    m_which->selectedId() );
	m_basket->setClipboardMaxItems( m_maxItems->value()   );
}

/** BasketPropertiesMirrorWidget */

BasketPropertiesMirrorWidget::BasketPropertiesMirrorWidget(Basket *basket, QWidget *parent, const char *name)
 : QWidget(parent, name)
{
	m_basket = basket;

	QVBoxLayout *vLay = new QVBoxLayout(this, 0, 6);
	QLabel *label = new QLabel(
		i18n("<qt>This basket mirror the content of the folder <b>%1</b>.<br>"
		     "Each files in the folder have an associated item.</qt>")
			.arg(m_basket->folderName()), this );
	vLay->addWidget(label);
	vLay->addSpacing(20);
	m_onlyNewFiles = new QCheckBox(i18n("Mi&rror only new files"), this);
	m_onlyNewFiles->setChecked(m_basket->mirrorOnlyNewFiles());
	vLay->addWidget(m_onlyNewFiles);
	vLay->addStretch();
}

BasketPropertiesMirrorWidget::~BasketPropertiesMirrorWidget()
{
}

void BasketPropertiesMirrorWidget::saveChanges()
{
	m_basket->setMirrorOnlyNewFiles( m_onlyNewFiles->isChecked() );
}

/** BasketPropertiesOnClicWidget */

BasketPropertiesOnClicWidget::BasketPropertiesOnClicWidget(Basket *basket, QWidget *parent, const char *name)
 : QWidget(parent, name)
{
	m_basket = basket;

	QVBoxLayout *vLay = new QVBoxLayout(this, 0, 6);

	m_contentOnClickAction = new QVButtonGroup(i18n("For &Text, Rich Text, Image, Animation, Color and Unknow Items"), this);
	if (Settings::singleClick()) {
		new QRadioButton(i18n("Click: copy, Ctrl+click: edit, Alt+click: open"), m_contentOnClickAction);
		new QRadioButton(i18n("Click: edit, Ctrl+click: copy, Alt+click: open"), m_contentOnClickAction);
		new QRadioButton(i18n("Click: open, Ctrl+click: edit, Alt+click: copy"), m_contentOnClickAction);
	} else {
		new QRadioButton(i18n("Double-click: copy, Ctrl+double-click: edit, Alt+double-click: open"), m_contentOnClickAction);
		new QRadioButton(i18n("Double-click: edit, Ctrl+double-click: copy, Alt+double-click: open"), m_contentOnClickAction);
		new QRadioButton(i18n("Double-click: open, Ctrl+double-click: edit, Alt+double-click: copy"), m_contentOnClickAction);
	}
	m_contentOnClickAction->setButton(basket->contentOnClickAction());
	vLay->addWidget(m_contentOnClickAction);

	m_fileOnClickAction = new QVButtonGroup(i18n("&For Sound, File, Link and Launcher Items"), this);
	if (Settings::singleClick()) {
		new QRadioButton(i18n("Click: copy, Ctrl+click: edit, Alt+click: open"), m_fileOnClickAction);
		new QRadioButton(i18n("Click: edit, Ctrl+click: copy, Alt+click: open"), m_fileOnClickAction);
		new QRadioButton(i18n("Click: open, Ctrl+click: edit, Alt+click: copy"), m_fileOnClickAction);
	} else {
		new QRadioButton(i18n("Double-click: copy, Ctrl+double-click: edit, Alt+double-click: open"), m_fileOnClickAction);
		new QRadioButton(i18n("Double-click: edit, Ctrl+double-click: copy, Alt+double-click: open"), m_fileOnClickAction);
		new QRadioButton(i18n("Double-click: open, Ctrl+double-click: edit, Alt+double-click: copy"), m_fileOnClickAction);
	}
	m_fileOnClickAction->setButton(basket->fileOnClickAction());
	vLay->addWidget(m_fileOnClickAction);

	vLay->addStretch();
}

BasketPropertiesOnClicWidget::~BasketPropertiesOnClicWidget()
{
}

void BasketPropertiesOnClicWidget::saveChanges()
{
	m_basket->setContentOnClickAction( m_contentOnClickAction->selectedId/*currentItem*/() );
	m_basket->setFileOnClickAction(    m_fileOnClickAction->selectedId/*currentItem*/()    );
}

/** BasketPropertiesDialog */

BasketPropertiesDialog::BasketPropertiesDialog(Basket *basket, QWidget *parent)
 : KDialogBase(KDialogBase::Tabbed, i18n("Basket Properties"),
               KDialogBase::Ok | KDialogBase::Apply | KDialogBase::Cancel, KDialogBase::Ok, parent)
{
	m_basket = basket;

	QFrame *page1 = addPage(i18n("&Properties"));
	QVBoxLayout *layout = new QVBoxLayout(page1);
	m_properties = new BasketPropertiesWidget(basket, page1);
	layout->addWidget(m_properties);

	QFrame *page2 = addPage(i18n("Add Item Polic&y"));
	layout = new QVBoxLayout(page2);
	m_addPolicy = new BasketPropertiesAddItemWidget(basket, page2);
	layout->addWidget(m_addPolicy);

	QFrame *page3 = addPage(i18n("C&lick Actions"));
	layout = new QVBoxLayout(page3);
	m_onClic = new BasketPropertiesOnClicWidget(basket, page3);
	layout->addWidget(m_onClic);

	if (m_basket->isAClipboard()) {
		QFrame *page5 = addPage(i18n("Clip&board"));
		layout = new QVBoxLayout(page5);
		m_clipboard = new BasketPropertiesClipboardWidget(basket, page5);
		layout->addWidget(m_clipboard);
	}

	QFrame *page4 = addPage( m_basket->isAMirror() ? i18n("&Mirror") : i18n("&Stack") );
	layout = new QVBoxLayout(page4);
	if (m_basket->isAMirror()) {
		m_mirror = new BasketPropertiesMirrorWidget(basket, page4);
		layout->addWidget(m_mirror);
	} else {
		m_stack = new BasketPropertiesStackWidget(basket, page4);
		layout->addWidget(m_stack);
		connect( m_addPolicy, SIGNAL(insertSideChanged(bool)), m_stack, SLOT(insertSideChanged(bool)) );
	}
}

BasketPropertiesDialog::~BasketPropertiesDialog()
{
}

void BasketPropertiesDialog::slotApply()
{
	m_properties->saveChanges();
	m_addPolicy->saveChanges();
	m_onClic->saveChanges();

	if (m_basket->isAMirror())
		m_mirror->saveChanges();
	else
		m_stack->saveChanges();

	if (m_basket->isAClipboard())
		m_clipboard->saveChanges();

	m_basket->save();

	if (Global::tray)
		Global::tray->updateToolTip();

	KDialogBase::slotApply();
}

void BasketPropertiesDialog::slotOk()
{
	slotApply();
	KDialogBase::slotOk();
}

#include "basketproperties.moc"
