# Arch Perl library, Copyright (C) 2004 Mikhael Goikhman
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

use 5.005;
use strict;

package AXP::ModuleFinder;

use Arch::Util qw(load_file);

sub new ($) {
	my $class = shift;
	my $self = {
		modules => {},
		modules_with_pod => {},
	};
	return bless $self, $class;
}

sub _get_cached_modules ($$;$) {
	my $self = shift;
	my $name = shift || die;
	my $sections = shift || [ qw(Arch AXP ArchZoom ArchWay) ];
	my $sections_index = join(',', @$sections);

	my $found = exists $self->{$name}->{$sections_index};
	my $modules = $self->{$name}->{$sections_index} ||= {};
	return ($found, $modules, $sections);
}

sub find_modules ($;$) {
	my $self = shift;
	my $sections0 = shift;

	my ($found, $modules, $sections) =
		$self->_get_cached_modules("modules", $sections0);
	return $modules if $found;

	foreach my $dir (@INC) {
		foreach my $section (@$sections) {
			my $subdir = "$dir/$section";
			next unless -d $subdir;
			my @files = map { glob "$subdir/$_*.pm" } ("", "*/", "*/*/");
			push @files, "$subdir.pm" if -f "$subdir.pm";
			foreach my $file (@files) {
				my $class = substr($file, length($dir) + 1);
				$class =~ s/\.pm$//;
				$class =~ s!/!::!g;
				$modules->{$class} = $file;
			}
		}
	}
	return wantarray ? %$modules : $modules;
}

sub find_modules_with_pod ($;$) {
	my $self = shift;
	my $sections0 = shift;
	my $remove_flag = shift || 0;

	my ($found, $modules, $sections) =
		$self->_get_cached_modules("modules_with_pod", $sections0);
	return $modules if $found;

	%$modules = ($self->find_modules($sections));
	foreach my $class (keys %$modules) {
		my $file = $modules->{$class};
		my $content = load_file($file);
		unless ($content =~ /^=head1/m) {
			$modules->{$class} = undef;
			delete $modules->{$class} if $remove_flag;
		}
	}
	return wantarray ? %$modules : $modules;
}

1;

__END__

=head1 NAME

AXP::ModuleFinder - find all Arch related perl modules

=head1 SYNOPSIS 

    use AXP::ModuleFinder;
    my $finder = AXP::ModuleFinder->new;
    my %modules = $finder->find_modules;

    while (my ($class, $file) = each %modules) {
        print "$class -> $file\n";
    }

    my $modules = $finder->find_modules_with_pod;
    $modules = $finder->find_modules([ "Arch", "ArchWay" ]);

=head1 DESCRIPTION

This class helps to find all Arch related perl modules in the include
directories, i.e. their names and locations on the disk.

=head1 METHODS

=over 4

=item B<find_modules> [I<limit>]

Return hash (or hashref in scalar context) of all visible perl modules.
The hash pairs are I<module_class> => I<module_file_name>.

By default the I<limit> is ["Arch", "AXP", "ArchZoom", "ArchWay"], but you
may supply another limit that is arrayref of perl class sections.

=item B<find_modules_with_pod> [I<limit>] [I<remove>]

The same as B<find_modules>, but only returns modules that have inline
documentation. If I<remove> flag is set then the modules without the
pod are removed, otherwise (by default) their file value is set to undef.

=back

=head1 AUTHORS

Mikhael Goikhman (migo@homemail.com--Perl-GPL/axp--devel).

=head1 SEE ALSO

For more information, see L<axp>.

=cut
