/*
 * trust_anchors.c - implements trust anchors.
 * Copyright (c) 2008, NLnet Labs. All rights reserved.
 * This software is open source.
 * For license see doc/LICENSE.
 */

#include <config.h>

#include "trust_anchors.h"
#include "log.h"
#include "util.h"

/** Check if a trust anchor is a DNSKEY */
static uint8_t
rr_is_dnskey(ldns_rr* rr)
{
	if (!rr)
		return 0;
	return (ldns_rr_get_type(rr) == LDNS_RR_TYPE_DNSKEY);
}

/** Check if a trust anchor has the specified flag */
uint16_t
rr_has_flag(ldns_rr* rr, uint16_t flag)
{
	ldns_rdf* rdf;
	uint16_t flags;

	if (!rr_is_dnskey(rr))
		return 0;

	rdf = ldns_rr_dnskey_flags(rr);
	flags = ldns_read_uint16(ldns_rdf_data(rdf));
	return (flags & flag);
}

/** revoke a trust anchor */
void
revoke_dnskey(ta_t* ta, int off)
{
	ldns_rdf* rdf;
	uint16_t flags;

	if (!ta || !ta->rr)
		return;

	if (rr_is_dnskey(ta->rr))
	{
		rdf = ldns_rr_dnskey_flags(ta->rr);
		flags = ldns_read_uint16(ldns_rdf_data(rdf));

		if (off && rr_has_flag(ta->rr, LDNS_KEY_REVOKE_KEY))
			flags ^= LDNS_KEY_REVOKE_KEY; /* flip */
		else
			flags |= LDNS_KEY_REVOKE_KEY;
		rdf = ldns_native2rdf_int16(LDNS_RDF_TYPE_INT16, flags);
		if (!ldns_rr_dnskey_set_flags(ta->rr, rdf))
		{
			char* str = ldns_rdf2str(ldns_rr_owner(ta->rr));
			if (off)
				error("trust anchor [%s, DNSKEY, id=%i] "
				      "unrevocation failed",
					str, ldns_calc_keytag(ta->rr));
			else
				error("trust anchor [%s, DNSKEY, id=%i] "
				      "revocation failed",
					str, ldns_calc_keytag(ta->rr));
			free(str);
		}
	}
}

/** Check if RR is a ZSK */
uint16_t
rr_is_zsk(ldns_rr* rr)
{
	if (!rr)
		return 0;
	return rr_has_flag(rr, LDNS_KEY_ZONE_KEY);
}


/** Check if RR is a KSK */
uint16_t
rr_is_dnskey_sep(ldns_rr* rr)
{
	if (!rr)
		return 0;
	return rr_has_flag(rr, LDNS_KEY_SEP_KEY);
}

/** Check if trust anchor is a DS */
uint16_t
rr_is_ds(ldns_rr* rr)
{
	if (!rr)
		return 0;
	return (ldns_rr_get_type(rr) == LDNS_RR_TYPE_DS);
}

/** Check if RR is a trust anchor */
uint16_t
rr_is_trust_anchor(ldns_rr* rr)
{
	return (rr_is_ds(rr) || rr_is_dnskey_sep(rr));
}

/** Create a new trust anchor */
ta_t*
create_trustanchor(ldns_rr* rr, ta_state_t s)
{
	ta_t* ta = (ta_t*) malloc(sizeof(ta_t));

	if (!ta)
	{
		ldns_rr_free(rr);
		return NULL;
	}
	ldns_rr2canonical(rr);
	ta->rr = rr;
	ta->s = s;
	ta->pending_count = 0;
	ta->fetched = 0;
	ta->revoked = 0;
	ta->last_state_change = time(NULL);
	return ta;
}

/** Clone a trust anchor */
ta_t*
ta_clone(ta_t* orig)
{
	ta_t* clone;
	clone = (ta_t*) malloc(sizeof(ta_t));
	if (!clone)
		return NULL;

	clone->rr = ldns_rr_clone(orig->rr);
	clone->s = orig->s;
        clone->last_state_change = orig->last_state_change;

	return clone;
}

/** Get the string version of the trust anchor state */
const char*
trustanchor_state2str(ta_state_t s)
{
	switch (s) {
		case STATE_START:	return "  START  ";
		case STATE_ADDPEND:	return " ADDPEND ";
		case STATE_VALID:	return "  VALID  ";
		case STATE_MISSING:	return " MISSING ";
		case STATE_REVOKED:	return " REVOKED ";
		case STATE_REMOVED:	return " REMOVED ";
		default:		return "UNDEFINED";
	}

	/* not reached */
	return " UNKNOWN ";
}

/* Check if the holddown time has already exceeded
 * setting: add-holddown: add holddown timer
 * setting: del-holddown: del holddown timer
 */
int
check_holddown(ta_t* ta, unsigned int holddown)
{
	unsigned int elapsed = 0;
	char* str;
	time_t now;
	if (!ta)
		return 0;
	now = time(NULL);
	elapsed = (unsigned int) difftime(now, ta->last_state_change);
	if (elapsed > holddown) {
		return (int) (elapsed-holddown);
	}
	str = ldns_rdf2str(ldns_rr_owner(ta->rr));
	verbos(1, "trust anchor [%s, DNSKEY, id=%i] holddown time %i seconds "
		  "to go", str, ldns_calc_keytag(ta->rr),
		(int) (holddown-elapsed));
	free(str);
	return 0;
}

/** Set last_state_change to now */
void
reset_holddown(ta_t* ta)
{
	if (ta)
		ta->last_state_change = time(NULL);
}

/** Set the state for this trust anchor */
void
set_trustanchor_state(ta_t* ta, ta_state_t s)
{
	if (ta) {
		char* str = ldns_rdf2str(ldns_rr_owner(ta->rr));
		notice("trust anchor [%s, DNSKEY, id=%i] update: %s -> %s",
			str, ldns_calc_keytag(ta->rr),
			trustanchor_state2str(ta->s), trustanchor_state2str(s));
		ta->s = s;
		reset_holddown(ta);
		free(str);
	}
}

/** Set fetched value */
void
seen_trustanchor(ta_t* ta, int seen)
{
	if (ta) {
		ta->fetched = seen;
		ta->pending_count++;
	}
}

/** Set revoked value */
void
seen_revoked_trustanchor(ta_t* ta, int revoked)
{
	if (ta)
		ta->revoked = revoked;
}

/** Get the state of this trust anchor */
ta_state_t
get_trustanchor_state(ta_t* ta)
{
	if (!ta)
		return -1;
	return ta->s;
}

/** Create an empty trust point */
tp_t*
create_trustpoint(char* rr_dname, ldns_rr_class rr_class)
{
	tp_t* tp = (tp_t*) malloc(sizeof(tp_t));
	if (!tp)
		return NULL;

	tp->name = rr_dname;
	tp->klass = rr_class;
	tp->dname = ldns_dname_new_frm_str((const char*) rr_dname);
	if (!tp->dname)
	{
		tp_free(tp);
		return NULL;
        }
	ldns_dname2canonical(tp->dname);

	tp->file = NULL;

	tp->count = 0;
	tp->capacity = 0;
	tp->anchorlist = NULL;

	tp->last_queried = 0;
	tp->query_failed = 0;
	tp->query_interval = 0;
	tp->retry_time = 0;
	tp->valid = 0;
	tp->missing = 0;
	return tp;
}

/** Create an empty trust point */
tp_set_t*
create_trustpoint_set(void)
{
	tp_set_t* tp_set = (tp_set_t*) malloc(sizeof(tp_set_t));
	if (!tp_set)
		return NULL;
	tp_set->sep = rbt_create(tp_compare);
	return tp_set;
}

/** Get the first trust point in the set */
int
tp_have_valid(tp_t* tp)
{
	if (!tp)
		return 0;
	return (tp->valid + tp->missing);
}

/** Get the number of keys in a trust point */
size_t
tp_tacount(tp_t* tp)
{
	if (!tp)
		return 0;
	return tp->count;
}

/** Get the capacity of a trust point */
size_t
tp_tacapacity(tp_t* tp)
{
	if (!tp)
		return 0;
	return tp->capacity;
}

/** Add a trust point */
tp_t*
tp_add(tp_set_t* tp_set, char* rr_dname, ldns_rr_class rr_class)
{
	tp_t* tp = create_trustpoint(rr_dname, rr_class);
        if (!tp)
                return NULL;
        if (rbt_insert(tp_set->sep, (const void*) tp) == 1)
                return tp;
        /* rbt_insert returned -1 or 0 */
	free(tp);
        return NULL;
}

/** Add a trust anchor to a trust point */
ta_t*
add_trustanchor(tp_t* tp, ta_t* ta)
{
	ta_t* ta_local;
	size_t tacount, tacapacity;

	if (!tp)
		return NULL;

	/* look for duplicate trust anchors */
	ta_local = search_trustanchor(tp, ta);
	if (ta_local)
		return NULL;

	/* enough capacity to add trust anchor? */
	tacount = tp_tacount(tp);
	tacapacity = tp_tacapacity(tp);
	if ((tacount+1) > tacapacity) {
		ta_t** tas;
		if (!tacapacity)
			tacapacity = TA_KEY_CAPACITY_INIT;
		else
			tacapacity *= 2;

		tas = (ta_t**) realloc((tp->anchorlist),
			tacapacity*sizeof(ta_t*));
		if (!tas)
			return NULL;
		tp->anchorlist = tas;
		tp->capacity = tacapacity;
	}

	/* add trust anchor */
	tp->anchorlist[tacount] = ta;
	tp->count = tacount+1;
	if (ta->s == STATE_VALID)
		tp->valid += 1;
	else if (ta->s == STATE_MISSING)
		tp->missing += 1;
	return tp->anchorlist[tacount];
}

/** Check if a line contains data (besides comments) */
int
str_contains_data(char* str, char comment)
{
	while (*str != '\0')
	{
		if (*str == comment || *str == '\n')
			return 0;
		if (*str != ' ' && *str != '\t')
                        return 1;
		str++;
        }
	return 0;
}

/** Add a trust anchor from RR */
ta_t*
add_trustanchor_frm_rr(tp_set_t* tp_set, ldns_rr* rr, tp_t** tpt, int state)
{
	uint16_t ksk = rr_is_dnskey_sep(rr);
	uint16_t zsk = rr_is_zsk(rr);
	uint16_t ds = rr_is_ds(rr);
	ldns_rdf* dname = NULL;
	char* dname_str = NULL;

	dname = ldns_rr_owner(rr);
	dname_str = ldns_rdf2str(dname);

	/* from statefile, only read KSKs, otherwise, read KSKs, ZSKs and DSs */
	if ((state && ksk) || (!state && (ksk || zsk || ds)))
	{
		tp_t* tp = NULL;
		ta_t* ta_local = NULL;
		ta_t* ta = create_trustanchor(ldns_rr_clone(rr), STATE_START);
		if (!ta)
			return NULL;

		/* search corresponding trust point */
		tp = tp_search(tp_set, dname, LDNS_RR_CLASS_IN);
		if (!tp)
			tp = tp_add(tp_set, dname_str, LDNS_RR_CLASS_IN);
		else /* dname_str not needed */
			free(dname_str);

		*tpt = tp;

		if (!tp)
		{
			debug(("could not find or create trust point %s",
				dname_str));
			free_trustanchor(ta);
			return NULL;
		}

		/* add the trust anchor */
		ta_local = search_trustanchor(tp, ta);
		if (!ta_local)
			ta_local = add_trustanchor(tp, ta);
		else
			free_trustanchor(ta);
		if (!ta_local)
		{
			debug(("could not find or add trust anchor "
				"[%s, DNSKEY, id=%i]", dname_str,
				ldns_calc_keytag(rr)));
			free_trustanchor(ta);
			return NULL;
		}

		if (zsk && !ksk)
			warning("trust anchor file contains ZSKs, will be "
				"removed if there are KSKs known");
		/* added or already present */
		return ta_local;
	}
	warning("trust anchor file contains non-trustanchors, will be "
			"removed");
	return NULL;
}

/** Add a trust anchor from string */
ta_t*
add_trustanchor_frm_str(tp_set_t* tp_set, char* str, tp_t** tpt, int state)
{
	ldns_rr* rr;
	ldns_status lstatus = LDNS_STATUS_OK;
	ta_t* ta = NULL;
	if (!str_contains_data(str, ';'))
		return NULL; /* empty line */
	if (LDNS_STATUS_OK !=
		(lstatus = ldns_rr_new_frm_str(&rr, str, 0, NULL, NULL)))
	{
		error("ldns error while converting string to RR");
		return NULL;
	}
	ta = add_trustanchor_frm_rr(tp_set, rr, tpt, state);

	/* rr cloned */
	ldns_rr_free(rr);
	return ta;
}

time_t
next_refresh(tp_t* tp)
{
	time_t refresh = time(NULL);

	if (!tp)
		return refresh;

	if (tp->query_failed)
		refresh = tp->last_queried + tp->retry_time;
	else
		refresh = tp->last_queried + tp->query_interval;

	return refresh;
}

/* Compare two RR buffers, skipping the REVOKED bit */
static int
ldns_rr_compare_wire_skip_revbit(ldns_buffer* rr1_buf, ldns_buffer* rr2_buf)
{
	size_t rr1_len, rr2_len, min_len, i, offset;
	rr1_len = ldns_buffer_capacity(rr1_buf);
	rr2_len = ldns_buffer_capacity(rr2_buf);
	/* jump past dname (checked in earlier part) and especially past TTL */
	offset = 0;
	while (offset < rr1_len && *ldns_buffer_at(rr1_buf, offset) != 0)
		offset += *ldns_buffer_at(rr1_buf, offset) + 1;
	/* jump to rdata section (PAST the rdata length field */
	offset += 11;
	min_len = (rr1_len < rr2_len) ? rr1_len : rr2_len;
	/* compare RRs RDATA byte for byte. */
	for(i = offset; i < min_len; i++)
	{
		uint8_t *rdf1, *rdf2;
		rdf1 = ldns_buffer_at(rr1_buf, i);
		rdf2 = ldns_buffer_at(rr2_buf, i);
		if (i==(offset+1))
		{
			/* this is the second part of the flags field */
			*rdf1 = *rdf1 | LDNS_KEY_REVOKE_KEY;
			*rdf2 = *rdf2 | LDNS_KEY_REVOKE_KEY;
		}
		if (*rdf1 < *rdf2)	return -1;
		else if (*rdf1 > *rdf2)	return 1;
        }
	return 0;
}

/** Compare two RRs skipping the REVOKED bit */
static int
ldns_rr_compare_skip_revbit(const ldns_rr* rr1, const ldns_rr* rr2)
{
	int result;
	size_t rr1_len, rr2_len;
	ldns_buffer* rr1_buf;
	ldns_buffer* rr2_buf;

	result = ldns_rr_compare_no_rdata(rr1, rr2);
	if (result == 0)
	{
		rr1_len = ldns_rr_uncompressed_size(rr1);
		rr2_len = ldns_rr_uncompressed_size(rr2);
		rr1_buf = ldns_buffer_new(rr1_len);
		rr2_buf = ldns_buffer_new(rr2_len);
		if (ldns_rr2buffer_wire_canonical(rr1_buf, rr1,
			LDNS_SECTION_ANY) != LDNS_STATUS_OK)
		{
			ldns_buffer_free(rr1_buf);
			ldns_buffer_free(rr2_buf);
			return 0;
		}
		if (ldns_rr2buffer_wire_canonical(rr2_buf, rr2,
			LDNS_SECTION_ANY) != LDNS_STATUS_OK) {
			ldns_buffer_free(rr1_buf);
			ldns_buffer_free(rr2_buf);
			return 0;
		}
		result = ldns_rr_compare_wire_skip_revbit(rr1_buf, rr2_buf);
		ldns_buffer_free(rr1_buf);
		ldns_buffer_free(rr2_buf);
	}
	return result;
}

/** Compare two trust anchors */
int
ta_compare(const void* a, const void *b)
{
	ta_t* ka = (ta_t*) a;
	ta_t* kb = (ta_t*) b;
	int result;
	/* only if a && b are not NULL and both are from type DNSKEY or DS,
	 * we can do something useful.
	 */
	if (!ka && !kb)		result = 0;
	else if (!ka || !kb)	result = -1;
	else if (ldns_rr_get_type(ka->rr) != ldns_rr_get_type(kb->rr))
		result = (int)ldns_rr_get_type(ka->rr) -
			(int)ldns_rr_get_type(kb->rr);
	else if (ldns_rr_get_type(ka->rr) == LDNS_RR_TYPE_DNSKEY)
		result = ldns_rr_compare_skip_revbit(ka->rr, kb->rr);
	else if (ldns_rr_get_type(ka->rr) == LDNS_RR_TYPE_DS)
		result = ldns_rr_compare(ka->rr, kb->rr);
	else	result = -1;
	return result;
}

/** Compare two trust points (dname, class) */
int
tp_compare(const void* a, const void *b)
{
	tp_t* ta = (tp_t*) a;
	tp_t* tb = (tp_t*) b;

	log_assert(ta != NULL && tb != NULL);

	if (ta->klass != tb->klass)
		return (int) ta->klass - (int) tb->klass;
	return ldns_dname_compare(ta->dname, tb->dname);
}

/** Search a trust anchor */
ta_t*
search_trustanchor(tp_t* tp, ta_t* ta)
{
	size_t i;
	if (!tp || !ta)
		return NULL;
	for(i=0; i < tp->count; i++)
	{
		if (!ta_compare((const void*) tp->anchorlist[i],
				(const void*) ta))
			return tp->anchorlist[i];
	}
	return NULL;
}

/** Search a trust point */
tp_t*
tp_search(tp_set_t* tp_set, ldns_rdf* rr_dname, ldns_rr_class rr_class)
{
        rbnode_t* n = NULL;
	tp_t* tp;
	char* str = ldns_rdf2str(rr_dname);
	tp = create_trustpoint(str, rr_class);
	if (tp_set && tp_set->sep)
		n = rbt_search(tp_set->sep, tp);
	tp_free((const void*) tp);
	if (!n)
		return NULL;
	return (tp_t*) n->key;
}

/** Remove a trust anchor from the memory */
void
free_trustanchor(ta_t* ta)
{
	if (ta)
	{
		if (ta->rr)
			ldns_rr_free(ta->rr);
		free(ta);
	}
}

/** Remove a trust point from the memory */
void
tp_free(const void* tp)
{
	if (tp)
	{
		size_t i;
		tp_t* tpdel = (tp_t*) tp;
		for (i=0; i < tpdel->count; i++)
			free_trustanchor(tpdel->anchorlist[i]);
		ldns_rdf_deep_free(tpdel->dname);
		free(tpdel->name);
		free(tpdel->anchorlist);
		free(tpdel);
	}
}

/** Remove a trust point set from the memory */
void
free_trustpoints(tp_set_t* tp_set)
{
	if (tp_set)
	{
		rbt_free(tp_set->sep, tp_free);
		free(tp_set);
	}
}

#ifdef TDEBUG

/** Calculate keytag prior to revocation */
static uint16_t
rr_calc_old_keytag(ta_t* ta)
{
	uint16_t keytag;
	if (ta->s != STATE_REVOKED && ta->s != STATE_REMOVED)
		return 0;

	/* Couldn't we just return keytag - 128? */
	/* No, because keytag does not change for RSA-MD5 */

	revoke_dnskey(ta, 1); /* unrevoke */
	keytag = ldns_calc_keytag(ta->rr);
	revoke_dnskey(ta, 0); /* revoke */
	return keytag;
}

/** Print all trust anchors */
void
debug_print_trustpoints(tp_set_t* tp_set)
{
	if (!tp_set)
		debug(("no trust anchors configured"));
	else
	{
		rbnode_t* node = rbt_first(tp_set->sep);
		size_t i;

		while (node)
		{
			tp_t* tp = (tp_t*) node->key;
			debug(("printing trust anchors for %s (%u ksk):", tp->name,
				(tp->valid + tp->missing)));

			for (i=0; i < tp->count; i++)
			{
				char* rr = ldns_rr2str(tp->anchorlist[i]->rr);
				int len = strlen(rr);
				uint16_t old_keytag =
				 rr_calc_old_keytag(tp->anchorlist[i]);
				char* was_keytag = (char*)
						malloc(sizeof(char)*10);
				rr[len-1] = '\0';
				sprintf(was_keytag, " (%i)", old_keytag);

				debug(("%s DNSKEY[%i]%s:\t%s;;[state:%i,%i,%u]",
				 trustanchor_state2str(tp->anchorlist[i]->s),
				 ldns_calc_keytag(tp->anchorlist[i]->rr),
				 old_keytag?was_keytag:"",
				 rr,
				 tp->anchorlist[i]->s,
				 tp->anchorlist[i]->pending_count,
				 (unsigned int) tp->anchorlist[i]->last_state_change
				));

				free(rr);
				free(was_keytag);
			}
			node = rbt_successor(node);
		}
        }
}
#endif /* TDEBUG */
