/****************************************************************
**
** Attal : Lords of Doom
**
** imageTheme.cpp
** manage images in theme
**
** Version : $Id: imageTheme.cpp,v 1.26 2004/12/18 21:02:20 lusum Exp $
**
** Author(s) : Pascal Audoux
**
** Date : 10/06/2001
**
** Licence :
**	This program is free software; you can redistribute it and/or modify
**   	it under the terms of the GNU General Public License as published by
**     	the Free Software Foundation; either version 2, or (at your option)
**      any later version.
**
**	This program is distributed in the hope that it will be useful,
** 	but WITHOUT ANY WARRANTY; without even the implied warranty of
**	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**	GNU General Public License for more details.
**
****************************************************************/

#include "imageTheme.h"

// generic include files
// include files for QT
#include <qimage.h>
#include <qptrlist.h>
#include <qpixmap.h>
#include <qpointarray.h>
#include <qcanvas.h>
// application specific include files
#include "libCommon/dataTheme.h"
#include "libCommon/genericMapCreature.h"
#include "libCommon/log.h"
#include "libCommon/unit.h"

#include "libClient/attalSound.h"

ImageTheme ImageTheme;

extern DataTheme DataTheme;
extern QString IMAGE_PATH;

//
// ----- CreaturePixmap -----
//

CreaturePixmap::CreaturePixmap( QPtrList<QPixmap> pix, QPtrList<QPoint> hotspots )
{
	_pixmaps[0] = new QCanvasPixmapArray( pix, hotspots );

	QPtrList<QPixmap> newList;
	for( uint i=0; i<pix.count(); i++ ) {
		QCanvasPixmap oldPix( *pix.at(i), *hotspots.at(i) );
		QCanvasPixmap * newPix;
		QImage newImg;
		if( !oldPix.isNull() ) {
			newImg = oldPix.convertToImage().mirror( true, false );
			newPix = new QCanvasPixmap( newImg );
			newList.append( newPix );
		} else {
			newList.append( new QPixmap() );
		}
	}
	_pixmaps[1] = new QCanvasPixmapArray( newList, hotspots );
}

CreaturePixmap::CreaturePixmap( const QValueList<QPixmap> & pix, const QPointArray & hotspots )
{
	_pixmaps[0] = new QCanvasPixmapArray( pix, hotspots );

	QValueList<QPixmap> newList;
	for( uint i=0; i<pix.count(); i++ ) {
		QCanvasPixmap oldPix( pix[ i ], hotspots.at(i) );
		QImage newImg;
		if( !oldPix.isNull() ) {
			newImg = (QImage) oldPix.convertToImage().mirror( true, false );
			QCanvasPixmap newPix(newImg);
			newList.append( newPix ) ;
		} else {
			newList.append( QPixmap() );
		}
	}
	_pixmaps[1] = new QCanvasPixmapArray( newList, hotspots );
}

CreaturePixmap::~CreaturePixmap()
{
	if( _pixmaps[0] ) {
		delete _pixmaps[0];
	}

	if( _pixmaps[1] ) {
		delete _pixmaps[1];
	}
}

QCanvasPixmapArray * CreaturePixmap::operator[]( uint i )
{
	return _pixmaps[i%2];
}

//
// ----- MapCreaturePixmap -----
//

MapCreaturePixmap::MapCreaturePixmap( QPtrList<QPixmap> pix, QPtrList<QPoint> hotspots )
{
	_pixmaps[0] = new QCanvasPixmapArray( pix, hotspots );

	QPtrList<QPixmap> newList;
	for( uint i=0; i<pix.count(); i++ ) {
		QCanvasPixmap oldPix( * pix.at( i ), * hotspots.at(i) );
		QCanvasPixmap * newPix;
		QImage newImg;
		if( !oldPix.isNull() ) {
			newImg = oldPix.convertToImage().mirror( true, false );
			newPix = new QCanvasPixmap( newImg );
			newList.append( newPix );
		} else {
			newList.append( new QPixmap() );
		}
	}
	_pixmaps[1] = new QCanvasPixmapArray( newList, hotspots );
}

MapCreaturePixmap::MapCreaturePixmap( const QValueList<QPixmap> & pix, const QPointArray & hotspots )
{
	_pixmaps[0] = new QCanvasPixmapArray( pix, hotspots );

	QValueList<QPixmap> newList;
	for( uint i=0; i<pix.count(); i++ ) {
		QCanvasPixmap oldPix( pix[ i ], hotspots.at(i) );
		QImage newImg;
		if( !oldPix.isNull() ) {
			newImg = oldPix.convertToImage().mirror( true, false );
			QCanvasPixmap newPix(newImg);
			newList.append( newPix ) ;
		} else {
			newList.append( QPixmap() );
		}
	}
	_pixmaps[1] = new QCanvasPixmapArray( newList, hotspots );

}

MapCreaturePixmap::~MapCreaturePixmap()
{
	if( _pixmaps[0] ) {
		delete _pixmaps[0];
	}

	if( _pixmaps[1] ) {
		delete _pixmaps[1];
	}
}

QCanvasPixmapArray * MapCreaturePixmap::operator[]( uint i )
{
	return _pixmaps[i%2];
}


//
// ----- ImageTheme -----
//

ImageTheme::ImageTheme()
{
	cells = 0;
	lords = 0;
	bases = 0;
	buildings = 0;
	insideBuildings = 0;
	skills = 0;
	transition = 0;
	decoration = 0;
	creatures = 0;
	artefacts = 0;
	events = 0;
	mapCreatures = 0;
	_insideBase = 0;
	_artefactIcon = 0;
	_resourceIcon = 0;
	_resourceSmallIcon = 0;
	_flags = 0;
	_mapFlags = 0;
	_sound = 0;
	_isLoaded = false;
}

ImageTheme::~ImageTheme()
{
	clear();
}

QPixmap * ImageTheme::getInsideBase( uint num )
{
	QPixmap * ret = 0;
	if( num < DataTheme.bases.count() ) {
		if( _insideBase[num] == 0 ) {
			QString name = IMAGE_PATH + "base/insideBase_" + QString::number( num ) + ".png";
			_insideBase[num] = new QPixmap( name );
		}
		ret = _insideBase[num];
	}

	return ret;
}

QPixmap * ImageTheme::getArtefactIcon( uint num )
{
	QPixmap * ret;
	if( num < DataTheme.artefacts.count() ) {
		if( _artefactIcon[num] == 0 ) {
			QString name = IMAGE_PATH + "artefacts/artefactIcon_" + QString::number( num ) + ".png";
			_artefactIcon[num] = new QPixmap( name );
		}
		ret = _artefactIcon[num];
	} else {
		logEE( "artefact %d pixmap not found", num );
		ret = new QPixmap(); /// XXX: memory leak...
	}

	return ret;
}

QPixmap * ImageTheme::getResourceIcon( uint num )
{
	QPixmap * ret;
	if( num < DataTheme.resources.count() ) {
		if( _resourceIcon[num] == 0 ) {
			QString name;
			name.sprintf( "ressources/ress_%02d.png", num + 1 );
			_resourceIcon[num] = new QPixmap( IMAGE_PATH + name );
		}
		ret = _resourceIcon[num];
	} else {
		logEE( "artefact %d pixmap not found", num );
		ret = new QPixmap();
	}

	return ret;
}

QPixmap * ImageTheme::getResourceSmallIcom( uint num )
{
	QPixmap * ret;
	if( num < DataTheme.resources.count() ) {
		if( _resourceSmallIcon[num] == 0 ) {
			QString name;
			name.sprintf( "ressources/smallResource_%02d.png", num + 1 );
			_resourceSmallIcon[num] = new QPixmap( IMAGE_PATH + name );
		}
		ret = _resourceSmallIcon[num];
	} else {
		logEE( "artefact %d pixmap not found", num );
		ret = new QPixmap();
	}

	return ret;
}

bool ImageTheme::init()
{
	_isLoaded = initSkills();
	_isLoaded = _isLoaded && initCreatures();
	_isLoaded = _isLoaded && initCells();
	_isLoaded = _isLoaded && initBuildings();
	_isLoaded = _isLoaded && initDecorations();
	_isLoaded = _isLoaded && initTransitions();
	_isLoaded = _isLoaded && initArtefacts();
	_isLoaded = _isLoaded && initResources();
	_isLoaded = _isLoaded && initBonus();
	_isLoaded = _isLoaded && initChest();
	_isLoaded = _isLoaded && initMapCreatures();
	_isLoaded = _isLoaded && initEvents();
	_isLoaded = _isLoaded && initTeams();
	_isLoaded = _isLoaded && initLords();
	_isLoaded = _isLoaded && initBases();
	_isLoaded = _isLoaded && initSound();

	return isLoaded();
}

void ImageTheme::save()
{

}

void ImageTheme::clear()
{
	if( cells ) {
		delete cells;
	}
	cells = 0;

	if( lords ) {
		delete lords;
	}
	lords = 0;
	
	if( bases ) {
		delete bases;
	}
	bases = 0;

	if( buildings ) {
		delete buildings;
	}
	buildings = 0;

	if( insideBuildings ) {
		delete insideBuildings;
	}
	insideBuildings = 0;

	if( skills ) {
		delete skills;
	}
	skills = 0;

	if( transition ) {
		delete transition;
	}
	transition = 0;

	if( decoration ) {
		for( uint i = 1; i < DataTheme.decorations.count(); i++ ) {
			delete decoration[ i - 1 ];
		}
		delete [] decoration;
	}
	decoration = 0;

	if( creatures ) {
		int cpt = 0;
		uint i, j;
		uint nbRaces = DataTheme.countRace();
		uint nbCreatures;
		for( i = 0; i < nbRaces; i++ ) {
			nbCreatures = DataTheme.creatures.getRace( i )->count();
			for( j = 0; j < nbCreatures; j++ ) {
				delete creatures[cpt];
				cpt++;
				//XXX: ?? delete creatures;
			}
		}
		delete [] creatures;
	}
	creatures = 0;

	if( artefacts ) {
		delete artefacts;
	}

	if( _artefactIcon ) {
		uint i;
		uint nbArtefacts = DataTheme.artefacts.count();
		for( i = 0; i < nbArtefacts; i++ ) {
			if( _artefactIcon[i] != 0 ) {
				delete _artefactIcon[i];
				_artefactIcon[i] = 0;
			}
		}
		delete [] _artefactIcon;
	}

	if( _resourceIcon ) {
		uint i;
		uint nbRes = DataTheme.resources.count();
		for( i = 0; i < nbRes; i++ ) {
			if( _resourceIcon[i] != 0 ) {
				delete _resourceIcon[i];
				_resourceIcon[i] = 0;
			}
		}
		delete [] _resourceIcon;
	}

	if( _resourceSmallIcon ) {
		uint i;
		uint nbRes = DataTheme.resources.count();
		for( i = 0; i < nbRes; i++ ) {
			if( _resourceSmallIcon[i] != 0 ) {
				delete _resourceSmallIcon[i];
				_resourceSmallIcon[i] = 0;
			}
		}
		delete [] _resourceSmallIcon;
	}

	if( bonus ) {
		/// XXX: improve delete of array
		delete bonus;
	}

	if( _insideBase ) {
		delete [] _insideBase;
	}

	if( _flags ) {
		delete [] _flags;
	}
	
	if( _mapFlags ) {
		/// TODO
	}

	delete _sound;
}

QString ImageTheme::getPhotoCreature( GenericFightUnit * unit )
{
	QString pix = IMAGE_PATH + "units/face_" + QString::number( unit->getRace() ) + "_" + QString::number( unit->getLevel() ) + ".png";
	//logDD("Pixmap: %s", pix.latin1() );
	return  pix;
}

QPixmap * ImageTheme::getFlag( uint num )
{
	QPixmap * ret = 0;

	if( num < DataTheme.teams.count() ) {
		if( ! _flags[num] ) {
			_flags[ num ] = new QPixmap( IMAGE_PATH + "players/flag_" + QString::number( num ) + ".png" );
		}
		ret = _flags[num];
	}

	return ret;
}

QPixmap * ImageTheme::getLordPixmap( uint num )
{
	if( num <= DataTheme.lords.count() ) {
		if( ! _lordPixmap[num] ) {
			QString name;
			name.sprintf( "lords/lord_%03d.png", num );
			_lordPixmap[num] = new QPixmap( IMAGE_PATH + name );
		}
		return _lordPixmap[num];
	} else {
		logEE("There is not so many lords");
	}

	return 0;
}

QPixmap * ImageTheme::getLordSmallPixmap( uint num )
{
	if( num <= DataTheme.lords.count() ) {
		if( ! _lordSmallPixmap[num] ) {
			QString name;
			name.sprintf( "lords/smallLord_%03d.png", num );
			_lordSmallPixmap[num] = new QPixmap( IMAGE_PATH + name );
		}
		return _lordSmallPixmap[num];
	} else {
		logEE("There is not so many lords");
	}

	return 0;
}

QPixmap * ImageTheme::getBaseSmallPixmap( uint /*num*/ )
{
	/// XXX: TODO
	logEE( "not finished" );
	return _baseSmallPixmap[ 0 ];
}

QCanvasPixmapArray * ImageTheme::getMapFlag( uint type, uint teamId )
{
	return _mapFlags[ type ][ teamId ];
}

const uint NB_FLAGTYPE = 1;

bool ImageTheme::initTeams()
{
	uint i, j;
	uint nbTeams = DataTheme.teams.count();
	QString fileName;
	
	_flags = new QPixmap * [ nbTeams ];
	for( i = 0; i < nbTeams; i++ ) {
		_flags[i] = 0;
	}
	
	/// XXX: NB_FLAGTYPE = 1 = nb of type of flags (only one for the moment but will be extended if managed in dataTheme
	_mapFlags = new QCanvasPixmapArray ** [ NB_FLAGTYPE ];
	
	for( i = 0; i < NB_FLAGTYPE; i++ ) {
		_mapFlags[ i ] = new QCanvasPixmapArray * [ nbTeams ];
		for( j = 0; j < nbTeams; j++ ) {
			QPointArray listPointFlag( 1 );
			QValueList<QPixmap> listPixmapFlag;
			/// XXX: for the moment, only one pixmap, but maybe one day it will be animated...
			fileName.sprintf( "players/mapFlag_%d_%d_0.png", i, j );
			listPixmapFlag.append( QPixmap( IMAGE_PATH + fileName.latin1() ) );
			listPointFlag.setPoint( 0, 0, 0 );
			_mapFlags[ i ][ j ] = new QCanvasPixmapArray( listPixmapFlag, listPointFlag );
		}
	}

	return true;
}

bool ImageTheme::initSkills()
{
	QPointArray listPointSkill( 2 );
	QValueList<QPixmap> listPixmapSkill;
	listPixmapSkill.append( QPixmap( IMAGE_PATH + "technics/technic_1_1.png" ) );
	listPointSkill.setPoint( 0, 0, 0 );
	listPixmapSkill.append( QPixmap( IMAGE_PATH + "technics/technic_1_2.png" ) );
	listPointSkill.setPoint( 1, 0, 0 );
	skills = new QCanvasPixmapArray( listPixmapSkill, listPointSkill );

	return true;
}

bool ImageTheme::initCells()
{
	uint i, j;
	QString name;

	cells = new QCanvasPixmapArray * [DataTheme.tiles.count()];

	QPointArray listPointUnknown( 1 ); 
	QValueList<QPixmap> listPixmapUnknown;
	listPixmapUnknown.append( QPixmap( IMAGE_PATH + "tiles/unknown.png" ) );
	listPointUnknown.setPoint( 0, 0, 0 );
	cells[ 0 ] = new QCanvasPixmapArray( listPixmapUnknown, listPointUnknown );

	for( i = 1; i < DataTheme.tiles.count(); i++ ) {
		uint nbTiles = DataTheme.tiles.at( i )->getDiversificationNumber();
		
		QPointArray listPointCell( nbTiles );
		QValueList<QPixmap> listPixmapCell;

		for( j = 0; j < nbTiles; j++ ) {
			name.sprintf( "tiles/tile_%d_%d.png", i, j );
			listPixmapCell.append( QPixmap( IMAGE_PATH + name.latin1() ) );
			listPointCell.setPoint( j, 0, 0 );
		}
		cells[i] = new QCanvasPixmapArray( listPixmapCell, listPointCell );
	}


	QPointArray listPointPathCell( 66 );
	QValueList<QPixmap> listPixmapPathCell;
	
	for( i = 0; i < 33; i++ ) {
		QString name = "misc/arrow-" + QString::number( i ) + ".png";
		listPixmapPathCell.append( QPixmap( IMAGE_PATH + name.latin1() ) );
		listPointPathCell.setPoint( i, 0, 0 );
	}
	for( i = 0; i < 33; i++ ) {
		QString name = "misc/arrowFar-" + QString::number( i ) + ".png";
		listPixmapPathCell.append( QPixmap( IMAGE_PATH + name.latin1() ) );
		listPointPathCell.setPoint( 33 + i, 0, 0 );
	}
	pathCells = new QCanvasPixmapArray( listPixmapPathCell, listPointPathCell );

	return true;
}

bool ImageTheme::initCreatures()
{
	uint i, j;
	int cpt = 0;
	creatures = new CreaturePixmap * [DataTheme.creatures.count()];
	for( i = 0; i < DataTheme.countRace(); i++ ) {
		for( j = 0; j < DataTheme.creatures.getRace( i )->count(); j++ ) {
			QPtrList<QPixmap> listPixmapFightUnit;
			listPixmapFightUnit.setAutoDelete( true );
			QPtrList<QPoint> listPointFightUnit;
			listPointFightUnit.setAutoDelete( true );
			
			if ( DataTheme.creatures.at (i , j)->isAnimated () )
			{
				for ( int fr = 0; fr < DataTheme.creatures.at ( i , j )->getNumFrames (); fr++ )
				{
					QString numFrame = "00";
					numFrame.sprintf ("%02d",fr + 1);
					QString fileName = "units/cre_" + QString::number( i ) + "_" + QString::number( j ) + "_" + numFrame + ".png";
					listPixmapFightUnit.append( new QPixmap( IMAGE_PATH + fileName ) );
					listPointFightUnit.append( new QPoint( 0, 0 ) );
				}
			}
			else
			{
				listPixmapFightUnit.append( new QPixmap( IMAGE_PATH + "units/cre_" + QString::number( i ) + "_" + QString::number( j ) + ".png" ) );
				listPointFightUnit.append( new QPoint( 0, 0 ) );
			}
			creatures[cpt] = new CreaturePixmap( listPixmapFightUnit , listPointFightUnit );
			cpt++;
		}
	}

	QPointArray listPointDead( 1 );
	QValueList<QPixmap> listPixmapDead;
	listPixmapDead.append( QPixmap( IMAGE_PATH + "units/dead.png" ) );
	listPointDead.setPoint( 0, 0, -70 );
	deadCreature = new QCanvasPixmapArray( listPixmapDead, listPointDead );

	return true;
}

bool ImageTheme::initLords()
{
	uint nbLord = DataTheme.lords.count();
	uint i;

	QPointArray listPointLord( 1 );
	QValueList<QPixmap> listPixmapLord;
	listPixmapLord.append( QPixmap( IMAGE_PATH + "units/lord.png" ) );
	listPointLord.setPoint( 0, 0, 0 );
	lords = new QCanvasPixmapArray( listPixmapLord , listPointLord );

	_lordPixmap = new QPixmap * [ nbLord ];
	for( i = 0; i < nbLord; i++ ) {
		_lordPixmap[i] = 0;
	}

	_lordSmallPixmap = new QPixmap * [ nbLord ];
	for( i = 0; i < nbLord; i++ ) {
		_lordSmallPixmap[i] = 0;
	}

	return true;
}

bool ImageTheme::initBases()
{
	uint i;
	uint nbBases = DataTheme.bases.count();
	
	_insideBase = new QPixmap * [DataTheme.bases.count()];
	for( i = 0; i < nbBases; i++ ) {
		_insideBase[i] = 0;
	}

	QPointArray listPointBase( nbBases );
	QValueList<QPixmap> listPixmapBase;
	for( i = 0; i < nbBases; i++ ) {
		listPixmapBase.append( QPixmap( IMAGE_PATH + "base/base_" + QString::number( i ) + ".png" ) );
		listPointBase.setPoint( i, 0, 0 );
	}
	bases = new QCanvasPixmapArray( listPixmapBase, listPointBase );

	_baseSmallPixmap = new QPixmap * [1];
	_baseSmallPixmap[0] = new QPixmap( IMAGE_PATH + "base/smallBase.png" );

	return true;
}

bool ImageTheme::initArtefacts()
{
	uint i;
	uint nbArtefacts = DataTheme.artefacts.count();
	 
	QPointArray listPointArtefact( nbArtefacts );
	QValueList<QPixmap> listPixmapArtefact;
	for( i = 0; i < nbArtefacts; i++ ) {
		listPixmapArtefact.append( QPixmap( IMAGE_PATH + "artefacts/artefactItem_" + QString::number( i ) + ".png" ) );
		listPointArtefact.setPoint( i, 0, 0 );
	}
	artefacts = new QCanvasPixmapArray( listPixmapArtefact, listPointArtefact );

	_artefactIcon = new QPixmap * [ nbArtefacts ];
	for( i = 0; i < nbArtefacts; i++ ) {
		_artefactIcon[i] = 0;
	}

	return true;
}

bool ImageTheme::initResources()
{
	uint i;

	_resourceIcon = new QPixmap * [DataTheme.resources.count()];
	_resourceSmallIcon = new QPixmap * [DataTheme.resources.count()];
	for( i = 0; i < DataTheme.resources.count(); i++ ) {
		_resourceIcon[i] = 0;
		_resourceSmallIcon[i] = 0;
	}

	return true;
}

bool ImageTheme::initBonus()
{
	uint j;
	uint nbResources = DataTheme.resources.count();
	QString name;
	bonus = new QCanvasPixmapArray * [ 4 ];

	QPointArray listPointBonus0( nbResources );
	QValueList<QPixmap> listPixmapBonus0;
	for( j = 0; j < nbResources; j++ ) {
		name.sprintf( "ressources/smallResource_%02d.png", j+1 );
		listPixmapBonus0.append( QPixmap( IMAGE_PATH + name ) );
		listPointBonus0.setPoint( j, 0, 0 );
	}
	bonus[0] = new QCanvasPixmapArray( listPixmapBonus0, listPointBonus0 );

	QPointArray listPointBonus1( 1 );
	QValueList<QPixmap> listPixmapBonus1;
	name = "events/event1.png";
	listPixmapBonus1.append( QPixmap( IMAGE_PATH + name ) );
	listPointBonus1.setPoint( 0, 0, 0 );
	bonus[1] = new QCanvasPixmapArray( listPixmapBonus1, listPointBonus1 );

	QPointArray listPointBonus2( 1 );
	QValueList<QPixmap> listPixmapBonus2;
	name = "events/event1.png";
	listPixmapBonus2.append( QPixmap( IMAGE_PATH + name ) );
	listPointBonus2.setPoint( 0, 0, 0 );
	bonus[2] = new QCanvasPixmapArray( listPixmapBonus2, listPointBonus2 );

	QPointArray listPointBonus3( 1 );
	QValueList<QPixmap> listPixmapBonus3;
	name = "events/scroll.png";
	listPixmapBonus3.append( QPixmap( IMAGE_PATH + name ) );
	listPointBonus3.setPoint( 0, 0, 0 );
	bonus[3] = new QCanvasPixmapArray( listPixmapBonus3, listPointBonus3 );

	return true;
}

bool ImageTheme::initChest()
{
	QPointArray listPointChest( 1 );
	QValueList<QPixmap> listPixmapChest;
	listPixmapChest.append( QPixmap( IMAGE_PATH + "events/chest.png" ) );
	listPointChest.setPoint( 0, 0, 0 );
	chest = new QCanvasPixmapArray( listPixmapChest, listPointChest );

	return true;
}

bool ImageTheme::initBuildings()
{
	uint i, j, nbBuildings, nbBases;
	buildings = new QCanvasPixmapArray * [ DataTheme.buildings.count() ];
	for( i = 0; i < DataTheme.buildings.count(); i++ ) {
		nbBuildings = DataTheme.buildings.at( i )->getNbFrame();
		QPointArray listPointBuilding( nbBuildings );
		QValueList<QPixmap> listPixmapBuilding;
		for( j = 0; j < nbBuildings; j++ ) {
			QString name = "building/building_" + QString::number( i ) + "_" + QString::number( j ) + ".png";
			listPixmapBuilding.append( QPixmap( IMAGE_PATH + name ) );
			listPointBuilding.setPoint( j, 0, 0 );
		}
		buildings[i] = new QCanvasPixmapArray( listPixmapBuilding, listPointBuilding );
	}

	nbBases = DataTheme.bases.count();
	insideBuildings = new QCanvasPixmapArray * [ nbBases ];
	for( i = 0; i < nbBases; i++ ) {
		nbBuildings = DataTheme.bases.at( i )->getBuildingCount();
		QPointArray listPointInside( 2 * nbBuildings );
		QValueList<QPixmap> listPixmapInside;
		for( j = 0; j < nbBuildings; j++ ) {
			listPixmapInside.append( QPixmap( IMAGE_PATH + "base/inside_" + QString::number( i ) + "_" + QString::number( j ) + ".png" ) );
			listPointInside.setPoint( 2 * j, 0, 0 );
			listPixmapInside.append( QPixmap( IMAGE_PATH + "base/insideSel_" + QString::number( i ) + "_" + QString::number( j ) + ".png" ) );
			listPointInside.setPoint( (2 * j) + 1, 0, 0 );
		}
		insideBuildings[i] = new QCanvasPixmapArray( listPixmapInside, listPointInside );
	}

	return true;
}

bool ImageTheme::initTransitions()
{
	uint i, j;
	uint nbTransitions = DataTheme.tiles.count();
	
	transition = new QCanvasPixmapArray * [ nbTransitions ];
	for( i = 1; i <= nbTransitions; i++ ) {
		QPointArray listPointTransition( NB_TRANSITION );
		QValueList<QPixmap> listPixmapTransition;
		for( j = 1; j <= NB_TRANSITION; j++ ) {
			QString name = "tiles/transi_" + QString::number( i ) + "_" + QString::number( j ) + ".png";
			listPixmapTransition.append( QPixmap( IMAGE_PATH + name.latin1() ) );
			listPointTransition.setPoint( j - 1, 0, 0 );
		}
		transition[i-1] = new QCanvasPixmapArray( listPixmapTransition, listPointTransition );
	}

	return true;
}

bool ImageTheme::initDecorations()
{
	uint i, j;
	uint nbDecoGroups = DataTheme.decorations.count();
	uint nbDeco;
	
	decoration = new QCanvasPixmapArray * [ nbDecoGroups ];
	for( i = 1; i < nbDecoGroups; i++ ) {
		DecorationGroup * group = DataTheme.decorations.at( i );
		nbDeco = group->count();
		QPointArray listPointDeco( nbDeco );
		QValueList<QPixmap> listPixmapDeco;
		
		for( j = 0; j < nbDeco; j++ ) {
			listPixmapDeco.append( QPixmap( IMAGE_PATH + "decor/decoration_" + QString::number( i ) + "_" + QString::number( j ) + ".png" ) );
			listPointDeco.setPoint( j, 0, 0 );
		}
		
		decoration[ i-1 ] = new QCanvasPixmapArray( listPixmapDeco, listPointDeco );
	}
	

	return true;
}

bool ImageTheme::initEvents()
{
	uint i;
	events = new QCanvasPixmapArray * [ 3 ];
	QString fileName;

	QPointArray listPointEvent( 1 );
	QValueList<QPixmap> listPixmapEvent;
	fileName = "events/event1.png";
	listPixmapEvent.append( QPixmap( IMAGE_PATH + fileName ) );
	listPointEvent.setPoint( 0, 0, 0 );
	events[ 0 ] = new QCanvasPixmapArray( listPixmapEvent, listPointEvent );

	uint nbRes = DataTheme.resources.count();
	QPointArray listPointEvent2( nbRes );
	QValueList<QPixmap> listPixmapEvent2;
	for( i = 0; i < nbRes; ++i ) {
		fileName.sprintf( "ressources/smallResource_%02d.png", i+1 );
		listPixmapEvent2.append( QPixmap( IMAGE_PATH + fileName ) );
		listPointEvent2.setPoint( i, 0, 0 );
	}
	events[ 1 ] = new QCanvasPixmapArray( listPixmapEvent2, listPointEvent2 );

	QPointArray listPointEvent3( 1 );
	QValueList<QPixmap> listPixmapEvent3;
	fileName = "events/chest.png";
	listPixmapEvent3.append( QPixmap( IMAGE_PATH + fileName ) );
	listPointEvent3.setPoint( 0, 0, 0 );
	events[ 2 ] = new QCanvasPixmapArray( listPixmapEvent3, listPointEvent3 );

	return true;
}

bool ImageTheme::initMapCreatures()
{
	uint i, j;

	mapCreatures = new MapCreaturePixmap ** [ DataTheme.countRace() ];
	for( i = 0; i < DataTheme.countRace(); i++ ) {
		mapCreatures[i] = new MapCreaturePixmap * [ DataTheme.creatures.getRace( i )->count() ];
		for( j = 0; j < DataTheme.creatures.getRace( i )->count(); j++ ) {
			QPointArray listPointMapCreature( 1 );
			QValueList<QPixmap> listPixmapMapCreature;
			QString name = "units/creMap_" + QString::number( i ) + "_" + QString::number( j ) + ".png";
			listPixmapMapCreature.append( QPixmap( IMAGE_PATH + name ) );
			listPointMapCreature.setPoint( 0, 0, 0 );
			mapCreatures[i][j] = new MapCreaturePixmap( listPixmapMapCreature, listPointMapCreature );
		}
	}

	return true;
}

QPixmap * ImageTheme::getMapCreature( uint race, uint level, bool rightLooking )
{
	return ( ( * ( mapCreatures[race][level] ) )[0] )->image( rightLooking?1:0 );
}

QPixmap * ImageTheme::getMapCreature( GenericMapCreature * creature, bool rightLooking )
{
	return getMapCreature( creature->getRace(), creature->getLevel(), rightLooking );
}

CreaturePixmap * ImageTheme::getCreature ( uint race, uint level )
{
	int nCnt = 0;
	for( uint i = 0; i < race; i++ ) {
		nCnt += DataTheme.creatures.getRace( i )->count();
	}

	nCnt += level;

	return ( creatures[nCnt] );
}

bool ImageTheme::initSound()
{
	_sound = new AttalSound();
	_sound->init();

	return true;
}

void ImageTheme::playMusicMap()
{
	_sound->playMusicMap();
}

void ImageTheme::playMusicBase( uint race )
{
	_sound->playMusicBase( race );
}

void ImageTheme::playMusicFight()
{
	_sound->playMusicFight();
}

void ImageTheme::playSound( AttalSound::SoundType snd )
{
	_sound->playSound( snd );
}





