------------------------------------------------------------------------------
--                                                                          --
--                          GNATCHECK COMPONENTS                            --
--                                                                          --
--       G N A T C H E C K . R U L E S . M E T R I C S . C H E C K S        --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                       Copyright (C) 2008, AdaCore                        --
--                                                                          --
-- GNATCHECK  is  free  software;  you can redistribute it and/or modify it --
-- under terms of the  GNU  General Public License as published by the Free --
-- Software Foundation;  either version 2, or ( at your option)  any  later --
-- version.  GNATCHECK  is  distributed in the hope that it will be useful, --
-- but  WITHOUT  ANY  WARRANTY;   without  even  the  implied  warranty  of --
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General --
-- Public License for more details.  You should have received a copy of the --
-- GNU  General Public License distributed with GNAT; see file  COPYING. If --
-- not,  write to the  Free Software Foundation,  51 Franklin Street, Fifth --
-- Floor, Boston, MA 02110-1301, USA.                                       --
--                                                                          --
-- GNATCHECK is maintained by AdaCore (http://www.adacore.com).             --
--                                                                          --
------------------------------------------------------------------------------

with ASIS_UL.Metrics.Compute;     use ASIS_UL.Metrics.Compute;
with ASIS_UL.Misc;                use ASIS_UL.Misc;
with ASIS_UL.Output;              use ASIS_UL.Output;
with ASIS_UL.Utilities;           use ASIS_UL.Utilities;

package body Gnatcheck.Rules.Metrics.Checks is

   Complexity_Metrics : Complexity_Metric_Counter;
   --  This variable collects complexity metrics. We make it global to allow to
   --  call the routine that computes all the complexity metrics for a given
   --  element only once and then to use the results for both cyclomatic
   --  complexity and essential complexity checks.

   --------------------------------------------
   -- Operations belonging to the root types --
   -- of the metric check hierarchy          --
   --------------------------------------------

   ------------------------------
   -- Analyze_Metric_Parameter --
   ------------------------------

   procedure Analyze_Metric_Parameter
     (Rule : in out Metric_Checks_Rule_Type'Class;
      Par  : String)
   is
      First_Idx : Natural := Par'First;
      Last_Idx  : Positive;
      Success   : Boolean := False;
   begin

      if Par (First_Idx) = '>'
        and then
         Rule.Metric_Value_Check = Smaller_Then
      then
         Rule.Rule_State := Disabled;
         Error ("for " & Rule_Name (Rule) & " lower limit " &
                "cannot be specified");
         return;
      elsif
         Par (First_Idx) = '<'
        and then
         Rule.Metric_Value_Check = Greater_Then
      then
         Rule.Rule_State := Disabled;
         Error ("for " & Rule_Name (Rule) & " upper limit " &
                "cannot be specified");
         return;
      end if;

      for J in First_Idx + 1 .. Par'Last loop
         if not Is_White_Space (Par (J)) then
            First_Idx := J;
            exit;
         end if;
      end loop;

      if First_Idx = Par'First then
         Rule.Rule_State := Disabled;
         Error ("no limit specified for " & Rule_Name (Rule));
         return;
      end if;

      for J in reverse  First_Idx .. Par'Last loop
         if not Is_White_Space (Par (J)) then
            Last_Idx := J;
            exit;
         end if;
      end loop;

      Set_Metric_Limit
        (Rule    => Rule,
         Par     => Par (First_Idx .. Last_Idx),
         Success => Success);

      if Success then
         Metrics_Violation_Rule.Rule_State := Enabled;
      end if;

   end Analyze_Metric_Parameter;

   ----------------------------------------
   -- Set_Metric_Limit (integer metrics) --
   ----------------------------------------

   procedure Set_Metric_Limit
     (Rule    : in out Integer_Metric_Rule_Type;
      Par     :        String;
      Success :    out Boolean)
   is
      Val : Metric_Count;
   begin
      Val               := Metric_Count'Value (Par);
      Rule.Metric_Limit := Val;
      Rule.Rule_State   := Enabled;
      Success           := True;
   exception
      when Constraint_Error =>
         Rule.Rule_State := Disabled;
         Error ("wrong limit specification for " & Rule_Name (Rule));
         Success := False;
   end Set_Metric_Limit;

   -------------------------------------
   -- Set_Metric_Limit (real metrics) --
   -------------------------------------

   procedure Set_Metric_Limit
     (Rule    : in out Real_Metric_Rule_Type;
      Par     :        String;
      Success :    out Boolean)
   is
      Val : Float;
   begin
      Val               := Float'Value (Par);
      Rule.Metric_Limit := Val;
      Rule.Rule_State   := Enabled;
      Success           := True;
   exception
      when Constraint_Error =>
         Rule.Rule_State := Disabled;
         Error ("wrong limit specification for " & Rule_Name (Rule));
         Success := False;
   end Set_Metric_Limit;

   -----------------------------------------------
   -- Implementation of the metric-based checks --
   -----------------------------------------------

   ---------------------------
   -- Cyclomatic_Complexity --
   ---------------------------

   -----------------------------------------------
   -- Rule_Check_Pre_Op (Cyclomatic_Complexity) --
   -----------------------------------------------

   procedure Rule_Check_Pre_Op
     (Rule    : in out Cyclomatic_Complexity_Rule_Type;
      Element :        Asis.Element;
      Control : in out Traverse_Control;
      State   : in out Rule_Traversal_State)
   is
      pragma Unreferenced (Control);
      Cyclomatic_Complexity : Metric_Count;
   begin

      if Is_Executable_Body (Element) then
         Compute_Complexity_Metrics (Element, Complexity_Metrics);
         Cyclomatic_Complexity := Complexity_Metrics.Statement_Complexity +
            Complexity_Metrics.Short_Circuit_Complexity;

         if Cyclomatic_Complexity > Rule.Metric_Limit then
            State.Detected    := True;
            State.Diag_Params :=
              Enter_String ("%1%" & Cyclomatic_Complexity'Img);
         end if;

      end if;

   end Rule_Check_Pre_Op;

   ---------------------------------------
   -- Init_Rule (Cyclomatic_Complexity) --
   ---------------------------------------

   procedure Init_Rule (Rule : in out Cyclomatic_Complexity_Rule_Type) is
   begin
      --  Common rule fields:
      Rule.Name       := new String'("Cyclomatic_Complexity");
      Rule.Rule_State := Disabled;
      Rule.Help_Info  := new String'("(metrics) high cyclomatic complexity");
      Rule.Diagnosis  := new String'("cyclomatic complexity is too high: %1%");

      --  Field specific for internal rules
      Rule.Implements := Gnatcheck.Rules.Metrics.Metrics_Violation_Rule'Access;

      --  Field specific for metrics rules
      Rule.Metric_Value_Check := Greater_Then;
   end Init_Rule;

   --------------------------
   -- Essential_Complexity --
   --------------------------

   ----------------------------------------------
   -- Rule_Check_Pre_Op (Essential_Complexity) --
   ----------------------------------------------

   procedure Rule_Check_Pre_Op
     (Rule    : in out Essential_Complexity_Rule_Type;
      Element :        Asis.Element;
      Control : in out Traverse_Control;
      State   : in out Rule_Traversal_State)
   is
      pragma Unreferenced (Control);
   begin
      if Is_Executable_Body (Element) then

         if not Is_Enable (Cyclomatic_Complexity_Rule) then
            Compute_Complexity_Metrics (Element, Complexity_Metrics);
         end if;

         if Complexity_Metrics.Essential_Complexity > Rule.Metric_Limit then
            State.Detected    := True;
            State.Diag_Params :=
              Enter_String ("%1%" &
                            Complexity_Metrics.Essential_Complexity'Img);
         end if;
      end if;

   end Rule_Check_Pre_Op;

   --------------------------------------
   -- Init_Rule (Essential_Complexity) --
   --------------------------------------

   procedure Init_Rule (Rule : in out Essential_Complexity_Rule_Type) is
   begin
      --  Common rule fields:
      Rule.Name       := new String'("Essential_Complexity");
      Rule.Rule_State := Disabled;
      Rule.Help_Info  := new String'("(metrics) high essential complexity");
      Rule.Diagnosis  := new String'("essential complexity is too high: %1%");

      --  Field specific for internal rules
      Rule.Implements := Gnatcheck.Rules.Metrics.Metrics_Violation_Rule'Access;

      --  Field specific for metrics rules
      Rule.Metric_Value_Check := Greater_Then;
   end Init_Rule;

   -----------
   -- LSLOC --
   -----------

   ------------------------------
   -- Rule_Check_Pre_Op LSLOC) --
   ------------------------------

   procedure Rule_Check_Pre_Op
     (Rule    : in out LSLOC_Rule_Type;
      Element :        Asis.Element;
      Control : in out Traverse_Control;
      State   : in out Rule_Traversal_State)
   is
      pragma Unreferenced (Control);
      Counter   : Syntax_Metric_Counter;
      LSLOC     : Metric_Count;
   begin

      if Is_Program_Unit (Element) then
         Compute_Syntaxt_Metrics (Element, Counter);
         LSLOC := Counter.All_Statements + Counter.All_Declarations;

         if LSLOC > Rule.Metric_Limit then
            State.Detected    := True;
            State.Diag_Params := Enter_String ("%1%" & LSLOC'Img);
         end if;

      end if;

   end Rule_Check_Pre_Op;

   procedure Init_Rule (Rule : in out LSLOC_Rule_Type) is
   begin
      --  Common rule fields:
      Rule.Name       := new String'("LSLOC");
      Rule.Rule_State := Disabled;
      Rule.Help_Info  := new String'("(metrics) high LSLOC value");
      Rule.Diagnosis  := new String'("LSLOC is too high: %1%");

      --  Field specific for internal rules
      Rule.Implements := Gnatcheck.Rules.Metrics.Metrics_Violation_Rule'Access;

      --  Field specific for metrics rules
      Rule.Metric_Value_Check := Greater_Then;
   end Init_Rule;

end Gnatcheck.Rules.Metrics.Checks;
