------------------------------------------------------------------------------
--                                                                          --
--                            GNATPP COMPONENTS                             --
--                                                                          --
--                       G N A T P P . O P T I O N S                        --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--                    Copyright (C) 2001-2006, AdaCore                      --
--                                                                          --
-- GNATPP is free software; you can redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNATPP is  distributed in the  hope that it will  be  useful, but --
-- WITHOUT ANY WARRANTY; without even the implied warranty of  MERCHANTABI- --
-- LITY or  FITNESS  FOR A  PARTICULAR  PURPOSE. See the GNU General Public --
-- License  for more details. You  should  have  received a copy of the GNU --
-- General Public License  distributed with GNAT; see file COPYING. If not, --
-- write  to  the Free  Software  Foundation,  59 Temple Place - Suite 330, --
-- Boston,                                                                  --
--                                                                          --
-- GNATPP is maintained by AdaCore (http://www.adacore.com)                 --
--                                                                          --
------------------------------------------------------------------------------

--  This package contains all the GNATPP options and control parameters, and
--  some auxiliary flags and objects needed to control pretty-printing

with GNAT.OS_Lib; use GNAT.OS_Lib;

package GNATPP.Options is

   PP_Indentation : Positive range 1 .. 9 := 3;
   --  Indentation level

   PP_Cont_Line_Indentation : Positive range 1 .. 9 := 3;
   --  Indentation for continuation lines

   type PP_Casing is
   --  Defines the casing of identifiers and keywords generated by gnatpp
     (Lower_Case,
      --  All letters are lowercase
      Upper_Case,
      --  All letters are uppercase
      Mixed,
      --  For both defining and usage occurrences of identifiers The first
      --  letter and each letter which immediately follows the underscore are
      --  uppercase, and all the other letters are lowercase
      --  operator
      As_Declared);
      --  All the usage occurrences of identifiers have the same casing as
      --  defining occurrences, defining occurrences have the same casing as
      --  the corresponding defining occurrences in the argument source.

   PP_Attribute_Casing : PP_Casing range Lower_Case .. Mixed :=
      Mixed;

   PP_Keyword_Casing : PP_Casing range Lower_Case .. Upper_Case := Lower_Case;

   PP_Pragma_Casing : PP_Casing range Lower_Case .. Mixed := Mixed;
   --  Specifies the casing of pragma names and identifiers specific to
   --  pragmas

   PP_Name_Casing : PP_Casing := As_Declared;
   --  Defines the casing for both defining and usage occurrences of the names

   Use_Predefined_Casing : Boolean := True;
   --  This flag specifies if for the predefined names should be used the same
   --  casing as given in RM95

   Use_Dictionary : Boolean := False;
   --  This flag specifies if the exception dictionary should be used for
   --  defining the name casing

   Format_Comments : Boolean := True;
   --  If this flag is set OFF, all the comments (comment lines and end-of-line
   --  comments) are moved into the result unchanged (no indentation or long
   --  line splitting is performed).

   GNAT_Comment_Inden : Boolean := True;
   --  Comment lines are indented in GNAT style. The difference with
   --  Standard_Comment_Indent is that comment lines preceding if and case
   --  statements alternatives and 'begin/ keywords are indented as the
   --  corresponding alternatives or keywords, but not as enclosing statements.

   Standard_Comment_Indent : Boolean := False;
   --  Comment lines are indented as the corresponding code lines.

   GNAT_Comment_Start : Boolean := False;
   --  The comment (if non-empty) should have at least two space characters
   --  after '--'

   Reformat_Comment_Block : Boolean := False;
   --  For sequences of comment lines (separated by space lines or empty
   --  comment lines (lines containing only two minuses) the attempt should be
   --  made to reformat the text of the comment in a word processor style -
   --  that is, to put as many words in the line as possible, using only one
   --  space as a separator.

   No_Tab_In_Comments : Boolean := False;
   --  Remove HT and VT from the content of the comments. If this flag is set
   --  ON, all the VT characters are removed from the comment text and replaced
   --  with spaces to get to the nearest Tab stop (the Tab step is supposed to
   --  be equal to 8), and after that the comment line may be further
   --  reformatted to get the indentation and maximum line length rules.
   --  As for now, reformattable comment blocks can not contain HT characters,
   --  and VT are removed from reformattable blocks as a part of reformatting.

   End_Labels : Boolean := True;
   --  Do set end/exit labels even if missed in the argument source;

   Add_FF : Boolean := False;
   --  Add Form Feed after a pragma Page.

   Compact_Layout : Boolean := True;
   --  Use compact layout for records and named statements;

   Separate_Line_For_IS : Boolean := True;
   --  Use a separate sine for IS in subprogram body in case if we need
   --  more then one line for subprogram specification

   Add_Empty_Lines : Boolean := True;
   --  Add empty lines (if needed to separate compound statements, bodies and
   --  return statements

   Line_Len_Limit  : constant Natural                   := 256;
   Max_Line_Length : Natural range 32 .. Line_Len_Limit := 79;

   Align_Colons_In_Decl : Boolean := False;
   Align_Asign_In_Decl  : Boolean := False;
   Align_Asign_In_Stmts : Boolean := False;
   Align_Arrows         : Boolean := False;
   Allign_Ats           : Boolean := False;

   Alignment_On         : Boolean := True;
   --  If this flag is set OFF, then only explicitly specified alignments are
   --  performed

   Case_Threshold : Natural := 10;
   --  Starting from this number an extra indentation level is not used for
   --  variants in record variant part and case statement alternatives in case
   --  statements, the value 0 means that the extra level is used for any
   --  number of variants and case alternatives

   type Output_Modes is
   --  Defines the where and how gnatpp places the result source
     (Pipe,
      --  Sends the output into Stderr
      Create_File,
      --  Creates the file with the name specified in 'o' option. If the
      --  file with the given name already exists, does not erase it and gives
      --  up
      Force_Create_File,
      --  Creates the file with the name specified in 'o' option. If the
      --  file with the given name already exists, erases the old file and
      --  replaces it with the pretty-printed source.
      Replace,
      --  Replaces the argument source with the pretty-printed source. The
      --  original source is stored in the file <arg_source>.npp
      --  (<arg_source>$NPP if on OpenVMS host). If the file with such a name
      --  already exists, gnatpp gives up
      Force_Replace,
      --  Replaces the argument source with the pretty-printed source. The
      --  original source is stored in the file <arg_source>.npp
      --  (<arg_source>$NPP if on OpenVMS host). If the file with such a name
      --  already exists, gnatpp overrides it
      Replace_No_Backup,
      --  Replaces the argument source with the pretty-printed source. The
      --  original source is not stored in any back-up file.
      Default);
      --  Put the result source into <arg_source>.pp (<arg_source>$PP if on
      --  OpenVMS host), overriding the existing file if any

   Output_Mode : Output_Modes := Default;

   type Output_Encodings is
   --  Defines the encodings used for the output file
     (Hex_ESC,
      Upper_Half,
      Shift_JIS,
      EUC,
      UTF_8,
      Brackets,
      Default);

   Output_Encoding : Output_Encodings := Default;
   --  Defines the encoding used for the result file(s).

   Form_String : String_Access := new String'("");
   --  Used as the value for the Form parameter of Open and Create procedures,
   --  defines the encoding of the result file

   Verbose_Mode : Boolean := False;
   --  If this flag is set on, gnatpp prints out the version info and generates
   --  the trace of the processing of the argument source

   Warnings_ON : Boolean := False;
   --  If this flag is set ON, GNATPP generates a warning message each time it
   --  has to violate the formatting rules in the result source

   Quiet_Mode : Boolean := False;
   --  In quiet mode gnatpp does not print anything in Standard Error
   --  except the diagnostic and warning messages

   Progress_Indicator_Mode : Boolean := False;
   --  Generates the output to be used for GPS progress indicator.

   Multiple_File_Mode : Boolean := False;
   --  This option is not set by means of gnatpp parameters, we compute it when
   --  we know how many sources we have to output

   type Out_File_Formats is (
      Default,
      CRLF,
      LF);

   Out_File_Format : Out_File_Formats := Default;
   --  The format of the gnatpp result file

   ----------------------------------------------
   --  Options which are not set by parameters --
   ----------------------------------------------

   Max_Enums_On_Line : Positive := 5;
   --  The maximum number of enumeration literals in enumeration type
   --  definition when all the literals can be placed into one line

   Too_Short_Comment : Positive := 30;
   --  If the comment (starting from '--' and ending with the last non-blank
   --  character) is shorter then this value, then this comment line can not be
   --  considered as the start line of the reformattable comment block

   Max_Commment_Indentation : Positive := 5;
   --  If the indentation in the comment (that is, the distance between '--'
   --  and the first non-blank character) is bigger then this value, then this
   --  comment line can not be considered as the start line of the
   --  reformattable comment block (We suppose that this line contains a kind
   --  of header)

end GNATPP.Options;
