#include <unistd.h>
#include <sys/stat.h>

#include <dsp.h>
#include <source-default.h>
#include <asd.h>
#include <sample-convert.h>

#include "source-dsp.h"

static gboolean _source_dsp_open(Source *s)
{
  SourceDefaultPrivate *p;
  g_assert(s);
  g_assert(p = (SourceDefaultPrivate*) s->private_data);
  g_assert(p->filename && !p->is_open);
  g_assert(s->mode == SOURCE_DISABLED);

  if ((p->is_open = (p->fd = dsp_open(p->filename, TRUE, &s->sample_type)) >= 0))
    {
      g_message("Source '%s' successfully opened.", s->shortname);
      
      if (!sample_type_equal(&s->sample_type, &default_sample_type))
	{
	  p->buffer_size = sample_convert_length(&default_sample_type, &s->sample_type, default_block_size, FALSE);
	  p->buffer = g_new(guint8, p->buffer_size);
	}
    }
  else
    g_message("Source '%s' open failed.", s->shortname);

  source_set_mode(s, p->is_open ? SOURCE_RUNNING : SOURCE_DISABLED);

  return p->is_open;
}

static void _source_dsp_close(Source*s)
{
  SourceDefaultPrivate *p;
  g_assert(s);
  g_assert(p = (SourceDefaultPrivate*) s->private_data);
  
  if (p->is_open)
    {
      dsp_close(p->filename, TRUE);
      p->is_open = FALSE;  
      g_message("Source '%s' closed.", s->shortname);
    }

  source_set_mode(s, SOURCE_DISABLED);
}

Source* source_dsp_new(gchar *shortname, gchar *fname)
{
  Source *s;
  char name[ASD_NAME_LENGTH];

  g_snprintf(name, sizeof(name), "DSP %s", fname);
  g_assert(s = source_new(shortname, name, TRUE));

  g_assert(s->private_data = source_default_alloc_private(fname));

  s->type = "DSP";
  s->open = _source_dsp_open;
  s->close = _source_dsp_close;

  s->flags &= ~SOURCE_PUSH_WAIT;

  return s;
}
