#include <stdlib.h>
#include <string.h>
#include <libxml/tree.h>

#include <gc.h>
#include <recursive-lock.h>

#include <session.h>
#include <asd.h>

static GStaticMutex _session_mutex = G_STATIC_MUTEX_INIT;
static GStaticPrivate _session_private = G_STATIC_PRIVATE_INIT;

#define _session_lock() recursive_lock(&_session_mutex, &_session_private)
#define _session_unlock() recursive_unlock(&_session_mutex, &_session_private)

static GSList *_sources = NULL;
static GSList *_sinks = NULL;

static gchar _session_path[PATH_MAX] = "";

void session_register_source(Source *s)
{
  g_assert(s);
  _session_lock();
  _sources = g_slist_append(_sources, gc_ref_inc(s));
  _session_unlock();
}

void session_register_sink(Sink *s)
{
  g_assert(s);
  _session_lock();
  _sinks = g_slist_append(_sinks, gc_ref_inc(s));
  _session_unlock();
}

void session_unregister_source(Source *s)
{
  g_assert(s);
  _session_lock();
  _sources = g_slist_remove(_sources, s);
  gc_ref_dec(s);
  _session_unlock();
}

void session_unregister_sink(Sink *s)
{
  g_assert(s);
  _session_lock();
  _sinks = g_slist_remove(_sinks, s);
  gc_ref_dec(s);
  _session_unlock();
}

void session_reset()
{
  _session_lock();

  while (_sources)
    {
      gpointer p = _sources->data;
      _sources = g_slist_remove(_sources, p);
      gc_ref_dec(p);
    }

  while (_sinks)
    {
      gpointer p = _sinks->data;
      _sinks = g_slist_remove(_sinks, p);
      gc_ref_dec(p);
    }

  _session_unlock();
}

void session_save(gchar *f)
{
  FILE*fd;

  if (!f) f = getenv("ASDSESSION");

  if (f)
    strncpy(_session_path, f, sizeof(_session_path));
  else
    g_snprintf(_session_path, sizeof(_session_path), "%s/.asd-session", g_get_home_dir());
      
  g_assert(fd = fopen(_session_path, "w"));
  session_save_fd(fd);
  fclose(fd);
}

void session_save_fd(FILE *f)
{
  xmlDocPtr doc;
  xmlNodePtr tree;
  gchar c[256];
  GSList *l;

  doc = xmlNewDoc("1.0");
  doc->children = xmlNewDocNode(doc, NULL, "asd-session", NULL);

  xmlSetProp(doc->children, "version", "1.1");

  tree = xmlNewChild(doc->children, NULL, "global-configuration", NULL);

  g_snprintf(c, sizeof(c), "%i", def_block_size);
  xmlNewChild(tree, NULL, "fragment-size", c);

  g_snprintf(c, sizeof(c), "%u", def_sample_type);
  xmlNewChild(tree, NULL, "sample-type", c);

  _session_lock();

  tree = xmlNewChild(doc->children, NULL, "sources", NULL);
  l = _sources;
  while (l)
    {
      Source *s;
      g_assert(s = (Source*) l->data);
      if (s->save)
	s->save(s, tree);
      l = l->next;
    }

  tree = xmlNewChild(doc->children, NULL, "sinks", NULL);
  l = _sinks;
  while (l)
    {
      Sink *s;
      g_assert(s = (Sink*) l->data);
      if (s->save)
	s->save(s, tree);
      l = l->next;
    }

  _session_unlock();

  xmlDocDump(f, doc);
  xmlFreeDoc(doc);
}
