#include <string.h>

#include <asdutil.h>

#include <asd.h>
#include <namespace.h>
#include <conjunction.h>

#include "protocol-esound-impl.h"
#include "protocol-esound.h"
#include "idgen.h"

#include "../sources/source-socket.h"
#include "../sinks/sink-socket.h"

#define MAYBE_SWAP(x,b) ((b) ? GUINT32_SWAP_LE_BE(x) : (x))
#define SAMPLE_TYPE_TO_ESD(t) (((t).channels >= 2 ? ESD_STEREO : ESD_MONO) | ((t).bits >= 16 ? ESD_BITS16 : ESD_BITS8))

gint protocol_esound_STREAM_PLAY(ProtocolEsoundClient *client, ProtocolEsoundCommand command);
gint protocol_esound_LATENCY(ProtocolEsoundClient *client, ProtocolEsoundCommand command);
gint protocol_esound_SERVER_INFO(ProtocolEsoundClient *client, ProtocolEsoundCommand command);
gint protocol_esound_ALL_INFO(ProtocolEsoundClient *client, ProtocolEsoundCommand command);
gint protocol_esound_LOCK(ProtocolEsoundClient *client, ProtocolEsoundCommand command);

ProtocolEsoundHandler protocol_esound_handler_map[] = {
  { ESD_PROTO_STREAM_PLAY, protocol_esound_STREAM_PLAY, "Play stream" },
  { ESD_PROTO_STREAM_REC, protocol_esound_STREAM_PLAY, "Record stream" },
  { ESD_PROTO_STREAM_MON, protocol_esound_STREAM_PLAY, "Monitor stream" },
  { ESD_PROTO_LATENCY, protocol_esound_LATENCY, "Get latency" },
  { ESD_PROTO_SERVER_INFO, protocol_esound_SERVER_INFO, "Server info" },
  { ESD_PROTO_ALL_INFO, protocol_esound_ALL_INFO, "All info" },
  { ESD_PROTO_LOCK, protocol_esound_LOCK, "Lock" },
  { ESD_PROTO_UNLOCK, protocol_esound_LOCK, "Unlock" },
  { 0, NULL, NULL }
};

ProtocolEsoundHandler* protocol_esound_find_handler(ProtocolEsoundCommand c)
{
  guint i;
  for (i = 0; ; i++)
    if (protocol_esound_handler_map[i].proc == NULL)
      return NULL;
    else if (protocol_esound_handler_map[i].command == c)
      return &protocol_esound_handler_map[i];
}

gint protocol_esound_STREAM_PLAY(ProtocolEsoundClient *client, ProtocolEsoundCommand command)
{
  ProtocolEsoundStream stream;
  Sink *dev_sink = NULL;
  Source *dev_source = NULL;

  g_assert(client);
  
  if (atomic_read(client->fd, &stream, sizeof(stream)) == sizeof(stream))
    {
      gchar shortname[ASD_SHORTNAME_LENGTH];
      SampleType st;
      gchar t[256];

      stream.name[sizeof(stream.name)-1] = 0;
      stream.format = MAYBE_SWAP(stream.format, client->swap);

      st.be = FALSE;
      st.bits = ((stream.format & ESD_MASK_BITS) == ESD_BITS16) ? 16 : 8;
      st.sign = st.bits >= 16;
      st.channels = ((stream.format & ESD_MASK_CHAN) == ESD_STEREO) ? 2 : 1;
      st.rate = MAYBE_SWAP(stream.rate, client->swap);
      
      if (!sample_type_valid(&st))
	return 2;

      g_snprintf(shortname, sizeof(shortname), "esound%u", client->id);
      if (!namespace_register(shortname))
        return 2;

      if (command == ESD_PROTO_STREAM_PLAY || command == ESD_PROTO_STREAM_MON) // Play or monitor
	dev_sink = (Sink*) namespace_lookup_with_type("play0", NAMESPACE_SINK);
      else // Capture
	dev_source = (Source*) namespace_lookup_with_type("capt0", NAMESPACE_SOURCE);

      if (!dev_sink && !dev_source)
        {
          namespace_unregister(shortname);
          return 2;
        }

      pthread_cleanup_push(gc_ref_dec, dev_sink ? (gpointer) dev_sink : (gpointer) dev_source);
      pthread_cleanup_push(MAKE_CLEANUP_HANDLER(namespace_unregister), &shortname[0]);

      g_message("ESOUND: New client '%s' registered.", shortname);

      g_message("ESOUND: Client wants to %s a stream.", (command == ESD_PROTO_STREAM_PLAY ? "play" : (command == ESD_PROTO_STREAM_MON ? "monitor" : "capture")));

      sample_type_to_string(&st, t, sizeof(t));
      g_message("ESOUND: Sample format is %s.", t);

      if (command == ESD_PROTO_STREAM_PLAY) // Play
	{
	  Source *s;
	  g_assert(s = source_socket_new(shortname, stream.name, client->fd, &st));
	  link_source_sink(s, dev_sink, 4, 0);
	  gc_ref_dec(dev_sink);
	  source_start(s);
	  gc_ref_dec(s);
	}
      else if (command == ESD_PROTO_STREAM_REC) // Capture
	{
	  Sink *s;
	  g_assert(s = sink_socket_new(shortname, stream.name, client->fd, &st));
	  link_source_sink(dev_source, s, 4, 0);
	  gc_ref_dec(dev_source);
	  sink_start(s);
	  gc_ref_dec(s);
	}
      else // Monitor
	{
	  Sink *s;
	  g_assert(s = sink_socket_new(shortname, stream.name, client->fd, &st));
	  link_sink_sink(dev_sink, s, 4, 0);
	  gc_ref_dec(dev_sink);
	  sink_start(s);
	  gc_ref_dec(s);
	}
      
      return 1;
      
      pthread_cleanup_pop(1);
      pthread_cleanup_pop(1);
    }
  
  return 2;
}

gint protocol_esound_LATENCY(ProtocolEsoundClient *client, ProtocolEsoundCommand command)
{
  guint32 l;
  g_assert(client);
  
  l = MAYBE_SWAP(latency_global_average(), client->swap);

  if (atomic_write(client->fd, &l, sizeof(l)) != sizeof(l))
    return 2;

  return 0;
}

gint protocol_esound_SERVER_INFO(ProtocolEsoundClient *client, ProtocolEsoundCommand command)
{
  ProtocolEsoundServerInfo i;
  guint32 version;
  g_assert(client);
  
  if (atomic_read(client->fd, &version, sizeof(version)) != sizeof(version))
    return 2;

  if (version != 0)
    return 2;

  i.version = MAYBE_SWAP(0x1337BABE, client->swap);
  i.rate = MAYBE_SWAP(default_sample_type.rate, client->swap);
  i.format = MAYBE_SWAP(SAMPLE_TYPE_TO_ESD(default_sample_type), client->swap);

  if (atomic_write(client->fd, &i, sizeof(i)) != sizeof(i))
    return 2;

  return 0;
}


typedef struct {
  guint id;
  ProtocolEsoundClient *client;
  gboolean source;
} _EnumStruct;

static void _proc(gchar *name, gpointer data, gpointer userdata)
{
  ProtocolEsoundPlayerInfo info;
  _EnumStruct *e;
  Volume *v;
  SampleType *t;

  g_assert(name && userdata && data);
  e = (_EnumStruct*) userdata;

  if (e->source)
    {
      Source *s = (Source*) data;
      strncpy(info.name, s->name, sizeof(info.name));
      v = &s->volume;
      t = &s->sample_type;
      info.format = ESD_PLAY;
    }
  else
    {
      Sink *s = (Sink*) data;
      strncpy(info.name, s->name, sizeof(info.name));
      v = &s->volume;
      t = &s->sample_type;
      info.format = ESD_RECORD;
    }
  
  info.source_id = MAYBE_SWAP(e->id++, e->client->swap);
  info.rate = MAYBE_SWAP(t->rate, e->client->swap);
  info.left_vol_scale = MAYBE_SWAP(v->factor[0]*256/0xFFFF, e->client->swap);
  info.right_vol_scale = MAYBE_SWAP(v->factor[1]*256/0xFFFF, e->client->swap);
  info.format = MAYBE_SWAP(SAMPLE_TYPE_TO_ESD(*t)|info.format, e->client->swap);

  atomic_write(e->client->fd, &info, sizeof(info));
}

gint protocol_esound_ALL_INFO(ProtocolEsoundClient *client, ProtocolEsoundCommand command)
{
  ProtocolEsoundPlayerInfo pinfo;
  ProtocolEsoundSampleInfo sinfo;
  _EnumStruct e;

  if (protocol_esound_SERVER_INFO(client, command) != 0)
    return 2;

  e.id = 1;
  e.client = client;
  e.source = TRUE;
  namespace_foreach(_proc, NAMESPACE_SOURCE, &e);

  e.source = FALSE;
  namespace_foreach(_proc, NAMESPACE_SINK, &e);

  pinfo.source_id = 0;
  if (atomic_write(client->fd, &pinfo, sizeof(pinfo)) != sizeof(pinfo))
    return 2;
 
  sinfo.sample_id = 0;
  if (atomic_write(client->fd, &sinfo, sizeof(sinfo)) != sizeof(sinfo))
    return 2;

  return 0;
  
}

gint protocol_esound_LOCK(ProtocolEsoundClient *client, ProtocolEsoundCommand command)
{
  ProtocolEsoundAuthenticate auth;
  guint32 ok;

  g_assert(client);

  ok = MAYBE_SWAP(1, client->swap);

  // I would like to know why a second authentication is needed? We ignore it
  if (atomic_read(client->fd, &auth, sizeof(auth)) != sizeof(auth))
    return 2;
    
  if (atomic_write(client->fd, &ok, sizeof(ok)) != sizeof(ok))
    return 2;

  if (atomic_write(client->fd, &ok, sizeof(ok)) != sizeof(ok))
    return 2;
    
  protocol_esound_auth_locked = command == ESD_PROTO_LOCK;

  g_message("ESOUND %slocked.", protocol_esound_auth_locked ? "" : "un");

  return 0;
}
