#include <glib.h>

#include <gc.h>
#include <recursive-lock.h>

static GHashTable *_gc = NULL;
static GStaticMutex _gc_mutex = G_STATIC_MUTEX_INIT;
static GStaticPrivate _gc_private = G_STATIC_PRIVATE_INIT;

static GMemChunk *_gc_memory = NULL;
static guint _gc_memory_count = 0;

typedef struct
{
  gpointer data;
  guint ref;
  GcFreeProc proc;
  gchar *name;
} GcEntry;

#define _gc_lock() recursive_lock(&_gc_mutex, &_gc_private)
#define _gc_unlock() recursive_unlock(&_gc_mutex, &_gc_private)

static GcEntry* _gc_entry_new(gpointer data, GcFreeProc proc, gchar *name)
{
  GcEntry *e;

  if (!_gc_memory)
    g_assert(_gc_memory = g_mem_chunk_create(GcEntry, 10, G_ALLOC_AND_FREE));

  g_assert(e = g_chunk_new(GcEntry, _gc_memory));
  _gc_memory_count++;

  e->data = data;
  e->ref = 1;
  e->proc = proc;
  e->name = g_strdup(name);
  return e;
}

static void _gc_entry_free(GcEntry *e)
{
  if (e->proc)
    e->proc(e->data);
  
  g_free(e->name);

  g_assert(e && _gc_memory);
  g_chunk_free(e, _gc_memory);
  _gc_memory_count--;

  if (_gc_memory_count == 0)
    {
      g_mem_chunk_destroy(_gc_memory);
      _gc_memory = NULL;
    }
}

gpointer gc_add(gpointer data, GcFreeProc proc, gchar *name)
{
  g_assert(data);

  _gc_lock();

  if (_gc)
    g_assert(!g_hash_table_lookup(_gc, data));
  else
    _gc = g_hash_table_new(g_direct_hash, g_direct_equal);

  g_hash_table_insert(_gc, data, _gc_entry_new(data, proc, name));

  _gc_unlock();

  return data;
}

gpointer gc_ref_inc(gpointer data)
{
  GcEntry *e;
  g_assert(data && _gc);

  _gc_lock();
  
  g_assert(e = g_hash_table_lookup(_gc, data));
  e->ref++;

  _gc_unlock();
  return data;
}

void gc_ref_dec(gpointer data)
{
  GcEntry *e;
  
  if (!data)
    return;

  g_assert(_gc);

  _gc_lock();
 
  g_assert(e = g_hash_table_lookup(_gc, data));
  g_assert(e->ref > 0);
  
  e->ref--;

  if (e->ref == 0)
    {
      g_hash_table_remove(_gc, data);
      _gc_entry_free(e);

      if (g_hash_table_size(_gc) == 0)
	{
	  g_hash_table_destroy(_gc);
	  _gc = NULL;
	}
    }
      
  _gc_unlock();
}

static gboolean _proc(gpointer key, gpointer value, gpointer user_data)
{
  GcEntry *e;
  g_assert(e = (GcEntry*) value);
  g_printerr("Removing forcibly %p. (%s by %u)\n", e->data, e->name, e->ref);
  _gc_entry_free(e);
  return TRUE;
}

void gc_cleanup()
{
  _gc_lock();
    
  if (_gc)
    {
      guint l = g_hash_table_size(_gc);
      g_printerr("%i objects not removed from gc.\n", l);

      g_hash_table_foreach_remove(_gc, _proc, NULL);
      g_hash_table_destroy(_gc);
      _gc = NULL;
    }

  _gc_unlock();
}

guint gc_ref_count(gpointer data)
{
  GcEntry *e;
  guint ref;

  _gc_lock();
  g_assert(e = g_hash_table_lookup(_gc, data));
  ref = e->ref;
  _gc_unlock();

  return ref;
}

gboolean gc_ref_single(gpointer data)
{
  return gc_ref_count(data) == 1;
}

