# 
# add-pristine.sh: locally cache a pristine revision
################################################################
# Copyright (C) 2001, 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 

################################################################
# special options
# 
# Some options are special:
# 
#       --version | -V
#       --help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
                printf "locally cache a pristine revision\\n"
                printf "usage: add-pristine [options] [archive/]revision\\n"
                printf "\\n"
                printf " -V --version                  print version info\\n"
                printf " -h --help                     display help\\n"
                printf "\\n"
                printf " -R --root root                specify the local archive root\\n"
                printf " -A --archive archive          specify the archive name\\n"
		printf "  --cache DIR                  cache directory for locally cached\\n"
                printf "                                 revisions\\n"
                printf "\\n"
                printf " -d --dir DIR                  cd to DIR first\\n"
                printf "\\n"
		printf " --silent                      no output (except odd errors)\\n"
		printf " --quiet                       brief output\\n"
		printf " --report                      default output\\n"
		printf " --verbose                     maximal output\\n"
		printf " --debug                       debugging output\\n"
		printf "\\n"
                printf " --lock                        lock the pristine revision\\n"
                printf "\\n"
		printf "Extract REVISION from an archive, creating a pristine copy.\\n"
		printf "Store that pristine copy in the current directories project tree\\n"
		printf "\\n"
                exit 0
                ;;

      *)
                ;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 

archroot=
archive=
cache_dir=
lock=

dir=.

quiet=--quiet
report=--report
verbose=
silent_opt=
quiet_opt=
report_opt=
verbose_opt=
debug_opt=
__restart=

while test $# -ne 0 ; do

  case "$1" in 

    -d|--dir)		shift
			if test $# -eq 0 ; then
			  printf "add-pristine: -d and --dir require an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			dir="$1"
			shift
			;;

    ----restart) shift
    		__restart=----restart
		;;

    --lock)		shift
			lock=--lock
			;;

    --silent)	shift
    		quiet=
		report=
		verbose=
		silent_opt=--silent
		quiet_opt=
		report_opt=
		verbose_opt=
		;;

    --quiet)	shift
    		quiet=--quiet
		report=
		verbose=
		silent_opt=
		quiet_opt=--quiet
		report_opt=
		verbose_opt=
		;;

    --report)	shift
    		quiet=--quiet
		report=--report
		verbose=
		silent_opt=
		quiet_opt=
		report_opt=--report
		verbose_opt=
		;;

    --verbose)	shift
    		quiet=--quiet
		report=--report
		verbose=--verbose
		silent_opt=
		quiet_opt=
		report_opt=
		verbose_opt=--verbose
		;;

    --debug)	shift
    		larch heading "add-pristine: debugging output enabled\\n"
    		set -x
		debug_opt=--debug
		;;


    -R|--root)          shift
                        if test $# -eq 0 ; then
                          printf "add-pristine: -R and --root require an argument\\n" 1>&2
                          printf "try --help\\n" 1>&2
                          exit 1
                        fi
                        archroot="$1"
                        shift
                        ;;

    -A|--archive)       shift
                        if test $# -eq 0 ; then
                          printf "add-pristine: -A and --archive require an argument\\n" 1>&2
                          printf "try --help\\n" 1>&2
                          exit 1
                        fi
                        archive="$1"
                        shift
                        ;;

    --cache)            shift
                        if test $# -eq 0 ; then
                          printf "add-pristine: --cache requires an argument\\n" 1>&2
                          printf "try --help\\n" 1>&2
                          exit 1
                        fi
                        cache_dir="$1"
                        shift
                        ;;

    -*)                 printf "add-pristine: unrecognized option (%s)\\n" "$1" 1>&2
                        printf "try --help\\n" 1>&2
                        exit 1
                        ;;

    *)                  break
                        ;;
  esac

done



################################################################
# Ordinary Arguments
# 

if test $# -ne 1 ; then
  printf "usage: add-pristine [options] [archive/]revision\\n" 1>&2
  printf "try --help\\n" 1>&2
  exit 1
fi

rvnspec="$1"
shift

################################################################
# Sanity Check and Process Defaults
# 

cd "$dir"
dir="`pwd`"
wdroot="`larch tree-root`"

archive_revision=`larch indicated-revision -e add-pristine -R "$archroot" -A "$archive" "$rvnspec"`

archive=${archive_revision%%/*}
revision=${archive_revision#*/}

here="`pwd`"
tmpdir="$here/,,cache-revision.$$/"

if test -z "$cache_dir" ; then
  cache_dir="$here"
fi

  
################################################################
# Greetings
# 

if test "(" -z "$__restart" -a ! -z "$quiet" ")" -o ! -z "$report" ; then
  larch heading "add-pristine\\n"
  printf "arguments: %s\\n"  "$command_line" | fold -w 60 | larch body-indent
  larch heading --sub "add-pristine start time: %s\\n" "`date`"
  larch heading --sub "project tree: %s\\n" "$wdroot"
  larch heading --sub "archive: %s\\n" "$archive"
  larch heading --sub "revision: %s\\n" "$revision"
fi

################################################################
# Ensure that We Have an Archive Connection 
# 

if test "$WITHARCHIVE" != "$archive" ; then

  if test ! -z "$quiet" ; then
    larch heading --sub "restarting with connection to archive\\n"
  fi

  exec larch with-archive -A "$archive"  \
	larch add-pristine --cache "$cache_dir" \
		  	  $silent_opt $quiet_opt $report_opt $verbose_opt $debug_opt \
			  ----restart \
			  "$archive/$revision"

fi

################################################################
# Make a Temp Dir
# 

bail()
{
  cd "$here"
  rm -rf "$tmpdir"
  exit 1
}

trap "printf \"add-pristine: interrupted -- cleaning up\\n\" 1>&2 ; bail" INT

mkdir "$tmpdir"
cd "$tmpdir"



################################################################
# Get the Revision
# 

if test ! -z "$quiet" ; then
  larch heading --sub "getting pristine copy\\n"
fi

cd "$tmpdir"
mkdir ,,pristine
cd ,,pristine

if ! larch nested larch build-revision -R "$archroot" -A "$archive" --cache "`dirname \"$wdroot\"`" "$revision" ; then
  printf "add-pristine: unable to get revision\\n" 1>&2
  printf "  archive: %s\\n" "$archive" 1>&2
  printf "  revision: %s\\n" "$revision" 1>&2
  printf "\\n"
  bail
fi

cd "$tmpdir"

if test ! -z "$quiet" ; then
  larch heading --sub "storing pristine revision\\n"
fi

larch wd-mv-pristine $lock "$archive" "$revision" "$tmpdir/,,pristine" "$wdroot"
cd "$here"
rm -rf "$tmpdir"


if test ! -z "$quiet" ; then
  larch heading --sub "add-pristine finish time: %s\\n" "`date`"
fi

# tag: Tom Lord Tue Dec 11 12:12:43 2001 (local-cache/add-pristine.sh)
#
