#
# bts.rb - ruby interface for debian bts
# Copyright (c) 2002 Masato Taruishi <taru@debian.org>
# Copyright (c) 2006 Junichi Uekawa <dancer@debian.org>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# Currently, this interface has only acquires to create bugs.

require 'debian/bug'
require 'net/http'
require 'uri.rb'
require 'find'
require 'ftools'
require 'zlib'
require 'thread'

module Debian
  module BTS
    class Acquire
      module Cache

	def create_cache_dir(cache_dir)
	  ::File.mkpath(cache_dir)
	end

	def expire(cache_dir, timer)
	  now = Time.now
	  create_cache_dir(cache_dir)
	  Find.find(cache_dir) { |path|
	    if FileTest.file?(path)
	      mtime = ::File.mtime(path)
	      if now - mtime > timer
		::File.delete(path)
	      end
	    end
          }
	end

	def encode(path)
	  URI.escape(path, /\/|&/)
	end

	def read(cache_dir, path)
	  cache_path = cache_dir + "/" + encode(path)
	  if FileTest.exist?(cache_path)
	    open(cache_path) { |file|
	      file.read
            }
	  else
	    nil
	  end
	end

	def write(cache_dir, path, body)
	  cache_path = cache_dir + "/" + encode(path)
	  open(cache_path, "w") { |file|
            file.write body
          }
	end

	module_function :expire, :read, :write, :encode,
	  :create_cache_dir

      end

      def initialize(cache_dir = nil, timer = nil)
	@cache_dir = cache_dir
	begin
          Cache.expire(@cache_dir, timer) if @cache_dir != nil
        rescue Errno::EACCES
          $stderr.puts "W: #{$!}"
          @cache_dir = nil
        end
	@use_cache = (@cache_dir != nil)
	@mutex = Mutex.new
        @mutex_hash = {}
	@buf = {}
	@user_agent = nil
      end

      def read (path, use_cache = @use_cache)
	$stderr.puts "Reading #{path}... " if $DEBUG
	@mutex.synchronize {
          @mutex_hash[path] = Mutex.new if @mutex_hash[path] == nil 
	}
        @mutex_hash[path].synchronize {
          if @buf[path] == nil && use_cache
	    @buf[path] = Cache.read(@cache_dir, encode(path))
          end
	  if @buf[path] == nil
	    @buf[path] = read_real(path)
	    begin
              Cache.write(@cache_dir, encode(path), @buf[path]) if @cache_dir
            rescue Errno::EACCES
              $stderr.puts "W: #{$!}"
            end
	  end
        }
	@buf[path]
      end

      def write(path, buf)
	Cache.write(@cache_dir, encode(path), buf)
      end

      def cache_path(path)
	@cache_dir + "/" + encode(path)
      end

      def read_real(path)
	raise "Not Implemented"
      end

      def encode(path)
	raise "Not Implemented"
      end

      attr_accessor :use_cache, :host, :port

      class HTTP < Acquire
	def initialize(host = "bugs.debian.org", port = 80, cache_dir = nil, timer = nil)
	  super(cache_dir, timer)
	  @host = host
	  @port = port
	  @extra_headers = {}
	  if ENV["http_proxy"] != nil
	    uri = URI::split(ENV["http_proxy"])
            @proxy_user, @proxy_pass = uri[1].split(/:/) if uri[1] != nil
	    @proxy_host = uri[2]
	    @proxy_port = uri[3]
	    begin
	      Net::HTTP::Proxy(@proxy_host, @proxy_port).start(@host,@port)
	    rescue
	      $stderr.puts "Disabling unavailable proxy configuration: #{ENV["http_proxy"]}: #{$!}"
              @proxy_host = @proxy_port = @proxy_user = @proxy_pass = nil
	    end
	    puts "http_proxy: #{ENV['http_proxy']} : #{@proxy_user},#{@proxy_pass},#{@proxy_host},#{@proxy_port}" if $DEBUG
	  end
          puts "http://#{host}:#{port}/, user-auth=#{@proxy_user}:#{@proxy_pass}" if $DEBUG
	end
        
        def fetch( uri_str, extra_headers, limit = 10 )
          # internal: fetch the given URL with HTTP redirect support.
          
          puts "  fetch #{uri_str}" if $DEBUG
          raise ArgumentError, 'http redirect too deep' if limit == 0
          
          response = Net::HTTP::Proxy(@proxy_host, @proxy_port, @proxy_user, @proxy_pass).get_response(URI.parse(uri_str))
          case response
          when Net::HTTPSuccess     
            response
          when Net::HTTPRedirection 
            fetch(response['Location'], extra_headers, limit - 1)
          else
            response.error!
          end
        end

	def read_real(path)
          # HTTP get.

          puts "getting #{path}" if $DEBUG
          val = nil
          val = fetch("http://#{@host}:#{@port}#{path}", @extra_headers).body
          val
	end

	def encode(path)
	  URI.escape(path, /\/|&/)
	end

	def user_agent
	  @extra_headers['User-Agent']
	end

	def user_agent=(agent)
	  @extra_headers['User-Agent'] = agent
	end

      end

      class File < Acquire

        def initialize(dir=".")
          super( )
          @dir = dir
        end

        def read_real(path)
          $stderr.puts "File: read #{path}" if $DEBUG
          if @buf[path] == nil
            open(@dir + "/" + path) { |io|
              @buf[path] = io.read
            }
          end
          @buf[path]
        end

      end
    end


    class Parser

      def initialize(acquire)
	@acquire = acquire
      end

      attr_reader :acquire

      class Index < Parser

        class StringIO
          def initialize(buf)
            @buf = buf
	    @ptr = 0
	  end

          def read ( size = nil )
	    if @buf == nil
	      return nil
	    end 
            if size
              buf = @buf[@ptr,size]
              @ptr += size
              if @buf.size < @ptr
                @buf = nil
              end
	    else
	      buf = @buf
	      @buf = nil
	    end
            buf
	  end
	end

	# pkg   num  date? status submitter severity tags
	REGEX = /^(\S+) (\d+) (\d+) (\S+) \[(.+)\] (\S+)( (.+))?/o

	def initialize(acquire, indexdir = "/")
	  super(acquire)
	  @indexes = {}
	  @buf = nil
	  @indexdir = indexdir
	  puts "indexdir = #{@indexdir}" if $DEBUG
	end

        ParseStep = 200

	def parse(pkgs, severities = ["critical", "grave"])
          require 'debian/btssoap'
          soap = Debian::BTSSOAP::Soap.new(@acquire.host, @acquire.port)
          sa = Debian::BTSSOAP::StringArray.new
          fetched = Debian::Bugs.new
          bugs = Debian::Bugs.new
          offset_i = 0 

          # very slow part of the process; try to give feedback.
          # obtains BTS contents via SOAP.
          max = severities.inject(0) { |result, severity|
            result + number_of_bugs(severity)
          }
          severities.each { |severity|
            each_bug_with_index(severity, pkgs) { |bug, i|
              sa << bug.bug_number.to_s
              if sa.length > ParseStep
                soap.get_status(sa).each { |b| bugs << b }
                sa = Debian::BTSSOAP::StringArray.new
                yield "#{((offset_i+i).to_f*100/max.to_f).to_i}%"
              end
            }
            offset_i += number_of_bugs(severity)
          }
          soap.get_status(sa).each { |b| bugs << b }
          bugs
	end

        private
        def number_of_bugs(severity)
          read_index(severity).to_a.size
        end

        def read_index(severity)
          if @indexes[severity]
            @indexes[severity]
          else
            path=@indexdir + "index.db-#{severity}.gz"
            puts "reading #{path}.. " if $DEBUG
            io = StringIO.new( @acquire.read(path) )
            index = Zlib::GzipReader.wrap(io) { |input|
              input.read
            }
            @indexes[severity] = index
            index
          end
        end

        def each_bug_with_index(severity, pkgs=[])
          return if pkgs.empty?
          read_index(severity).each_with_index { |line, i|
            REGEX =~ line
            pkgname = $1
            next unless pkgs.include?(pkgname)
            id = $2
            time = $3
            sev = $6
            stat = $4
            tag = $7.split(' ') if $7 != nil
            #             pkg, id, sev, st, desc, tag, merge 
            bug = Bug.new(pkgname, id,  sev, stat, "", tag, [], Time::at(time.to_i))
            yield(bug, i)
          }
        end
      end
    end      
  end
end
