#! /usr/bin/perl
# This is a library file for Apt-cacher to allow code
# common to Apt-cacher itself plus its supporting scripts
# (apt-cacher-report.pl and apt-cacher-cleanup.pl) to be
# maintained in one location.

# This function reads the given config file into the
# given hash ref. The key and value are separated by
# a '=' and will have all the leading and trailing 
# spaces removed.
sub read_config
{
	# set the default config variables
	my $config = {
			cache_dir => '/var/log/cache/apt-cacher',
			logdir => '/var/log/apt-cacher',
			admin_email => 'root@localhost',
			generate_reports => 0,
			expire_hours => 36,
			http_proxy => 'proxy.example.com:8080',
			use_proxy => 0,
			debug => 0,
			clean_cache => 0,
	};

	($config_file) = @_;

	open CONFIG, $config_file or die $!;

	while (<CONFIG>)
	{
		s/#.*$//;	# kill off comments
		s/^\s+//;	# kill off leading spaces
		s/\s+$//;	# kill off trailing spaces
		if ($_)
		{
			my ($key, $value) = split(/\s*=\s*/);	# split into key and value pair
			$value = 0 unless ($value);
			#print "key: $key, value: $value\n";
			$config->{$key} = $value;
			#print "$config{$key}\n";
		}
	}

	close CONFIG;

	return $config;
}

# Convert a human-readable IPv4 address to raw form (4-byte string)
# Returns undef if the address is invalid
sub ipv4_normalise ($)
{
	return undef if $_[0] =~ /:/;
	my @in = split (/\./, $_[0]);
	return '' if $#in != 3;
	my $out = '';
	foreach my $num (@in)
	{
		return undef if $num !~ /^[[:digit:]]{1,3}$/o;
		$out .= pack ("C", $num);
	}
	return $out;
}

# Convert a human-readable IPv6 address to raw form (16-byte string)
# Returns undef if the address is invalid
sub ipv6_normalise ($)
{
	return "\0" x 16 if $_[0] eq '::';
	return undef if $_[0] =~ /^:[^:]/  || $_[0] =~ /[^:]:$/ || $_[0] =~ /::.*::/;
	my @in = split (/:/, $_[0]);
	return undef if $#in > 7;
	shift @in if $#in >= 1 && $in[0] eq '' && $in[1] eq ''; # handle ::1 etc.
	my $num;
	my $out = '';
	my $tail = '';
	while (defined ($num = shift @in) && $num ne '')
	{
		return undef if $num !~ /^[[:xdigit:]]{1,4}$/o;
		$out .= pack ("n", hex $num);
	}
	foreach $num (@in)
	{
		return undef if $num !~ /^[[:xdigit:]]{1,4}$/o;
		$tail .= pack ("n", hex $num);
	}
	my $l = length ($out.$tail);
	return $out.("\0" x (16 - $l)).$tail if $l < 16;
	return $out.$tail if $l == 16;
	return undef;
}

# Make a netmask from a CIDR network-part length and the IP address length
sub make_mask ($$)
{
	my ($mask, $bits) = @_;
	return undef if $mask < 0 || $mask > $bits;
	my $m = ("\xFF" x ($mask / 8));
	$m .= chr ((-1 << (8 - $mask % 8)) & 255) if $mask % 8;
	return $m . ("\0" x ($bits / 8 - length ($m)));
}

sub extract_sums {
   $_=shift;
   $hashref=shift;

   my ($cat, $listpipe);
   $cat = (/bz2$/ ? "bzcat" : (/gz$/ ? "zcat" : "cat"));

   # lock it or wait
   open($lck, $_);
   flock($lck, LOCK_EX);

   open($listpipe, "-|", $cat, $_);
   my $file;
   while(<$listpipe>) {
      if(/^\s(\w{32})\s\d+\s(\S+)\n/) {
         $sum=$1;
         $file=$2;
      }
      elsif(/^MD5sum:\s+(.*)$/) {
         $sum=$1;
      }
      elsif(/^Filename:\s+(.*)$/) {
         $file=$1;
         $file=~s/.*\///;
      }
      if(defined($file) && defined($sum)) {
         $$hashref{$file}=$sum;
         undef $file;
         undef $sum;
      }
   };
   flock($lck, LOCK_UN);
}

######### HOOKS ###########
#
# arg: file to be scanned and added to DB
sub import_sums {
   return 1;
}

# purpose: ?create?, lock the DB file and establish DB connection
sub db_init {
   return 1;
}

# purpose: create hasher object
sub data_init {
   return 1;
}

# purpose: append data to be scanned
sub data_feed {
   return 1;
}

# args: filename only or filename and sum
sub check_sum {
   return 1;
}



1;
