/* Copyright (C) 2002, 2003, 2004 Jan Wedekind.
   This file is part of the recipe database application AnyMeal.

   AnyMeal is free software; you can redistribute it and/or modify it under
   the terms of the GNU GENERAL PUBLIC LICENSE as published by the Free
   Software Foundation; either version 2 of the License, or (at your option)
   any later version.

   AnyMeal is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTIBILITY or FITNESS
   FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
   details.

   You should have received a copy of the GNU General Public License along
   with AnyMeal; if not, contact one of the authors of this software. */
#ifndef __PARSEERROR_HPP
#define __PARSEERROR_HPP

#include <boost/shared_ptr.hpp>
#include <iostream>
#include "error.hpp"

/** Exception for an error during parsing.
    @date Sun Jan 09 2005
    @author Jan Wedekind (wedesoft@users.sourceforge.net) */
class ParseError: public Error {
public:
  /** Constructor.
      The exception-object contains the information, which is necessary
      to extract the erroneous recipe from the stream. Also an
      error-description is provided.
      @param _recipeStartLine Number of line, where recipe is starting.
      @param _recipeEndLine Number of line, where recipe is ending.
      @param _errorLineNumber Number of line, where error was found.
      @param _stream Pointer to stream, where the recipes are read from.
      @param _recipeStartPos Starting position of erroneous recipe.
      @param _recipeEndPos Ending position of erroneous recipe. */
  ParseError( int _recipeStartLine, int _recipeEndLine,
              int _errorLineNumber, std::istream *_stream,
              std::streampos _recipeStartPos,
              std::streampos _recipeEndPos );
  ///
  virtual ~ParseError(void) throw() {}
  /// Get starting position of recipe.
  std::streampos getRecipeStartPos(void) const { return recipeStartPos; }
  /// Get ending position of recipe.
  std::streampos getRecipeEndPos(void) const { return recipeEndPos; }
  /// Get line-number of error.
  int getErrorLineNumber(void) const { return errorLineNumber; }
  /// Get access to input stream.
  std::istream &getInputStream(void) const { return *stream; }
  /// Get starting position of recipe.
  int getRecipeStartLine(void) const { return recipeStartLine; }
  /// Get ending position of recipe.
  int getRecipeEndLine(void) const { return recipeEndLine; }
protected:
  /// Number of first line of recipe.
  int recipeStartLine;
  /// Line number of recipe footer.
  int recipeEndLine;
  /// Line number of line, where syntax error was found.
  int errorLineNumber;
  /// Pointer to input stream, where the recipes are read from.
  std::istream *stream;
  /// Starting position of erroneous recipe.
  std::streampos recipeStartPos;
  /// Ending position of erroneous recipe.
  std::streampos recipeEndPos;
};

///
typedef boost::shared_ptr< ParseError > ParseErrorPtr;

#endif
