
/*
    card-als100.c - driver for Avance Logic ALS100 based soundcards.
    Copyright (C) 1999-2000 by Massimo Piccioni <dafastidio@libero.it>

    Thanks to Pierfrancesco 'qM2' Passerini.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#define __SND_OSS_COMPAT__
#define SND_MAIN_OBJECT_FILE
#include "../include/driver.h"
#include "../include/initval.h"
#include "../include/mpu401.h"
#include "../include/opl3.h"
#include "../include/sb.h"

EXPORT_NO_SYMBOLS;
MODULE_DESCRIPTION("\
Driver: Avance Logic ALS1X0\n\
Card: Avance Logic ALS100\n\
Card: Avance Logic ALS110\n\
Card: Avance Logic ALS120\n\
Card: Avance Logic ALS200\n\
Card: 3D Melody MF1000\n\
Card: Digimate 3D Sound\n\
ISAPNP: ALS0001=@@@0001,@X@0001,@H@0001\n\
ISAPNP: ALS0110=@@@1001,@X@1001,@H@1001\n\
ISAPNP: ALS0120=@@@2001,@X@2001,@H@2001\n\
ISAPNP: ALS0200=@@@0020,@X@0020,@H@0001\n\
ISAPNP: RTL3000=@@@2001,@X@2001,@H@2001\n\
");
MODULE_LICENSE("GPL");

int snd_index[SND_CARDS] = SND_DEFAULT_IDX;		/* Index 0-MAX */
char *snd_id[SND_CARDS] = SND_DEFAULT_STR;		/* ID for this card */
int snd_port[SND_CARDS] = SND_DEFAULT_PORT;		/* PnP setup */
int snd_mpu_port[SND_CARDS] = SND_DEFAULT_PORT;		/* PnP setup */
int snd_fm_port[SND_CARDS] = SND_DEFAULT_PORT;		/* PnP setup */
int snd_irq[SND_CARDS] = SND_DEFAULT_IRQ;		/* PnP setup */
int snd_mpu_irq[SND_CARDS] = SND_DEFAULT_IRQ;		/* PnP setup */
int snd_dma8[SND_CARDS] = SND_DEFAULT_DMA;		/* PnP setup */
int snd_dma8_size[SND_CARDS] = SND_DEFAULT_DMA_SIZE;	/* 8,16,32,64,128 */
int snd_dma16[SND_CARDS] = SND_DEFAULT_DMA;		/* PnP setup */
int snd_dma16_size[SND_CARDS] = SND_DEFAULT_DMA_SIZE;	/* 8,16,32,64,128 */

#ifdef MODULE_PARM
MODULE_PARM(snd_index, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_index, "Index value for als100 based soundcard.");
MODULE_PARM(snd_id, "1-" __MODULE_STRING(SND_CARDS) "s");
MODULE_PARM_DESC(snd_id, "ID string for als100 based soundcard.");
MODULE_PARM(snd_port, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_port, "Port # for als100 driver. [PORT12]");
MODULE_PARM(snd_mpu_port, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_mpu_port, "MPU-401 port # for als100 driver. [PORT12]");
MODULE_PARM(snd_fm_port, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_fm_port, "FM port # for als100 driver. [PORT12]");
MODULE_PARM(snd_irq, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_irq, "IRQ # for als100 driver. [IRQ]");
MODULE_PARM(snd_mpu_irq, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_mpu_irq, "MPU-401 IRQ # for als100 driver. [IRQ]");
MODULE_PARM(snd_dma8, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_dma8, "8-bit DMA # for als100 driver. [DMA8]");
MODULE_PARM(snd_dma16, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_dma16, "16-bit DMA # for als100 driver. [DMA16]");
MODULE_PARM(snd_dma8_size, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_dma8_size, "Size of 8-bit DMA # for als100 driver. [DMA8_SIZE]");
MODULE_PARM(snd_dma16_size, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_dma16_size, "Size of 16-bit DMA # for als100 driver. [DMA16_SIZE]");
#endif	/* MODULE_PARM */

static struct snd_card_als100 {
	snd_irq_t *irqptr;
	snd_irq_t *mpuirqptr;
	snd_dma_t *dma8ptr;
	snd_dma_t *dma16ptr;
	snd_card_t *card;
	snd_pcm_t *pcm;
	snd_kmixer_t *mixer;
	snd_rawmidi_t *rmidi;
	snd_hwdep_t *synth;
#ifdef __ISAPNP__
	struct isapnp_dev *dev;
	struct isapnp_dev *devmpu;
	struct isapnp_dev *devopl;
#endif	/* __ISAPNP__ */
} *snd_card_als100_cards[SND_CARDS] = SND_DEFAULT_PTR;

#ifdef __ISAPNP__
static unsigned int snd_card_als100_pnpids[] = {
	/* ALS100 - PRO16PNP */
	(ISAPNP_VENDOR('A','L','S')<<16)|ISAPNP_DEVICE(0x0001),    /* device */
	(ISAPNP_VENDOR('@','@','@')<<16)|ISAPNP_FUNCTION(0x0001),  /* audio */
	(ISAPNP_VENDOR('@','X','@')<<16)|ISAPNP_FUNCTION(0x0001),  /* mpu401 */
	(ISAPNP_VENDOR('@','H','@')<<16)|ISAPNP_FUNCTION(0x0001),  /* opl */
	/* ALS110 - MF1000 - Digimate 3D Sound */
	(ISAPNP_VENDOR('A','L','S')<<16)|ISAPNP_DEVICE(0x0110),    /* device */
	(ISAPNP_VENDOR('@','@','@')<<16)|ISAPNP_FUNCTION(0x1001),  /* audio */
	(ISAPNP_VENDOR('@','X','@')<<16)|ISAPNP_FUNCTION(0x1001),  /* mpu401 */
	(ISAPNP_VENDOR('@','H','@')<<16)|ISAPNP_FUNCTION(0x1001),  /* opl */
	/* ALS120 */
	(ISAPNP_VENDOR('A','L','S')<<16)|ISAPNP_DEVICE(0x0120),    /* device */
	(ISAPNP_VENDOR('@','@','@')<<16)|ISAPNP_FUNCTION(0x2001),  /* audio */
	(ISAPNP_VENDOR('@','X','@')<<16)|ISAPNP_FUNCTION(0x2001),  /* mpu401 */
	(ISAPNP_VENDOR('@','H','@')<<16)|ISAPNP_FUNCTION(0x2001),  /* opl */
	/* ALS200 */
	(ISAPNP_VENDOR('A','L','S')<<16)|ISAPNP_DEVICE(0x0200),    /* device */
	(ISAPNP_VENDOR('@','@','@')<<16)|ISAPNP_FUNCTION(0x0020),  /* audio */
	(ISAPNP_VENDOR('@','X','@')<<16)|ISAPNP_FUNCTION(0x0020),  /* mpu401 */
	(ISAPNP_VENDOR('@','H','@')<<16)|ISAPNP_FUNCTION(0x0001),  /* opl */
	/* RTL3000 */
	(ISAPNP_VENDOR('R','T','L')<<16)|ISAPNP_DEVICE(0x3000),    /* device */
	(ISAPNP_VENDOR('@','@','@')<<16)|ISAPNP_FUNCTION(0x2001),  /* audio */
	(ISAPNP_VENDOR('@','X','@')<<16)|ISAPNP_FUNCTION(0x2001),  /* mpu401 */
	(ISAPNP_VENDOR('@','H','@')<<16)|ISAPNP_FUNCTION(0x2001),  /* opl */
	0       /* end */
};
#endif	/* __ISAPNP__ */

#define DRIVER_NAME	"snd-card-als100"

#define snd_als100_printk(format, args...)	snd_printk(__FILE__": " format, ##args)


static void snd_card_als100_use_inc(snd_card_t *card)
{
	MOD_INC_USE_COUNT;
}

static void snd_card_als100_use_dec(snd_card_t *card)
{
	MOD_DEC_USE_COUNT;
}

static void snd_card_als100_interrupt(int irq, void *dev_id,
				      struct pt_regs *regs)
{
	struct snd_card_als100 *acard;
	snd_pcm_t *pcm;
	sbdsp_t *sb16dsp;
	unsigned long flags;
	unsigned short status;

	if (!(acard = (struct snd_card_als100 *) dev_id))
		return;

	if (!(pcm = acard->pcm))
		return;

	sb16dsp = snd_magic_cast(sbdsp_t, pcm->private_data, );
	if (sb16dsp == NULL)
		return;

	spin_lock_irqsave(&sb16dsp->mixer.lock, flags);
	status = snd_sb16mixer_read(&sb16dsp->mixer, SB_DSP4_IRQSTATUS);
	spin_unlock_irqrestore(&sb16dsp->mixer.lock, flags);

	if (status & 0x03)
		snd_sb16dsp_interrupt(pcm, status);
}

static void snd_card_als100_mpu_interrupt(int irq, void *dev_id,
					  struct pt_regs *regs)
{
	struct snd_card_als100 *acard;

	if (!(acard = (struct snd_card_als100 *) dev_id))
		return;

	snd_mpu401_uart_interrupt(acard->rmidi);
}

static int __init snd_card_als100_resources(int dev,
					    struct snd_card_als100 *acard,
					    snd_card_t *card)
{
	int error;
	static long no_alternatives[] = {-1};

	if (snd_register_ioport(card, snd_mpu_port[dev], 2,
			DRIVER_NAME" - MPU-401", NULL) < 0)
		snd_mpu_port[dev] = -1;
	else if (snd_register_interrupt(card,
			DRIVER_NAME" - MPU-401", snd_mpu_irq[dev],
			SND_IRQ_TYPE_ISA, snd_card_als100_mpu_interrupt,
			acard, no_alternatives, &acard->mpuirqptr) < 0) {
		snd_unregister_ioports(card);
		snd_mpu_port[dev] = -1;
	}

	if ((error = snd_register_ioport(card, snd_port[dev], 16,
			DRIVER_NAME" - DSP", NULL)) < 0)
		return error;
	if ((error = snd_register_interrupt(card,
			DRIVER_NAME" - DSP", snd_irq[dev],
			SND_IRQ_TYPE_ISA, snd_card_als100_interrupt,
			acard, no_alternatives, &acard->irqptr)) < 0)
		return error;
	if ((error = snd_register_dma_channel(card,
			DRIVER_NAME" - DSP", snd_dma8[dev],
			SND_DMA_TYPE_ISA, snd_dma8_size[dev],
			no_alternatives, &acard->dma8ptr)) < 0)
		return error;
	if ((error = snd_register_dma_channel(card,
			DRIVER_NAME" - DSP", snd_dma16[dev],
			SND_DMA_TYPE_ISA, snd_dma16_size[dev],
			no_alternatives, &acard->dma16ptr)) < 0)
		return error;
	
	if (snd_register_ioport(card, snd_fm_port[dev], 4,
			DRIVER_NAME" - OPL", NULL) < 0)
		snd_fm_port[dev] = -1;

	return 0;
}

#ifdef __ISAPNP__
static int __init snd_card_als100_isapnp(int dev, struct snd_card_als100 *acard)
{
	static int idx = 0;
	static struct isapnp_card *card = NULL;
        unsigned int tmp;
	struct isapnp_dev *pdev;

      __again:
	while ((tmp = snd_card_als100_pnpids[idx]) != 0) {
		if ((card = isapnp_find_card(tmp >> 16, tmp & 0xffff, card)))
			break;
		idx += 4;
	}
	if (card == NULL)
		return -ENODEV;

	tmp = snd_card_als100_pnpids[idx+1];
	acard->dev = isapnp_find_dev(card, tmp >> 16, tmp & 0xffff, NULL);
	if (acard->dev == NULL)
		goto __again;
	if (acard->dev->active)
		goto __again;

	tmp = snd_card_als100_pnpids[idx+2];
	acard->devmpu = isapnp_find_dev(card, tmp >> 16, tmp & 0xffff, NULL);
	tmp = snd_card_als100_pnpids[idx+3];
	acard->devopl = isapnp_find_dev(card, tmp >> 16, tmp & 0xffff, NULL);

	pdev = acard->dev;
	if (pdev->prepare(pdev)<0)
		return -EAGAIN;

	if (snd_port[dev] != SND_AUTO_PORT)
		isapnp_resource_change(&pdev->resource[0], snd_port[dev], 16);
	if (snd_dma8[dev] != SND_AUTO_DMA)
		isapnp_resource_change(&pdev->dma_resource[0], snd_dma8[dev],
			1);
	if (snd_dma16[dev] != SND_AUTO_DMA)
		isapnp_resource_change(&pdev->dma_resource[1], snd_dma16[dev],
			1);
	if (snd_irq[dev] != SND_AUTO_IRQ)
		isapnp_resource_change(&pdev->irq_resource[0], snd_irq[dev], 1);

	if (pdev->activate(pdev)<0) {
		snd_als100_printk("AUDIO isapnp configure failure\n");
		return -EBUSY;
	}

	snd_port[dev] = pdev->resource[0].start;
	snd_dma8[dev] = pdev->dma_resource[1].start;
	snd_dma16[dev] = pdev->dma_resource[0].start;
	snd_irq[dev] = pdev->irq_resource[0].start;

	pdev = acard->devmpu;
	if (pdev == NULL || pdev->prepare(pdev)<0) {
		snd_mpu_port[dev] = -1;
		return 0;
	}

	if (snd_mpu_port[dev] != SND_AUTO_PORT)
		isapnp_resource_change(&pdev->resource[0], snd_mpu_port[dev],
			2);
	if (snd_mpu_irq[dev] != SND_AUTO_IRQ)
		isapnp_resource_change(&pdev->irq_resource[0], snd_mpu_irq[dev],
			1);

	if (pdev->activate(pdev)<0) {
		snd_als100_printk("MPU-401 isapnp configure failure\n");
		snd_mpu_port[dev] = -1;
		acard->devmpu = NULL;
	} else {
		snd_mpu_port[dev] = pdev->resource[0].start;
		snd_mpu_irq[dev] = pdev->irq_resource[0].start;
	}

	pdev = acard->devopl;
	if (pdev == NULL || pdev->prepare(pdev)<0) {
		snd_fm_port[dev] = -1;
		return 0;
	}

	if (snd_fm_port[dev] != SND_AUTO_PORT)
		isapnp_resource_change(&pdev->resource[0], snd_fm_port[dev], 4);

	if (pdev->activate(pdev)<0) {
		snd_als100_printk("OPL isapnp configure failure\n");
		snd_fm_port[dev] = -1;
		acard->devopl = NULL;
	} else {
		snd_fm_port[dev] = pdev->resource[0].start;
	}

	return 0;
}

static void snd_card_als100_deactivate(struct snd_card_als100 *acard)
{
	if (acard->dev)
		acard->dev->deactivate(acard->dev);
	if (acard->devmpu)
		acard->devmpu->deactivate(acard->devmpu);
	if (acard->devopl)
		acard->devopl->deactivate(acard->devopl);
}
#endif	/* __ISAPNP__ */

static int __init snd_card_als100_probe(int dev, struct snd_card_als100 *acard)
{
	int error;
	sbdsp_t *codec;
	snd_card_t *card;
	snd_pcm_t *pcm;
	snd_kmixer_t *mixer;
	snd_rawmidi_t *rmidi;
	snd_hwdep_t *synth;

	if (!(card = snd_card_new(snd_index[dev], snd_id[dev],
			snd_card_als100_use_inc, snd_card_als100_use_dec)))
		return -ENOMEM;
	card->type = SND_CARD_TYPE_ALS100;

#ifdef __ISAPNP__
	if ((error = snd_card_als100_isapnp(dev, acard)))
		goto __on_error;
#else
	snd_als100_printk("you have to enable PnP support ...\n");
	return -ENOSYS;
#endif	/* __ISAPNP__ */

	if ((error = snd_card_als100_resources(dev, acard, card)))
		goto __on_error;

	if (snd_sb16dsp_new_pcm(card, 0, snd_port[dev], acard->irqptr,
		        acard->dma8ptr, acard->dma16ptr,
		        SB_HW_ALS100, &pcm) < 0)
		goto __on_error;

	codec = snd_magic_cast(sbdsp_t, pcm->private_data, -ENXIO);
	if (snd_sb16dsp_new_mixer(codec, 0, pcm, &mixer) < 0)
		goto __on_error;

	rmidi = NULL;
	if (snd_mpu_port[dev] > 0) {
		if (snd_mpu401_uart_new(card, 0, MPU401_HW_ALS100,
				snd_mpu_port[dev], snd_mpu_irq[dev],
				&rmidi) < 0)
			snd_als100_printk("no MPU-401device at 0x%x ?\n",
				snd_mpu_port[dev]);
	}

	synth = NULL;
	if (snd_fm_port[dev] > 0) {
		if (snd_opl3_new(card, 0,
				snd_fm_port[dev], snd_fm_port[dev] + 2,
				OPL3_HW_AUTO, 0, &synth) < 0)
			snd_als100_printk("no OPL device at 0x%x-0x%x ?\n",
				snd_fm_port[dev], snd_fm_port[dev] + 2);
	}

	strcpy(card->abbreviation, "ALS100");
	strcpy(card->shortname, "Avance Logic ALS100");
	sprintf(card->longname, "%s soundcard, %s at 0x%lx, irq %li, dma %li&%li",
		card->shortname, codec->name, codec->port,
		acard->irqptr->irq, acard->dma8ptr->dma, acard->dma16ptr->dma);
	if (!snd_card_register(card)) {
		acard->card = card;
		acard->pcm = pcm;
		acard->mixer = mixer;
		acard->rmidi = rmidi;
		acard->synth = synth;
		return 0;
	}

      __on_error:
	snd_card_free(card);
	return error ? error : -ENXIO;
}

#ifdef MODULE

static void __exit snd_card_als100_free(int dev)
{
	struct snd_card_als100 *acard;

	if ((acard = snd_card_als100_cards[dev])) {
		snd_card_als100_cards[dev] = NULL;
		snd_card_unregister(acard->card);
#ifdef __ISAPNP__
		snd_card_als100_deactivate(acard);
#endif	/* __ISAPNP__ */
		snd_kfree(acard);
	}
}

#endif

static int __init alsa_card_als100_init(void)
{
	int cards, dev;
	struct snd_card_als100 *acard;

	for (dev = cards = 0; dev < SND_CARDS; dev++) {
		acard = (struct snd_card_als100 *) snd_kcalloc(
			sizeof(struct snd_card_als100), GFP_KERNEL);
		if (acard == NULL)
			break;

		if (snd_card_als100_probe(dev, acard) < 0) {
#ifdef __ISAPNP__
			snd_card_als100_deactivate(acard);
#endif	/* __ISAPNP__ */
			snd_kfree(acard);
			continue;
		}
		snd_card_als100_cards[cards++] = acard;
	}

#ifdef MODULE
	if (!cards)
		snd_als100_printk("no ALS100 based soundcards found\n");
#endif
	return cards ? 0 : -ENODEV;
}

static void __exit alsa_card_als100_exit(void)
{
	int dev;

	for (dev = 0; dev < SND_CARDS; dev++)
		snd_card_als100_free(dev);
}

module_init(alsa_card_als100_init)
module_exit(alsa_card_als100_exit)
