/*
 * OSS compatible sequencer driver
 *
 * Copyright (C) 1998,99 Takashi Iwai <iwai@ww.uni-erlangen.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "seq_oss_device.h"
#include "seq_oss_synth.h"
#include "seq_oss_midi.h"
#include "seq_oss_event.h"
#include "../../../include/seq_oss_legacy.h"
#include "seq_oss_readq.h"
#include "seq_oss_writeq.h"


/*
 * prototypes
 */
static int extended_event(seq_oss_devinfo_t *dp, evrec_t *q, snd_seq_event_t *ev);
static int chn_voice_event(seq_oss_devinfo_t *dp, evrec_t *event_rec, snd_seq_event_t *ev);
static int chn_common_event(seq_oss_devinfo_t *dp, evrec_t *event_rec, snd_seq_event_t *ev);
static int timing_event(seq_oss_devinfo_t *dp, evrec_t *event_rec, snd_seq_event_t *ev);
static int local_event(seq_oss_devinfo_t *dp, evrec_t *event_rec, snd_seq_event_t *ev);
static int old_event(seq_oss_devinfo_t *dp, evrec_t *q, snd_seq_event_t *ev);
static int note_on_event(seq_oss_devinfo_t *dp, int dev, int ch, int note, int vel, snd_seq_event_t *ev);
static int note_off_event(seq_oss_devinfo_t *dp, int dev, int ch, int note, int vel, snd_seq_event_t *ev);
static int set_note_event(seq_oss_devinfo_t *dp, int dev, int type, int ch, int note, int vel, snd_seq_event_t *ev);
static int set_control_event(seq_oss_devinfo_t *dp, int dev, int type, int ch, int param, int val, snd_seq_event_t *ev);
static int set_echo_event(seq_oss_devinfo_t *dp, unsigned int echo, snd_seq_event_t *ev);


/*
 * convert an OSS event to ALSA event
 * return 0 : enqueued
 *        non-zero : invalid - ignored
 */

int
snd_seq_oss_process_event(seq_oss_devinfo_t *dp, evrec_t *q, snd_seq_event_t *ev)
{
	switch (q->s.code) {
	case SND_OSS_SEQ_EXTENDED:
		return extended_event(dp, q, ev);

	case SND_OSS_EV_CHN_VOICE:
		return chn_voice_event(dp, q, ev);

	case SND_OSS_EV_CHN_COMMON:
		return chn_common_event(dp, q, ev);

	case SND_OSS_EV_TIMING:
		return timing_event(dp, q, ev);

	case SND_OSS_EV_SEQ_LOCAL:
		return local_event(dp, q, ev);

	case SND_OSS_EV_SYSEX:
		return snd_seq_oss_synth_sysex(dp, q->x.dev, q->x.buf, ev);

	case SND_OSS_SEQ_MIDIPUTC:
		/* put a midi byte */
		if (! is_write_mode(dp->file_mode))
			break;
		if (snd_seq_oss_midi_open(dp, q->s.dev, SND_SEQ_OSS_FILE_WRITE))
			break;
		if (snd_seq_oss_midi_filemode(dp, q->s.dev) & SND_SEQ_OSS_FILE_WRITE)
			return snd_seq_oss_midi_putc(dp, q->s.dev, q->s.parm1, ev);
		break;

	case SND_OSS_SEQ_ECHO:
		return set_echo_event(dp, q->echo >> 8, ev);

	case SND_OSS_SEQ_PRIVATE:
		return snd_seq_oss_synth_raw_event(dp, q->c[1], q->c, ev);

	default:
		return old_event(dp, q, ev);
	}
	return -EINVAL;
}

/* old type events */
static int
old_event(seq_oss_devinfo_t *dp, evrec_t *q, snd_seq_event_t *ev)
{
	switch (q->s.code) {
	case SND_OSS_SEQ_NOTEOFF:
		return note_off_event(dp, 0, q->n.chn, q->n.note, q->n.vel, ev);

	case SND_OSS_SEQ_NOTEON:
		return note_on_event(dp, 0, q->n.chn, q->n.note, q->n.vel, ev);

	case SND_OSS_SEQ_WAIT:
		/* skip */
		break;

	case SND_OSS_SEQ_PGMCHANGE:
		return set_control_event(dp, 0, SND_SEQ_EVENT_PGMCHANGE,
					 q->n.chn, 0, q->n.note, ev);

	case SND_OSS_SEQ_SYNCTIMER:
		/* start timer -- already parsed */
		break;

	}

	return -EINVAL;
}

/* 8bytes extended event */
static int
extended_event(seq_oss_devinfo_t *dp, evrec_t *q, snd_seq_event_t *ev)
{
	int val;

	switch (q->e.cmd) {
	case SND_OSS_SEQ_NOTEOFF:
		return note_off_event(dp, q->e.dev, q->e.chn, q->e.p1, q->e.p2, ev);

	case SND_OSS_SEQ_NOTEON:
		return note_on_event(dp, q->e.dev, q->e.chn, q->e.p1, q->e.p2, ev);

	case SND_OSS_SEQ_PGMCHANGE:
		return set_control_event(dp, q->e.dev, SND_SEQ_EVENT_PGMCHANGE,
					 q->e.chn, 0, q->e.p1, ev);

	case SND_OSS_SEQ_AFTERTOUCH:
		return set_control_event(dp, q->e.dev, SND_SEQ_EVENT_CHANPRESS,
					 q->e.chn, 0, q->e.p1, ev);

	case SND_OSS_SEQ_BALANCE:
		/* convert -128:127 to 0:127 */
		val = (char)q->e.p1;
		val = (val + 128) / 2;
		return set_control_event(dp, q->e.dev, SND_SEQ_EVENT_CONTROLLER,
					 q->e.chn, SND_OSS_CTL_PAN, val, ev);

	case SND_OSS_SEQ_CONTROLLER:
		val = ((short)q->e.p3 << 8) | (short)q->e.p2;
		switch (q->e.p1) {
		case SND_OSS_CTRL_PITCH_BENDER: /* SEQ1 V2 control */
			/* -0x2000:0x1fff */
			return set_control_event(dp, q->e.dev,
						 SND_SEQ_EVENT_PITCHBEND,
						 q->e.chn, 0, val, ev);
		case SND_OSS_CTRL_PITCH_BENDER_RANGE:
			/* conversion: 100/semitone -> 128/semitone */
			return set_control_event(dp, q->e.dev,
						 SND_SEQ_EVENT_REGPARAM,
						 q->e.chn, 0, val*128/100, ev);
		default:
			return set_control_event(dp, q->e.dev,
						  SND_SEQ_EVENT_CONTROL14,
						  q->e.chn, q->e.p1, val, ev);
		}

	case SND_OSS_SEQ_VOLMODE:
		return snd_seq_oss_synth_raw_event(dp, q->e.dev, q->c, ev);

	}
	return -EINVAL;
}

/* channel voice events */
static int
chn_voice_event(seq_oss_devinfo_t *dp, evrec_t *q, snd_seq_event_t *ev)
{
	switch (q->v.cmd) {
	case SND_OSS_MIDI_NOTEON:
		return note_on_event(dp, q->v.dev, q->v.chn, q->v.note, q->v.parm, ev);

	case SND_OSS_MIDI_NOTEOFF:
		return note_off_event(dp, q->v.dev, q->v.chn, q->v.note, q->v.parm, ev);

	case SND_OSS_MIDI_KEY_PRESSURE:
		return set_note_event(dp, q->v.dev, SND_SEQ_EVENT_KEYPRESS,
				       q->v.chn, q->v.note, q->v.parm, ev);

	}
	return -EINVAL;
}

/* channel common events */
static int
chn_common_event(seq_oss_devinfo_t *dp, evrec_t *q, snd_seq_event_t *ev)
{
	switch (q->l.cmd) {
	case SND_OSS_MIDI_PGM_CHANGE:
		return set_control_event(dp, q->l.dev, SND_SEQ_EVENT_PGMCHANGE,
					  q->l.chn, 0, q->l.p1, ev);

	case SND_OSS_MIDI_CTL_CHANGE:
		return set_control_event(dp, q->l.dev, SND_SEQ_EVENT_CONTROLLER,
					  q->l.chn, q->l.p1, q->l.val, ev);

	case SND_OSS_MIDI_PITCH_BEND:
		/* conversion: 0:0x3fff -> -0x2000:0x1fff */
		return set_control_event(dp, q->l.dev, SND_SEQ_EVENT_PITCHBEND,
					  q->l.chn, 0, q->l.val - 8192, ev);
		
	case SND_OSS_MIDI_CHN_PRESSURE:
		return set_control_event(dp, q->l.dev, SND_SEQ_EVENT_CHANPRESS,
					  q->l.chn, 0, q->l.val, ev);
	}
	return -EINVAL;
}

/* timer events */
static int
timing_event(seq_oss_devinfo_t *dp, evrec_t *q, snd_seq_event_t *ev)
{
	if (q->t.cmd == SND_OSS_TMR_ECHO)
		return set_echo_event(dp, q->t.time, ev);

	return -EINVAL;
}

/* local events */
static int
local_event(seq_oss_devinfo_t *dp, evrec_t *q, snd_seq_event_t *ev)
{
	return -EINVAL;
}

/*
 * process note-on event for OSS synth
 * three different modes are available:
 * - SND_SEQ_OSS_PROCESS_EVENTS  (for one-voice per channel mode)
 *	Accept note 255 as volume change.
 * - SND_SEQ_OSS_PASS_EVENTS
 *	Pass all events to lowlevel driver anyway
 * - SND_SEQ_OSS_PROCESS_KEYPRESS  (mostly for Emu8000)
 *	Use key-pressure if note >= 128
 */
static int
note_on_event(seq_oss_devinfo_t *dp, int dev, int ch, int note, int vel, snd_seq_event_t *ev)
{
	seq_oss_synthinfo_t *info = &dp->synths[dev];
	switch (info->arg.event_passing) {
	case SND_SEQ_OSS_PROCESS_EVENTS:
		if (! info->ch || ch < 0 || ch >= info->nr_voices) {
			/* pass directly */
			return set_note_event(dp, dev, SND_SEQ_EVENT_NOTEON, ch, note, vel, ev);
		}

		if (note == 255 && info->ch[ch].note >= 0) {
			/* volume control */
			int type;
			//if (! vel)
				/* set volume to zero -- note off */
			//	type = SND_SEQ_EVENT_NOTEOFF;
			//else
				if (info->ch[ch].vel)
				/* sample already started -- volume change */
				type = SND_SEQ_EVENT_KEYPRESS;
			else
				/* sample not started -- start now */
				type = SND_SEQ_EVENT_NOTEON;
			info->ch[ch].vel = vel;
			return set_note_event(dp, dev, type, ch, info->ch[ch].note, vel, ev);
		} else if (note >= 128)
			return -EINVAL; /* invalid */

		if (note != info->ch[ch].note && info->ch[ch].note >= 0)
			/* note changed - note off at beginning */
			set_note_event(dp, dev, SND_SEQ_EVENT_NOTEOFF, ch, info->ch[ch].note, 0, ev);
		/* set current status */
		info->ch[ch].note = note;
		info->ch[ch].vel = vel;
		if (vel) /* non-zero velocity - start the note now */
			return set_note_event(dp, dev, SND_SEQ_EVENT_NOTEON, ch, note, vel, ev);
		return -EINVAL;
		
	case SND_SEQ_OSS_PASS_EVENTS:
		/* pass the event anyway */
		return set_note_event(dp, dev, SND_SEQ_EVENT_NOTEON, ch, note, vel, ev);

	case SND_SEQ_OSS_PROCESS_KEYPRESS:
		if (note >= 128) /* key pressure: shifted by 128 */
			return set_note_event(dp, dev, SND_SEQ_EVENT_KEYPRESS, ch, note - 128, vel, ev);
		else /* normal note-on event */
			return set_note_event(dp, dev, SND_SEQ_EVENT_NOTEON, ch, note, vel, ev);
	}
	return -EINVAL;
}

/*
 * process note-off event for OSS synth
 */
static int
note_off_event(seq_oss_devinfo_t *dp, int dev, int ch, int note, int vel, snd_seq_event_t *ev)
{
	seq_oss_synthinfo_t *info = &dp->synths[dev];
	switch (info->arg.event_passing) {
	case SND_SEQ_OSS_PROCESS_EVENTS:
		if (! info->ch || ch < 0 || ch >= info->nr_voices) {
			/* pass directly */
			return set_note_event(dp, dev, SND_SEQ_EVENT_NOTEON, ch, note, vel, ev);
		}

		if (info->ch[ch].note >= 0) {
			note = info->ch[ch].note;
			info->ch[ch].vel = 0;
			info->ch[ch].note = -1;
			return set_note_event(dp, dev, SND_SEQ_EVENT_NOTEOFF, ch, note, vel, ev);
		}
		return -EINVAL; /* invalid */

	case SND_SEQ_OSS_PASS_EVENTS:
	case SND_SEQ_OSS_PROCESS_KEYPRESS:
		/* pass the event anyway */
		return set_note_event(dp, dev, SND_SEQ_EVENT_NOTEOFF, ch, note, vel, ev);

	}
	return -EINVAL;
}

/*
 * create a note event
 */
static int
set_note_event(seq_oss_devinfo_t *dp, int dev, int type, int ch, int note, int vel, snd_seq_event_t *ev)
{
	if (! snd_seq_oss_synth_is_valid(dp, dev))
		return -ENXIO;
	
	ev->type = type;
	snd_seq_oss_synth_addr(dp, dev, ev);
	ev->dest.channel = ch;
	ev->data.note.note = note;
	ev->data.note.velocity = vel;

	return 0;
}

/*
 * create a control event
 */
static int
set_control_event(seq_oss_devinfo_t *dp, int dev, int type, int ch, int param, int val, snd_seq_event_t *ev)
{
	if (! snd_seq_oss_synth_is_valid(dp, dev))
		return -ENXIO;
	
	ev->type = type;
	snd_seq_oss_synth_addr(dp, dev, ev);
	ev->dest.channel = ch;
	ev->data.control.param = param;
	ev->data.control.value = val;

	return 0;
}

/*
 * create an echo event
 */
static int
set_echo_event(seq_oss_devinfo_t *dp, unsigned int echo, snd_seq_event_t *ev)
{
	ev->type = SND_SEQ_EVENT_ECHO;
	/* echo back to itself */
	snd_seq_oss_fill_addr(dp, ev, dp->addr.client, dp->addr.port);
	ev->data.raw32.d[0] = echo;
	ev->data.raw32.d[1] = 0; /* for user event */

	return 0;
}

/*
 * event input callback from ALSA sequencer:
 * the echo event is processed here.
 */
int
snd_seq_oss_event_input(snd_seq_event_t *ev, int direct, void *private_data,
			int atomic, int hop)
{
	seq_oss_devinfo_t *dp = (seq_oss_devinfo_t *)private_data;

	if (ev->type != SND_SEQ_EVENT_ECHO)
		return snd_seq_oss_midi_input(ev, direct, private_data);

	if (ev->data.raw32.d[1]) {
		/* sync echo back */
		snd_seq_oss_writeq_wakeup(dp->writeq, ev->data.raw32.d[0]);
		
	} else {
		/* echo back event */
		evrec_t rec;
		if (dp->readq == NULL)
			return 0;
		rec.echo = (ev->data.raw32.d[0] << 8) | SND_OSS_SEQ_ECHO;
		snd_seq_oss_readq_put_event(dp->readq, &rec);
	}
	return 0;
}

