// ---------------------------------------------------------------------------
// - Enum.cpp                                                                -
// - aleph engine - enumeration class implementation                         -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2003 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Cons.hpp"
#include "Item.hpp"
#include "Enum.hpp"
#include "Vector.hpp"
#include "Lexical.hpp"
#include "Builtin.hpp"
#include "Runnable.hpp"
#include "Exception.hpp"

namespace aleph {

  // the enum supported quarks
  static const long QUARK_ADD  = String::intern ("add");

  // create a new enumeration object

  Enum::Enum (void) {
  }

  // return the class name

  String Enum::repr (void) const {
    return "Enum";
  }

  // add a new item by string

  void Enum::add (const String& name) {
    // check for validity
    if (Lexical::valid (name) == false)
      throw Exception ("name-error", "invalid enumeration name", name);
    // map to quark and add
    add (name.toquark ());
  }

  // add a new item by quark
  
  void Enum::add (const long quark) {
    wrlock ();
    try {
      if (d_enum.exists (quark) == false) {
	d_enum.add (quark);
      }
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // create a new enumeration in a generic way

  Object* Enum::mknew (Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();
    Enum* result = new Enum;

    // add the enumeration string
    try {
      for (long i = 0; i < argc; i++) result->add (argv->getstring (i));
    } catch (...) {
      delete result;
      throw;
    }
    return result;
  }

  // evaluate an enumeration member by quark

  Object* Enum::eval (Runnable* robj, Nameset* nset, const long quark) {
    if (d_enum.exists (quark) == true) {
      Object* result = new Item (this, quark);
      robj->post (result);
      return result;
    }
    return Object::eval (robj, nset, quark);
  }

  // apply this enumeration with a set of arguments and a quark

  Object* Enum::apply (Runnable* robj, Nameset* nset, const long quark,
		       Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch 1 argument
    if (argc == 1) {
      if (quark == QUARK_ADD) {
	String name = argv->getstring (0);
	add (name);
	return nilp;
      }
    }
    // call the object method
    return Object::apply (robj, nset, quark, argv);
  }

  // create a new enumeration object - this is the builtin function

  Object* builtin_enum (Runnable* robj, Nameset* nset, Cons* args) {
    Enum* result = new Enum;
    // loop into the cons cell
    while (args != nilp) {
      Lexical* lex = dynamic_cast <Lexical*> (args->getcar ());
      if (lex == nilp) {
	delete result;
	throw Exception ("argument-error", 
			 "only symbol can be used as argument");
      }
      result->add (lex->toquark ());
      args = args->getcdr ();
    }
    return result;
  }
}
