// ---------------------------------------------------------------------------
// - UdpSocket.hpp                                                           -
// - aleph:net library - udp socket class definition                         -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2001 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  ALEPH_UDPSOCKET_HPP
#define  ALEPH_UDPSOCKET_HPP

#ifndef  ALEPH_DATAGRAM_HPP
#include "Datagram.hpp"
#endif

namespace aleph {

  /// The UdpSocket class is a base class for the udp ip protocol. When a 
  /// udp socket is created, the base socket methods can be called to
  /// perform its setup. The standard i/o methods are implemented here. 
  /// The class implements also the accept method which can be used to get
  /// a datagram.
  /// @author amaury darsch

  class UdpSocket : public Socket {
  private:
    /// the receiving buffer
    char* p_buf;

  protected:
    /// the source address
    Address d_addr;
    /// the source port
    t_word  d_port;

  public:
    /// create a default udp socket
    UdpSocket (void);

    /// create a udp socket by id
    /// @param sid the socket id
    UdpSocket (const int sid);

    /// destroy this udp socket
    ~UdpSocket (void);

    /// @return the class name
    String repr (void) const;

    /// join a multicast group by address
    /// @param addr the multicast address
    bool join (const Address& addr);

    /// drop from group by address
    /// @param addr the multicast address
    bool drop (const Address& addr);

    /// read one character from a udp message
    char read (void);

    /// read a buffer by size from a udp message
    /// @param size the number of character to read
    Buffer* read (const long size);

    /// write one character in a udp message
    /// @param value the character to send
    void write (const char value);

    /// write a buffer character in a udp message
    /// @param value the buffer to send
    void write (const char* value);

    /// throw an exception - since udp is connection less
    bool iseof (void) const;

    /// check if we can read a character
    /// @param tout the timeout value
    bool valid (const long tout) const;

    /// @return a datagram from this udp socket
    Datagram* accept (void) const;

    /// generate a new udp socket
    /// @param argv the argument vector
    static Object* mknew (Vector* argv);

    /// apply this udp socket with a set of arguments and a quark
    /// @param robj  robj the current runnable
    /// @param nset  the current nameset    
    /// @param quark the quark to apply these arguments
    /// @param argv  the arguments to apply
    Object* apply (Runnable* robj, Nameset* nset, const long quark,
                   Vector* argv);

  private:
    // make the copy constructor private
    UdpSocket (const UdpSocket&);
    // make the assignment operator private
    UdpSocket& operator = (const UdpSocket&);
  };
}

#endif
