// ---------------------------------------------------------------------------
// - Symbol.cpp                                                              -
// - aleph engine - symbol class implementation                              -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2001 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Vector.hpp"
#include "Symbol.hpp"
#include "Lexical.hpp"
#include "Boolean.hpp"
#include "Promise.hpp"
#include "Runnable.hpp"
#include "Exception.hpp"

namespace aleph {
  // the symbol supported quarks
  static const long QUARK_GETCONST  = String::intern ("get-const");
  static const long QUARK_GETOBJECT = String::intern ("get-object");
  static const long QUARK_SETCONST  = String::intern ("set-const");
  static const long QUARK_SETOBJECT = String::intern ("set-object");

  // create a new symbol by name

  Symbol::Symbol (const String& name) {
    if (Lexical::valid (name) == false)
      throw Exception ("name-error", "invalid symbol name", name);
    d_quark  = name.toquark ();
    p_object = nilp;
    d_const  = false;
  }

  // create a new symbol by quark

  Symbol::Symbol (const long quark) {
    d_quark  = quark;
    p_object = nilp;
    d_const  = false;
  }

  // create a symbol by name with an object

  Symbol::Symbol (const String& name, Object* object) {
    if (Lexical::valid (name) == false)
      throw Exception ("name-error", "invalid symbol name", name);
    d_quark  = name.toquark ();
    p_object = Object::iref (object);
    d_const  = false;
  }

  // create a symbol by quark with an object

  Symbol::Symbol (const long quark, Object* object) {
    d_quark  = quark;
    p_object = Object::iref (object);
    d_const  = false;
  }

  // copy constructor for this symbol

  Symbol::Symbol (const Symbol& that) {
    that.rdlock ();
    try {
      d_quark  = that.d_quark;
      p_object = Object::iref (that.p_object);
      d_const  = that.d_const;
      that.unlock ();
    } catch (...) {
      that.unlock ();
      throw;
    }
  }

  // destroy this symbol

  Symbol::~Symbol (void) {
    Object::dref (p_object);
  }

  // return the class name

  String Symbol::repr (void) const {
    return "Symbol";
  }

  // clone this symbol

  Object* Symbol::clone (void) const {
    return new Symbol (*this);
  }

  // make this symbol a shared object

  void Symbol::mksho (void) {
    if (p_shared != nilp) return;
    Object::mksho ();
    if (p_object != nilp) p_object->mksho ();
  }

  // get a literal representation of this symbol

  String Symbol::toliteral (void) const {
    rdlock ();
    String result = String::qmap (d_quark);
    unlock ();
    return result;
  }

  // get a string representation of this symbol

  String Symbol::tostring (void) const {
    rdlock ();
    String result = String::qmap (d_quark);
    unlock ();
    return result;
  }

  // get the symbol quark

  long Symbol::toquark (void) const {
    rdlock ();
    long result = d_quark;
    unlock ();
    return result;
  }

  // set the const flag for this symbol

  void Symbol::setconst (const bool flag) {
    wrlock ();
    d_const = flag;
    unlock ();
  }

  // get the const flag for this symbol

  bool Symbol::getconst (void) const {
    rdlock ();
    bool result = d_const;
    unlock ();
    return result;
  }

  // set the object for this symbol

  void Symbol::setobj (Object* object) {
    // get the write lock
    wrlock ();
    // check for the const flag
    if (d_const == true) {
      const String& name = String::qmap (d_quark);
      unlock ();
      throw Exception ("const-error", "const violation for symbol", name);
    }
    Object::iref (object);
    Object::dref (p_object);
    p_object = object;
    if ((p_shared != nilp) && (p_object != nilp)) p_object->mksho ();
    unlock ();
  }

  // get the symbol object

  Object* Symbol::getobj (void) const {
    rdlock ();
    Object* result = p_object;
    unlock ();
    return result;
  }

  // create a new symbol in a generic way

  Object* Symbol::mknew (Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // map just a name
    if (argc == 1) {
      String name = argv->getstring (0);
      return new Symbol (name);
    }

    // map a name and an object
    if (argc == 2) {
      String name = argv->getstring (0);
      return new Symbol (name, argv->get (1));
    }
    // illegal arguments
    throw Exception ("argument-error", "too many arguments with symbol");
  }

  // set this symbol as a const object

  Object* Symbol::cdef (Runnable* robj, Nameset* nset, Object* object) {
    // get thr write lock
    wrlock ();
    // define constant
    try {
      setobj   (object);
      d_const = true;
      robj->post (object);
      unlock ();
      return object;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // set this symbol with an object

  Object* Symbol::vdef (Runnable* robj, Nameset* nset, Object* object) {
    wrlock ();
    try {
      setobj (object);
      robj->post (object);
      unlock ();
      return object;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // evaluate this object - this is like get object

  Object* Symbol::eval (Runnable* robj, Nameset* nset) {
    rdlock ();
    try {
      Object* result = p_object;
      if (dynamic_cast <Promise*> (p_object) != nilp) 
	result = p_object->eval (robj, nset);
      robj->post (result);
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // apply this symbol with a set of arguments and a quark

  Object* Symbol::apply (Runnable* robj, Nameset* nset, const long quark,
			 Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch 0 argument
    if (argc == 0) {
      if (quark == QUARK_GETCONST)  return new Boolean (getconst ());
      if (quark == QUARK_GETOBJECT) {
	rdlock ();
	Object* result = getobj ();
	robj->post (result);
	unlock ();
	return result;
      }
    }

    // dispatch 1 argument
    if (argc == 1) {
      if (quark == QUARK_SETCONST) {
	bool bval = argv->getbool (0);
	setconst (bval);
	return nilp;
      }
      if (quark == QUARK_SETOBJECT) {
	setobj (argv->get (0));
	return nilp;
      }
    }
    // call the literal method
    return Literal::apply (robj, nset, quark, argv);
  }
}
