// ---------------------------------------------------------------------------
// - Interp.cpp                                                              -
// - aleph engine - interpreter class implementation                         -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2001 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Real.hpp"
#include "Meta.hpp"
#include "List.hpp"
#include "Graph.hpp"
#include "Queue.hpp"
#include "Mutex.hpp"
#include "Regex.hpp"
#include "Thread.hpp"
#include "Interp.hpp"
#include "Return.hpp"
#include "Reader.hpp"
#include "Module.hpp"
#include "System.hpp"
#include "Method.hpp"
#include "BitSet.hpp"
#include "Symbol.hpp"
#include "Lexical.hpp"
#include "Relatif.hpp"
#include "Condvar.hpp"
#include "Library.hpp"
#include "Printer.hpp"
#include "Builtin.hpp"
#include "Boolean.hpp"
#include "Closure.hpp"
#include "Librarian.hpp"
#include "Function.hpp"
#include "Character.hpp"
#include "Exception.hpp"

namespace aleph {
  // global mutex for the shared libraries
  static Mutex shlmtx;

  // the interpreter supported quarks
  static const long QUARK_URL        = String::intern ("aleph-url");
  static const long QUARK_LOAD       = String::intern ("load");
  static const long QUARK_ARGV       = String::intern ("argv");
  static const long QUARK_CLONE      = String::intern ("clone");
  static const long QUARK_MAJOR      = String::intern ("major-version");
  static const long QUARK_MINOR      = String::intern ("minor-version");
  static const long QUARK_PATCH      = String::intern ("patch-version");
  static const long QUARK_LAUNCH     = String::intern ("launch");
  static const long QUARK_DAEMON     = String::intern ("daemon");
  static const long QUARK_OSNAME     = String::intern ("os-name");
  static const long QUARK_OSTYPE     = String::intern ("os-type");
  static const long QUARK_VERSION    = String::intern ("version");
  static const long QUARK_PGMNAME    = String::intern ("program-name");
  static const long QUARK_GETPREC    = String::intern ("get-real-precision");
  static const long QUARK_SETPREC    = String::intern ("set-real-precision");
  static const long QUARK_LIBRARY    = String::intern ("library");
  static const long QUARK_GETIS      = String::intern ("get-input-stream");
  static const long QUARK_GETOS      = String::intern ("get-output-stream");
  static const long QUARK_GETES      = String::intern ("get-error-stream");

  // this procedure initialize the interpreter global nameset
  static void gset_init (Interp* interp) {
    // initialize the global set
    Nameset* nset = interp->getgset ();

    // bind standard constants
    nset->symcst  ("...",         nset);
    nset->symcst  ("nil",         (Object*) nilp);
    nset->symcst  ("true",        new Boolean (true));
    nset->symcst  ("false",       new Boolean (false));

    //  builtin functions
    interp->mkrsv ("if",          new Function (builtin_if));
    interp->mkrsv ("do",          new Function (builtin_do));
    interp->mkrsv ("for",         new Function (builtin_for));
    interp->mkrsv ("try",         new Function (builtin_try));
    interp->mkrsv ("eval",        new Function (builtin_eval));
    interp->mkrsv ("sync",        new Function (builtin_sync));
    interp->mkrsv ("loop",        new Function (builtin_loop));
    interp->mkrsv ("const",       new Function (builtin_const));
    interp->mkrsv ("trans",       new Function (builtin_trans));
    interp->mkrsv ("class",       new Function (builtin_class));
    interp->mkrsv ("block",       new Function (builtin_block));
    interp->mkrsv ("while",       new Function (builtin_while));
    interp->mkrsv ("gamma",       new Function (builtin_gamma));
    interp->mkrsv ("throw",       new Function (builtin_throw));
    interp->mkrsv ("force",       new Function (builtin_force));
    interp->mkrsv ("delay",       new Function (builtin_delay));
    interp->mkrsv ("lambda",      new Function (builtin_lambda));
    interp->mkrsv ("switch",      new Function (builtin_switch));
    interp->mkrsv ("return",      new Function (builtin_return));
    interp->mkrsv ("launch",      new Function (builtin_launch));
    interp->mkrsv ("daemon",      new Function (builtin_daemon));
    interp->mkrsv ("protect",     new Function (builtin_protect));
    interp->mkrsv ("nameset",     new Function (builtin_nameset));

    // builtin operators
    interp->mkrsv ("+",           new Function (builtin_add));
    interp->mkrsv ("-",           new Function (builtin_sub));
    interp->mkrsv ("*",           new Function (builtin_mul));
    interp->mkrsv ("/",           new Function (builtin_div));
    interp->mkrsv ("==",          new Function (builtin_eql));
    interp->mkrsv ("!=",          new Function (builtin_neq));
    interp->mkrsv (">=",          new Function (builtin_geq));
    interp->mkrsv (">",           new Function (builtin_gth));
    interp->mkrsv ("<=",          new Function (builtin_leq));
    interp->mkrsv ("<",           new Function (builtin_lth));
    interp->mkrsv ("assert",      new Function (builtin_assert));

    // logical operators
    interp->mkrsv ("or",          new Function (builtin_or));
    interp->mkrsv ("not",         new Function (builtin_not));
    interp->mkrsv ("and",         new Function (builtin_and));
    
    // standard printer objects
    interp->mkrsv ("print",       new Printer  (Printer::OUTPUT));
    interp->mkrsv ("println",     new Printer  (Printer::OUTPUTLN));
    interp->mkrsv ("error",       new Printer  (Printer::ERROR));
    interp->mkrsv ("errorln",     new Printer  (Printer::ERRORLN));

    // standard predicates
    interp->mkrsv ("nil-p",       new Function (builtin_nilp));
    interp->mkrsv ("cons-p",      new Function (builtin_consp));
    interp->mkrsv ("list-p",      new Function (builtin_listp));
    interp->mkrsv ("real-p",      new Function (builtin_realp));
    interp->mkrsv ("node-p",      new Function (builtin_nodep));
    interp->mkrsv ("edge-p",      new Function (builtin_edgep));
    interp->mkrsv ("graph-p",     new Function (builtin_graphp));
    interp->mkrsv ("regex-p",     new Function (builtin_regexp));
    interp->mkrsv ("queue-p",     new Function (builtin_queuep));
    interp->mkrsv ("class-p",     new Function (builtin_clsp));
    interp->mkrsv ("string-p",    new Function (builtin_strp));
    interp->mkrsv ("thread-p",    new Function (builtin_thrp));
    interp->mkrsv ("vector-p",    new Function (builtin_vecp));
    interp->mkrsv ("bitset-p",    new Function (builtin_bitsp));
    interp->mkrsv ("number-p",    new Function (builtin_nump));
    interp->mkrsv ("symbol-p",    new Function (builtin_symp));
    interp->mkrsv ("lexical-p",   new Function (builtin_lexp));
    interp->mkrsv ("condvar-p",   new Function (builtin_condp));
    interp->mkrsv ("promise-p",   new Function (builtin_prmp));
    interp->mkrsv ("boolean-p",   new Function (builtin_boolp));
    interp->mkrsv ("integer-p",   new Function (builtin_intp));
    interp->mkrsv ("relatif-p",   new Function (builtin_rltp));
    interp->mkrsv ("literal-p",   new Function (builtin_litp));
    interp->mkrsv ("closure-p",   new Function (builtin_clop));
    interp->mkrsv ("nameset-p",   new Function (builtin_nstp));
    interp->mkrsv ("instance-p",  new Function (builtin_instp));
    interp->mkrsv ("resolver-p",  new Function (builtin_rslvp));
    interp->mkrsv ("hashtable-p", new Function (builtin_ashp));
    interp->mkrsv ("character-p", new Function (builtin_charp));
    interp->mkrsv ("qualified-p", new Function (builtin_qualp));
    interp->mkrsv ("librarian-p", new Function (builtin_lbrnp));

    // standard objects
    interp->mkrsv ("Real",        new Meta (Real::mknew));
    interp->mkrsv ("Cons",        new Meta (Cons::mknew));
    interp->mkrsv ("List",        new Meta (List::mknew));
    interp->mkrsv ("Node",        new Meta (Node::mknew));
    interp->mkrsv ("Edge",        new Meta (Edge::mknew));
    interp->mkrsv ("Regex",       new Meta (Regex::mknew));
    interp->mkrsv ("Graph",       new Meta (Graph::mknew));
    interp->mkrsv ("Queue",       new Meta (Queue::mknew));
    interp->mkrsv ("String",      new Meta (String::mknew));
    interp->mkrsv ("Buffer",      new Meta (Buffer::mknew));
    interp->mkrsv ("Vector",      new Meta (Vector::mknew));
    interp->mkrsv ("BitSet",      new Meta (BitSet::mknew));
    interp->mkrsv ("Symbol",      new Meta (Symbol::mknew));
    interp->mkrsv ("Lexical",     new Meta (Lexical::mknew));
    interp->mkrsv ("Condvar",     new Meta (Condvar::mknew));    
    interp->mkrsv ("Relatif",     new Meta (Relatif::mknew));    
    interp->mkrsv ("Integer",     new Meta (Integer::mknew));
    interp->mkrsv ("Boolean",     new Meta (Boolean::mknew));
    interp->mkrsv ("Closure",     new Meta (Closure::mknew));
    interp->mkrsv ("Resolver",    new Meta (Resolver::mknew));
    interp->mkrsv ("Character",   new Meta (Character::mknew));
    interp->mkrsv ("Librarian",   new Meta (Librarian::mknew));
  }

  // this procedure look for a shared library
  static Library* getshl (Vector* shlib, const String& name) {
    long len = (shlib == nilp) ? 0 : shlib->length ();
    if (len == 0) return nilp;
    for (long i = 0; i < len; i++) {
      Library* lib = dynamic_cast <Library*> (shlib->get (i));
      if (lib == nilp) continue;
      if (lib->getname () == name) return lib;
    }
    return nilp;
  }

  // this procedure clears all the globalset as long as long
  // it is called from the master thread
  static void clrgset (Vector* vnset, Nameset* gset) {
    //clear all globalsets
    long len = (vnset == nilp) ? 0 : vnset->length ();
    for (long i = 0; i < len; i++) {
      Nameset* nset = dynamic_cast <Nameset*> (vnset->get (i));
      if (nset == nilp) continue;
      nset->clear ();
    }
    if (gset != nilp) gset->clear ();
  }

  // create a default interpreter
  
  Interp::Interp (void) {
    // initialize default values
    d_assert = false;
    d_cloned = false;
    // create a default terminal
    Object::iref (p_term = new Terminal);
    // save streams
    Object::iref (p_is = p_term);
    Object::iref (p_os = p_term);
    Object::iref (p_es = new OutputTerm (OutputTerm::ERROR));
    // clear the post object
    p_posted = nilp;
    // initialize the arguments vector
    Object::iref (p_argv = new Vector);
    // initialize the resolver
    Object::iref (p_rslv = new Resolver);
    // initialize the global nameset
    Object::iref (p_gset = new Globalset);
    gset_init (this);
    // bind the interpreter
    p_gset->symcst ("interp", this);
    // create the execution stack
    p_stk = new Stack;
    // reset the runnable form
    p_rform = nilp;
    // create the dynamic library vector
    Object::iref (p_shlib = new Vector);
    // create the vector of namesets
    Object::iref (p_vnset = new Vector);
  }

  // create a new interpreter with the specified streams

  Interp::Interp (Input* is, Output* os, Output* es) {
    // initialize default values
    d_assert = false;
    d_cloned = false;
    p_term   = nilp;
    // save streams
    Object::iref (p_is = is);
    Object::iref (p_os = os);
    Object::iref (p_es = es);
    // clear the post object
    p_posted = nilp;
    // initialize the arguments vector
    Object::iref (p_argv = new Vector);
    // initialize the resolver
    Object::iref (p_rslv = new Resolver);
    // initialize the global nameset
    Object::iref (p_gset = new Globalset);
    gset_init (this);
    // bind the interpreter
    p_gset->symcst ("interp", this);
    // create the execution stack
    p_stk = new Stack;
    // reset the runnable form
    p_rform = nilp;
    // create the dynamic library vector
    Object::iref (p_shlib = new Vector);
    // create the vector of namesets
    Object::iref (p_vnset = new Vector);
  }

  // copy constructor

  Interp::Interp (const Interp& that) {
    // initialize default value
    d_assert = that.d_assert;
    d_cloned = true;
    // copy the terminal
    Object::iref (p_term = that.p_term);
    // copy the streams - but not the input one
    Object::iref (p_is = that.p_is);
    Object::iref (p_os = that.p_os);
    Object::iref (p_es = that.p_es);
    // clear the post object
    p_posted = nilp;
    // copy the vector arguments
    Object::iref (p_argv = that.p_argv);
    // copy the file path
    Object::iref (p_rslv = that.p_rslv);
    // copy the global nameset
    Object::iref (p_gset = that.p_gset);
    // create the execution stack
    p_stk = new Stack;
    // reset the runnable form
    p_rform = nilp;
    // copy the dynamic library vector
    Object::iref (p_shlib = that.p_shlib);
    // copy the vector of namesets
    Object::iref (p_vnset = that.p_vnset);
  }

  // delete this interpreter

  Interp::~Interp (void) {
    // protect us
    Object::iref (this);
    // clear the posted object
    Object::dref (p_posted);
    p_posted = nilp;
    // eventually clear the finalizer and the global sets
    if (d_cloned == false) {
      Object::clrfnl ();
      clrgset (p_vnset, p_gset);
    }
    // clean the objects first
    Object::dref (p_is);
    Object::dref (p_os);
    Object::dref (p_es);
    Object::dref (p_term);
    Object::dref (p_gset);
    Object::dref (p_argv);
    Object::dref (p_rslv);
    Object::dref (p_rform);
    Object::dref (p_shlib);
    Object::dref (p_vnset);
    // clean the rest
    delete p_stk;
  }

  // return the class name

  String Interp::repr (void) const {
    return "Interp";
  }

  // make this interpreter a shared object

  void Interp::mksho (void) {
    if (p_shared != nilp) return;
    Object::mksho ();
    if (p_term  != nilp) p_term->mksho  ();
    if (p_is    != nilp) p_is->mksho    ();
    if (p_os    != nilp) p_os->mksho    ();
    if (p_es    != nilp) p_es->mksho    ();
    if (p_argv  != nilp) p_argv->mksho  ();
    if (p_rslv  != nilp) p_rslv->mksho  ();
    if (p_gset  != nilp) p_gset->mksho  ();
    if (p_shlib != nilp) p_shlib->mksho ();
    if (p_vnset != nilp) p_vnset->mksho ();
  }

  // post an object in this interpreter
  void Interp::post (Object* object) {
    if (object == p_posted) return;
    Object::iref (object);
    Object::dref (p_posted);
    p_posted = object;
  }

  // clone this interpreter

  Interp* Interp::clone (void) {
    return new Interp (*this);
  }

  // clone this interpreter and set the runnable form

  Interp* Interp::clone (Object* form) {
    Interp* interp = new Interp (*this);
    interp->p_rform = Object::iref (form);
    return interp;
  }

  // evaluate the runnable form

  Object* Interp::run (void) {
    Object* result = nilp;
    try {
      result = (p_rform == nilp) ? nilp : p_rform->eval (this, p_gset);
    } catch (const Exception& e) {
      getes()->errorln (e);
      if (e.getabf () == true) Thread::exit ();
    } catch (const Return& r) {
      result = r.getobj ();
    } catch (...) {
      getes()->errorln ("fatal: unknown exception trapped");
      result = nilp;
    }
    this->post (result);
    return result;
  }

  // execute a form in a thread by cloning this interpreter

  Object* Interp::launch (Object* form) {
    // mark everything shared
    mksho ();
    // make the form a shared object
    if (form != nilp) form->mksho ();
    // create a new thread
    Interp* interp = clone (form);
    return new Thread (Thread::NORMAL, interp);
  }

  // execute a form in a daemon thread by cloning this interpreter

  Object* Interp::daemon (Object* form) {
    // mark everything shared
    mksho ();
    // make the form a shared object
    if (form != nilp) form->mksho ();
    // create a new thread
    Interp* interp = clone (form);
    return new Thread (Thread::DAEMON, interp);
  }

  // return the interpreter input stream

  Input* Interp::getis (void) const {
    wrlock ();
    if (p_is != nilp) {
      unlock ();
      return p_is;
    }
    Object::iref (p_is = new InputTerm);
    unlock ();
    return p_is;
  }

  // return the interpreter output stream

  Output* Interp::getos (void) const {
    wrlock ();
    if (p_os != nilp) {
      unlock ();
      return p_os;
    }
    Object::iref (p_os = new OutputTerm (OutputTerm::OUTPUT));
    unlock ();
    return p_os;
  }
  
  // return the interpreter error stream

  Output* Interp::getes (void) const {
    wrlock ();
    if (p_es != nilp) {
      unlock ();
      return p_es;
    }
    Object::iref (p_es = new OutputTerm (OutputTerm::ERROR));
    unlock ();
    return p_es;
  }

  // return the interpreter stack

  Stack* Interp::getstk (void) const {
    return p_stk;
  }

  // return the interpreter global nameset

  Nameset* Interp::getgset (void) const {
    return p_gset;
  }

  // set the assert flag

  void Interp::setassert (const bool flag) {
    d_assert = flag;
  }

  // return the assert flag

  bool Interp::getassert (void) const {
    return d_assert;
  }

  // set the interpreter arguments

  void Interp::setargs (const Strvec& args) {
    long len = args.length ();
    for (long i = 0; i < len; i++) p_argv->append (new String (args.get (i)));
  }

  // add a new path entry for the interpreter

  void Interp::setpath (const Strvec& path) {
    long len = path.length ();
    for (long i = 0; i < len; i++) p_rslv->add (path.get (i));
  }

  // create a new reserved entry in the global nameset
  
  void Interp::mkrsv (const String& name, Object* object) {
    p_gset->symcst (name, object);
    Token::mkrsv   (name);
  }

  // create a new nameset in reference to another one

  Nameset* Interp::mknset (const String& name, Nameset* parent) {
    // look for parent
    if (parent == nilp) parent = p_gset;
    // look for an existing one
    if (parent->exists (name) == true) {
      Object* object = parent->eval (this, parent, name.toquark ());
      Nameset* nset = dynamic_cast <Nameset*> (object);
      if (nset == nilp)
	throw Exception ("type-error", "name does not evaluate as a nameset",
			 Object::repr (object));
      return nset;
    }
    Nameset* result = new Globalset (parent);
    parent->symcst (name, result);
    p_vnset->append (result);
    return result;
  }

  // open a new dynamic library by name
 
  Object* Interp::library (const String& name, Vector* argv) {
    // this procedure is shared by threads so lock it
    shlmtx.lock ();
    try {
      // check if the library already exists
      Library* lib = getshl (p_shlib, name);
      if (lib != nilp) {
	shlmtx.unlock ();
	return lib;
      }
      // the library does no exists, so create it
      lib = new Library (name);
      p_shlib->append (lib);
      // call the initialize procedure now
      Object::cref (lib->dlinit (this, argv));
      shlmtx.unlock ();
      return lib;
    } catch (...) {
      shlmtx.unlock ();
      throw;
    }
  }
 
  // loop on the standard input by doing a read-eval loop
  // this procedure return false is something bad happen

  bool Interp::loop (void) {
    bool status = true;
    // create a new reader 
    Reader* rd = new Reader (getis ());
    // loop until we have an eof
    while (true) {
      Cons* cons = nilp;
      try {
	cons = rd->parse ();
	if (cons == nilp) break;
	Object::cref (cons->eval (this, p_gset));
	Object::dref (cons);
      } catch (const Exception& e) {
	Object::dref (cons);
	status = false;
	getes()->errorln (e);
	if (e.getabf () == true) System::exit (1);
      } catch (const Return& r) {
	Object::dref (cons);
	this->post (r.getobj ());
      } catch (...) {
	Object::dref (cons);
	status = false;
	getes()->errorln ("fatal: unknown exception trapped");
	break;
      }
    }
    // clean and return
    delete rd;
    // wait for all threads to terminate - this works only for the
    // master thread - for other threads, this does nothing
    Thread::waitall ();
    return status;
  }

  // loop with an input file by doing a read-eval loop - no thread wait
  
  void Interp::load (const String& fname) {
    // try to open the module
    Module* mp = new Module (p_rslv->alplkp (fname));
    // loop until we have an eof
    while (true) {
      Cons* cons = nilp;
      try {
	cons = mp->parse ();
	if (cons == nilp) break;
	Object::cref (cons->eval (this, p_gset));
	Object::dref (cons);
      } catch (Exception& e) {
	e.setfname (fname);
	e.setlnum  (mp->getlnum ());
	delete mp;
	throw;
      } catch (const Return& r) {
	Object::dref (cons);
	this->post (r.getobj ());
      } catch (...) {
	Object::dref (cons);
	delete mp;
	throw;
      }
    }
    // clean the module and return status
    delete mp;
  }

  // loop with an input file by doing a read-eval loop
  // this procedure return false is something bad happen
  
  bool Interp::loop (const String& fname) {
    // load the file
    bool status = true;
    try {
      load (fname);
    } catch (const Exception& e) {
      status = false;
      getes()->errorln (e);
      if (e.getabf () == true) System::exit (1);
    } catch (...) {
      status = false;
      getes()->errorln ("fatal: unknown exception trapped");
    }
    // wait for all threads to terminate
    if (status == true) Thread::waitall ();
    return status;
  }

  // compile from an input stream into an output stream
  
  void Interp::compile (const String& name, Output& os) {
    // try to open the module and write
    Module* mp = new Module (p_rslv->lookup (name));
    try {
      mp->write (os);
    } catch (const Exception& e) {
      Integer line = mp->getlnum ();
      String  msg  = name + ':' + line.tostring ();
      delete mp;
      throw Exception (msg, e.getval ());
    }
  }

  // evaluate an object in this interpreter

  Object* Interp::eval (Object* object) {
    if (object == nilp) return nilp;
    Object* result = object->eval (this, p_gset);
    post (result);
    return result;
  }

  // evaluate an interpreter method name

  Object* Interp::eval (Runnable* robj, Nameset* nset, const long quark) {
    // standard builtin name
    if (quark == QUARK_ARGV)    return p_argv;
    if (quark == QUARK_URL)     return new String (System::geturl  ());
    if (quark == QUARK_MAJOR)   return new String (System::major   ());
    if (quark == QUARK_MINOR)   return new String (System::minor   ());
    if (quark == QUARK_PATCH)   return new String (System::patch   ());
    if (quark == QUARK_OSTYPE)  return new String (System::ostype  ());
    if (quark == QUARK_OSNAME)  return new String (System::osname  ());
    if (quark == QUARK_VERSION) return new String (System::version ());
    if (quark == QUARK_PGMNAME) return new String (System::getpgm  ());
    // look the object
    return Object::eval (robj, nset, quark);
  }

  // apply this interpreter with a set of arguments and a quark

  Object* Interp::apply (Runnable* robj, Nameset* nset, const long quark,
			 Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // special case for library with any arguments
    if ((quark == QUARK_LIBRARY) && (argc > 0)) {
      String lname = argv->getstring (0);
      return library (lname, argv);
    }

    // check for 0 argument
    if (argc == 0) {
      if (quark == QUARK_GETIS)   return getis ();
      if (quark == QUARK_GETOS)   return getos ();
      if (quark == QUARK_GETES)   return getes ();
      if (quark == QUARK_CLONE)   return clone ();
      if (quark == QUARK_GETPREC) return new Real (Real::getprecision ());
    }

    // check for one argument
    if (argc == 1) {
      if (quark == QUARK_SETPREC) {
	t_real val = argv->getreal (0);
	Real::setprecision (val);
	return nilp;
      }
      if (quark == QUARK_LOAD) {
	String val = argv->getstring (0);
	load (val);
	return nilp;
      }
      if (quark == QUARK_LAUNCH) {
	Object* form = argv->get(0);
	Object* result = this->launch (form);
	return result;
      }
      if (quark == QUARK_DAEMON) {
	Object* form = argv->get(0);
	Object* result = this->daemon (form);
	return result;
      }
    }

    // call the object method
    return Object::apply (robj, nset, quark, argv);
  }
}
