/**********************************************************************
 ** Moveable class: Objects that can be moved from one location to another, 
 **                 and usually placed in other objects by the players
 **
 ** Last reviewed: version 0.14
 **
 **
 ** Copyright (C) 2000 George Noel (Slate)
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the 
 **   License, or any later version. 
 **
 **   This program is distributed in the hope that it will be useful, but 
 **   WITHOUT ANY WARRANTY; without even the implied warranty of 
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 **   General Public License for more details. 
 ** 
 **   You should have received a copy of the GNU General Public License 
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 **
 **********************************************************************/

#ifndef GETTABLE_C
#define GETTABLE_C

#include "config.h"
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include "strings.h"
#include "mudtypes.h"
#include "mudobject.h"
#include "objtype.h"
#include "moveable.h"
#include "builder.h"
#include "flags.h"
#include "utils.h"
#include "merger.h"
#include "player.h"
#include "newfuncts.h"
#include "sysdep.h"

/***********************************************************************
 ** Moveable (constructor)
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Moveable::Moveable(void)
{
   size = weight = 0;
   num_allowed = percent_allowed = 0;
}


/***********************************************************************
 ** Moveable (constructor) - creates the item
 **
 ** Parameters: the_name - the name of the marker
 **             the_area - the area the marker belongs to
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Moveable::Moveable(char *the_name, char *the_area)
{
   /* this should be the only place that this is set */
   obj_type = OBJ_TYPE_MOVEABLE;
   
   if (the_name != NULL)
      set_name(the_name);
   
   if (the_area != NULL)
      set_area(the_area);

   size = weight = 0;
   num_allowed = percent_allowed = 0;

   set_brief("None.", 0);
   set_brief("None.", 1);
}

/***********************************************************************
 ** ~Moveable (destructor) - destroys it
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Moveable::~Moveable()
{
}


/***********************************************************************
 ** write_object - writes the location to a specified file in builder
 **                file format
 **
 ** Parameters: the_file - the file to write to
 **
 ** Returns:  1 if successful
 **          -1 if failed
 **
 ***********************************************************************/
   
int Moveable::write_moveable_attrib(FILE *the_file)
{
   fprintf(the_file, "^%s^\n", get_brief(0));
   fprintf(the_file, "^%s^\n", get_brief(1));
   fprintf(the_file, "%d\n%d\n", get_size(), get_weight());
   fprintf(the_file, "%d\n%d\n", get_num_allowed(), get_percent_allowed());
   return 1;
}


/***********************************************************************
 ** write_object - writes the location to a specified file in builder
 **                file format
 **
 ** Parameters: the_file - the file to write to
 **
 ***********************************************************************/
   
void Moveable::write_object(FILE *the_file, int build_format)
{
   fprintf(the_file, "\nmoveable %s\n", get_name());
   if (build_format)
      fprintf(the_file, "%d\n", is_modified());

   write_mudobject_attrib(the_file);
   write_item_attrib(the_file);
   write_moveable_attrib(the_file);
}

/***********************************************************************
 ** describe - describes the marker to a builder
 **
 ** Parameters: the_builder - the person to send all the data to
 **
 ***********************************************************************/

void Moveable::describe(Builder *the_builder)
{
   the_builder->send_bldr("\n&+GMoveable: \t&+M%s&*\n", get_name());
   the_builder->send_bldr("&+GTitle: \t\t&+w%s&*\n", get_title());
   the_builder->send_bldr("&+GAltnames: \t&+g%s&*\n", get_altname());
   the_builder->send_bldr("&+GClones: \t&+g%s&*\n", get_clones());
   the_builder->send_bldr("&+GSpecials: \t&+g%s&*\n", get_special_str());
   the_builder->send_bldr("&+GGuards: \t&+g%s&*\n", get_guards());
   the_builder->send_bldr("&+GLocation: \t&+M%s&*\n", get_location());
   the_builder->send_bldr("&+GSize: \t\t&+w%d&*\n", get_size());
   the_builder->send_bldr("&+GWeight:\t\t&+w%d&*\n", get_weight());
   the_builder->send_bldr("&+GNumAllow:\t&+w%d&*\n", get_num_allowed());
   the_builder->send_bldr("&+GPercentAllow:\t&+w%d&*\n", get_percent_allowed());
   if (itemflags->get_flag(ITEMFLAG_CONTAINER))
      the_builder->send_bldr("&+GCapacity:\t&+w%d&*\n", get_capacity());
   the_builder->send_bldr("&+GBrief0:\n&*%s\n", get_brief(0));
   the_builder->send_bldr("&+GBrief1:\n&*%s\n", get_brief(1));
   the_builder->send_bldr("&+GDesc:&*\n%s\n", get_desc());

   the_builder->send_bldr("\n");
}


/***********************************************************************
 ** describe - describes the moveable to a player
 **
 ** Parameters: the_builder - the person to send all the data to
 **
 ***********************************************************************/

void Moveable::describe(Player *the_player)
{
   MudObject *tmp_container;
   Strings   container_name;

   the_player->send_plr("\n&+GMoveable: \t&+M%s&*\n", get_name());
   the_player->send_plr("&+GTitle: \t\t&+w%s&*\n", get_title());
   the_player->send_plr("&+GAltnames: \t&+g%s&*\n", get_altname());
   the_player->send_plr("&+GClones: \t&+g%s&*\n", get_clones());
   the_player->send_plr("&+GSpecials: \t&+g%s&*\n", get_special_str());
   the_player->send_plr("&+GGuards: \t&+g%s&*\n", get_guards());
   the_player->send_plr("&+GStartLoc: \t&+M%s&*\n", get_location());

   tmp_container = get_contained_by();
   if (tmp_container == NULL)
      container_name = "nowhere";
   else
      container_name.sprintf("%s@%s", tmp_container->get_name(), 
                                         tmp_container->get_area());

   the_player->send_plr("&+GCurrentLoc: \t&+M%s&*\n", 
                                                 container_name.str_show());
   the_player->send_plr("&+GSize: \t\t&+w%d&*\n", get_size());
   the_player->send_plr("&+GWeight:\t\t&+w%d&*\n", get_weight());
   the_player->send_plr("&+GNumAllow:\t&+w%d&*\n", get_num_allowed());
   the_player->send_plr("&+GPercentAllow:\t&+w%d&*\n", get_percent_allowed());
   if (itemflags->get_flag(ITEMFLAG_CONTAINER))
      the_player->send_plr("&+GCapacity:\t&+w%d&*\n", get_capacity());
   the_player->send_plr("&+GBrief0:\n&*%s\n", get_brief(0));
   the_player->send_plr("&+GBrief1:\n&*%s\n", get_brief(1));
   the_player->send_plr("&+GDesc:&*\n%s\n", get_desc());
   the_player->send_plr("&+YSize: \t\t\t&+W%d&*\n", get_mem_size());

   list_specials(the_player);

   the_player->send_plr("\n");
}


/***********************************************************************
 ** set_attrib - sets a specified attribute to a specified value
 **
 ** Parameters: the_builder - the builder who is changing this attribute
 **
 ** Returns:  1 if successful
 **          -1 if failed
 **
 ***********************************************************************/
   
int Moveable::set_attrib(Builder *the_builder, Parse *the_parsed){

   if (the_parsed->get_target1() == NULL)
   {   the_builder->
        send_bldr("You can set the following attributes on a moveable.\n"
                  "   title, weight, size, altnames, desc, brief0, brief1\n"
                  "   guards, itemflags, location, specials, capacity\n"
                  "   numallow, percentallow, and merger\n");
       return -1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "title",
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_title(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "weight",
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_weight(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "size",
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_size(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "altnames",
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_altnames(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "clones",
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_clones(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "desc", 
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_desc(the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "itemflags",
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_itemflags(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "guards",
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_guard(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "specials", 
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_special(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "brief0", 
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_brief(the_builder, 0);
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "brief1", 
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_brief(the_builder, 1);
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "location", 
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_loc(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "capacity", 
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_capacity(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "numallow", 
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_num_allow(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "percentallow", 
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_percent_allow(the_parsed, the_builder);
   }

   the_builder->send_bldr("The attribute '%s' is not a moveable attribute.\n",
                                           the_parsed->get_target1());
   return -1;
}

/***********************************************************************
 ** set_size - sets the size of the moveable object
 **
 ** Parameters: new_size - the new size of the object
 **
 ** Returns:  1 if success
 **          -1 if failure
 **
 ***********************************************************************/
int Moveable::set_size(int new_size)
{   size = new_size;
    return 1;
}

/***********************************************************************
 ** set_weight - sets the weight of the moveable object
 **
 ** Parameters: new_weight - the new weight of the object
 **
 ** Returns:  1 if success
 **          -1 if failure
 **
 ***********************************************************************/
int Moveable::set_weight(int new_weight)
{   weight = new_weight;
    return 1;
}

/***********************************************************************
 ** set_num_allowed - sets the number of this object that are allowed in
 **                   the game at one time, counting those in user files
 **
 ** Parameters: new_val - the new value to set this to
 **
 ***********************************************************************/
void Moveable::set_num_allowed(int new_val)
{   num_allowed = new_val;
}

/***********************************************************************
 ** get_num_allowed - gets the number of this object that are allowed in
 **                   the game at one time, counting those in user files
 **
 ***********************************************************************/
int Moveable::get_num_allowed(void)
{
   return num_allowed;
}

/***********************************************************************
 ** set_percent_allowed - sets the number allowed based on the percent of
 **                       users in the userbase
 **
 ** Parameters: new_val - the new value to set this to
 **
 ***********************************************************************/
void Moveable::set_percent_allowed(int new_val)
{   
   percent_allowed = new_val;
}

/***********************************************************************
 ** get_percent_allowed - sets the number allowed based on the percent of
 **                       users in the userbase
 **
 ***********************************************************************/
int Moveable::get_percent_allowed(void)
{
   return percent_allowed;
}


/***********************************************************************
 ** set_brief - sets the brief description of the object
 **
 ** Parameters: the_desc - the new description of the object
 **
 ** Returns:  1 if success
 **          -1 if failure
 **
 ***********************************************************************/
int Moveable::set_brief(char *the_desc, int the_brief)
{
    if (the_brief == 1)
       brief_desc1 = the_desc;
    else if (the_brief == 0)
       brief_desc0 = the_desc;
    else
       return -1;

    return 1;
}

/***********************************************************************
 ** get_brief - gets the brief description of the object
 **
 ** Parameters: none.
 **
 ** Returns:  1 if success
 **          -1 if failure
 **
 ***********************************************************************/
char *Moveable::get_brief()
{   
    if (get_state())
       return brief_desc1.str_show();
    else
       return brief_desc0.str_show();
}


/***********************************************************************
 ** get_brief - gets the brief description of the object
 **
 ** Parameters: the_brief - the brief number to return
 **
 ** Returns: pointer to the string if success
 **          NULL if failure
 **
 ***********************************************************************/
char *Moveable::get_brief(int the_brief)
{   
    if (the_brief == 0)
       return brief_desc0.str_show();
    else if (the_brief == 1)
       return brief_desc1.str_show();
    else
       return NULL;
}


/***********************************************************************
 ** set_moved - set that the object has been moved
 **
 ** Parameters: none.
 **
 ** Returns:  Nothing
 **
 ***********************************************************************/
void Moveable::set_moved()
{
  set_state(1);
}


/***********************************************************************
 ** is_moved - returns if the object has been moved yet
 **
 **
 ***********************************************************************/
int Moveable::is_moved()
{
   return get_state();
}

/***********************************************************************
 ** copy_moveable_attrib - copies all the moveable attributes over from 
 **                        another moveable
 **
 ** Parameters: copy_from - the object we copy attributes from
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Moveable::copy_moveable_attrib(Moveable *copy_from)
{
   set_brief(copy_from->get_brief(0), 0);
   set_brief(copy_from->get_brief(1), 1);
   set_size(copy_from->get_size());
   set_weight(copy_from->get_weight());
   set_num_allowed(copy_from->get_num_allowed());
   set_percent_allowed(copy_from->get_percent_allowed());

   return 1;
}


/***********************************************************************
 ** copy_object - copies the moveable to an object of a different name
 **
 ** Parameters: copy_obj - copy attributes from this object
 **
 ** Returns:  1 if succeeded 
 **           0 if failed
 **
 ***********************************************************************/
int Moveable::copy_object(Entity *copy_obj)
{
   Moveable *copy_from;

   if (copy_obj->get_type() != OBJ_TYPE_MOVEABLE)
      return 0;

   copy_from = (Moveable *) copy_obj;

   /****** set the mudobject attrib *******/
   copy_mudobject_attrib((MudObject *) copy_from);

   /****** set item attributes **************/
   copy_item_attrib((Item *) copy_from);

   /****** set the moveable attributes ******/
   copy_moveable_attrib(copy_from);

   return 1;
}


/***********************************************************************
 ** operator = - copies an object to this object
 **
 ** Parameters: None
 **
 ** Returns: a pointer to this object copied to
 **
 ***********************************************************************/

Moveable *Moveable::operator = (Moveable *copy_from)
{
   if (!STRCASECMP(copy_from->get_name(), get_name()))
      return NULL;

   copy_object(copy_from);
   return this;
}


/***********************************************************************
 ** set_attrib_weight - for set attribute command, sets the weight attribute
 **
 ** Parameters: the_parsed - the parsed list from the user
 **             the_builder - the builder doing this command
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Moveable::set_attrib_weight(Parse *the_parsed, Builder *the_builder)
{
   if (the_parsed->get_speech() == NULL)
   {
      the_builder->send_bldr("You need to specify a number as well.\n");
      return -1;
   }
 
   if (!isdigit(*(the_parsed->get_speech()))) 
   {
      the_builder->send_bldr("You need to specify a number as weight.\n");
      return -1;
   }

   set_weight(atoi(the_parsed->get_speech()));
   the_builder->send_bldr("Weight set to %d on moveable object %s.\n",
                                                  get_weight(), get_name());
   return 1;
}


/***********************************************************************
 ** set_attrib_size - for set attribute command, sets the size attribute
 **
 ** Parameters: the_parsed - the parsed list from the user
 **             the_builder - the builder doing this command
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Moveable::set_attrib_size(Parse *the_parsed, Builder *the_builder)
{
   if (the_parsed->get_speech() == NULL)
   {
      the_builder->send_bldr("You need to specify a number as well.\n");
      return -1;
   }
 
   if (!isdigit(*(the_parsed->get_speech()))) 
   {
      the_builder->send_bldr("You need to specify a number as size.\n");
      return -1;
   }

   set_size(atoi(the_parsed->get_speech()));
   the_builder->send_bldr("Size set to %d on moveable object %s.\n",
                                               get_size(), get_name());
   return 1;
}


/***********************************************************************
 ** set_attrib_num_allow - for set attribute command, sets the num_allowed
 **                        attribute
 **
 ** Parameters: the_parsed - the parsed list from the user
 **             the_builder - the builder doing this command
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Moveable::set_attrib_num_allow(Parse *the_parsed, Builder *the_builder)
{
   if (the_parsed->get_speech() == NULL)
   {
      the_builder->send_bldr("You need to specify a number as well.\n");
      return -1;
   }
 
   if (!isdigit(*(the_parsed->get_speech()))) 
   {
      the_builder->send_bldr("You need to specify a number as NumAllow.\n");
      return -1;
   }

   set_num_allowed(atoi(the_parsed->get_speech()));
   the_builder->send_bldr("NumAllow set to %d on moveable object %s.\n",
                                      get_num_allowed(), get_name());
   return 1;
}


/***********************************************************************
 ** set_attrib_percent_allow - for set attribute command, sets the
 **                            percent_allowed attribute
 **
 ** Parameters: the_parsed - the parsed list from the user
 **             the_builder - the builder doing this command
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Moveable::set_attrib_percent_allow(Parse *the_parsed, Builder *the_builder)
{
   if (the_parsed->get_speech() == NULL)
   {
      the_builder->send_bldr("You need to specify a number as well.\n");
      return -1;
   }
 
   if (!isdigit(*(the_parsed->get_speech()))) 
   {
      the_builder->send_bldr("You need to specify a number as PercentAllow.\n");
      return -1;
   }

   set_percent_allowed(atoi(the_parsed->get_speech()));
   the_builder->send_bldr("PercentAllow set to %d on moveable object %s.\n",
                                      get_percent_allowed(), get_name());
   return 1;
}


/***********************************************************************
 ** set_attrib_brief - for set attribute command, sets the brief attribute
 **
 ** Parameters: the_parsed - the parsed list from the user
 **             the_builder - the builder doing this command
 **             brief_no - the briefing number we are setting
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Moveable::set_attrib_brief(Builder *the_builder, int brief_no)
{
   if (brief_no)
   {
      if (the_builder->get_long_input(&brief_desc1) < 0)
      {
         the_builder->send_bldr("Error reading in input, failed!\n");
         return -1;
      }
   }
   else
      if (the_builder->get_long_input(&brief_desc0) < 0)
      {
         the_builder->send_bldr("Error reading in input, failed!\n");
         return -1;
      }

   return 1;
}


/***********************************************************************
 ** get_size - gets the size of the mudobject
 **
 **
 ***********************************************************************/

int Moveable::get_size() 
{ 
   if ((is_merger()) && ((Merger *)this)->get_number_of() != 0)
      return (size * ((Merger *)this)->get_number_of());
   else
      return size; 
}


/***********************************************************************
 ** get_weight - gets the weight of the mudobject
 **
 **
 ***********************************************************************/

int Moveable::get_weight() 
{ 
   return weight; 
}


/***********************************************************************
 ** read_moveable_attrib - reads in door attributes from the file
 **
 ** Parameters: read_file - the file to read in from
 **             areaname - the area that we are reading
 **             error_log - the error log to write any errors to
 **
 ** Returns:  1 for successful read
 **          -1 for errors in the read
 **
 ***********************************************************************/

int Moveable::read_moveable_attrib(FILE *read_file, ErrLog *error_log)
{
   token_record *the_token;
   Strings      holder;
   char         *tmp_charholder;

   /* set the brief desc */
   tmp_charholder = 
           read_desc_type(read_file, error_log, (MudObject *) this); 
   set_brief(tmp_charholder, 0);
   delete tmp_charholder;

   /* set the brief */
   tmp_charholder = 
           read_desc_type(read_file, error_log, (MudObject *) this); 
   set_brief(tmp_charholder, 1);
   delete tmp_charholder;


   /* Set size */
   the_token = get_token(read_file, '\0');
     
   if (the_token->token_type != T_NUMERICAL)
   {
      holder.
         sprintf("Invalid format for attribute size in mudobject %s", 
                                                               get_name());

      error_log->log_err(holder.str_show(), "read_moveable_attrib");
      return -1;
   }
   set_size(atoi(the_token->the_string));

   /* Set weight */
   the_token = get_token(read_file, '\0');
     
   if (the_token->token_type != T_NUMERICAL)
   {
      holder.
           sprintf("Invalid format for attribute weight in mudobject %s", 
                                                            get_name());
      error_log->log_err(holder.str_show(), "read_moveable_attrib");
      return -1;
   }
   set_weight(atoi(the_token->the_string));

   /* Set num_allowed */
   the_token = get_token(read_file, '\0');
     
   if (the_token->token_type != T_NUMERICAL)
   {
      holder.
           sprintf("Invalid format for attribute num_allowed in mudobject %s@%s", 
                                                get_name(), get_area());
      error_log->log_err(holder.str_show(), "read_moveable_attrib");
      return -1;
   }
   set_num_allowed(atoi(the_token->the_string));

   /* Set percent_allowed */
   the_token = get_token(read_file, '\0');
     
   if (the_token->token_type != T_NUMERICAL)
   {
      holder.
           sprintf("Invalid format for attribute percent_allow in mudobject %s", 
                                                            get_name());
      error_log->log_err(holder.str_show(), "read_moveable_attrib");
      return -1;
   }
   set_percent_allowed(atoi(the_token->the_string));

   return 1;
}


/***********************************************************************
 ** get_mem_size_moveable - gets how much memory is being taken up by the
 **                         moveable part of the object
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Moveable::get_mem_size_moveable()
{
   int size = 0;

   size += brief_desc0.get_mem_size_dynamic();
   size += brief_desc1.get_mem_size_dynamic();

   return size;
}



#endif


