// ---------------------------------------------------------------------------
// - Mail.hpp                                                                -
// - afnix:net module - mail class definition                                -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2007 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  AFNIX_MAIL_HPP
#define  AFNIX_MAIL_HPP

#ifndef  AFNIX_STRVEC_HPP
#include "Strvec.hpp"
#endif

#ifndef  AFNIX_TCPCLIENT_HPP
#include "TcpClient.hpp"
#endif

namespace afnix {

  /// The Mail class is a client class which provides a simple interface
  /// to send mail. In its simplest mode, only a recipient address is required 
  /// to send a message. In ts most complex form, the class can almost mimic
  /// the requirement of RFC822 for mail header description. This class 
  /// contains several methods to create a complete message and specify 
  /// hot to send it and where. More about this below.
  /// @author amaury darsch

  class Mail : public Object {
  private:
    /// the mta address
    String d_addr;
    /// the mta port
    t_word d_port;
    /// the host name
    String d_host;
    /// the from address
    String d_from;
    /// the "to" recipients list
    Strvec d_torl;
    /// the "cc" recipients list
    Strvec d_ccrl;
    /// the "bcc" recipients list
    Strvec d_bcrl;
    /// the message subject
    String d_subj;
    /// the message body
    Buffer d_mesg;

  public:
    /// create an empty mail message
    Mail (void);

    /// @return the class name
    String repr (void) const;

    /// set the mta address
    /// @param addr the address to set
    void setaddr (const String& addr);

    /// @return the mta address
    String getaddr (void) const;

    /// set the mta port
    /// @param port the port to set
    void setport (const t_word port);

    /// @return the mta port
    t_word getport (void) const;

    /// add a recipient in the "to" list
    /// @param rcpt the recipient address
    void addto (const String& rcpt);

    /// add a recipient in the "cc" list
    /// @param rcpt the recipient address
    void addcc (const String& rcpt);

    /// add a recipient in the "bcc" list
    /// @param rcpt the recipient address
    void addbcc (const String& rcpt);

    /// set the message subject
    /// @param subj  the subject to set
    void setsubj (const String& subj);

    /// add some data to the message body
    /// @param data the data to add to the message
    void addmsg (const String& data);

    /// send this message to the recipients
    void send (void) const;

  private:
    // make the copy constructor private
    Mail (const Mail&);
    // make the assignment operator private
    Mail& operator = (const Mail&);
    // private mail methods
    void gethead (Buffer& head) const;

  public:
    /// create a new object in a generic way
    /// @param argv the argument vector
    static Object* mknew (Vector* argv);

    /// @return true if the given quark is defined
    bool isquark (const long quark, const bool hflg) const;

    /// apply this object with a set of arguments and a quark
    /// @param robj  the current runnable
    /// @param nset  the current nameset    
    /// @param quark the quark to apply these arguments
    /// @param argv  the arguments to apply
    Object* apply (Runnable* robj, Nameset* nset, const long quark,
		   Vector* argv);
  };
}

#endif
