// ---------------------------------------------------------------------------
// - BitSet.hpp                                                              -
// - standard object library - bit set class definition                      -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2007 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  AFNIX_BITSET_HPP
#define  AFNIX_BITSET_HPP

#ifndef  AFNIX_STRING_HPP
#include "String.hpp"
#endif

namespace afnix {

  /// The BitSet class is a resizable bit set object which can be used
  /// to mark a particular bit in a defined field. Standard boolean methods
  /// are provided to manipulate bit sets.
  /// @author amaury darsch

  class BitSet : public virtual Object {
  private:
    /// the bitset size
    long    d_size;
    /// the array of bytes
    t_byte* p_byte;

  public:
    /// create a default bitset
    BitSet (void);

    /// create a bitset with a size
    /// @param size the bitset size
    BitSet (const long size);

    /// copy constructor for this bitset
    /// @param that the bitset to copy 
    BitSet (const BitSet& that);

    /// destroy this bitset
    ~BitSet (void);

    /// @return the class name
    String repr (void) const;

    /// assign a bitset to this one
    /// @param that the bitset to assign
    BitSet& operator = (const BitSet& that);

    /// @return true if two bitset are equals
    bool operator == (const BitSet& bset) const;

    /// @return the bitset size
    long length (void) const;

    /// get a bit at a certain position
    /// @param pos the bit position
    bool get (const long pos) const;

    /// mark a bit at a certain position
    /// @param pos the bit position
    void mark (const long pos);

    /// clear a bit at a certain position
    /// @param pos the bit position
    void clear (const long pos);

    /// set a bit at a certain position
    /// @param pos the bit position
    /// @param bit the bit value to set
    void set (const long pos, const bool bit);

    /// set a byte at a certain index
    /// @param idx the byte index
    /// @param val the byte value
    void set (const long idx, const t_byte val);

    /// reserve a new size for this bitset
    /// @param size the new size to reserve
    void reserve (const long size);

  public:
    /// create a new object in a generic way
    /// @param argv the argument vector
    static Object* mknew (Vector* argv);

    /// @return true if the given quark is defined
    bool isquark (const long quark, const bool hflg) const;

    /// apply this object with a set of arguments and a quark
    /// @param robj  the current runnable
    /// @param nset  the current nameset    
    /// @param quark the quark to apply these arguments
    /// @param args  the arguments to apply
    Object* apply (Runnable* robj, Nameset* nset, const long quark,
		   Vector* argv);
  };
}

#endif
