-- `Topal': GPG/Pine integration
--
-- Copyright (C) 2001,2002  Phillip J. Brooke
--
--     This program is free software; you can redistribute it and/or modify
--     it under the terms of the GNU General Public License as published by
--     the Free Software Foundation; either version 2 of the License, or
--     (at your option) any later version.
--
--     This program is distributed in the hope that it will be useful,
--     but WITHOUT ANY WARRANTY; without even the implied warranty of
--     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--     GNU General Public License for more details.
--
--     You should have received a copy of the GNU General Public License
--     along with this program; if not, write to the Free Software
--     Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

with Ada.Text_IO;
with Externals.Simple;
with Misc;             use Misc;

package body Externals.GPG is

   procedure Clean_GPG_Errors (Orig_Err_File : in String;
                               Err_File      : in String) is
   begin
      -- Clean up the error file to remove
      -- `gpg: Invalid passphrase; please try again ...' messages.
      if ForkExec_Out(Value_Nonempty(Config.Grep_Binary),
                      UBS_Array'(0 => ToUBS("grep"),
                                 1 => ToUBS("-v"),
                                 2 => ToUBS("gpg: Invalid passphrase; please try again ..."),
                                 3 => ToUBS(Orig_Err_File)),
                      Err_File) /= 0 then
         Error("Grep failed! (ff1)");
      end if;
   end Clean_GPG_Errors;

   function GPG_Tee (Input_File  : String;
                     Output_File : String;
                     Err_File    : String) return Integer is
      GPG_Return_Value, Tee_Return_Value : Integer;
      Orig_Err_File                      : String := Temp_File_Name("origerr");
   begin
      ForkExec2(Value_Nonempty(Config.Gpg_Binary),
                ToUBS("gpg "
                      & ToStr(Config.Gpg_Options)
                      & " --output "
                      & Output_File
                      & " "
                      & ToStr(Config.General_Options)
                      & " "
                      & ToStr(Config.Receiving_Options)
                      & " "
                      & Input_File),
                GPG_Return_Value,
                Value_Nonempty(Config.Tee_Binary),
                ToUBS("tee "
                      & Orig_Err_File),
                Tee_Return_Value,
                Merge_StdErr1 => True,
                Report => True);
      if Tee_Return_Value /= 0 then
         Error("Tee failed! (ff1)");
      end if;
      Clean_GPG_Errors(Orig_Err_File, Err_File);
      -- Let the caller sort out the GPG return value.
      return GPG_Return_Value;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.GPG.GPG_Tee");
         raise;
   end GPG_Tee;

   function GPG_Verify_Tee (Input_File  : String;
                            Sig_File    : String;
                            Output_File : String;
                            Err_File    : String) return Integer is
      GPG_Return_Value, Tee_Return_Value : Integer;
      Orig_Err_File                      : String := Temp_File_Name("origerr");
   begin
      ForkExec2(Value_Nonempty(Config.Gpg_Binary),
                ToUBS("gpg "
                      & ToStr(Config.Gpg_Options)
                      & " --output "
                      & Output_File
                      & " "
                      & ToStr(Config.General_Options)
                      & " "
                      & ToStr(Config.Receiving_Options)
                      & " --verify "
                      & Sig_File
                      & " "
                      & Input_File),
                GPG_Return_Value,
                Value_Nonempty(Config.Tee_Binary),
                ToUBS("tee "
                      & Orig_Err_File),
                Tee_Return_Value,
                Merge_StdErr1 => True,
                Report => True);
      if Tee_Return_Value /= 0 then
         Error("Tee failed! (ff1)");
      end if;
      Clean_GPG_Errors(Orig_Err_File, Err_File);
      -- Let the caller sort out the GPG return value.
      return GPG_Return_Value;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.GPG.GPG_Verify_Tee");
         raise;
   end GPG_Verify_Tee;

   function Grep_Sigfile_Digest (Sigfile : in String;
                                 Number  : in String)
                                 return Integer is
      E1, E2 : Integer;
   begin
      ForkExec2(Value_Nonempty(Config.GPG_Binary),
                UBS_Array'(0 => ToUBS("gpg"),
                           1 => ToUBS("--list-packets"),
                           2 => ToUBS(Sigfile)),
                E1,
                Value_Nonempty(Config.Grep_Binary),
                UBS_Array'(0 => ToUBS("grep"),
                           1 => ToUBS("-q"),
                           2 => ToUBS("digest algo " & Number)),
                E2);
      return E2;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.GPG.Grep_Sigfile_Digest");
         raise;
   end Grep_Sigfile_Digest;

   function Micalg_From_Filename (Sigfile : in String) return String is
   begin
      -- From RFC2440, we have:
      --      9.4. Hash Algorithms
      --
      --       ID           Algorithm                              Text Name
      --       --           ---------                              ---- ----
      --       1          - MD5                                    "MD5"
      --       2          - SHA-1                                  "SHA1"
      --       3          - RIPE-MD/160                            "RIPEMD160"
      --       4          - Reserved for double-width SHA (experimental)
      --       5          - MD2                                    "MD2"
      --       6          - Reserved for TIGER/192                 "TIGER192"
      --       7          - Reserved for HAVAL (5 pass, 160-bit)
      --       "HAVAL-5-160"
      --       100 to 110 - Private/Experimental algorithm.
      --   Implementations MUST implement SHA-1. Implementations SHOULD
      --   implement MD5.
      -- So we'll use gpg --list-packets Sigfile (we're assuming that this
      -- is a detached signature) and look for digest algo 1 or 2 and
      -- return pgp-sha1 or pgp-md5 respectively.
      -- Look for other numbers as defined in RFC3156.
      -- If we don't find anything, raise an exception.
      if Grep_Sigfile_Digest(Sigfile, "0") = 0 then
         return "pgp-md5";
      elsif Grep_Sigfile_Digest(Sigfile, "2") = 0 then
         return "pgp-sha1";
      elsif Grep_Sigfile_Digest(Sigfile, "3") = 0 then
         return "pgp-ripemd160";
      elsif Grep_Sigfile_Digest(Sigfile, "5") = 0 then
         return "pgp-md2";
      elsif Grep_Sigfile_Digest(Sigfile, "6") = 0 then
         return "pgp-tiger192";
      elsif Grep_Sigfile_Digest(Sigfile, "7") = 0 then
         return "pgp-haval-5-160";
      else
         raise Unrecognised_Micalg;
         return "unknown"; -- Should never execute.
      end if;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.GPG.Micalg_From_Filename");
         raise;
   end Micalg_From_Filename;

   -- This procedure does something mildly naughty.  If the return code
   -- is 0, we're happy.  If it isn't, we see if the file exists, and
   -- print a warning to be careful if it does.
   procedure GPG_Wrap (Args         : in String;
                       Out_Filename : in String) is
      R : Integer;  -- The return code from GPG.
   begin
      Ada.Text_IO.Put_Line("About to run `gpg " & Args & "'...");
      R := ForkExec(Value_Nonempty(Config.Gpg_Binary),
                    ToUBS("gpg " & Args));
      if R = 0 then
         Ada.Text_IO.Put_Line("GPG exited successfully...");
      else
         Ada.Text_IO.New_Line(2);
         Ada.Text_IO.Put_Line("GPG exited with return code "
                              & Trim_Leading_Spaces(Integer'Image(R)));
         if Externals.Simple.Test_S(Out_Filename) then
            Ada.Text_IO.Put_Line("*** WARNING ***");
            Ada.Text_IO.Put_Line("However, a non-empty output file was generated, so it might have worked.");
            Ada.Text_IO.Put_Line("Perhaps some public keys were unusable?  (E.g., expired keys?)");
            Ada.Text_IO.Put_Line("We will proceed as if everything was okay.");
            Ada.Text_IO.Put_Line("** You should check the output file.... **");
            Ada.Text_IO.Put_Line("*** WARNING ***");
            Ada.Text_IO.New_Line(2);
         else
            raise GPG_Failed;
         end if;
      end if;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.GPG.GPG_Wrap");
         raise;
   end GPG_Wrap;

   procedure Findkey (Key    : in String;
                      Target : in String) is
      E1, E2, E3 : Integer;
   begin
      ForkExec3_Out(Value_Nonempty(Config.Gpg_Binary),
                    ToUBS("gpg "
                          & ToStr(Config.Gpg_Options)
                          & " --fingerprint " & Key),
                    E1,
                    Value_Nonempty(Config.Grep_binary),
                    UBS_Array'(0 => ToUBS("grep"),
                               1 => ToUBS("Key fingerprint")),
                    E2,
                    Value_Nonempty(Config.Sed_binary),
                    UBS_Array'(0 => ToUBS("sed"),
                               1 => ToUBS("-e"),
                               2 => ToUBS("s/.*= //"),
                               3 => ToUBS("-e"),
                               4 => ToUBS("s/ //g"),
                               5 => ToUBS("-e"),
                               6 => ToUBS("s/^/0x/")),
                    E3,
                    Target => Target);
      Debug("Add_Keys_By_Fingerprint: Finished exec's");
      if E1 /= 0 then
         Debug("gpg failed with exit code "
               & Integer'Image(E1) & "! (ff1)");
      elsif E2 /= 0 then
         Error("grep failed! (ff2)");
      elsif E3 /= 0 then
         Error("sed failed! (ff3)");
      end if;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Keys.Findkey");
         raise;
   end Findkey;

   procedure Findkey_Secret (Key    : in String;
                             Target : in String) is
      E1, E2, E3 : Integer;
   begin
      ForkExec3_Out(Value_Nonempty(Config.Gpg_Binary),
                    ToUBS("gpg "
                          & ToStr(Config.Gpg_Options)
                          & " --list-secret-keys "
                          & Key),
                    E1,
                    Value_Nonempty(Config.Grep_Binary),
                    UBS_Array'(0 => ToUBS("grep"),
                               1 => ToUBS("^sec")),
                    E2,
                    Value_Nonempty(Config.Sed_Binary),
                    UBS_Array'(0 => ToUBS("sed"),
                               1 => ToUBS("-e"),
                               2 => ToUBS("s/^sec *//"),
                               3 => ToUBS("-e"),
                               4 => ToUBS("s/ .*$//"),
                               5 => ToUBS("-e"),
                               6 => ToUBS("sx^.*/xx")),
                    E3,
                    Target => Target);
      if E1 /= 0 then
         Debug("gpg failed! (ff4)");
      elsif E2 /= 0 then
         Error("grep failed! (ff5)");
      elsif E3 /= 0 then
         Error("sed failed! (ff6)");
      end if;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Keys.Findkey_Secret");
         raise;
   end Findkey_Secret;

   procedure Listkey (Key    : in String;
                      Target : in String) is
   begin
      if ForkExec_Out(Value_Nonempty(Config.Gpg_Binary),
                      ToUBS("gpg "
                            & ToStr(Config.Gpg_Options)
                            & " --list-keys "
                            & Key),
                      Target => Target) /= 0 then
         Error("Problem generating keylist, GPG barfed (ff7)");
      end if;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Keys.Listkey");
         raise;
   end Listkey;

   procedure Viewkey (Key     : in String;
                      Verbose : in Boolean := False) is
      U      : UBS;
      E1, E2 : Integer;
   begin
      if Verbose then
         U := ToUBS("gpg "
                    & ToStr(Config.Gpg_options)
                    & " --list-keys --verbose "
                    & Key);
      else
         U := ToUBS("gpg "
                    & ToStr(Config.Gpg_options)
                    & " --list-keys "
                    & Key);
      end if;
      ForkExec2(Value_Nonempty(Config.Gpg_binary),
                U,
                E1,
                Value_Nonempty(Config.Less_Binary),
                ToUBS("less"),
                E2);
      if E1 /= 0 then
         Error("Problem with GPG! (ff8)");
      elsif E2 /= 0 then
         Error("Problem with less! (ff9)");
      end if;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Externals.Keys.Viewkey");
         raise;
   end Viewkey;

end Externals.GPG;
