/*
 * snes9express
 * s9xskin.cc
 * Copyright (C) 2002  David Nordlund
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * For further details, please read the included COPYING file,
 * or go to http://www.gnu.org/copyleft/gpl.html
 */

#include "s9xskin.h"
#include <iostream>

#ifdef S9X_SKINABLE

#include <sstream>
#include <fstream>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string>
#include <list>
#include "frend.h"

const char *headstart = "snes9express skin\x09";
const char *headfinish = "\x0A\x04\x0D\x1A";
const int headskinver = 0x02;


s9x_SkinFileBase::~s9x_SkinFileBase()
{
	if(skinfptr)
		fclose(skinfptr);
	index::iterator i = idx.begin(), e = idx.end();
	for(; i!=e; i++)
	{
		s9x_SkinFileIndexRecord *rec = (*i).second;
		free(rec);
	}
}

/**
 * Open the skin file for reading.
 */
s9x_SkinReader::s9x_SkinReader(const std::string filespec)
{
	skinfptr = fopen(filespec.c_str(), "r");
	if(!skinfptr)
		throw std::string("failed to open skin file\n" + filespec);

	readHeader();
	loadIndex();
}

/**
 * Read file header to verify that the file is a snes9express skin file
 */
void s9x_SkinReader::readHeader()
{
	char headbuf[32];

	int l = strlen(headstart);
	int r = fread(headbuf, 1, l, skinfptr);
	if((r!=l)||(memcmp(headbuf, headstart, l)!=0))
		throw std::string("not a s9xskin file");

	char v = fgetc(skinfptr);
	if(v > headskinver)
		throw std::string("unsupported s9xskin version");

	l = strlen(headfinish);
	r = fread(headbuf, 1, l, skinfptr);
	if((r!=l)||(memcmp(headbuf, headfinish, l)!=0))
		throw std::string("corrupt s9xskin file");
}

/**
 * Load the index of files
 */
void s9x_SkinReader::loadIndex()
{
	s9x_SkinFileIndexRecord *rec;
	fseek(skinfptr, sizeof(s9x_SkinFileIndexRecord)-1, SEEK_SET);
	int c = fgetc(skinfptr);
	for(; c; c--)
	{
		rec = (s9x_SkinFileIndexRecord*)malloc(sizeof(s9x_SkinFileIndexRecord));
		if(!rec)
			throw std::string("out of memory");
		int l = fread(rec, 1, sizeof(s9x_SkinFileIndexRecord), skinfptr);
		if(l!=sizeof(s9x_SkinFileIndexRecord))
			throw std::string("corrupt s9xskin index");
		rec->filename[sizeof(rec->filename)-1] = 0;
		rec->fileoffset = ntohl(rec->fileoffset);
		rec->filesize = ntohl(rec->filesize);
		std::string filename(rec->filename);
		idx[filename] = rec;
		/*std::cout << "read index: " << rec->filename
			<< "\tofs:" << rec->fileoffset << "\t" << rec->filesize << " bytes"
			<< std::endl;*/
	}
}

/**
 * return the number of bytes in the filename requested, 0 if file not found
 */
uint32_t s9x_SkinReader::getFileSize(std::string filename)
{
	index::iterator e = idx.end(), f = idx.find(filename);
	if(e==f)
		return 0;
	s9x_SkinFileIndexRecord *rec = f->second;
	return rec->filesize;
}

/**
 * return an istream for the data in a given filename
 */
std::istream& s9x_SkinReader::getIstream(std::string filename)
{
	char *data;
	uint32_t len;

	index::iterator e = idx.end(), f = idx.find(filename);
	if(e==f)
		throw std::string("file not found: " + filename);
	s9x_SkinFileIndexRecord *rec = (*f).second;

	data = new char[rec->filesize];
	if(!data) throw std::string("not enough memory");
	/*
	 * load the whole file into an istringstream, unless you can think of a better idea.
	 * *sigh* the file contents get copied in memory 4 times! crazy.
	 * ...that's in addition to being allocated 4 times, and then 3 frees.
	 */
	fseek(skinfptr, rec->fileoffset, SEEK_SET);
	len = fread(data, 1, rec->filesize, skinfptr); // 1
	std::string sdata(data, len); // 2
	std::istringstream *i = new std::istringstream(sdata); // 3
	delete[] data;

	return *i; // 4
}

/**
 * Return a FILE pointer starting at the beginning of the requested file.
 * Unfortunately, there's no way to fake an EOF(correct me if I'm wrong, please)
 * so a reader of this can read beyond the file boundry. :/
 */
FILE* s9x_SkinReader::seekFile(std::string filename)
{
	index::iterator e = idx.end(), f = idx.find(filename);
	if(e==f)
		throw std::string("file not found: " + filename);
	s9x_SkinFileIndexRecord *rec = f->second;

	fseek(skinfptr, rec->fileoffset, SEEK_SET);
	return skinfptr;
}





s9x_SkinWriter::s9x_SkinWriter(const std::string& inputdir)
{
	skindir = inputdir;
	stylesheet = "snes9express.skin";
}

void s9x_SkinWriter::scanStyleSheet()
{
	char oldpwd[FR_PATH_MAX];
	getcwd(oldpwd, FR_PATH_MAX);
	if(chdir(skindir.c_str())!=0)
		throw std::string("could not cd to skin directory: " + skindir);

	addFile(stylesheet);
	std::ifstream skinstream;
	skinstream.open(stylesheet.c_str(), std::ios::in);
	if(!skinstream.is_open())
	{
		chdir(oldpwd);
		throw std::string("could not open stylesheet: " + stylesheet);
	}
	std::string t;
	while(skinstream >> t)
	{
		unsigned int p = t.rfind('.');
		if(p==t.npos) continue;
		p = t.find(':');
		if(p==t.npos) p=0;
		unsigned int e = t.rfind(';');
		if(e==t.npos) e = t.size();
		std::string filename(t, p, e);
		if(fr_FileExists(filename))
			addFile(filename);
	}
	skinstream.close();
	chdir(oldpwd);
}

void s9x_SkinWriter::addFile(const std::string& filename)
{
	struct stat statbuf;
	s9x_SkinFileIndexRecord *rec;

	index::const_iterator e = idx.end();
	if(idx.find(filename)!=e)  // already listed
		return;
	rec = (s9x_SkinFileIndexRecord*)malloc(sizeof(s9x_SkinFileIndexRecord));
	if(!rec) throw std::string("not enough memory");

	snprintf(rec->filename, sizeof(rec->filename), "%s", filename.c_str());

	if(stat(rec->filename, &statbuf)!=0) throw std::string(filename + ": cannot access file");
	rec->fileoffset = 0;
	rec->filesize = statbuf.st_size;
	files.push_back(filename);
	idx[filename] = rec;
}

void s9x_SkinWriter::writeIndex()
{
	int c = files.size();
	s9x_SkinFileIndexRecord rec, *r;
	uint32_t ofs = (c + 1) * sizeof(rec);

	fputc(c, skinfptr);
	filelist::iterator i = files.begin(), e = files.end();
	for(; i!=e; i++)
	{
		r = idx[*i];
		memset(&rec, 0, sizeof(rec));
		snprintf(rec.filename, sizeof(rec.filename), "%s", r->filename);
		r->fileoffset = ofs;
		rec.fileoffset = htonl(ofs);
		ofs += r->filesize;
		rec.filesize = htonl(r->filesize);
		fwrite(&rec, sizeof(rec), 1, skinfptr);
		/*std::cout << "write index: " << r->filename
			<< "\tofs:" << r->fileoffset << "\t" << r->filesize << " bytes"
			<< std::endl;*/

	}
}

void s9x_SkinWriter::writeFile(const std::string& outputfilename)
{
	scanStyleSheet();

	skinfptr = fopen(outputfilename.c_str(), "w");
	if(!skinfptr)
		throw std::string("could not open output file: " + outputfilename);

	char oldpwd[FR_PATH_MAX];
	getcwd(oldpwd, FR_PATH_MAX);
	if(chdir(skindir.c_str())!=0)
		throw std::string("could not cd to skin directory: " + skindir);

	fprintf(skinfptr, "%s%c%s", headstart, headskinver, headfinish);
	fputc(0, skinfptr);
	long fillsize = sizeof(s9x_SkinFileIndexRecord) - ftell(skinfptr) - 1;
	char *filler = new char[fillsize];
	memset(filler, 0x99, fillsize);
	fwrite(filler, 1, fillsize, skinfptr);
	delete[] filler;

	writeIndex();

	filelist::iterator i = files.begin(), e = files.end();
	for(; i!=e; i++)
	{
		s9x_SkinFileIndexRecord *rec = idx[*i];

		FILE* fptr = fopen(rec->filename, "r");
		char buf[4096];
		int l = fread(buf, 1, sizeof(buf), fptr);
		while(l>0)
		{
			fwrite(buf, 1, l, skinfptr);
			l = fread(buf, 1, sizeof(buf), fptr);
		}
		fclose(fptr);
	}
	chdir(oldpwd);
}


int s9xskin_main(int argc, char**argv)
{
	bool quiet = false;
	bool inputmode = true;

	if(argc < 2) return -1;
	std::list<std::string> inputdirs;
	std::string outputdir(".");
	bool showusage = false;
	for(int a=2; a < argc; a++)
	{
		if(argv[a][0]==0) continue;
		std::string arg(argv[a]);
		if((arg=="-h")||(arg=="--help")||(arg=="--usage"))
		{
			showusage = true;
			break;
		} else if(arg=="-q") {
			quiet = true;
		} else if(arg=="-i") {
			inputmode = true;
		} else if(arg=="-o") {
			inputmode = false;
		} else if(fr_DirExists(arg)) {
			if(inputmode)
				inputdirs.push_back(arg);
			else if(outputdir==".")
				outputdir = arg;
			else
			{
				std::cerr << "output directory set too many times" << std::endl;
				showusage = true;
			}
		} else {
			std::cerr << "unrecognized option: '" << arg << "'" << std::endl;
			showusage = true;
		}
	}

	if(inputdirs.size()==0)
	{
		if(fr_FileExists("snes9express.skin"))
			inputdirs.push_back(".");
		else
			showusage = true;
	}
	if(showusage)
	{
		std::cerr << "usage: snes9express skin [-q] -i inputdirs -o outputdir" << std::endl;
		std::cerr << "Create a .s9xskin file" << std::endl;
		std::cerr << "Options:" << std::endl;
		std::cerr << " -h  help, shows this usage message" << std::endl;
		std::cerr << " -q  quiet, does not print output to screen" << std::endl;
		std::cerr << " -i  inputdirs is a list of one or more s9x skin directories" << std::endl;
		std::cerr << " -o  outputdir is where to create the .s9xskin files" << std::endl;
		return 1;
	}

	std::string errmsg;
	std::list<std::string>::iterator i = inputdirs.begin(), e = inputdirs.end();
	try
	{
		for(; i!=e; i++)
		{
			std::string inputdir = fr_RealPath(*i);
			std::string outputfile = outputdir + '/' + fr_Basename(inputdir) + ".s9xskin";
			if(!quiet)
				std::cout << " * "  << inputdir
					  << " --> " << outputfile << std::flush;
			s9x_SkinWriter sw(inputdir);
			sw.writeFile(outputfile);
			if(!quiet)
				std::cout << "\t[ ok ]" << std::endl;
		}
	} catch(const std::string& e) {
		errmsg = e;
	} catch(const char*e) {
		errmsg = e;
	} catch(...) {
		errmsg = "unidentified exception";
	}

	if(errmsg.size())
	{
		std::cerr << "\t[ not ok ]\ncaught exception: \"" << errmsg << '"' << '\n'
			  << "aborting." << std::endl;
		return -1;
	}
	return 0;
}
#else
int s9xskin_main(int argc, char**argv)
{
	std::cerr << PROG " was compiled without skin support."
		  << std::endl;
	return 1;
}
#endif
